#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
#
# FILE:
# GTrigger.py
#
# DESCRIPTION:
# Provides the basic classes needed by the generic trigger system
#
# NOTES:
#
import sys
import types
import string
import copy
from gnue.common.definitions.GObjects import GObj
from gnue.common.apps import GDebug

from gnue.common.formatting import GTypecast
from xml.sax import saxutils
from gnue.common.definitions.GParserHelpers import GContent

class TriggerError:
  def __init__(self, msg):
    self.msg = msg


class TriggerAbort:
  pass
class TriggerStop:
  pass
class TriggerSuccess:
  pass



#######################################################################
#
# Trigger instance classes
#
# Classes in here are used to represent the triggers defined in
# <trigger> tags in an xml file
#
#
# Note: this is a partial graft of the old GFTrigger system
#       only here so that the new namespace code could be
#       put to use right away


#
# GTrigger
#
# Class used to implement triggers
#
class GTrigger(GObj):
  def __init__(self, parent=None, type=None, name=None, src=None, text=None, language='python'):

    GObj.__init__(self, parent, 'GCTrigger')

    self._text=''
    self._triggerns={}
    self._inits   = [None,self.initialize]

    self.language=language
    self.src = src
    self.type = type and string.upper(type) or None
    self.name = name
    if text != None:
      GContent(self, text)
    if self.type != None:
      self._buildObject()

    
  #
  # Must be at least a phase 2 init
  #
  # TODO: this function merges the local namespace
  # TODO: not sure if this is doing the right thing
  # TODO: with regard to NAMED triggers.  It should
  # TODO: merge the local namespace of the object that
  # TODO: fired the trigger.
  def initialize(self):
    self._root = self.findParentOfType(None)
    self._triggerns.update( self._root._triggerns )
    self._globalns = self._root._globalRuntimeNamespace
    self.__call__ = self.dummyFunction

    if self.type != "NAMED":
      if self.getParent ():
        self.getParent ().associateTrigger (self.type, self)
        self._triggerns.update(self.getParent ()._localTriggerNamespace)
    else:
      self._root._triggerDictionary[self.name] = self
      self._triggerns.update(self._root._localTriggerNamespace)

    if self.src == None:
      self.setFunction( self.getChildrenAsContent(), self.language )
    else:
      self.setFunctionFrom(self._root._triggerDictionary[self.src])


  def setFunctionFrom(self, object):
    self.__call__=object.__call__

  def setFunction(self, text, language):
    self._text = text
    self.language = language
    if self.language != 'python':
      GDebug.printMesg(0, "Language %s not implemented" % self.language)
      sys.exit()


    # TODO: this isn't working?????
    # Verify that the trigger contains no tabs
    if string.find('\t', self._text) >= 0:
      GDebug.printMesg(0, "The trigger named %s contains a tab character which is not allowed at pos %s"
                       % ( self.name, string.find('\t', self._text) ))
      sys.exit()

    # Remove whitespace from last line
    self._text = string.rstrip(self._text)

    # First compile the trigger:
    #   Compilation at this stage has the useful side effect that
    #   syntax errors are spotted during XML parsing rather than
    #   during execution.

    # Get the indentation level of the first line of code so
    # we can indent our imports to the same level
    indentLevel = 0
    for line in string.split(string.replace(self._text,'\r',''),'\n'):
      if len(string.strip(line)) and string.lstrip(line)[0] != '#':
        indentLevel = len(line) - len(string.lstrip(line))
        break

    revisedTriggerCode = \
      "from gnue.common.logic.GTrigger import TriggerError\n"
    for line in string.split(string.replace(self._text,'\r',''),'\n'):
      revisedTriggerCode = revisedTriggerCode + line[indentLevel:] + '\n'
    revisedTriggerCode = revisedTriggerCode + \
      "pass\n"

    try:
      self._code = compile(revisedTriggerCode, '<string>', 'exec')
    except SyntaxError, err:
      GDebug.printMesg(0, "Syntax error in line %s of trigger in element ??" \
            % ( err.lineno ))
      sys.exit()

    def thisTrigger(myself, code = self._code,
                    triggerns = self._triggerns,
                    globalns = self._globalns):

      # Merge the trigger's namespace with the runtime global namespace
      # (Which can be set via the "global myvar" construct)
      try:
        del globalns['__builtins__']
      except KeyError:
        pass
      try:
        localns = copy.copy(triggerns)
        localns.update(globalns)

        # And execute our code
        exec code in globalns, localns
      except TriggerError:
        raise
      except:
        # May be better to deal with this in GTriggerExtension
        raise
        import sys
        GDebug.printMesg(0, "%s in trigger code, value: %s" % (sys.exc_type, sys.exc_value))

    self.__call__ = thisTrigger

  def dummyFunction(self, myself):
    GDebug.printMesg(0, "Trigger not implemented")

  #
  # getDescription
  #
  # Return a useful description of this object
  # for use by designer
  #
  def getDescription(self):
    if self.type == 'NAMED':
      return self.name
    else:
      return string.upper(self.type)

  #
  # dumpXML
  #
  # Dumps an XML representation of the object
  # used in saving
  #
  def dumpXML(self, lookupDict, treeDump=None, gap=None,
              textEncoding='<locale>', xmlnamespaces={}, stripPrefixes = None):
    try:
      escape = not int(gConfig('StoreTriggersAsCDATA'))
    except:
      escape = 1
    xmlEntity = "trigger"
    xmlString = "%s<%s" % (gap[:-2],xmlEntity)

    indent = len(xmlString)
    pos = indent
    for attribute in self.__dict__.keys():

      # variables beginning with _ are never saved out to file
      # they are internal to the program
      if attribute[0] == "_":
        continue

      val = self.__dict__[attribute]
      if lookupDict[xmlEntity].has_key('Attributes') and \
         lookupDict[xmlEntity]['Attributes'].has_key(attribute):
        if val != None and \
           (not lookupDict[xmlEntity]['Attributes'][attribute].has_key('Default') or \
            (lookupDict[xmlEntity]['Attributes'][attribute]['Default']) != (val)):
          typecast = lookupDict[xmlEntity]['Attributes'][attribute]['Typecast']
          if typecast == GTypecast.boolean \
             and val == 1:
            addl = ' %s=""' % (attribute)
          elif typecast == GTypecast.names:
            addl = ' %s="%s"' % \
                (attribute, string.join(val,','))
          else:
            addl = ' %s="%s"' % (attribute, saxutils.escape('%s' % val))
          if len(addl) + pos > 78:
            xmlString += "\n" + " " * indent + addl
            pos = indent
          else:
            xmlString = xmlString + addl
            pos += len(addl)

    if len(self._children):
      hasContent = 0
      for child in self._children:
        hasContent = hasContent or isinstance(child,GContent)
      if hasContent:
        xmlString += ">"
      else:
        xmlString += ">\n"

      if treeDump:
        if hasContent and not escape:
          xmlString += "<![CDATA["
        for child in self._children:
          xmlString += child.dumpXML(lookupDict, 1,gap+"  ",escape=escape,
              stripPrefixes = stripPrefixes)
        if hasContent and not escape:
          xmlString += "]]>"

      if hasContent:
        xmlString += "</%s>\n" % (xmlEntity)
      else:
        xmlString += "%s</%s>\n" % (gap[:-2], xmlEntity)
    else:
      xmlString += "/>\n"
    return xmlString





#######################################################################
#
# Trigger processor classes
#
# GTriggerExtension
#
# Objects that inherit this class will be
# capable of processing triggers
#
class GTriggerExtension:
  def __init__(self):
    self._trigger = {}
    
    #self._validTriggers = validTriggers 
                                        
  # associateTrigger
  #
  # Associates a trigger with the object.  More than one trigger of a specific type
  # can be associated with an object
  #
  def associateTrigger(self, key, function):
    key = string.upper(key)
    if key in self._validTriggers.keys():
      if not self._trigger.has_key(key):
        self._trigger[string.upper(key)] = []
      self._trigger[string.upper(key)].append(function)
    else:
      print _("Invalid trigger "),key

  # processTrigger
  #
  # "fires" the trigger
  def processTrigger(self, key):
    key = string.upper(key)
    if key in self._validTriggers.keys():
      if self._trigger.has_key(key):
        for function in self._trigger[key]:
          function(self)
      else:
        GDebug.printMesg(7, "No triggers to fire")
    else:
      print self._type,": ",_("Invalid trigger "),key


#
# Return any XML elements associated with
# GDataSources.  Bases is a dictionary of tags
# whose values are update dictionaries.
# For example: bases={'datasource': {'BaseClass':myDataSource}}
# sets xmlElements['datasource']['BaseClass'] = myDataSource
#
def getXMLelements(updates={}):

  xmlElements = {
      'trigger': {
         'BaseClass': GTrigger,
         'Importable': 1,
         'Attributes': {
            'name': {
               'Unique': 1,
               'Typecast': GTypecast.name },
            'type': {
               'Typecast': GTypecast.uppername },
            'src': {
               'References': 'trigger.name',
               'Typecast': GTypecast.name },
            'language': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'python': {} },
               'Default': 'python' } },
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'UsableBySiblings': 1,
         'ParentTags': None },
      }

  for alteration in updates.keys():
    xmlElements[alteration].update(updates[alteration])

  return xmlElements

