#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# $Id: mssql.py 5614 2004-04-02 10:53:54Z johannes $
#

from gnue.common.schema.scripter.processors.SQL import SQLProcessor
from string import join

name        = "MSSQL"
description = "MS SQL Server (7.x/2000)"

# =============================================================================
# GSD processor for MS SQL Server
# =============================================================================
class Processor (SQLProcessor):

  MAX_NAME_LENGTH = 31
  END_BATCH       = "GO"


  # ---------------------------------------------------------------------------
  # Process a GSField instance
  # ---------------------------------------------------------------------------
  def _processField (self, tableDef, gsField, isLast):
    field = "  %s" % self._qualify (gsField)

    # build a default value for this field
    if gsField.defaultwith == "serial":
      field += " IDENTITY"

    elif gsField.defaultwith == "timestamp":
      field += " DEFAULT GETDATE ()"

    elif hasattr (gsField, "default") and len (gsField.default):
      field += " DEFAULT %s" % gsField.default

    if not gsField.nullable:
      field += " NOT NULL"

    if not isLast:
      field += ","

    tableDef.getPhase (0).body.append (field)


  # ===========================================================================
  # Datatype translation
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # A key field is of type 'int'
  # ---------------------------------------------------------------------------

  def key (self, gsField):
    """
    A key is allways an 'int'
    """
    return "int"


  # ---------------------------------------------------------------------------
  # A number needs special treatment
  # ---------------------------------------------------------------------------
  def number (self, gsField):
    """
    A number translates to 'smallint', 'int', 'bigint' or 'decimal' according
    to it's precision and length.
    """
    if gsField.precision == 0:
      if gsField.length <= 4:
        return "smallint"

      elif gsField.length <= 9:
        return "int"

      elif gsField.length <= 18:
        return "bigint"

      else:
        return "decimal (%s,0)" % gsField.length
    else:
      return "decimal (%s,%s)" % (gsField.length + gsField.precision,
                                  gsField.precision)


  # ---------------------------------------------------------------------------
  # Keep boolean as 'boolean'
  # ---------------------------------------------------------------------------

  def boolean (self, gsField):
    """
    A boolean is translated to 'bit'.
    """
    return "bit"


  # ---------------------------------------------------------------------------
  # datetime is not equivalent to timestamp
  # ---------------------------------------------------------------------------

  def datetime (self, gsField):
    """
    'datetime' is the one and only date-time type of SQL Server.
    """
    return "datetime"


  # ---------------------------------------------------------------------------
  # date becomes datetime
  # ---------------------------------------------------------------------------

  def date (self, gsField):
    """
    date is mapped to 'datetime'
    """
    return self.datetime (gsField)


  # ---------------------------------------------------------------------------
  # time becomes datetime
  # ---------------------------------------------------------------------------

  def time (self, gsField):
    """
    time is mapped to 'datetime'
    """
    return self.datetime (gsField)


  # ===========================================================================
  # Data transformation services
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # convert datetime values using the convert method
  # ---------------------------------------------------------------------------

  def dts_datetime (self, gsField):
    """
    Datetime values will be converted using SQL Server's CONVERT function.
    """
    return "CONVERT (DATETIME, '%s', 102)" % gsField.value


  # ---------------------------------------------------------------------------
  # map date to datetime
  # ---------------------------------------------------------------------------

  def dts_date (self, gsField):
    """
    Call dts_datetime ().
    """
    return self.dts_datetime (gsField)


  # ---------------------------------------------------------------------------
  # call dts_datetime ()
  # ---------------------------------------------------------------------------

  def dts_time (self, gsField):
    """
    Call dts_datetime ().
    """
    return self.dts_datetime (gsField)

