#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: HTML.py 5614 2004-04-02 10:53:54Z johannes $

from gnue.common.schema.scripter.processors.Base import BaseProcessor
from string import join
from mx.DateTime import now

name        = "HTML"
description = "HTML 4.01"

# =============================================================================
# HTML processor for GNUe Schema Definitions
# =============================================================================

class Processor (BaseProcessor):

  END_COMMAND = ""
  END_BATCH   = ""

  COMMENT_BEGIN      = "<!-- "
  COMMENT_END        = "-->"
  COMMENT_SINGLELINE = False

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, destination, source):
    """
    Since this class generates a list of tables at the top of the page, we need
    another private property.
    """
    BaseProcessor.__init__ (self, destination, source)
    self.__tables = []


  # ---------------------------------------------------------------------------
  # Process all fields
  # ---------------------------------------------------------------------------

  def _processFields (self, tableDef):
    pro = tableDef.getPhase (0).prologue
    hdr = tableDef.getPhase (0).header

    pro.append ("")
    pro.extend (self.comment ("Table definition '%s'" % tableDef.name))

    pro.append ('<H2 class="tabledef"><A name="%s">%s</A></H2>' % \
        (tableDef.name, _("Table definition of '%s'") % tableDef.name))

    hdr.append ('<H3 class="fieldlist">%s</H3>' % _("List of fields"))
    hdr.append ('<TABLE class="tabledef" width="90%" border="0" ' + \
                'cellpadding="3" cellspacing="1">')
    hdr.extend (self._tableHeader ())

    tableDef.getPhase (0).footer.append ('</TABLE>')

    BaseProcessor._processFields (self, tableDef)


  # ---------------------------------------------------------------------------
  # Translate all fields into a single row
  # ---------------------------------------------------------------------------

  def _processField (self, tableDef, gsField, isLast):
    tableDef.getPhase (0).body.extend (self._fieldRow (gsField))


  # ---------------------------------------------------------------------------
  # Process a primary key definition
  # ---------------------------------------------------------------------------

  def _processPrimaryKey (self, tableDef):
    pkDef = tableDef.primaryKey
    flist = join ([pkf.name for pkf in pkDef.fields], ", ")

    epi = tableDef.getPhase (0).epilogue

    epi.append ("")
    epi.extend (self.comment ("Primary key '%s'" % pkDef.name))

    epi.append ('<H3 class="primarykey">%s</H3>' % \
      _("Primary Key: %s") % pkDef.name)

    epi.append ('<UL>')
    epi.extend (["  <LI>%s</LI>" % pkf.name for pkf in pkDef.fields])
    epi.append ('</UL>')


      
  # ---------------------------------------------------------------------------
  # Process an IndexDefinition
  # ---------------------------------------------------------------------------

  def _processIndex (self, tableDef, indexDef):
    if indexDef.unique:
      uniq = _("Unique Index")
    else:
      uniq = _("Index")

    epi = tableDef.getPhase (0).epilogue

    epi.append ("")
    epi.extend (self.comment ("Index '%s'" % indexDef.name))
    epi.append ('<H3 class="index">%s: %s</H3>' % (uniq, indexDef.name))

    epi.append ('<UL>')
    epi.extend (["  <LI>%s</LI>" % ixf.name for ixf in indexDef.fields])
    epi.append ('</UL>')


  # ---------------------------------------------------------------------------
  # Process a constraint definition
  # ---------------------------------------------------------------------------

  def _processConstraint (self, tableDef, constraint):
    epi = tableDef.getPhase (0).epilogue

    epi.append ("")
    epi.extend (self.comment ("Constraint '%s'" % constraint.name))
    if constraint.kind == "foreignkey":
      epi.append ('<H3 class="constraint">%s: %s</H3>' % 
        (_("Foreign Key"), constraint.name))

      epi.append ('<TABLE class="constraintdef" width="90%" border="0" ' + \
                  'cellpadding="3" cellspacing="1">')
      epi.append ('<TR>')
      epi.append ('  <TH class="fields">%s</TH>' % tableDef.name)
      epi.append ('  <TH class="fields">&nbsp;</TH>')
      epi.append ('  <TH class="fields"><A HREF="#%s">%s</A></TH>' % \
        (constraint.reftable, constraint.reftable))
      epi.append ('</TR>')

      epi.append ('<TR>')
      epi.append ('  <TD class="fields">')
      epi.extend (["    %s<BR>" % cf.name for cf in constraint.fields])
      epi.append ('  </TD>')
      epi.append ('  <TD class="fields">%s</TD>' % _("references"))
      epi.append ('  <TD class="fields">')
      epi.extend (["    %s<BR>" % rf.name for rf in constraint.reffields])
      epi.append ('  </TD>')
      epi.append ('</TR>')
      epi.append ('</TABLE>')



  # ---------------------------------------------------------------------------
  # Create a sequence with a header row for field tables
  # ---------------------------------------------------------------------------

  def _tableHeader (self):
    """
    Create a table header for field rows
    """
    res = []

    res.append ('<TR>')
    res.append ('  <TH class="fields">%s</TH>' % "Field")
    res.append ('  <TH class="fields">%s</TH>' % "Type")
    res.append ('  <TH class="fields">%s</TH>' % "Nullable")
    res.append ('  <TH class="fields">%s</TH>' % "Default")
    res.append ('</TR>')

    return res


  # ---------------------------------------------------------------------------
  # Create a sequence for a table row, describing a single field
  # ---------------------------------------------------------------------------

  def _fieldRow (self, gsField):
    """
    Create a HTML sequence describing the given gsField instance.
    """
    res = []

    if gsField.nullable:
      null = _("yes")
    else:
      null = _("no")

    if gsField.defaultwith in ["serial", "timestamp"]:
      default = gsField.defaultwith

    elif hasattr (gsField, "default"):
      default = gsField.default

    else:
      default = "&nbsp;"

    res.append ('<TR>')
    res.append ('  <TD class="fields">%s</TH>' % gsField.name)
    res.append ('  <TD class="fields">%s</TH>' % \
                self._translateType (gsField))
    res.append ('  <TD class="fields">%s</TH>' % null)
    res.append ('  <TD class="fields">%s</TH>' % default)
    res.append ('</TR>')

    return res


  # ---------------------------------------------------------------------------
  # start a new file
  # ---------------------------------------------------------------------------

  def startDump (self):
    """
    On start of a dump just create the HTML- and HEAD-tags
    """
    self._writeText (['<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 ' + \
       'Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">',
      '<HTML>', 
      ' <HEAD>', 
      '  <TITLE>GNUe Schema Definition: %s</TITLE>' % self.source,
      '  <META name="author" content="gnue-schema">'])


  # ---------------------------------------------------------------------------
  # Set the client encoding in the html head
  # ---------------------------------------------------------------------------

  def client_encoding (self, encoding):
    """
    Set the client encoding in the html head
    """
    self.encoding = encoding
    self._writeText ([ \
      '  <META HTTP-EQUIV="content-type" content="text/html; charset=%s">' % \
      self.encoding])


  # ---------------------------------------------------------------------------
  # Starting a schema dump means finishing the head and starting the body
  # ---------------------------------------------------------------------------

  def startSchema (self):
    """
    Close the HEAD section and start with the BODY.
    """
    self._writeText (['<style type="text/css">',
      '  BODY { font-family: helvetica, arial, sans-serif; }',
      '  TH { text-align: left; background: #AAAAAA; }',
      '  TD { text-align: left; background: #CCCCCC; }',
      '</style>'])

    self._writeText (['</HEAD>', '<BODY>'])


  def writePhase (self, tableDef, phase):
    pass

  # ---------------------------------------------------------------------------
  # stop the dump
  # ---------------------------------------------------------------------------

  def finishDump (self):
    """
    Create the table of contents section and write all definitions to the
    destination.
    """
    self._writeText (self.comment ('List of tables'))
    self._writeText ('<H1 class="toc">%s</H1>' % \
      _("Table definitions in '%s'") % self.source)

    self._writeText ('<UL>\n')

    for table in self.__tables:
      self._writeText ('  <LI><A HREF="#%s">%s</A></LI>\n' % \
        (table.name, table.name))

    self._writeText ('</UL>\n')

    for table in self.__tables:
      # table.writeDefinition (self.destination, self.encoding)
      BaseProcessor.writePhase (self, table, 0)
      self._writeText ('<HR>')

    self._writeText (['<P>%s %s UTC</P>' % (_("Generated on"), now ())])
    self._writeText (['</BODY>'])


  # ---------------------------------------------------------------------------
  # write a table definition
  # ---------------------------------------------------------------------------

  def translateTableDefinition (self, tableDef):
    """
    This function call all _process* () functions on the table definition given
    and adds the processed definition to the internal list of tables.
    """
    self._processFields (tableDef)

    if tableDef.primaryKey is not None:
      self._processPrimaryKey (tableDef)

    if len (tableDef.indices.keys ()):
      self._processIndices (tableDef)

    if len (tableDef.constraints.keys ()):
      self._processConstraints (tableDef)

    # Note: we won't write the table definition right now, but append it to the
    # list of tables. We would let stopDump () do the write out instead.
    self.__tables.append (tableDef)


  # ---------------------------------------------------------------------------
  # Datatype translation methods
  # ---------------------------------------------------------------------------

  def string (self, gsField):
    if hasattr (gsField, "length"):
      return "string (%s)" % gsField.length
    else:
      return "string"

  # ---------------------------------------------------------------------------
  # Datatype translation methods
  # ---------------------------------------------------------------------------

  def date (self, gsField):
    return "date"

  # ---------------------------------------------------------------------------
  # Datatype translation methods
  # ---------------------------------------------------------------------------

  def time (self, gsField):
    return "time"

  # ---------------------------------------------------------------------------
  # Datatype translation methods
  # ---------------------------------------------------------------------------

  def datetime (self, gsField):
    return "datetime"

  # ---------------------------------------------------------------------------
  # Datatype translation methods
  # ---------------------------------------------------------------------------

  def boolean (self, gsField):
    return "boolean"

  # ---------------------------------------------------------------------------
  # Datatype translation methods
  # ---------------------------------------------------------------------------

  def key (self, gsField):
    return "key"

  # ---------------------------------------------------------------------------
  # Datatype translation methods
  # ---------------------------------------------------------------------------

  def number (self, gsField):
    return "number (%s, %s)" % (gsField.length + gsField.precision, \
                                gsField.precision)
