#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: GSData.py 5619 2004-04-02 19:28:47Z reinhard $

import re
import mx.DateTime.ISO
from string import join


# =============================================================================
# Constants
# =============================================================================

_LENGTH_SCALE = re.compile ('^\w+\s*\((\d+)[\.]{0,1}(\d*)\)\s*')
_VALID_TYPES  = ["string", "number", "boolean", "date", "time", "datetime"]
_TRANS_TYPES  = { "text": "string",
                  "timestamp": "datetime" }


# =============================================================================
# Base Exception class
# =============================================================================

class EGSData (Exception):
  """
  This class is the ancestor of all exceptions used by this module.
  """
  def __init__ (self, text):
    self._text = text
    Exception.__init__ (self, text)


  # ---------------------------------------------------------------------------
  # Object representation
  # ---------------------------------------------------------------------------

  def __repr__ (self):
    return self._text


# =============================================================================
# Invalid or unknown datatype 
# =============================================================================

class EInvalidType (EGSData):
  """
  This exception is raised, if a invalid or unknown datatype was found.
  """
  def __init__ (self, gsValue, dataType):
    fName = ""
    if hasattr (gsValue, "field"):
      fName = "%s: " % gsValue.field

    msg = _("%sInvalid datatype '%s'.") % (fName, dataType)
    EGSData.__init__ (self, msg)


# =============================================================================
# Invalid or unknown value
# =============================================================================

class EInvalidValue (EGSData):
  """
  This exception is raised, if a native object has an invalid value/type
  """
  def __init__ (self, value, text):
    EGSData.__init__ (self, "'%s' %s" % (str (value), text))



# =============================================================================
# Fieldvalue out of range
# =============================================================================

class EOutOfRange (EGSData):
  """
  This exception is raised if a fieldvalue is out of range, e.g. a string
  exceeds it's length or a number exceeds it's numeric bounds.
  """
  def __init__ (self, gsValue, fieldValue, length, precision = None):
    fName = ""
    if hasattr (gsValue, "field"):
      fName = "%s: " % gsValue.field

    if precision is not None:
      rDef = "%d.%d" % (length, precision)
    else:
      rDef = "%d" % length

    msg = _("%sValue '%s' out of Range (%s)") % (fName, fieldValue, rDef)

    EGSData.__init__ (self, msg)


# =============================================================================
# EInvalidFormat 
# =============================================================================

class EInvalidFormat (EGSData):
  """
  This exception is the base class of format errors.
  """
  def __init__ (self, gsValue, fieldValue, fmtString):
    fName = ""
    if hasattr (gsValue, "field"):
      fName = "%s: " % gsValue.field

    msg = _("%sValue '%s' %s") % (fName, fieldValue, fmtString)
    EGSData.__init__ (self, msg)


# =============================================================================
# Field value is not a number
# =============================================================================

class EInvalidNumber (EInvalidFormat):
  """
  This exception is raised if a string cannot be converted to a number.
  """
  def __init__ (self, gsValue, value, length, precision):
    fmt = _("is not a number (%d.%d)") % (length, precision)
    EInvalidFormat.__init__ (self, gsValue, value, fmt)


# =============================================================================
# Field value is not a valid boolean
# =============================================================================

class EInvalidBoolean (EInvalidFormat):
  """
  Exception class for invalid formatted booleans
  """
  def __init__ (self, gsValue, fieldValue):
    fmt = _("is not a boolean (TRUE or FALSE)")
    EInvalidFormat.__init__ (self, gsValue, fieldValue, fmt)


# =============================================================================
# Field value is not a valid date
# =============================================================================

class EInvalidDate (EInvalidFormat):
  """
  Exception class for invalid formatted dates
  """
  def __init__ (self, gsValue, fieldValue):
    fmt = _("is not a date (YYYY-MM-DD)")
    EInvalidFormat.__init__ (self, gsValue, fieldValue, fmt)


# =============================================================================
# Field value is not a valid time
# =============================================================================

class EInvalidTime (EInvalidFormat):
  """
  Exception class for invalid formatted times
  """
  def __init__ (self, gsValue, fieldValue):
    fmt = _("is not a time (HH:MM:SS.ss")
    EInvalidFormat.__init__ (self, gsValue, fieldValue, fmt)


# =============================================================================
# Field value is not a valid datetime
# =============================================================================

class EInvalidDateTime (EInvalidFormat):
  """
  Exception class for invalid formatted datetimes
  """
  def __init__ (self, gsValue, fieldValue):
    fmt = _("is not a datetime (YYYY-MM-DD HH:MM:SS.ss)")
    EInvalidFormat.__init__ (self, gsValue, fieldValue, fmt)



# -----------------------------------------------------------------------------
# verify a given datatype
# -----------------------------------------------------------------------------

def verifyDataType (gsColumn):
  """
  This function checks if the @gsColumn instance has a vaild type attribute.
  If the datatype is valid, this function creates a tuple of '(typename, lenght,
  precision)', otherwise an EInvalidType exception will be raised.
  """
  fieldType = gsColumn.type.lower ().strip ()
  tpMatch   = re.compile ('^(\w+)').match (fieldType)
  tpList    = _VALID_TYPES + _TRANS_TYPES.keys ()

  # check if we know the given type
  if tpMatch is None or not tpMatch.groups () [0] in tpList:
    raise EInvalidType (gsColumn, fieldType)

  # and eventually translate it
  typename = tpMatch.groups () [0]
  if _TRANS_TYPES.has_key (typename):
    typename = _TRANS_TYPES [typename]

  flength    = 0
  fprecision = 0

  # try to extract length and precision from fieldType
  lsMatch = _LENGTH_SCALE.match (fieldType)
  if lsMatch is not None:
    (lstr, sstr) = lsMatch.groups ()

    if len (lstr):
      flength = int (lstr)
    if len (sstr):
      fprecision = int (sstr)

  return (typename, flength, fprecision)

  


# -----------------------------------------------------------------------------
# Convert a value from a GSValue instance to a native python object
# -----------------------------------------------------------------------------

def valueToNative (gsValue, gsColumn):
  """
  This function takes a GSValue instance and creates a native python object
  representing it's value according to it's type. If the value is an empty
  string 'None' is returned. As a side effect, the property 'dataType' of the
  GSValue instance will be updated to the type used.
  """
  fieldValue = unquoteString (gsValue.getChildrenAsContent ())

  # if no gsValue has no value we return None
  if len (fieldValue) == 0:
    gsValue.dataType = None
    return None

  # if we have a gsColumn instance supplied, we use it's datatype definition
  if gsColumn is not None:
    gsValue.dataType  = gsColumn.typename
    gsValue.length    = gsColumn.length
    gsValue.precision = gsColumn.precision

  else:
    (fieldType, length, precision) = verifyDataType (gsValue)
    gsValue.dataType  = fieldType
    gsValue.length    = length
    gsValue.precision = precision


  # a string type must stay in it's bounds (if known)
  if gsValue.dataType == "string":
    if gsValue.length > 0 and len (fieldValue) > gsValue.length: 
      raise EOutOfRange (gsValue, fieldValue, gsValue.length)

    return fieldValue

  # create a number according to length and precision
  elif gsValue.dataType == "number":
    value = fieldValue.strip ()

    if gsValue.length or gsValue.precision:
      vmatch = re.compile ('^([+-]{0,1})(\d+)[\.]{0,1}(\d*)$').match (value)
      if vmatch is None:
        raise EInvalidNumber (gsValue, value, gsValue.length, gsValue.precision)

      sign = vmatch.groups () [0]
      pre  = vmatch.groups () [1]
      frac = vmatch.groups () [2]

      if len (pre) > (gsValue.length - gsValue.precision) or \
         len (frac) > gsValue.precision:
        raise EOutOfRange (gsValue, value, gsValue.length, gsValue.precision)

      if len (frac):
        return float ("%s%s.%s" % (sign, pre, frac))

      else:
        return int ("%s%s" % (sign, pre))

    # we know nothing about precision 
    else:
      if "." in value:
        return float (value)

      else:
        return int (value)
      

  # booleans must be 'TRUE' or 'FALSE', otherwise they're "None"
  elif gsValue.dataType == "boolean":
    bool = fieldValue.upper ().strip ()

    if bool in ["TRUE", "FALSE"]:
      return bool == "TRUE"
    else:
      raise EInvalidBoolean (gsValue, fieldValue)


  # Dates must conform with the ISO spec: YYYY-MM-DD
  elif gsValue.dataType == "date":
    try:
      return mx.DateTime.ISO.ParseDate (fieldValue.strip ())

    except ValueError:
      raise EInvalidDate (gsValue, fieldValue)

    except:
      raise

    
  # Times must conform with the ISO spec: HH:[MM[:SS[.ss]]]
  elif gsValue.dataType == "time":
    try:
      return mx.DateTime.ISO.ParseTime (fieldValue.strip ())

    except ValueError:
      raise EInvalidTime (gsValue, fieldValue)

    except:
      raise


  # DateTime values must conform with the ISO spec: YYYY-MM-DD HH:MM:SS.ss
  elif gsValue.dataType == "datetime":
    try:
      return mx.DateTime.ISO.ParseDateTime (fieldValue.strip ())

    except ValueError:
      raise EInvalidDateTime (gsValue, fieldValue)

    except:
      raise


  # unhandled types
  else:
    raise EInvalidType (gsValue, gsValue.dataType)






# -----------------------------------------------------------------------------
# Unquote a string
# -----------------------------------------------------------------------------

def unquoteString (aString):
  """
  This function strips away leading and trailling quotes (single or double)
  from a string.
  """
  if len (aString) > 1 and aString [0] in ['"', "'"]:
    if aString [-1] == aString [0]:
      aString = aString [1:-1]

  return aString
