# GNU Enterprise RPC interface - Py-XMLRPC type conversion rules
#
# Copyright 2001-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: typeconv.py 5377 2004-03-19 17:00:58Z reinhard $

import types
import mx.DateTime
import xmlrpc

# -----------------------------------------------------------------------------
# Convert native Python type to xmlrpc's type
# -----------------------------------------------------------------------------

def python_to_rpc (value, exception):

  # None
  if value is None:
    return ''

  # String
  elif isinstance (value, types.StringType):
    return value
  elif isinstance (value, types.UnicodeType):
    return value.encode ('utf-8')

  # Number
  elif isinstance (value, types.IntType):
    return value
  elif isinstance (value, types.FloatType):
    return value

  # Date/Time
  elif isinstance (value, mx.DateTime.DateTimeType):
    # mx.DateTime uses date 1/1/1 for time-only values. py_xmlrpc doesn't like
    # that.
    y = value.year
    m = value.month
    d = value.day
    if (y, m, d) == (1, 1, 1): (y, m, d) = (9999, 1, 1)

    return xmlrpc.dateTime (y, m, d, value.hour, value.minute,
                            int (value.second))

  # Boolean
  elif hasattr (types, 'BooleanType') and \
       isinstance (value, types.BooleanType):
    if value:
      return xmlrpc.boolean (True)
    else:
      return xmlrpc.boolean (False)

  # List
  elif isinstance (value, types.ListType):
    return [python_to_rpc (element, exception) for element in value]

  # Tuple
  elif isinstance (value, types.TupleType):
    result = ()
    for element in value:
      result += (python_to_rpc (element, exception), )
    return result

  # Dictionary
  elif isinstance (value, types.DictionaryType):
    result = {}
    for (key, val) in value.items ():
      result [python_to_rpc (key, exception)] = python_to_rpc (val, exception)
    return result

  else:
    raise exception, repr (value)

# -----------------------------------------------------------------------------
# Convert xmlrpc's type to native Python type
# -----------------------------------------------------------------------------

def rpc_to_python (value, exception):

  # None or String
  if isinstance (value, types.StringType):
    if value:
      return unicode (value, 'utf-8')
    else:
      return None

  # Number
  elif isinstance (value, types.IntType):
    return value
  elif isinstance (value, types.FloatType):
    return value

  # Date/Time
  elif type (value) == type (xmlrpc.dateTime (1999, 12, 31, 23, 59, 59)):
    (year, month, day, hour, minute, second) = value.date ()
    # 1/1/1 (i.e. time without date) is transported as 1/1/9999
    if (year, month, day) == (9999, 1, 1): (year, month, day) = (1, 1, 1)
    return mx.DateTime.DateTime (year, month, day, hour, minute, second)

  # Boolean
  elif type (value) == type (xmlrpc.boolean (0)):
    if value:
      return True
    else:
      return False

  # List
  elif isinstance (value, types.ListType):
    return [rpc_to_python (element, exception) for element in value]

  # Tuple
  elif isinstance (value, types.TupleType):
    result = ()
    for element in value:
      result += (rpc_to_python (element, exception), )
    return result

  # Dictionary
  elif isinstance (value, types.DictionaryType):
    result = {}
    for (key, val) in value.items ():
      result [python_to_rpc (key, exception)] = python_to_rpc (val, exception)
    return result

  else:
    raise exception, repr (value)
