#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# xmlrpc/ServerAdapter.py
#
# DESCRIPTION:
# Set of classes that implement the XML-RPC server driver for GNUe Comm.
#
# NOTES:
# Requires xmlrpclib from http://www.pythonware.com/products/xmlrpc/
# or Python 2.2+
#
# Server Parameters:
#
#    port        The port that the service is located on
#

from gnue.common.rpc import server
from gnue.common.apps import GDebug
from gnue.common.rpc.drivers._helpers import ObjectLibrarian, DirectoryServer

from BaseHTTPServer import BaseHTTPRequestHandler;
from SimpleHTTPServer import SimpleHTTPRequestHandler;
import SocketServer

import string, sys, os, posixpath, urllib, socket, types

try:
  import xmlrpclib
except ImportError:
  tmsg = _("\nUnable to load xmlrpclib.  To use the XML-RPC interface, \n"
           "please install xmlrpc from:\n"
           "    http://www.pythonware.com/products/xmlrpc/")
  raise server.AdapterInitializationError, tmsg

import typeconv

# Mapping from GRPC's datatype to XML-RPC datatypes
_datatypeMap = {
  'integer': 'int',
  'string': 'string',
  'boolean': 'boolean',
  'date': 'dateTime.iso8601',
  'number': 'double',
  'base64': 'base64',
  'binary': 'base64'
}

class MyThreadingHTTPServer(SocketServer.ThreadingTCPServer):
  def setParent(self,parent):
    self._parent=parent
    
  def server_bind(self):
    self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    self.socket.bind(self.server_address)

class MyForkingHTTPServer(SocketServer.ForkingTCPServer):
  def setParent(self,parent):
    self._parent=parent
    
  def server_bind(self):
    self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    self.socket.bind(self.server_address)


# TO BE CHECKED:
# Python 2.1: SocketServer.ThreadingMixIn
#SocketServer.ThreadingMixIn does not work properly
#since it tries to close the socket of a request two
#times.

#Workaround for using SocketServer.ThreadingMixIn under
#Python 2.1:

#class MyThreadingHTTPServer(
#SocketServer.ThreadingMixIn,
#MyHTTPServer
#):
#def close_request(self, request):
#pass

# an medusa like server 
#  should also be implemented: import asyncore, overwrite asyncore.dispatcher


##############################################################################
#
# ServerAdapter
#
class ServerAdapter(DirectoryServer.DirectoryServer):

  def __init__(self, rpcdef, bindings, params):
    try:
      self._port = params['port']
    except KeyError:
      tmsg = _('Required parameter "port" not supplied') 
      raise server.AdapterConfigurationError, tmsg 

    if params.has_key('bindto'):
      self._bindto = params['bindto']
    else:
      self._bindto = '' # bind to all interfaces

    if params.has_key('allowed_hosts'):
      # TODO: Remove spaces, etc.
      self._allowed_hosts = string.split(params['allowed_hosts'],',')
    else:
      self._allowed_hosts = [''] # allow access from all hosts

    if params.has_key('loglevel'):
      self._loglevel = params['loglevel']
    else:
      self._loglevel = 1

    if params.has_key('httpbind'):
      self._httpbind = params['httpbind']
      
    self._httpserver=MyThreadingHTTPServer
    
    if params.has_key('servertype'):
      if params['servertype']=='threading':
        self._httpserver=MyThreadingHTTPServer            
      elif params['servertype']=='forking':
        self._httpserver=MyForkingHTTPServer
#      elif params['servertype']=='asyncore':
#        self._httpserver=MyAsyncoreDispatcher
      else:
        raise server.AdapterConfigurationError, \
              u_("Value %(value)s for parameter 'servertype' is not supported. "
                 "Valid values are: %(valid)s") \
              % {'value': params ['servertype'],
                 'valid': 'threading, forking'}

    DirectoryServer.DirectoryServer.__init__(self, rpcdef, bindings, params)
    
    # To make "Object" definitions with w/non-object RPC,
    # the first parameters to any "Object" methods are
    # a "passed reference"
    # if parent._type == 'RpObject':
    #    signature.append(_datatypeMap['string'])
    
    # To simulate "Object" definitions with w/this
    # non-object RPC, the first parameters to any
    # "Object" methods are a "passed reference"

  #
  # Return an exception
  #
  def raiseException(self, exception, message, event=None):
    xmlrpclib.dumps(xmlrpclib.Fault(34543, '%s: %s' % (exception, message)))



  #                                                                        #
  #                                                                        #
  ##########################################################################
  def serve(self):
    if hasattr(self,'_httpbind'):
      self._requestHandler = EnhancedRequestHandler
    else:
      self._requestHandler = RequestHandler
      
    self._tcpserver = self._httpserver((self._bindto,self._port),
                                       self._requestHandler)
    self._tcpserver.setParent(self)
    try: 
      self._tcpserver.serve_forever()
    except KeyboardInterrupt:
      pass
    
  #
  # Call the requested method
  #
  def call(self, method, params):
    if self._loglevel>0:
      print _("Dispatching: "), method, params 
    

    ## Check if the Method is part of a service or a pointer to a
    ## single object
    ##
    ## Call to an object:  method="_Management:235423456_.getAttr"
    ##                     params=""
    ## Call to an method: (of a service=one object)
    ##                     method="DonutPlace.Management.Restart"

    if method[0]=='[':
      # call to an object
      # 1. get the object from the objectlibrarian
      # 2. check, if the object is supported by the gfd
      try:
        i=string.index(method,']',1)
        objhandle=method[1:i]
        method=method[i+2:]
      except ValueError:
        tmsg = _("Wrong format of object handle ")+\
              _("in method call %s") % method
        raise AttributeError, tmsg
      # TODO check in service dir, if obj is supported or not
      o=ObjectLibrarian.retrieveObject(objhandle)
      try:
        server_method=getattr(o,method)
        server_attribute=None
      except AttributeError:
        server_method=None
        try:
          server_attribute=getattr(o,method[4:])
        except AttributeError:
          
          if method!="_close":                      
            msg=_("Internal XMLRPC server error: method %s can be ")% \
                 method +\
                 _("found in the directory (build out of a .grpc file),")+\
                 _(" but the object doesn't contain this method/attribut.")+\
                 _(" Please check you .grpc file for wrong return types.")
            
            raise AttributeError, msg
        
      if method!="_close":
        direntry = self.getMethodDirEntry(o._type+"."+method)
        signature=direntry['signature']
      else:
        signature= ('string',)                

    else:

      # call to a service method or a helper call (get/set) for
      # a service attribut
      try:
        direntry = self.getMethodDirEntry(method)
        server_method = direntry['binding']
        server_attribute = None
        
        # check if it is an real method (binding -> method)
        # or an get/set method for an attribut (binding-> attribut)
        if (type(server_method)!=type(self.call)):
          server_attribute = server_method
          server_method=None
          
        signature=direntry['signature']

        if (server_method==None) and (server_attribute==None):
          tmsg = _("Server XML-RPC method %s  is not ")% method +\
              _("bound to real method") 
          raise AttributeError, tmsg
      except KeyError:
        tmsg = _("Server does not have XML-RPC ") +\
              _("procedure %s") % method      
        raise AttributeError, tmsg
    try:
      #
      pass
        # TODO:  Compare submitted attributs with signature
    except KeyError:
      tmsg = _("Server XML-RPC ") +\
            _("procedure %s accepts just %s as attributs") % (method,attr)
      raise AttributeError, tmsg
    

    # replace object handles in param with the real object
    counter=0
    while counter<len(params):
      p=params[counter]
      if type(p)==type(""):
        if (len(p)==42) and (p[0]=="[") and (p[41]=="]"):
          try:
            p=p[1:41]
            obj=ObjectLibrarian.retrieveObject(p)
            newp=params[0:counter-1]+(obj,)+params[counter+1:]
            params=newp
          except:
            pass
      counter=counter+1;

    # check if it is an real method (binding -> method)
    # or an get/set method for an attribut (binding-> attribut)

    __params = typeconv.rpc_to_python (params, server.InvalidParameter)

    if (server_method!=None):
            
      # call method with params
      try:
        result = server_method(*__params)
      except:
        return xmlrpclib.Fault (1, self._traceback (1))
      else:
        result = typeconv.python_to_rpc (result, server.InvalidParameter)

    # check if it's the close object method
    elif (method=="_close"):
      
      result=self.releaseDynamicObject(o)
      
    else:
      
      ## check wether its the set or the get method for the attribut
      mparts=string.splitfields(method,'.')
      mparts.reverse()
      calltype=mparts[0]
      calltype=calltype[:4]
      GDebug.printMesg(7,'method %s has calling type %s' %\
                       (method,calltype))
      if calltype=='set_':
        # setAttribut method
        server_attribute=params[0]
      elif calltype=='get_':
        # getAttribut method
        result=server_attribute
      else:
        tmsg = _("Internal Server XML-RPC error: method type") +\
            _("(get/set attribute) couldn't be detected (method %s)") \
            % method       
        raise AttributeError, tmsg
    

    # replace real object in param with an object handle
    if type(result)==type(self):  ## both should be instances
       ObjectLibrarian.archiveObject(result)

       # get the type of the result
       rtype=signature[0]
       # delete the surrounding brackets < >
       rtype=rtype[1:len(rtype)-1]
       # store typeinfo in new object
       result._type=rtype

       result=ObjectLibrarian.getObjectReference(result)
       self.registerDynamicObject(result,rtype)

    # check for empty results (not allowed for XMLRPC)
    if (result==None) or (result==[None]):
      GDebug.printMesg(3,'Transform result None into 1')
      result=1
      
    return result

  def registerDynamicObject(self,objhandle,type):
    #
    #  add new methods to the xmlrpc server
    #
    tl=len(type)
    for i in self.directory.keys():
      if i[0:tl]==type:
        method="["+objhandle+"]"+i[tl:]
        GDebug.printMesg(1,'Method %s registered to py-xmlrpc ' \
                         % method +\
                         ' internal directory.')
 #       self.server.addMethods({method:None})
     
    # add a method to close this object
  #  self.server.addMethods({"["+objhandle+"]._close":None})

  def deregisterDynamicObject(self,object):
    #
    #  remove the new methods from the xmlrpc server
    #
    objhandle=ObjectLibrarian.getObjectReference(object)
    type=object._type
    tl=len(type)
    for i in self.directory.keys():
      if i[0:tl]==type:
        method="["+objhandle+"]"+i[tl:]
        GDebug.printMesg(1,'Method %s is deleted from py-xmlrpc ' \
                         % method +\
                         ' internal directory.')
#        self.server.removeMethods({method:None})

    # remove the method to close this object
#    self.server.removeMethods({"["+objhandle+"]._close":None}) 

  def releaseDynamicObject(self,object):
    # delete bindings for this object in the xmlrpc method directory
    #    self.deregisterDynamicObject(object)
    # release object for Object Librarian
    ObjectLibrarian.deferenceObject(object)
    



##############################################################################
#
# Class to Handle the HTTP Post Request for XMLRPC only
#

class RequestHandler(BaseHTTPRequestHandler):

  # override basic handle to test allowed host condition
  #
  def handle(self):
    for i in self.server._parent._allowed_hosts:
      # get ip address of client and compare its first characters with i
      if self.client_address[0][:len(i)]==i:
        
        # continue with normal processing
        return BaseHTTPRequestHandler.handle(self)

    self.requestline="UNKNOWN"
    self.request_version="UNKNOWN"
    self.send_error(403, "Access to this server is forbidden!")
    return
      
    
  #
  # Handle the HTTP POST command
  #
  # This method is based largely on the
  # sample code in xmlrpcserver.
  #
  def do_POST(self):
    try:
      # get arguments
      data = self.rfile.read(int(self.headers["content-length"]))
      
      params, method = xmlrpclib.loads(data)
      
      # generate response
      response = self.server._parent.call(method, params)

      if isinstance (response, xmlrpclib.Fault):
        response = xmlrpclib.dumps (response)
      else:
        if response is None:
          response = ()
        elif not isinstance (response, types.TupleType):
          response = (response,)
        response = xmlrpclib.dumps (response, methodresponse = 1)
        
    except:
      # internal error, report as HTTP server error
      GDebug.printMesg(1,
                       'Unexpected Exception in XML-RPC code: %s:%s' % \
                       (sys.exc_type, sys.exc_value))
      # TODO: try to be a bit more informative (on client side)
      self.send_response(500)
      self.end_headers()
    else:
      # got a valid XML RPC response
      self.send_response(200)
      self.send_header("Content-type", "text/xml")
      self.send_header("Content-length", str(len(response)))
      self.end_headers()
      self.wfile.write(response)

      # shut down the connection
      self.wfile.flush()
      self.connection.shutdown(1)

##############################################################################
#
# Enhanced HTTP Request Handler serving XMLRPC Post requests and allows to
# access appserver data directory
#

class EnhancedRequestHandler(RequestHandler,SimpleHTTPRequestHandler):

  def send_head(self):
    # check for handler providing file content in string form
    try:
      # TODO: rewrite, add a hook instead of just filtering single pages
      content=self.server._parent._httpbind[self.path]()
      self.send_response(200)
      self.send_header("Content-type", "text/html")
      self.send_header("Content-Length", str(len(content)))
      self.end_headers()
      if self.command=="GET":
        self.wfile.write(content)
        self.wfile.flush()
      return None
    except:
      return SimpleHTTPRequestHandler.send_head(self)

  
  def translate_path(self, path):
    path = posixpath.normpath(urllib.unquote(path))
    words = path.split('/')
    words = filter(None, words)    
    httpbind = self.server._parent._httpbind
    
    try:
      path=httpbind["/"]
    except:
      path=httpbind

    ## security check      
    if (type(path)!=type("") or (len(path)<4)):
      return "/dev/null"
    
    for word in words:
      drive, word = os.path.splitdrive(word)
      head, word = os.path.split(word)
      if word in (os.curdir, os.pardir): continue
      path = os.path.join(path, word)
      
    return path
    

###########################
# The following part is not used at the moment
##############################################################################
#
# Used internally by the XML-RPC server to proxy
# between Object-by-Ref and actual objects.
#
class _ObjectByReferenceHandler:

  def __init__(self, oclass):
    self._class = oclass

  # If this makes absolutely no sense, move on and trust the code :)
  def requestHandler(self, method, *params, **args):

    if len(params):

      # If parameters were passed, the first one is always the handle
      refid = params[0]

      # The object's real method doesn't expect to receive the object handle
      params = params[1:]

    else:

      try:

        # If only named parameters are present, then
        # obj_handle should be the object handle...
        refid = args['obj_handle']

        # The object's real method doesn't expect to receive the object handle
        del args['obj_handle']

      except KeyError:
        tmsg = _('Object handle not returned') 
        raise StandardError, tmsg   # TODO

    try:
      return ObjectLibrarian. \
        retrieveObject(refid).__dict__[method](*params, **args)
    except KeyError:

      # Attempt to use an invalid objecthandle
      tmsg = _('Invalid object handle') 
      raise StandardError, tmsg # TODO



