#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
#
# FILE:
# GTrigger.py
#
# DESCRIPTION:
# Provides the basic classes needed by the generic trigger system
#
# NOTES:
#
import sys
import types
import string
import copy
from gnue.common.definitions.GObjects import GObj
from gnue.common.apps import GDebug

from gnue.common.formatting import GTypecast
from xml.sax import saxutils
from gnue.common.definitions.GParserHelpers import GContent
from gnue.common.logic.language import getLanguageAdapter, RuntimeError
from gnue.common.logic.language import AbortRequest

# class TriggerError(StandardError):
#   pass
# 
# class TriggerAbort:
#   pass
# class TriggerStop:
#   pass
# class TriggerSuccess:
#   pass

#######################################################################
#
# Trigger instance classes
#
# Classes in here are used to represent the triggers defined in
# <trigger> tags in an xml file
#
#
# Note: this is a partial graft of the old GFTrigger system
#       only here so that the new namespace code could be
#       put to use right away


#
# GTrigger
#
# Class used to implement triggers
#
class GTrigger(GObj):
  def __init__(self, parent=None, type=None, name=None, src=None, text=None, 
               language='python'):

    GObj.__init__(self, parent, 'GCTrigger')

    self._text=''
    self._triggerns={}
    self._inits   = [None,self.initialize]

    self.language=language
    self.src = src
    self.type = type and string.upper(type) or None
    self.name = name
    if text != None:
      GContent(self, text)
    if self.type != None:
      self._buildObject()


  #
  # Must be at least a phase 2 init
  #
  # TODO: this function merges the local namespace
  # TODO: not sure if this is doing the right thing
  # TODO: with regard to NAMED triggers.  It should
  # TODO: merge the local namespace of the object that
  # TODO: fired the trigger.
  def initialize(self):
    self._root = self.findParentOfType(None)
    self._triggerns.update( self._root._triggerns )
    self._globalns = self._root._globalRuntimeNamespace
    self.__call__ = self.dummyFunction

    if self.type != "NAMED":
      if self._parent:
        self._parent.associateTrigger( self.type, self )
        self._triggerns.update(self._parent._localTriggerNamespace)
    else:
      self._root._triggerDictionary[self.name] = self
      self._triggerns.update(self._root._localTriggerNamespace)

    if self.src == None:
      self.setFunction( self.getChildrenAsContent(), self.language )
    else:
      self.setFunctionFrom(self._root._triggerDictionary[self.src])


  def setFunctionFrom(self, object):
    self.__call__=object.__call__

  def setFunction(self, text, language):
    global runtime_list

    self._text = text
    self.language = language

    engine = getLanguageAdapter (self.language)
    cx = engine.createNewContext ()
    
    # define global namespace ('global' directive in python)
    cx.defineNamespace (self._globalns, asGlobal = True)

    # define local trigger namespace
    cx.defineNamespace (self._triggerns)

    # Build the name of an object, has to be integrated with the namespace
    # creation
    path=""
    p = self
    delim = ""
    while p:
      if hasattr (p, 'name'):
        if p.name:
          path = p.name + delim + path
          delim = "_"
      if hasattr (p, 'type'):
        if p.type:
          path = p.type + delim + path
          delim = "_"
      if p._triggerGlobal == 1:
        p=None
      else:
        p=p._parent
      
    method = cx.buildFunction (path, self._text)
    self.__call__ = method

  def dummyFunction(self, myself):
    GDebug.printMesg(0, "Trigger not implemented")

  def getDescription(self):
    """
    Return a useful description of this object
    for use by designer
    """
    print "Type", self.type
    if self.type == 'NAMED':
      return self.name
    else:
      return string.upper(self.type)

  #
  # dumpXML
  #
  #
  def dumpXML(self, lookupDict, treeDump=None, gap=None,
              textEncoding='<locale>',xmlnamespaces={}):
    """
    Dumps an XML representation of a trigger.
    used in saving.     
    """
    try:
      escape = not int(gConfig('StoreTriggersAsCDATA'))
    except:
      escape = 1
    xmlEntity = "trigger"
    xmlString = "%s<%s" % (gap[:-2],xmlEntity)

    indent = len(xmlString)
    pos = indent
    for attribute in self.__dict__.keys():

      # variables beginning with _ are never saved out to file
      # they are internal to the program
      if attribute[0] == "_":
        continue

      val = self.__dict__[attribute]
      if lookupDict[xmlEntity].has_key('Attributes') and \
         lookupDict[xmlEntity]['Attributes'].has_key(attribute):
        if val != None and \
           (not lookupDict[xmlEntity]['Attributes'][attribute].has_key('Default') or \
            (lookupDict[xmlEntity]['Attributes'][attribute]['Default']) != (val)):
          typecast = lookupDict[xmlEntity]['Attributes'][attribute]['Typecast']
          if typecast == GTypecast.boolean \
             and val == 1:
            addl = ' %s=""' % (attribute)
          elif typecast == GTypecast.names:
            addl = ' %s="%s"' % \
                (attribute, string.join(val,','))
          else:
            addl = ' %s="%s"' % (attribute, saxutils.escape('%s' % val))
          if len(addl) + pos > 78:
            xmlString += "\n" + " " * indent + addl
            pos = indent
          else:
            xmlString = xmlString + addl
            pos += len(addl)

    if len(self._children):
      hasContent = 0
      for child in self._children:
        hasContent = hasContent or isinstance(child,GContent)
      if hasContent:
        xmlString += ">"
      else:
        xmlString += ">\n"

      if treeDump:
        if hasContent and not escape:
          xmlString += "<![CDATA["
        for child in self._children:
          xmlString += child.dumpXML(lookupDict, 1,gap+"  ",escape=escape)
        if hasContent and not escape:
          xmlString += "]]>"

      if hasContent:
        xmlString += "</%s>\n" % (xmlEntity)
      else:
        xmlString += "%s</%s>\n" % (gap[:-2], xmlEntity)
    else:
      xmlString += "/>\n"
    return xmlString





#######################################################################
#
# Trigger processor classes
#
class GTriggerExtension:
  """
  Objects that inherit this class will be
  capable of processing triggers.
  """
  def __init__(self):
    self._trigger = {}
    
    #self._validTriggers = validTriggers 

  def associateTrigger(self, key, function):
    """
    Associates a trigger with the object.  More than one trigger of a specific type
    can be associated with an object.  This function is typically automatically
    called during tree construction from xml source (A GObj tree)
    """
    key = string.upper(key)
    if key in self._validTriggers.keys():
      if not self._trigger.has_key(key):
        self._trigger[key] = []
      self._trigger[key].append(function)
    else:
      GDebug.printMesg(0, "%s %s" % (_("Invalid trigger "),key))

  def processTrigger(self, key, ignoreAbort=True):
    """
    Fires the requested trigger if a trigger of that type
    has been associated with this object.
    
    B{Note:} You cannot fire a trigger before phase 3 of a gnue object
    tree phased init.
    
    @param key: The name of the trigger
    @type key: string
    @param ignoreAbort: If true then AbortRequests from the trigger are ignored
    @type ignoreAbort: boolean
    """
    key = string.upper(key)
    if key in self._validTriggers.keys():
      if self._trigger.has_key(key):
        for function in self._trigger[key]:
          # TODO: do we need to call "updateNamespace" here?
          #   function.updateNamespace()
          try:
            function(self)
          except AbortRequest:
            if not ignoreAbort:
              raise
          except RuntimeError, msg:
            # call my own exceptionHandler here.
            print "Runtime Error occured:\n %s" % msg
      else:
        GDebug.printMesg(10, "No triggers to fire")
    else:
      GDebug.printMesg(0, "%s: %s %s" % (self._type,_("Invalid trigger "),key))
      #print self._type,": ",_("Invalid trigger "),key

def getXMLelements(updates={}):
  """
  Return any XML elements associated with
  GTriggers.  Bases is a dictionary of tags
  whose values are update dictionaries.
  For example: bases={'datasource': {'BaseClass':myDataSource}}
  sets xmlElements['datasource']['BaseClass'] = myDataSource
  """
  xmlElements = {
      'trigger': {
         'BaseClass': GTrigger,
         'Importable': 1,
         'Attributes': {
            'name': {
               'Unique': 1,
               'Typecast': GTypecast.name },
            'type': {
               'Typecast': GTypecast.uppername },
            'src': {
               'Label': _('Source Trigger'),
               'References': 'trigger.name',
               'Typecast': GTypecast.name },
            'language': {
               'Typecast': GTypecast.name,
               'ValueSet': {
                   'python': {'Label': 'Python Script'} },
               'Default': 'python' } },
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'UsableBySiblings': 1,
         'ParentTags': None },
      }

  for alteration in updates.keys():
    xmlElements[alteration].update(updates[alteration])

  return xmlElements

