#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# _dbsig/DBdriver.py
#
# DESCRIPTION:
# Generic implementation of dbdriver using Python DB-SIG v2
# specification.
#
# NOTES:
# The classes below are meant to be extended
#

__all__ = ['RecordSet']

from types import *

from gnue.common.datasources import GConditions, Exceptions
from gnue.common.datasources.drivers.Base import RecordSet as BaseRecordSet
from gnue.common.apps import GDebug
import string


######################################################################
#
#
#
class RecordSet(BaseRecordSet):

  def _postChanges(self):
    if not self.isPending(): return
    if self._deleteFlag:
      s = self._buildDeleteStatement()
    elif self._insertFlag:
      s = self._buildInsertStatement()
    elif self._updateFlag:
      s = self._buildUpdateStatement()

    if isinstance (s, TupleType):
      # when useParameters is not set
      (statement, parameters) = s
    else:
      # when useParameters is set
      (statement, parameters) = (s, None)

    GDebug.printMesg(5, "_postChanges: statement=%s" % statement)

    try:
      self._parent._dataObject._connection.sql (statement, parameters)

      # Set _initialData to be the just-now posted values
      if not self._deleteFlag:
        self._initialData = {}
        self._initialData.update(self._fields)

    except self._parent._dataObject._DatabaseError, err:
      raise Exceptions.ConnectionError, err

    self._updateFlag = 0
    self._insertFlag = 0
    self._deleteFlag = 0

    return 1


  # If a vendor can do any of these more efficiently (i.e., use a known
  # PRIMARY KEY or ROWID, then override these methods. Otherwise, leave
  # as default.  Note that these functions are specific to DB-SIG based
  # drivers (i.e., these functions are not in the base RecordSet class)

  # This function is only used with "useParameters" set in gnue.conf
  def _where (self):
    if self._initialData.has_key(self._parent._dataObject._primaryIdField):
      where = [self._parent._dataObject._primaryIdFormat % \
               self._initialData [self._parent._dataObject._primaryIdField]]
      parameters = {}
    else:
      where = []
      parameters = {}
      for field in self._initialData.keys ():
        if self._parent.isFieldBound (field):
          if self._initialData [field] == None:
            where.append ("%s IS NULL" % field)
          else:
            key = 'old_' + field
            where.append ("%s=%%(%s)s" % (field, key))
            parameters [key] = self._initialData [field]

    return (string.join (where, ' AND '), parameters)

  def _buildDeleteStatement(self):
    if gConfig ('useParameters'):
      (where, parameters) = self._where ()
      statement = 'DELETE FROM %s WHERE %s' % \
                  (self._parent._dataObject.table, where)
      return (statement, parameters)
    else:
      if self._initialData.has_key(self._parent._dataObject._primaryIdField):
        where = [self._parent._dataObject._primaryIdFormat % \
            self._initialData[self._parent._dataObject._primaryIdField]  ]
      else:
        where = []
        for field in self._initialData.keys():
          if self._parent.isFieldBound(field):
            if self._initialData[field] == None:
              where.append ("%s IS NULL" % field)
            else:
              where.append ("%s=%s" % (field,
                self._parent._dataObject._toSqlString(self._initialData[field])))

      statement = "DELETE FROM %s WHERE %s" % \
         (self._parent._dataObject.table, string.join(where,' AND ') )
      return statement

  def _buildInsertStatement(self):
    if gConfig ('useParameters'):
      fields = []
      values = []
      parameters = {}

      for field in self._modifiedFlags.keys ():
        if self._parent.isFieldBound (field):
          key = 'new_' + field
          fields.append (field)
          values.append ('%%(%s)s' % key)
          parameters [key] = self._fields [field]

      statement = "INSERT INTO %s (%s) VALUES (%s)" % \
                  (self._parent._dataObject.table,
                   string.join (fields,', '),
                   string.join (values,', '))

      return (statement, parameters)
    else:
      vals = []
      fields = []

      for field in self._modifiedFlags.keys():
        if self._parent.isFieldBound(field):
          fields.append (field)
          if self._fields[field] == None or self._fields[field] == '':
            vals.append ("NULL") #  % (self._fields[field]))
          else:
            vals.append (self._parent._dataObject._toSqlString(self._fields[field]))

      return "INSERT INTO %s (%s) VALUES (%s)" % \
         (self._parent._dataObject.table, string.join(fields,','), \
          string.join(vals,',') )

  def _buildUpdateStatement(self):
    if gConfig ('useParameters'):
      (where, parameters) = self._where ()
      updates = []
      for field in self._modifiedFlags.keys():
        key = 'new_' + field
        updates.append ("%s=%%(%s)s" % (field, key))
        parameters [key] = self._fields [field]

      statement = "UPDATE %s SET %s WHERE %s" % \
                  (self._parent._dataObject.table,
                   string.join (updates, ', '),
                   where)
      return (statement, parameters)
    else:
      updates = []
      for field in self._modifiedFlags.keys():
        updates.append ("%s=%s" % (field,
           self._parent._dataObject._toSqlString(self._fields[field])))

      if self._parent._dataObject._primaryIdField:
        where = [self._parent._dataObject._primaryIdFormat % \
            self._initialData[self._parent._dataObject._primaryIdField]  ]
      else:
        where = []
        for field in self._initialData.keys():
          if self._initialData[field] == None:
            where.append ("%s IS NULL" % field)
          else:
            where.append ("%s=%s" % (field, self._parent._dataObject._toSqlString(self._initialData[field])))

      return "UPDATE %s SET %s WHERE %s" % \
         (self._parent._dataObject.table, string.join(updates,','), \
          string.join(where,' AND ') )
