#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# ResultSet.py
#
# DESCRIPTION:
#
# NOTES:
#

__all__ = ['ResultSet']

from gnue.common.apps import GDebug
from gnue.common.datasources import GConditions, Exceptions
import string

from RecordSet import RecordSet

###########################################################
#
#
#
###########################################################
class ResultSet:

  _recordSetClass = RecordSet

  def __init__(self, dataObject, cursor=None,defaultValues={},masterRecordSet=None):
     self._dataObject = dataObject
     self._cursor = cursor
     self._cachedRecords = []
     self._currentRecord = -1
     self._masterRecordSet = masterRecordSet
     self._readonly = 0
     self._recordCount = 0
     self._postingRecord = None

     self._defaultValues = {}
     self._defaultValues.update(defaultValues)

     self.current = None

     if masterRecordSet:
       masterRecordSet.addDetailResultSet(self)

  # Since we are overriding __len__
  def __nonzero__(self):
    return 1

  # Return the # of records
  def __len__(self):
    return self.getRecordCount()

  def __getitem__(self, index):
    rs = self.getRecord(index)
    if not rs:
      raise IndexError
    else:
      return rs


  # Returns whether this result set is read only or not
  def isReadOnly(self):
    return self._readonly


  # Returns 1=At first record, 0=Not first record
  def isFirstRecord(self):
    return (self._currentRecord == 0)


  # Returns 1=At last record, 0=Not last record
  def isLastRecord(self):
    if self._currentRecord < len(self._cachedRecords) - 1 or \
       self._cacheNextRecord():
      return 0
    else:
      return 1


  # returns -1=No records in memory, #=Current record #
  def getRecordNumber(self):
    return self._currentRecord


  # returns # of records currently loaded
  def getCacheCount(self):
    return len(self._cachedRecords)

  # returns # of records the
  def getRecordCount(self):
    return self._recordCount  > 0 and self._recordCount or self.getCacheCount()

  # Get a specific record (0=based)
  def getRecord(self, record):
    while (record + 1 > len(self._cachedRecords)) and self._cacheNextRecord():
      pass

    if record + 1 > len(self._cachedRecords):
      return None
    else:
      return self._cachedRecords[record]


  # move to record #, returns 1=New record loaded, 0=invalid #
  def setRecord(self, record):

    while (record > len(self._cachedRecords) -1) and self._cacheNextRecord():
      pass

    if record >= len(self._cachedRecords):
      return None
    else:
      self._currentRecord = record
      self.current = self._cachedRecords[self._currentRecord]
      self.notifyDetailObjects()
      return self.current

  # returns 1=New record loaded, 0=No more records
  def nextRecord(self):
    if self._currentRecord + 1 == len(self._cachedRecords):
      if not self._cacheNextRecord():
        return None

    self._currentRecord += 1
    self.current = self._cachedRecords[self._currentRecord]
    self.notifyDetailObjects()
    return self.current


  # returns 1=New record loaded, 0=At first record
  def prevRecord(self):
    if self._currentRecord < 1:
      return None
    else:
      self._currentRecord -= 1
      self.current = self._cachedRecords[self._currentRecord]
      self.notifyDetailObjects()
      return self.current


  # returns 1=at first record, 0=No records loaded
  def firstRecord(self):
    if self._currentRecord < 0:
      if not self._cacheNextRecord():
        return None

    self._currentRecord = 0
    self.current = self._cachedRecords[0]
    self.notifyDetailObjects()
    return self.current



  # returns 1=at last record, 0=No records loaded
  def lastRecord(self):
    if self._currentRecord == -1:
      return None
    else:
      while self._cacheNextRecord():
        pass
      self._currentRecord = len(self._cachedRecords) - 1
      self.current = self._cachedRecords[self._currentRecord]
      self.notifyDetailObjects()
      return self.current



  # Insert a blank record after the current record
  def insertRecord(self):
    if self.isReadOnly():
      # Provide better feedback??
      tmsg =  _("Attempted to insert into a read only datasource")
      raise Exceptions.ReadOnlyError, tmsg
    else:
      GDebug.printMesg(7,'Inserting a blank record')
      self._currentRecord += 1
      self._cachedRecords.insert(self._currentRecord, self._createEmptyRecord())
      self._recordCount += 1
      self.current = self._cachedRecords[self._currentRecord]

      # Set any dataobject-wide default values
      for field in self._dataObject._defaultValues.keys():
        self.current.setField(field, self._dataObject._defaultValues[field],0)

      # Set any resultset specific values
      for field in self._defaultValues.keys():
        self.current.setField(field, self._defaultValues[field],0)

      # Pull any primary keys from a master record set
      if self._masterRecordSet != None and hasattr(self._dataObject, '_masterfields'):
        i = 0
        for field in self._dataObject._masterfields:
          self.current.setField(self._dataObject._detailfields[i],self._masterRecordSet.getField(field),0)
          i += 1

      self.notifyDetailObjects()
      return self.current


  # Returns 1=DataObject, or a detail resultset, has uncommitted changes
  def isPending(self):
    for rec in (self._cachedRecords):
      if rec.isPending():
        return 1
      else:
        for detail in rec._cachedDetailResultSets.values():
          if detail.isPending():
            return 1
    return 0


  # Returns 1=DataObject has uncommitted changes
  def isRecordPending(self):
    return self.current.isPending()


  def getPostingRecordset(self):
    global postingRecordset
    return postingRecordset

  # Post changes to the database
  def post(self, foreign_keys={}):
    global postingRecordset
    # post our changes
    self._update_cursor = self._dataObject._dataConnection.cursor()

    recordPosition = 0
    while recordPosition < len(self._cachedRecords):
      self._postingRecord = self._cachedRecords[recordPosition]
      postingRecordset = self._postingRecord
      delete = self._postingRecord._emptyFlag or self._postingRecord._deleteFlag
      if not delete:
        # Flip the flag for 'default' values to true so that hidden
        # default fields are included in insert statements
        if self._postingRecord.isPending():
          for field in self._dataObject._defaultValues.keys():
            self._postingRecord._modifiedFlags[field] = 1

        for field in foreign_keys.keys():
          self._postingRecord._fields[field] = foreign_keys[field]
          # Some DBs will throw an exception if you update a Primary Key
          # (even if you are updating to the same value)
          if self._postingRecord._insertFlag:
            self._postingRecord._modifiedFlags[field] = 1

        recordPosition += 1
      else:
        # Adjust the current record if a preceding record
        # or the current record is deleted
        if recordPosition <= self._currentRecord:
          self._currentRecord -= 1
        self._cachedRecords.pop(recordPosition)
        self._recordCount -= 1

      self._postingRecord.post()

    # Move to record 0 if all preceding records were deleted
    # (or set to -1 if all records were deleted)
    if self._currentRecord < 0:
      if len(self._cachedRecords):
        self._currentRecord = 0
      else:
        self._currentRecord = -1
# TODO: I don't think we need this anymore
#    if self._currentRecord >= self._recordCount:
#      self._currentRecord = self._recordCount - 1

  def notifyDetailObjects(self):
    GDebug.printMesg(5,'Master record changed; Notifying Detail Objects')
    for detail in self._dataObject._detailObjects:
      if detail[1]:
        detail[1].masterResultSetChanged(self,
                                         detail[0]._masterRecordChanged(self))


  # Returns 1=Field is bound to a database field
  def isFieldBound(self, fieldName):
    if self._dataObject._fieldReferences.has_key(fieldName):
      return 1
    else:
      #TODO: the string.lower() line should never be called but is left
      #TODO: here untill the code is cleaned up
      return self._dataObject._fieldReferences.has_key(string.lower(fieldName))


  # Load cacheCount number of new records
  def _cacheNextRecord(self):
    rs = self._loadNextRecord()
    if rs:
      self._dataObject._dataSource._onRecordLoaded(self._cachedRecords[-1])
    return rs



  ###
  ### Methods below should be overridden by Vendor Specific functions
  ### (_createEmptyRecord may not need to be overridden in all cases)
  ###

  # Load cacheCount number of new records
  def _loadNextRecord(self):
    return 0

  # Create an empty recordset
  def _createEmptyRecord(self):
    return self._recordSetClass(self)

  # Iterator support (Python 2.2+)
  def __iter__(self):
    return _ResultSetIter(self)



# A simple resultset iterator
# Lets you use ResultSets as:
#
#   for record in myResultSet:
#      blah
#
# NOTE: Python 2.2+  (but it won't get called in
#    Python 2.1 or below, so not a problem)
#
class _ResultSetIter:
  def __init__(self, resultset):
    self.resultset = resultset
    self.used = 0
    self.done = 0

  def __iter__(self):
    return self

  def next(self):
    if self.done:
      raise StopIteration
    if self.used:
      rs = self.resultset.firstRecord()
    else:
      rs = self.resultset.nextRecord()

    if not rs:
      raise StopIteration
    else:
      return rs


# TODO: wtf?
postingRecordset = None

