# GNU Enterprise Application Server - Generators - Layout manager
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: layout.py 7463 2005-04-22 16:54:38Z reinhard $

import math
import operator
import sys

# =============================================================================
# This is the base class for layout managers
# =============================================================================

class Manager:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, classdef, maxWidth, maxHeight):
    """
    @param classdef: class definition to create a layout for
    @param maxWidth: maximum width for a form
    @param maxHeight: maximum height for a form
    """

    self._SPECIALS = {'gnue_createdate': {'Label': u_("Created"),
                                          'Buddy': 'gnue_createuser'},
                      'gnue_modifydate': {'Label': u_("Last modified"),
                                          'Buddy': 'gnue_modifyuser'}}

    self._maxWidth   = maxWidth
    self._maxHeight  = maxHeight
    self._classdef   = classdef
    self._specials   = self._prepareSpecials ()
    self._specialLoc = {}

    self.arrangeItems ()


  # ---------------------------------------------------------------------------
  # Create the visual pages and arrange all items
  # ---------------------------------------------------------------------------

  def arrangeItems (self):
    """
    This function creates the visual pages and arranges all properties on them.
    As a side effect after this function the properties 'maxPageHeight' and
    'maxPageWidth' are set.
    """

    (self.visualPages, self.maxPageHeight) = self._createVisualPages ()

    if len (self._specials):
      self._addSpecials ()

    pgSpace = [0]

    for (page, properties) in self.visualPages:
      pgSpace.append (self._arrangePage (page, properties))

      # arrange all specials
      for item in self._specials:
        location = self.getSpecialLocation (page, item)
        location ['labelPos'] = (1, location ['row'])
        location ['left']     = item.left

        if item.buddy is not None:
          buddyLocation = self.getSpecialLocation (page, item.buddy)
          buddyLocation ['labelPos'] = None
          buddyLocation ['left']     = item.buddy.left #maxLabel+1+item.width

    # make sure we have a bit empty room at the right side, but not too much
    self.maxPageWidth = min (max (pgSpace) + 2, self._maxWidth)


  # ---------------------------------------------------------------------------
  # Horizontally arrange all widgets of a single property
  # ---------------------------------------------------------------------------

  def arrangeWidgets (self, item, left, maxSpace = None):
    """
    This function arranges all widgets of a property by lining them up starting
    at the given position. If an item is a reference all reference items are
    positioned as well, where the reference-property get's an overall-width
    set.

    After this function is done, the 'left'- and 'width'-members of an item
    (and all it's references/buddies) are set.

    @param item: the property to arrange
    @param left: the x position of the item
    @param maxSpace: the maximum space left for the widget
    """

    item.left = left
    # make sure a dropdown-style item has at least a width of 7 characters
    width = max (item.fieldLength, item.style == 'dropdown' and 7 or None)

    if item.reference is None:
      # stretchabel items try to expand as most as possible
      if item.stretchable:
        width = maxSpace - 2

      else:
        # do not exceed the available space
        if maxSpace is not None:
          width = min (maxSpace, width)

      item.width = width

      # if we have a buddy, make sure it get's arranged too
      if hasattr (item, 'buddy') and item.buddy:
        self.arrangeWidgets (item.buddy, left + width + 1, maxSpace)


    else:
      item.width = 0

      for refItem in item.reference.properties:
        # for search-rerefences make sure to have a gap for the search-button
        # after the first reference field
        if not item.reference.isLookup and \
            refItem != item.reference.properties [0]:
          left += 3
          item.width += 3

        self.arrangeWidgets (refItem, left, maxSpace)
        left += (refItem.width + 1)
        width = refItem.width

        item.width += width

    return left + width


  # ---------------------------------------------------------------------------
  # Get the location dictionary for a special property on a given page
  # ---------------------------------------------------------------------------

  def getSpecialLocation (self, page, item):
    """
    This function returns the location-dictionary for a special property on a
    given page. Such a dictionary has a 'labelPos'-, a 'left'- and a 'row'-key.

    @param page: the visual page to get the location record for
    @param item: the item to get the location record for
    @return: dictionary with location information for the special property
    """

    return self._specialLoc [page][item.fullName]


  # ---------------------------------------------------------------------------
  # Prepare the special properties
  # ---------------------------------------------------------------------------

  def _prepareSpecials (self):
    """
    This function creates a sequence of special properties. Bound properties
    are removed from the sequence, so it only contains those properties having
    a label.

    @return: sequence with special property instances
    """

    result = []
    spec   = {}

    # First we build a dictionary of all special properties
    for item in self._classdef.specials:
      spec [item.dbField] = item
      item.buddy  = None
      item.height = 1

    # now update property information based on _SPECIALS
    for (name, item) in self._SPECIALS.items ():
      entry = spec.get (name)
      if entry is not None:
        entry.label = item ['Label']

        # if the property has buddy (a property without it's own label) connect
        # it and remove it from the dictionary
        if spec.has_key (item ['Buddy']):
          entry.buddy = spec [item ['Buddy']]
          del spec [item ['Buddy']]

    # at the end we create a sequence of special properties, keeping the same
    # order as before
    for item in self._classdef.specials:
      if spec.has_key (item.dbField):
        result.append (item)

    return result


  # ---------------------------------------------------------------------------
  # Add special properties to a visual page
  # ---------------------------------------------------------------------------

  def _addSpecials (self):
    """
    This function adds all special properties at the bottom of every visual
    page by creating a location record per property/page. Addidionally the
    maximum page height will be incremented by the number of lines needed to
    fit all special properties.
    """

    self.maxPageHeight += 1

    for (page, props) in self.visualPages:
      row = self.maxPageHeight

      for item in self._specials:
        if not self._specialLoc.has_key (page):
          self._specialLoc [page] = {}

        self._specialLoc [page] [item.fullName] = {'labelPos': None,
                                                   'row'     : row,
                                                   'left'    : 0}
        if item.buddy is not None:
          self._specialLoc [page] [item.buddy.fullName] = {'labelPos': None,
                                                           'row'     : row,
                                                           'left'    : 0}
        row += 1

    self.maxPageHeight = row



# =============================================================================
# This class implements a layout manager which arranges items vertically
# =============================================================================

class Vertical (Manager):

  _START_ROW  = 1
  _MIN_HEIGHT = 2


  # ---------------------------------------------------------------------------
  # Create all visual pages
  # ---------------------------------------------------------------------------

  def _createVisualPages (self):
    """
    This function transforms the virtual pages into visual pages. All available
    space will be distributed to stretchable widgets. Every item gets it's
    height and row attributes set. Additionaly the maximum height of the
    largest page will be determined.

    @return: tuple with the visual page sequence and the height of the largest
        page. The page sequence is a sequence of tuples with the page name and
        a list of all properties in the page.
    """

    result        = []

    # available space without specials and first row
    userSpace = self.__getUserSpace ()
    maxPageH  = userSpace + 1

    for (page, properties) in self._classdef.virtualPages:
      work = properties [:]
      name = page
      pnum = 1

      while len (work):
        cPage = self.__nextPageSet (name, work, userSpace)
        pnum += 1
        name  = "%s-%d" % (page, pnum)

        work = work [len (cPage [1]):]
        result.append (cPage)

    return (result, maxPageH)


  # ---------------------------------------------------------------------------
  # Arrange all properties on a single page
  # ---------------------------------------------------------------------------

  def _arrangePage (self, page, properties):
    """
    This function arranges all properties on the given page. All labels are
    lined up at the left margin and all entries are set into a second column.

    @param page: the name of the visual page
    @param properties: a sequence of properties to arrange in the page

    @return: outermost position on the right side (max. page width)
    """

    # first get the widest label (including the colon at the end)
    maxLabel = max ([len (p.label) + 1 for p in properties + self._specials])
    maxSpace = self._maxWidth - maxLabel - 1

    # determine label position and starting positions for all widgets
    widths = []
    for item in properties + self._specials:
      if not item.isSpecial:
        item.labelPos = (1, item.row)

      self.arrangeWidgets (item, maxLabel + 1, maxSpace)
      widths.append (item.left + item.width)

    # return the outermost right position on the page
    return max (widths)


  # ---------------------------------------------------------------------------
  # Calculate the maximum space to be filled with widgets for all pages
  # ---------------------------------------------------------------------------

  def __getUserSpace (self):
    """
    This function determines the number of rows available to place widgets in.

    @return: size of user space
    """

    # maximum page height minus special rows, minus starting row offset
    maxSpace = self._maxHeight - len (self._specials) - 1 - self._START_ROW
    result   = 0

    for (page, properties) in self._classdef.virtualPages:
      need = self.__sum ([i.minHeight or self._MIN_HEIGHT for i in properties])

      # if the needed space exceeds the longest page possible, or a page
      # contains stretchable items, use the maximum available space
      if need >= maxSpace or True in [item.stretchable for item in properties]:
        return maxSpace

      result = max (result, need)

    return result


  # ---------------------------------------------------------------------------
  # Create the sum of a sequence of numbers
  # ---------------------------------------------------------------------------

  def __sum (self, *args):

    if sys.version_info [0:2] >= (2, 3):
      return sum (*args)
    else:
      return reduce (operator.add, *args)


  # ---------------------------------------------------------------------------
  # Get the next set of properties for a given page
  # ---------------------------------------------------------------------------

  def __nextPageSet (self, page, properties, userSpace):
    """
    This function creates a set of visual properties to fit into a visual page.
    If there is space left on a page it will be distributed to the stretchable
    controls on the page. After this function is done all selected properties
    have their height- and row-members set.

    @param page: name of the visual page to use
    @param properties: sequence of available properties to arrange
    @param userSpace: number of available rows to fill

    @return: tuple with (pagename, propertysequence)
    """

    taken   = []
    stretch = []
    need    = 0

    for item in properties:
      item.height = item.minHeight or self._MIN_HEIGHT

      if need + item.height > userSpace:
        break

      need += item.height
      taken.append (item)

      if item.stretchable:
        stretch.append (item)

    # if there was space left, distribute it to all stretchable controls
    if userSpace - need > 0 and len (stretch):
      available = userSpace - need

      for item in stretch [:]:
        avg = int (math.ceil (float (available) / len (stretch)))
        if not avg:
          break

        item.height += avg
        available   -= avg
        stretch.remove (item)

    # at last we set the row members for all properties on the page
    row = self._START_ROW
    for item in taken:
      item.row = row
      row += item.height

      if item.reference is not None:
        for refItem in item.reference.properties:
          refItem.height = item.height
          refItem.row    = item.row

    return (page, taken)



# =============================================================================
# This class implements a layout manager which arranges items in a table
# =============================================================================

class Tabular (Manager):

  _MIN_HEIGHT = 1

  # ---------------------------------------------------------------------------
  # Create the visual pages
  # ---------------------------------------------------------------------------

  def _createVisualPages (self):
    """
    This function transform the virtual page into a visual page. Visual pages
    use all available vertical space leaving room for a title row and the
    special properties at the bottom.
    """

    result = []
    maxPageHeight = self._maxHeight - len (self._specials) - 2
    maxRowHeight  = 0

    for (page, properties) in self._classdef.virtualPages:

      for item in properties:
        item.row     = 1
        item.height  = item.minHeight or self._MIN_HEIGHT
        item.rows    = maxPageHeight
        maxRowHeight = max (maxRowHeight, item.height)

        if item.reference is not None:
          for refItem in item.reference.properties:
            refItem.height = item.height
            refItem.row    = item.row
            refItem.rows   = item.rows

      result.append ((page, properties))

    return (result, maxPageHeight + 1)


  # ---------------------------------------------------------------------------
  # Arrange all properties on a single page
  # ---------------------------------------------------------------------------

  def _arrangePage (self, page, properties):
    """
    This function arranges all properties on the given page. All properties are
    lined up in a grid.

    @param page: the name of the visual page
    @param properties: a sequence of properties to arrange in the page

    @return: outermost position on the right side (max. page width)
    """

    left     = 1
    widths   = []
    maxLabel = max ([len (s.label) + 1 for s in self._specials])

    for item in properties + self._specials:
      # special properties are located at the bottom of the page
      if item.isSpecial:
        maxSpace = self._maxWidth - maxLabel - 1
        self.arrangeWidgets (item, maxLabel + 1, maxSpace)
        widths.append (item.left + item.width)

      else:
        item.labelPos = (left, 0)

        self.arrangeWidgets (item, left, self._maxWidth - left)
        left += max (item.width, len (item.label) + 1) + 1

      widths.append (left + 1)

    return max (widths)
