# GNU Enterprise Application Server - Language interface: Object
#
# Copyright 2003-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Object.py 7104 2005-03-07 16:49:25Z johannes $

import string

from Procedure import Procedure
from gnue import appserver
from gnue.common.apps import errors


# ===========================================================================
# Exceptions
# ===========================================================================

class MemberNotFoundError (errors.ApplicationError):
  def __init__ (self, classname, member):
    msg = u_("Class '%(classname)s' has no member '%(member)s'") % \
          {"classname": classname, "member": member}
    errors.ApplicationError.__init__ (self, msg)


# ===========================================================================
# Implements a single business object
# ===========================================================================

class Object:

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------

  def __init__ (self, session, classname, objectId = None, definition = None):
    # we won't trigger __setattr__ here ...
    self.__dict__ ['objectId']            = objectId
    self.__dict__ ['_Object__session']    = session
    self.__dict__ ['_Object__class']      = classname
    self.__dict__ ['_Object__sid']        = session.getSessionId ()
    self.__dict__ ['_Object__definition'] = definition or {}

    if self.objectId is None:
      # Create new object
      sm  = session.getSessionManager ()
      res = sm.store (self.__sid, self.__class, [None], [], [[]])
      self.objectId = res [0]

    self.__dict__ ['gnue_id'] = self.objectId


  # -------------------------------------------------------------------------
  # Return an attribute's value; if not available in cache load from server
  # -------------------------------------------------------------------------

  def __getattr__ (self, attr):

    gEnter (6)

    name = self.__session.qualify (attr)
    key  = name.lower ()

    if key == "gnue_id":
      result = self.objectId
      return gLeave (6, result)

    sm = self.__session.getSessionManager ()

    try:
      if not self.__definition.has_key (key):
        self.__definition [key] = sm.load (self.__sid, self.__class, [''],
                                           [name]) [0][0]
      datatype = self.__definition [key]

    except KeyError:
      raise MemberNotFoundError, (self.__class, name)

    if datatype == 'procedure':
      result = Procedure (self.__session, self.objectId, self.__class, name)

    else:
      res = sm.load (self.__sid, self.__class, [self.objectId], [name])
      value = res [0][0]

      # Convert reference fields to object references
      if '_' in datatype and value is not None:
        cDef   = self.__session.getClassDefinition (datatype)
        result = Object (self.__session, datatype, value, cDef)

      elif value is None:
        if '_' in datatype:
          cDef   = self.__session.getClassDefinition (datatype)
          result = NullObject (self.__session, datatype, cDef)
        else:
          result = None

      else:
        result = value

    return gLeave (6, result)


  # -------------------------------------------------------------------------
  # Set an attribute's value
  # -------------------------------------------------------------------------

  def __setattr__ (self, attr, value):

    gEnter (6)

    if self.__dict__.has_key (attr):
      self.__dict__ [attr] = value
      # Make sure to keep in sync between objectId and gnue_id
      if attr == 'objectId':
        self.__dict__ ['gnue_id'] = value
      elif attr == 'gnue_id':
        self.__dict__ ['objectId'] = value

    if not self.__dict__.has_key (attr) or attr.lower () == 'gnue_id':
      name = self.__session.qualify (attr)
      key  = name.lower ()
      sm   = self.__session.getSessionManager ()

      try:
        if not self.__definition.has_key (key):
          self.__definition [key] = sm.load (self.__sid, self.__class,
                                                   [''], [name]) [0][0]
        datatype = self.__definition [key]

      except KeyError:
        raise MemberNotFoundError, (self.__class, name)

      if datatype == 'procedure':
        # Procedures not valid in __setattr__
        raise MemberNotFoundError, (self.__class, name)

      if isinstance (value, Object):
        __value = value.objectId
      else:
        __value = value

      sm.store (self.__sid, self.__class, [self.objectId], [name], [[__value]])

    gLeave (6)


  # -------------------------------------------------------------------------
  # Return the Object-ID as string representation
  # -------------------------------------------------------------------------

  def __str__ (self):

    if self.objectId:
      return self.objectId
    else:
      return ''


  # -------------------------------------------------------------------------
  # Return the Object as string representation
  # -------------------------------------------------------------------------

  def __repr__ (self):

    if self.objectId:
      return '<%s instance with id %s>' % (self.__class, self.objectId)
    else:
      return '<deleted %s instance>' % self.__class


  # -------------------------------------------------------------------------
  # Make object['property'] possible
  # -------------------------------------------------------------------------

  def __getitem__ (self, item):

    return self.__getattr__ (item)

  def __setitem__ (self, item, value):

    self.__setattr__ (item, value)


  # ---------------------------------------------------------------------------
  # Truth value testing
  # ---------------------------------------------------------------------------

  def __nonzero__ (self):

    return True


  # -------------------------------------------------------------------------
  # Delete instance and clear object-id
  # -------------------------------------------------------------------------

  def delete (self):

    gEnter (6)

    sm = self.__session.getSessionManager ()
    sm.delete (self.__sid, self.__class, [self.objectId])
    self.objectId = None

    gLeave (6)



# =============================================================================
# Class implementing None values for reference-properties
# =============================================================================

class NullObject:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, session, classname, definition):

    self.__session    = session
    self.__class      = classname
    self.__sid        = session.getSessionId ()
    self.__definition = definition


  # ---------------------------------------------------------------------------
  # Property access method
  # ---------------------------------------------------------------------------

  def __getattr__ (self, attr):
    """
    This function simulates access to a property of the instance. While 'attr'
    is a reference property the instance itself will be returned, changing the
    class to the referenced by the property. For all other property-types None
    will be return

    @return: self for reference properties, None for non-reference properties.
    """

    gEnter (6)

    name = self.__session.qualify (attr)
    key  = name.lower ()
    if not self.__definition.has_key (key):
      sm = self.__session.getSessionManager ()
      self.__definition [key] = sm.load (self.__sid, self.__class,
                                               [''], [name]) [0][0]
    if '_' in self.__definition [key]:
      self.__class      = self.__definition [key]
      self.__definition = self.__session.getClassDefinition (self.__class)

      result = self
    else:
      result = None

    return gLeave (6, result)


  # ---------------------------------------------------------------------------
  # Truth-test
  # ---------------------------------------------------------------------------

  def __nonzero__ (self):
    """
    A NullObject instance always has a truth-value of 'False' (like None has).
    This is used for constructs like: "if x: ..."
    """
    return False


  # ---------------------------------------------------------------------------
  # Comparisonf of two NullObject instances
  # ---------------------------------------------------------------------------

  def __cmp__ (self, other):
    """
    This function compares an instance with another object.
    """

    if self.__eq__ (other):
      return 0

    if self.__lt__ (other):
      return -1

    return 1


  # ---------------------------------------------------------------------------
  # Official string representation
  # ---------------------------------------------------------------------------

  def __repr__ (self):
    """
    This function returns the official string representation of an instance
    """

    return "<NullObject '%s' at %s>" % (self.__class, hex (id (self)))


  # ---------------------------------------------------------------------------
  # Informal string represenation
  # ---------------------------------------------------------------------------

  def __str__ (self):
    """
    The informal representation is just 'empty'
    """
    return ''


  # ---------------------------------------------------------------------------
  # Rich comparison: equality
  # ---------------------------------------------------------------------------

  def __eq__ (self, other):
    return other is None or isinstance (other, NullObject)


  # ---------------------------------------------------------------------------
  # Rich comparison: inequality
  # ---------------------------------------------------------------------------

  def __ne__ (self, other):
    return not self.__eq__ (other)


  # ---------------------------------------------------------------------------
  # Less than
  # ---------------------------------------------------------------------------

  def __lt__ (self, other):
    return True


  # ---------------------------------------------------------------------------
  # Greater than
  # ---------------------------------------------------------------------------

  def __gt__ (self, other):
    return False


  # ---------------------------------------------------------------------------
  # Coercing is not implemented for NullObjects
  # ---------------------------------------------------------------------------

  def __coerce__ (self, attr):
    return NotImplemented

  def __call__ (self, *args):
    pass
