# GNU Enterprise Application Server - Session Manager Object
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: geasSessionManager.py 7104 2005-03-07 16:49:25Z johannes $

from types import *

import geasSession
import geasAuthentication
import classrep
import geasFilter
import os.path
import fnmatch
import time

from gnue import paths
from gnue.common.datasources import GConnections
from gnue.common.apps import i18n, GConfig, errors
from gnue.appserver.gcd import readgcd
from gnue.appserver.gld import readgld

# =============================================================================
# Exceptions
# =============================================================================

class SessionNotFoundError (errors.SystemError):
  def __init__ (self, sessionId):
    msg = u_("Cannot find a session with ID '%s'") % sessionId
    errors.SystemError.__init__ (self, msg)

# =============================================================================
# Session Manager class
# =============================================================================

class geasSessionManager:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, modulepath = None, scanModules = False):
    self._connections = connections
    self._sessions    = {}
    self._buildInternalSession ()
    self._modulepath  = modulepath or \
                        os.path.join (paths.data, "share", "gnue", "appserver")

    #import profile
    #prof = profile.Profile ()
    #prof.runctx ('self.updateRepository (haltOnError = True)', globals (),
        #locals ())
    #prof.dump_stats ('appserver.profile')
    self.updateRepository (scanModules, haltOnError = True)

    cfg = gConfigDict (section = 'appserver')
    dbauth = cfg.get ('authentication', 'False')
    if dbauth.lower () in ['true', 'yes', 'y']:
      gDebug (1, "Using DB Auth Agent")
      self._authAdapter = geasAuthentication.geasDBAuthAgent ( \
                                                         self._internalSession)
    else:
      gDebug (1, "Using dummy Auth Agent")
      self._authAdapter = geasAuthentication.geasAuthAgent ()

    self._langRuntimes = {}


  # ---------------------------------------------------------------------------
  # Build an internal session
  # ---------------------------------------------------------------------------

  def _buildInternalSession (self):
    self._internalSession = geasSession.geasSession (self._connections,
      geasAuthentication.geasAuthAgent(), self, {})
    self._internalSession.login (None, None) # fake login

    # Some parts of the code access with session ID 0, so we have to store it
    # as "session zero", too.  The correct session id is used for procedure
    # calls.
    self._sessions [id (self._internalSession)] = self._internalSession
    self._sessions [0] = self._internalSession

  # ---------------------------------------------------------------------------
  # Find a session from session ID
  # ---------------------------------------------------------------------------

  def _getSession (self, sess_id):
    if self._sessions.has_key (sess_id):
      return self._sessions [sess_id]
    else:
      raise SessionNotFoundError, (sess_id)


  # ---------------------------------------------------------------------------
  # Update the class repository
  # ---------------------------------------------------------------------------

  def updateRepository (self, scanModules, haltOnError = False):
    """
    This function updates the class repository used by the session manager.
    First the modulepath will be scanned for gcd- and gld-files. All files
    found will be integrated and finally the repository get's updated.
    """

    if not scanModules:
      self._loadRepository ()
      return

    try:
      (gcd, gld) = self._scanModulePath ()

      if len (gcd):
        reader = readgcd.gcdReader (self._connections,
                                    gConfig ('connection'), gcd)
        reader.run ()

        modRS = reader.moduleRS
        clsRS = reader.classRS
        prpRS = reader.propRS
        prcRS = reader.procRS
      else:
        (modRS, clsRS, prpRS, prcRS) = (None, None, None, None)

      if len (gld):
        reader = readgld.gldReader (self._connections,
                                    gConfig ('connection'), gld,
                                    modRS, clsRS, prpRS, prcRS)
        reader.run ()

    except:
      if haltOnError:
        raise

      msg = u_("Failed reloading repository: %s") % errors.getException () [3]
      gDebug (1, msg)
      print o(msg)

    else:
      self._loadRepository ()


  # ---------------------------------------------------------------------------
  # Load the class repository
  # ---------------------------------------------------------------------------

  def _loadRepository (self):
    """
    This function loads the class repository
    """

    print o(u_("Reloading class repository ..."))

    self._internalSession.rollback ()

    classrep.init (self)

    self._filter = geasFilter.geasFilter (self)

    print o(u_("Class repository loaded"))


  # ---------------------------------------------------------------------------
  # Scan the module path for all GCD- and GLD-files
  # ---------------------------------------------------------------------------

  def _scanModulePath (self):
    """
    This function splits the modulepath by semicolons and scans all parts for
    gcd- and gld-files.

    @return: tuple of sequences (gcd, gld)
    """

    parts = [os.path.normcase (p) for p in self._modulepath.split (';')]
    gcds, glds = [], []

    for root in parts:
      gDebug (1, "Scanning path '%s'" % root)

      arg = {'gcd': [], 'gld': []}
      os.path.walk (root, self.__visitPath, arg)
      gcds.extend (arg ['gcd'])
      glds.extend (arg ['gld'])

    return (gcds, glds)


  # ---------------------------------------------------------------------------
  # Search a given path for gcd- and gld-files
  # ---------------------------------------------------------------------------

  def __visitPath (self, arg, dirname, files):
    """
    This function get's called by os.path.walk () and processes all files in a
    given directory.
    """

    for name in files:
      fullname = os.path.normpath (os.path.join (dirname, name))

      # Since os.path.walk () does *not* walk symbolic links we start another
      # run for every linked directory. 
      if os.path.islink (fullname) and os.path.isdir (fullname):
        os.path.walk (fullname, self.__visitPath, arg)
      else:
        if fnmatch.fnmatch (name, '*.gcd'):
          arg ['gcd'].append (fullname)
        elif fnmatch.fnmatch (name, '*.gld'):
          arg ['gld'].append (fullname)


  # ===========================================================================
  # official API functions
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Open the connection
  # ---------------------------------------------------------------------------

  def open (self, authentication):

    gEnter (4)

    loginHandler = self._connections._loginHandler
    loginOptions = self._connections._loginOptions
    location     = self._connections._location
    
    conn = GConnections.GConnections(location, loginHandler, loginOptions)
   
    sess = geasSession.geasSession (conn, self._authAdapter, self,
                                    authentication)

    sess.login (authentication ['user'], authentication ['password'])

    # We use the memory address of the geasSession object as session id.  This
    # is always unique and even thread safe.
    session_id = id (sess)
    self._sessions [session_id] = sess

    return gLeave (4, session_id)

  # ---------------------------------------------------------------------------
  # Close the connection
  # ---------------------------------------------------------------------------

  def close (self, session_id, commit):

    gEnter (4)

    s = self._getSession (session_id)
    if commit:
      s.commit ()
    s.logout ()
    del self._sessions [session_id]

    gLeave (4)

  # ---------------------------------------------------------------------------
  # Commit current transaction
  # ---------------------------------------------------------------------------

  def commit (self, session_id):

    gEnter (4)

    s = self._getSession (session_id)
    s.commit ()

    gLeave (4)

  # ---------------------------------------------------------------------------
  # Rollback current transaction
  # ---------------------------------------------------------------------------

  def rollback (self, session_id):

    gEnter (4)

    s = self._getSession (session_id)
    s.rollback ()

    gLeave (4)


  # ---------------------------------------------------------------------------
  # Build list from query
  # ---------------------------------------------------------------------------

  def request (self, session_id, classname, conditions, sortorder,
               propertylist):

    gEnter (4)

    s = self._getSession (session_id)
    result = s.request (classname, conditions, sortorder, propertylist)

    return gLeave (4, result)


  # ---------------------------------------------------------------------------
  # Count number of objects in list
  # ---------------------------------------------------------------------------

  def count (self, session_id, list_id):

    gEnter (4)

    s = self._getSession (session_id)
    result = s.count (list_id)

    return gLeave (4, result)

  # ---------------------------------------------------------------------------
  # Fetch objects from list
  # ---------------------------------------------------------------------------

  def fetch (self, session_id, list_id, start, count, close = False):

    gEnter (4)
    s = self._getSession (session_id)
    result = s.fetch (list_id, start, count)

    return gLeave (4, result)


  # ---------------------------------------------------------------------------
  # Load objects from ids
  # ---------------------------------------------------------------------------

  def load (self, session_id, classname, obj_id_list, propertylist):

    gEnter (4)

    s = self._getSession (session_id)
    result = s.load (classname, obj_id_list, propertylist)

    return gLeave (4, result)

  # ---------------------------------------------------------------------------
  # Store objects
  # ---------------------------------------------------------------------------

  def store (self, session_id, classname, obj_id_list, propertylist, data):

    gEnter (4)

    s = self._getSession (session_id)
    result = s.store (classname, obj_id_list, propertylist, data)

    return gLeave (4, result)

  # ---------------------------------------------------------------------------
  # Delete objects
  # ---------------------------------------------------------------------------

  def delete (self, session_id, classname, obj_id_list):

    gEnter (4)

    s = self._getSession (session_id)
    s.delete (classname, obj_id_list)

    gLeave (4)


  # ---------------------------------------------------------------------------
  # Call procedure
  # ---------------------------------------------------------------------------

  def call (self, session_id, classname, obj_id_list, procedurename,
            parameters):

    gEnter (4)

    s = self._getSession (session_id)
    result = s.call (classname, obj_id_list, procedurename, parameters)

    return gLeave (4, result)


  # ---------------------------------------------------------------------------
  # Get a structure describing all available filters
  # ---------------------------------------------------------------------------

  def getFilters (self, language):

    gEnter (4)

    result = self._filter.getFilter (language)

    return gLeave (4, result)
