# GNU Enterprise Application Server - Class Repository: Procedure
#
# Copyright 2003-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Procedure.py 7104 2005-03-07 16:49:25Z johannes $

from Base import *
from Namespace import createName, splitName
from Parameter import *
import helpers

# =============================================================================
# Exceptions
# =============================================================================

class ProcedureNotFoundError (ClassRepositoryError):
  def __init__ (self, classname, procedure):
    msg = u_("Class '%(classname)s' has no procedure '%(procedure)s'") \
          % { "classname": classname, "procedure": procedure }
    ClassRepositoryError.__init__ (self, msg)

class ProcedureValidationError (ItemValidationError):
  def __init__ (self, classname, procedure, message):
    pName = "%s.%s" % (classname, procedure)
    ItemValidationError.__init__ (self, pName, message)

# =============================================================================
# Dictionary with all properties of a given class
# =============================================================================
class ProcedureDict (BaseDictionary):

  # ---------------------------------------------------------------------------
  # Construct a Procedure-Dictionary for class aClass
  # ---------------------------------------------------------------------------
  def __init__ (self, session, aClass, predefs = None, modules = None):
    BaseDictionary.__init__ (self, session, 'gnue_procedure')

    self.__class = aClass

    if aClass is not None:
      self.__module  = self.__class.module
      self.__modules = modules or self.__module.modules
    else:
      self.__module  = None
      self.__modules = modules


  # ---------------------------------------------------------------------------
  # Create a new instance of a dictionary item
  # ---------------------------------------------------------------------------
  def _getNewItem (self, aObject):
    pMod = self.__modules.find (aObject.gnue_module.objectId)
    return Procedure (self._session, pMod, aObject,
                      {"gnue_id": aObject.objectId})


  # ---------------------------------------------------------------------------
  # A reload () returns only the procedures of the class
  # ---------------------------------------------------------------------------
  def _getReloadCondition (self):
    if self.__class is not None:
      result = ['eq', ['field', u'gnue_class'], ['const', self.__class.gnue_id]]
    else:
      result = []

    return gLeave (6, result)


  # ---------------------------------------------------------------------------
  # Get a condition to match a given procedure
  # ---------------------------------------------------------------------------
  def _getSingleCondition (self, key):
    (module, name) = splitName (key)

    return ['and', self._getReloadCondition (),
                   ['eq', ['field', u'gnue_module.gnue_name'],
                          ['const', module]],
                   ['eq', ['field', u'gnue_name'], ['const', name]]]


  # ---------------------------------------------------------------------------
  # Get a list of columns to be prepared on a find () 
  # ---------------------------------------------------------------------------
  def _getColumns (self):
    result = [u"gnue_module", u"gnue_class", u"gnue_name", u"gnue_language",
            u"gnue_type", u"gnue_length", u"gnue_scale", u"gnue_comment",
            u"gnue_nullable"]
    return gLeave (6, result)

  # ---------------------------------------------------------------------------
  # Create a key-not-found exception
  # ---------------------------------------------------------------------------
  def _itemNotFoundError (self, key):
    return ProcedureNotFoundError (self.__class.fullName, key)

  def _validationError (self, item, verr):
    return ProcedureValidationError (self.__class.fullName, item.fullName,
                                     verr.message)

# =============================================================================
# A Business Object Procedure
# =============================================================================
class Procedure (BaseObject):

  # ---------------------------------------------------------------------------
  # Construct a new procedure from module, class and business object
  # ---------------------------------------------------------------------------
  def __init__ (self, session, module, object, pDefs = None):
    BaseObject.__init__ (self, session, 'gnue_procedure', object, pDefs)

    self.module       = module
    self.fullName     = createName (module.gnue_name, self.gnue_name)
    self.isCalculated = False
    self.isValidated  = False
    self.calcFullName = None

    self.parameters   = ParameterDict (session, self)


  # ---------------------------------------------------------------------------
  # Validate a procedure
  # ---------------------------------------------------------------------------

  def validate (self):
    """
    This function verifies the procedure definition and validates all given
    parameters.
    """

    helpers.verifyProcedure (self)

    self.isCalculated = self.gnue_type is not None and \
                        len (self.parameters.keys ()) == 0 and \
                        self.gnue_name [:3].lower () == 'get'
    if self.isCalculated:
      self.calcName     = self.gnue_name [3:]
      self.calcFullName = createName (self.module.gnue_name, self.calcName)
    else:
      self.calcName     = None
      self.calcFullName = None

    self.isValidated = True


  # ---------------------------------------------------------------------------
  # Complete the procedure instance
  # ---------------------------------------------------------------------------

  def complete (self):
    gDebug (2, "Completing procudure %s" % self.fullName)
    self.parameters.reload ()
