# GNU Enterprise Application Server - RPC Server Application
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: geasRpcServer.py 5471 2004-03-23 09:21:36Z reinhard $

import time
import os
import sys

import gnue.paths

from gnue.common.apps import GConfig
from gnue.common.apps.GServerApp import GServerApp
from gnue.common.datasources import GLoginHandler
from gnue.common.rpc import server
from gnue.common.utils.FileUtils import openResource

from gnue.appserver import VERSION
from gnue.appserver import geasSessionManager
from gnue.appserver import geasConfiguration

# =============================================================================
# RPC application class
# =============================================================================

class geasRpcServerApp (GServerApp):

  NAME = "GNUe Application Server"
  VERSION = VERSION
  USAGE = GServerApp.USAGE
  COMMAND = 'gnue-appserver'
  SUMMARY = _(
"""The GNUe Application Server is the core of the n-tier variant of the
GNU Enterprise system. To the front end (be it GNUe Forms, GNUe Reports or
any other tool), it provides user-defineable business objects with arbitary
fields and methods. While transforming access to those fields and methods
into database communication and calling of scripts, it cares about stability,
security, speed, and consistency.""")

  COMMAND_OPTIONS = [

    ['rpctype', None, 'rpc-type', 1, None, 'type', _(
"""Set the GNURPC connection type.  The currently supported values for <type>
are 'xmlrpc', 'xmlrpc.py_xmlrpc', 'xmlrpc.pw_xmlrpc' and 'pyro'.  For more
information on GNURPC connection types have a look at
common/doc/RPC-abstraction.""")],

    ['rpcport', None, 'rpc-port', 1, None, 'port', _(
"""Set the GNURPC port.  For more information on GNURPC have a look at
common/doc/RPC-abstraction.""")],

    ['database', None, 'database', 1, None, 'name', _(
"""Set the Database to use for loading and storing data to <name>.
gnue-appserver will search for it in connections.conf.""")],

    ['username', 'u', 'username', 1, None, 'user', _(
"""Set the username for the used database.""")],

    ['password', 'p', 'password', 1, None, 'password', _(
"""Set the password for the used database.""")],

    ['selftest', None, 'selftest', 0, None, None, _(
"""Test appservers connection to the backend database, check correctness of
global settings, etc.""")],

    ['web-frontend', None, 'web-frontend', 0, None, None, _(
"""Enable appservers web frontend. Just works for the rpc-type XMLRPC.  The
webfrontend is at the same port as XMLRPC.""")]]

#  USE_DATABASE_OPTIONS = 1 # Conflicts with the existing username and password

  # ---------------------------------------------------------------------------
  # Initialize the object
  # ---------------------------------------------------------------------------

  def __init__ (self, connections=None):
    GServerApp.__init__ (self, connections, 'appserver',
                         geasConfiguration.ConfigOptions)
    self.configurationManager.registerAlias ('gConfig', 'appserver')

    # overwrite configuration settings by command line parameters
    cparser = self.configurationManager._loadedConfigs ['gnue.conf']

    if self.OPTIONS ["rpctype"] != None:
      cparser.set ('appserver', 'rpctype', self.OPTIONS ["rpctype"])

    if self.OPTIONS ["rpcport"] != None:
      cparser.set ('appserver', 'rpcport', self.OPTIONS ["rpcport"])

    if self.OPTIONS["database"] != None:
      cparser.set ('appserver', 'database', self.OPTIONS ["database"])

  # ---------------------------------------------------------------------------
  # Set a list of transports
  # ---------------------------------------------------------------------------

  def setTransports (self, transports):
    self._transports = transports

  # ---------------------------------------------------------------------------
  # Initialize the server
  # ---------------------------------------------------------------------------

  def phaseInit (self):
    rpctype = gConfig ("rpctype")

    if rpctype in ('xmlrpc','xmlrpc.pw_xmlrpc','xmlrpc.py_xmlrpc'):
      port = gConfig ("rpcport")
      print _("Exporting our services via %(rpctype)s (port %(port)s) ...") % \
               {"rpctype": rpctype, "port": port}
      params = {'port': int (port),
                'allowed_hosts': gConfig ('allowed_hosts')}

      if self.OPTIONS ["web-frontend"]:
        httpbind = {'/': gConfig ('httpdir'),
                    '/status': self.htmlStatus}
        params.update ({'httpbind': httpbind})

      self.setTransports({rpctype: params})

    elif rpctype == "pyro":
      print _("Exporting our services via Pyro (EXPERIMENTAL!) ...")
      self.setTransports ({'pyro':{}})

    elif rpctype == "sockets":
      # Sockets not working yet
      print _("Exporting our services via sockets (EXPERIMENTAL!) ...")
      self.setTransports ({'sockets':{}})

    else:
      # wrong transport protocol set. exiting
      print _("The protocol you've set is currently not supported.")
      sys.exit (-1)

  # ---------------------------------------------------------------------------
  # Show current status via html
  # ---------------------------------------------------------------------------

  def htmlStatus (self):
    out = "<HTML><HEAD></HEAD><BODY>"
    out += _("Status: %s Sessions opened") % self.sm._sessNo 
    out += "</BODY></HTML>"
    return out

  # ---------------------------------------------------------------------------
  # Run the server
  # ---------------------------------------------------------------------------

  def run (self):
    if self.OPTIONS ["selftest"]:
      self.selftest ()
      return

    # Create the various servers
    servers = server.bind (gnue.paths.data + '/share/gnue/grpc/appserver.grpc',
                           self._transports,
                           {'SessionManager': self.requestSessionManager})

    # be verbose
    print _("\n... GNUe Application Server up and running ...\n")

    # Daemonize (if appropriate)
    GServerApp.run (self)

    # Start the server for the different protocolls
    for key in servers.keys ():
      servers [key].serveAsNewThread ()

    # wait for the servers shut down
    try:
      while 1 == 1:
        time.sleep (1999999)
    except KeyboardInterrupt:
      print _("Appserver is shutting down....ok")

  # ---------------------------------------------------------------------------
  # Request a session manager (called once for every connection)
  # ---------------------------------------------------------------------------

  def requestSessionManager (self):
    if hasattr (self, "sm"):
      return self.sm

    loginhandler = geasLoginHandler ()
    if self.OPTIONS ["username"]:
      loginhandler.setLoginData (self.OPTIONS ["username"],
                                 self.OPTIONS ["password"])
    self.connections.setLoginHandler (loginhandler)
    self.sm = geasSessionManager.geasSessionManager (self.connections)
    return self.sm

  # ---------------------------------------------------------------------------
  # Self test
  # ---------------------------------------------------------------------------

  def selftest (self):
    ## Starting Appserver selftest
    print _(
"""
GNUe Application Server is running a simple self test
=====================================================

PREREQUISITE: You have to populate the backend db with
'the "address_person" example.
""")

    print _("Step 1: Starting Session Manager ...")
    sm = self.requestSessionManager ()

    print _("Step 2: Opening session (user 'hacker', password 'secret') ...")
    session=sm.open ({'user':"hacker", 'password':"secret"})

    print _("Step 3: Creating object list ...")
    list = sm.request (session, "address_person", [], ["address_zip"],
                       ["address_name", "address_street", "address_city"])

    print _("Step 4: Retrieving first instance ...")
    rset = sm.fetch (session,list,0,1)

    print o(u_("""
These are the values of the first instance:
  Name  : %(name)s
  Street: %(street)s
  City  : %(city)s
""") % { "name": rset[0][1], "street": rset[0][2], "city": rset[0][3] })

    print _('Selftest passed!')

# =============================================================================
# Login Handler
# =============================================================================

class geasLoginHandler (GLoginHandler.LoginHandler):
  def setLoginData (self,username,password):
    self._username = username
    self._password = password

  def getLogin (self, *arguments):
    if hasattr (self, "_username"):
      return {"_username": self._username, "_password": self._password}
    return {"_username": "gnue", "_password": None}

# =============================================================================
# Main program
# =============================================================================

if __name__ == '__main__':
  app = geasRpcServerApp ();
  app.phaseInit ()
  app.run ()
