#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: SchemaSupport.py 5619 2004-04-02 19:28:47Z reinhard $

import sys
import types
import mx.DateTime.ISO

from gnue.common.schema import Objects
from gnue.common.definitions import GParserHelpers
from gnue.appserver.classrep import Namespace

# =============================================================================
# Exceptions
# =============================================================================

# -----------------------------------------------------------------------------
# Base exception
# -----------------------------------------------------------------------------

class SchemaSupportError (gException):
  pass

# -----------------------------------------------------------------------------
# Invalid export type
# -----------------------------------------------------------------------------

class ExportTypeError (SchemaSupportError):
  def __init__ (self, exportType):
    msg = _("Invalid export type: '%s'") % exportType
    SchemaSupportError.__init__ (self, msg)

# -----------------------------------------------------------------------------
# Invalid value for given type of 
# -----------------------------------------------------------------------------

class ValueError (SchemaSupportError):
  pass


# =============================================================================
# 
# =============================================================================

class SchemaSupport:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, modules, classes):
    self.__modules = modules
    self.__classes = classes

    self.__exportModules = {}
    self.__exportClasses = {}

    self.__schema   = None
    self.__tables   = None
    self.__data     = None
    self.__dataRows = {}


  # ---------------------------------------------------------------------------
  # Write the requested items to a GNUe Schema Definition
  # ---------------------------------------------------------------------------

  def writeSchemaToFile (self, filename = None, items = None, wType = None):
    """
    This function creates an XML object tree for all the requested @items,
    containing a tabledefinition and a datadefinition as given by @wType, and
    dumps this XML object tree to filename. If filename is omitted sys.stdout
    will be used.
    """
    checktype (filename, [types.NoneType, types.StringType, types.UnicodeType])
    checktype (items,    [types.NoneType, types.ListType])
    checktype (wType,    [types.NoneType, types.StringType, types.UnicodeType])

    # do we have a usable export type given
    if wType is None:
      wType = 'both'

    elif wType.lower () not in ['schema', 'data', 'both']:
      raise ExportTypeError, wType.lower ()

    # build a todo-list: if no specific item is reqested, we will use all
    # classes. Requested items are separated into modules and classes
    self.__exportModules = {}
    self.__exportClasses = {}

    if items is None or len (items) == 0:
      self.__exportModules = self.__modules
    else:
      for item in items:
        # item is a module
        if Namespace.getNamespaceId (item) == Namespace.NSID_MODULE:
          aModule = self.__modules [item]
          self.__exportModules [aModule.fullName] = aModule

        # or a class
        elif Namespace.getNamespaceId (item) == Namespace.NSID_CLASS:
          aClass = self.__classes [item]
          self.__exportClasses [aClass.fullName] = aClass

        else:
          # or invalid
          raise Namespace.InvalidNameError, item

    # create the top-level objects of the XML tree
    self.__schema = Objects.GSSchema ()
    self.__schema.title   = 'Appserver Schema Dump'
    self.__schema.author  = 'Appserver SchemaSupport'
    self.__schema.version = '1.0'

    self.__tables = None
    self.__data   = None

    if wType.lower () in ['both', 'schema']:
      self.__tables = Objects.GSTables (self.__schema)
    if wType.lower () in ['both', 'data']:
      self.__data = Objects.GSData (self.__schema)

    # if there is a list of requested modules, add them to the gnue_module_dump
    # before class dumps
    if self.__data is not None and len (self.__exportModules.keys ()):
      self.__addModuleDump ()

    # iterate over all available classes in the repository
    for aClass in self.__classes.values ():

      # class is requested explicitly, so do a 'full create export'
      if self.__exportClasses.has_key (aClass.fullName):
        self.__export (aClass, True)

      # if the module of a class is requested, export all stuff of this module
      # plus gnue_*-things (using create action)
      elif self.__exportModules.has_key (aClass.gnue_module.gnue_name):
        self.__export (aClass, True, self.__exportModules.keys () + ['gnue'])

      else:
        # if aClass has a property or procedure of a requested module, export
        # all stuff of the requested modules (using an extend action)
        xModules = [p.module.fullName for p in aClass.properties.values () + \
                                               aClass.procedures.values ()]
        for module in xModules:
          if module == 'gnue': 
            continue
          if self.__exportModules.has_key (module):
            self.__export (aClass, False, self.__exportModules.keys ())
            break


    # The XML object tree is complete now, so we're going to dump it
    if filename is not None:
      destination = open (filename, 'w')
    else:
      destination = sys.stdout

    header = """<?xml version="1.0" encoding="UTF-8"?>
<!-- Schema definition created by GNUe Appserver's Schema Support. 
     Run this file through gnue-schema to create SQL scripts       -->
"""
    destination.write (header)
    destination.write (self.__schema.dumpXML ().encode ('utf-8'))
    destination.close ()



  # ---------------------------------------------------------------------------
  # Export a given class (schema and data)
  # ---------------------------------------------------------------------------

  def __export (self, aClass, create, modules = None):
    """
    This function exports schema- and data-definition of a given class. If
    @create is True, the schema-definition is given as 'create'-action,
    otherwise an 'extend'-action will be used. @modules is a list of
    modulenames, whose properties and procedures should be exported. If no list
    is given, all properties and procedures will be exported.
    """
    if self.__tables is not None:
      self.__exportSchema (aClass, create, modules)

    if self.__data is not None:
      self.__exportData (aClass, create, modules)



  # ---------------------------------------------------------------------------
  # Add a class to the table-definition tree
  # ---------------------------------------------------------------------------

  def __exportSchema (self, aClass, create, modules = None):
    """
    This function adds a classes schema to the XML object tree.
    """

    schema = Objects.GSTable (self.__tables)
    schema.name = aClass.table
    if not create:
      schema.action = "extend"

    fields      = Objects.GSFields (schema)
    indexes     = Objects.GSIndexes (schema)
    constraints = Objects.GSConstraints (schema)

    # populate the fields-collection
    for cProp in self.__filterByModules (aClass.properties.values (), modules):
      field = Objects.GSField (fields)
      field.name        = cProp.column
      field.description = cProp.gnue_comment
      field.type        = cProp.dbType
      if cProp.dbLength:
        field.length    = cProp.dbLength
      if cProp.dbScale:
        field.precision = cProp.dbScale
      if not cProp.gnue_nullable is None and not cProp.gnue_nullable:
        field.nullable  = False

      # if a type has an underscore we assume it's a reference-type, which
      # means to create a foreign key reference to the gnue_id of that class
      if "_" in cProp.gnue_type:
        constraint = Objects.GSConstraint (constraints)
        constraint.name = "%s_%s_fk" % (aClass.table, cProp.column)
        constraint.type = "foreignkey"

        constField = Objects.GSConstraintField (constraint)
        constField.name = cProp.fullName

        constRef = Objects.GSConstraintRef (constraint)
        constRef.name  = "gnue_id"
        constRef.table = cProp.gnue_type

    # in create-mode look if a primary key could be constructed
    if create and aClass.properties.has_key ('gnue_id'):
      if aClass.properties ['gnue_id'].gnue_type == 'id':
        primarykey = Objects.GSPrimaryKey (schema)
        primarykey.name = "gnue_id_pk_%s" % aClass.table

        pkField = Objects.GSPKField (primarykey)
        pkField.name = 'gnue_id'



  # ---------------------------------------------------------------------------
  # Add a class to the data definition 
  # ---------------------------------------------------------------------------

  def __exportData (self, aClass, create, modules = None):
    """
    This function adds @aClass to several gnue-table rows.
    """
    if create:
      self.__dumpObject (aClass, 'gnue_class')

    for prop in self.__filterByModules (aClass.properties.values (), modules):
      self.__dumpObject (prop, 'gnue_property')

    for proc in self.__filterByModules (aClass.procedures.values (), modules):
      self.__dumpObject (proc, 'gnue_procedure')

      for param in proc.parameters.values ():
        self.__dumpObject (param, 'gnue_parameter')


  # ---------------------------------------------------------------------------
  # Dump an object to the given table
  # ---------------------------------------------------------------------------

  def __dumpObject (self, aClass, table):
    """
    This function creates a new row of @table, and adds all values from @aClass
    according to the classdefinition of @table to it.
    """
    row = Objects.GSRow (self.__getRows (table))
    
    for cProp in self.__classes [table].properties.values ():
      value = getattr (aClass, cProp.fullName)
      
      # resolve reference-properties
      if "_" in cProp.fullType:
        value = value.gnue_id

      self.__buildValue (row, cProp, value)


  # ---------------------------------------------------------------------------
  # Get the rows object of an tabledata-object
  # ---------------------------------------------------------------------------

  def __getRows (self, classname):
    """
    This function returns the rows-collection of a given table. If it does not
    exist already it will be created (with a column definition included).
    """

    if not self.__dataRows.has_key (classname):
      table = Objects.GSTableData (self.__data)
      table.name      = "%s_dump" % classname
      table.tablename = classname

      columns = Objects.GSDefinition (table)

      for prop in self.__classes [classname].properties.values ():
        column = Objects.GSColumn (columns)
        column.field = prop.column
        column.type  = prop.dbFullType

      self.__dataRows [classname] = Objects.GSRows (table)

    return self.__dataRows [classname]


  # ---------------------------------------------------------------------------
  # Create a GContents XML object for a data value in a row
  # ---------------------------------------------------------------------------

  def __buildValue (self, row, prop, data):
    """
    If @data is not None this function adds a GContents object with the
    normalized value of @data to the @row, where @prop specifies the datatype
    for normalization.
    """
    if data is not None:
      value = Objects.GSValue (row)
      value.field = prop.column
      GParserHelpers.GContent (value, \
                               self.__nativeToString (data, prop.dbFullType))


  # ---------------------------------------------------------------------------
  # Get a subsequence of items matching a modulelist
  # ---------------------------------------------------------------------------

  def __filterByModules (self, items, modules):
    """
    This function returns a sub-sequence of all @items which match the given
    module list (@modules). If this list is None, all @items are returned.
    """
    result = []
    for element in items:
      if modules is not None and not element.module.fullName in modules:
        continue

      result.append (element)

    return result



  # ---------------------------------------------------------------------------
  # add all requested modules to the module dump
  # ---------------------------------------------------------------------------

  def __addModuleDump (self):
    """
    This function creates a dump of the gnue_module table, by adding all
    requested modules.
    """
    for module in self.__exportModules.values ():
      self.__dumpObject (module, 'gnue_module')



  # ---------------------------------------------------------------------------
  # Convert a native python object to a string according to datatype
  # ---------------------------------------------------------------------------

  def __nativeToString (self, native, datatype):
    """
    This function creates a unicode string to be used in a <value>-tag of a GSD
    file.  The native python object will be treated and theirfore converted as
    'datatype'.
    """
    if datatype [:6] == "string" or datatype == "id":
      checktype (native, [types.NoneType, types.UnicodeType])

      if native is None:
        return u''

      if isinstance (native, types.UnicodeType):
        return native

    elif datatype [:6] == "number":
      if native is None:
        return "0"
      else:
        return str (native)


    elif datatype == "boolean":
      if native is not None and native:
        return u'TRUE'
      else:
        return u'FALSE'


    elif datatype == "date":
      if isinstance (native, mx.DateTime.DateTimeType):
        return native.date

      else:
        raise ValueError (_("%s is not a valid date object") % repr (native))


    elif datatype == "time":
      if isinstance (native, mx.DateTime.DateTimeType):
        return native.time

      elif isinstance (native, mx.DateTime.DateTimeDeltaType):
        return str (native)

      else:
        raise ValueError (_("%s is not a valid time object") % repr (native))


    elif datatype == "datetime":
      if isinstance (native, mx.DateTime.DateTimeType):
        return str (native)

      else:
        raise ValueError (_("%s is not a valid datetime object") % \
                                    repr (native))

    else:
      # must be reference property
      return native.gnue_id
