# GNU Enterprise Application Server - Class Repository: Module
#
# Copyright 2003-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Module.py 5329 2004-03-16 14:54:04Z johannes $

from Base import *
from Class import ClassDict


# =============================================================================
# Exceptions
# =============================================================================

class ModuleNotFoundError (ClassRepositoryError):
  """
  This exception is raised if a non-existend module was called
  """
  def __init__ (self, module):
    msg = _("Module '%s' not found in class repository.") % module
    ClassRepositoryError.__init__ (self, msg)
    

# =============================================================================
# A List Of Business Object Modules
# =============================================================================

class ModuleDict (BaseDictionary):
  """
  This class implements a dictionary of all modules in the class repository.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, session, predefines = None):
    BaseDictionary.__init__ (self, session, 'gnue_module')
    self.classdict = None

    # Note: this 'predefines' is a RepositoryDefinition
    if predefines is not None:
      for mDict in predefines.modules ():
        module = Module (self._session, self, None, mDict)
        self._items [module.fullName] = module


  # ---------------------------------------------------------------------------
  # Create a new module instance
  # ---------------------------------------------------------------------------

  def _getNewItem (self, aObject):
    """
    Create a new module instance, using @aObject's objectId as
    'gnue_id'-predefine.
    """
    return Module (self._session, self, aObject, {"gnue_id": aObject.objectId})


  # ---------------------------------------------------------------------------
  # Create a condition to reload a single module
  # ---------------------------------------------------------------------------

  def _getSingleCondition (self, key):
    """
    Match a single module by it's name.
    """
    return [["eq", ""], ["field", "gnue_name"], ["const", key]]


  # ---------------------------------------------------------------------------
  # Create an appropriate itemNotFound error
  # ---------------------------------------------------------------------------

  def _itemNotFoundError (self, key):
    """
    An _itemNotFoundError is a 'ModuleNotFoundError'
    """
    return ModuleNotFoundError (key)


  # ---------------------------------------------------------------------------
  # Find a listed module by its id
  # ---------------------------------------------------------------------------

  def find (self, moduleId):
    """
    This function searches for a module which gnue_id is @moduleId.
    """
    for mod in self.values ():
      if mod.gnue_id == moduleId:
        return mod
    return None



# =============================================================================
# A Business Object Module
# =============================================================================

class Module (BaseObject):
  """
  This class implements a module in the class dictionary.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, session, mDict, aObject = None, pDefs = None):
    BaseObject.__init__ (self, session, 'gnue_module', aObject, pDefs)

    self.modules  = mDict
    self.fullName = self.gnue_name


  # ---------------------------------------------------------------------------
  # Return an attribute's value
  # ---------------------------------------------------------------------------
  def __getattr__ (self, attr):
    """
    The first time the attribute 'classes' is used, a class dictionary with all
    classes of this module will be created. All other attributes are delegated
    to the BaseObject's __getattr__ ().
    """
    if attr == 'classes':
      self.__dict__ ['classes'] = ClassDict (self._session, self.modules, self)

      # Make sure the created class-dictionary is up to date
      self.classes.reload ()
      return self.classes

    else:
      return BaseObject.__getattr__ (self, attr)


  # ---------------------------------------------------------------------------
  # replace module references
  # ---------------------------------------------------------------------------

  def replaceReferences (self, classes):
    """
    This function iterates over all classes in the given class dictionary and
    replaces their 'string-type' references to this module instance with this
    modules language interface object.
    """
    me = self._getObject ()

    for aClass in classes.values ():
      aClass.replaceReferences (me)

