# GNU Enterprise Application Server - Class Repository Definition
#
# Copyright 2003-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Definition.py 5329 2004-03-16 14:54:04Z johannes $

import ConfigParser
import string
import os

from Base import ClassRepositoryError
from Namespace import *

# =============================================================================
# Exceptions
# =============================================================================

class InvalidDefinitionError (ClassRepositoryError):
  """
  This exception is raised on an invalid repository definition file
  """
  def __init__ (self, text):
    ClassRepositoryError (text)
    

# =============================================================================
# A wrapper around a class repository definition file
# =============================================================================
class RepositoryDefinition:

  # ---------------------------------------------------------------------------
  # Load the definition from filename
  # ---------------------------------------------------------------------------
  def __init__ (self, filename):
    self.filename = filename
    if not os.path.isfile (filename):
      raise ClassRepositoryError (_("Cannot find file '%s'") % filename)

    self.parser   = ConfigParser.ConfigParser ()
    self.parser.read (self.filename)

    self._modules    = {}
    self._classes    = {}
    self._properties = {}

    self._loadConfig ()


  # ---------------------------------------------------------------------------
  # Load the definition from the configuration file
  # ---------------------------------------------------------------------------
  def _loadConfig (self):
    for section in self.parser.sections ():
      data = {}
      for option in self.parser.options (section):
	u = unicode (self.parser.get (section, option))
        if u == "":
          u = None
	data [string.lower (option)] = u

      identifier = unicode (string.lower (section))
      nsid = getNamespaceId (identifier)

      if nsid == NSID_MODULE:
	self._modules [identifier] = data
  
      elif nsid == NSID_CLASS:
	self._classes [identifier] = data

      elif nsid == NSID_PROPERTY:
	self._properties [identifier] = data

      else:
	raise InvalidDefinitionError ( \
                _("Invalid identifier '%s'") % identifier)

    # Make some basic integrity checks
    for klass in self._classes.keys ():
      (module, classname) = splitName (klass)
      if not self._modules.has_key (module):
	raise InvalidDefinitionError (_("Module '%s' not found") % module)

    for prop in self._properties.keys ():
      (klass, property) = string.split (prop, '.')
      if not self._classes.has_key (klass):
	raise InvalidDefinitionError (_("Class '%s' not found") % klass)

      (module, item) = splitName (property)
      if not self._modules.has_key (module):
	raise InvalidDefinitionError (_("Module '%s' not found") % module)

  # ---------------------------------------------------------------------------
  # return a list of modules defined in the configuration file
  # ---------------------------------------------------------------------------
  def modules (self):
    res = []
    for module in self._modules.keys ():
      res.append (self._modules [module])

    return res
  
  # ---------------------------------------------------------------------------
  # Return a list of classes (from specified module)
  # ---------------------------------------------------------------------------
  def classes (self, module = None):
    res = []
    for classname in self._classes.keys ():
      if (module is not None) and splitName (classname) [0] <> module:
        continue

      res.append (self._classes [classname])

    return res

  # ---------------------------------------------------------------------------
  # Return a list of properties
  # ---------------------------------------------------------------------------
  def properties (self, klass = None):
    res = []
    for propname in self._properties.keys ():
      if klass:
        if string.split (propname, '.') [0] <> klass and \
           self._properties [propname] ['gnue_class'] <> klass:
          continue

      res.append (self._properties [propname])

    return res


# =============================================================================
# Modules self-test code
# =============================================================================
if __name__ == '__main__':

  x = RepositoryDefinition ('repository.ini')

  print "\nModules:"
  for m in x.modules ():
    print m


  print "\nClasses:"
  for c in x.classes ():
    print c

  print "\nProperties:"
  for p in x.properties ():
    print p

  print "\n\n"

  for p in x.properties ('gnue_module'):
    print "gnue_module::", p


  print "----"

  print x.classes ('gnue')

  print "=============="
  print x.properties ('00000000000000000000000000000010')
