# GNU Enterprise Application Server - Class Repository: Class
#
# Copyright 2003-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Class.py 5329 2004-03-16 14:54:04Z johannes $

from Base import *
from Property import *
from Procedure import *
from Namespace import createName, splitName


# =============================================================================
# Exceptions
# =============================================================================

class ClassNotFoundError (ClassRepositoryError):
  """
  Referenced class not found in repository
  """
  def __init__ (self, classname):
    msg = _("Class '%s' not found in class repository") % classname
    ClassRepositoryError.__init__ (self, msg)


# =============================================================================
# Dictionary with classes
# =============================================================================
class ClassDict (BaseDictionary):
  """
  This class implements a dictionary of classes in the class repository.
  """

  # ---------------------------------------------------------------------------
  # Create a dictionary with all classes [from a specified module]
  # ---------------------------------------------------------------------------

  def __init__ (self, session, moduleDict, module = None, predefines = None):
    BaseDictionary.__init__ (self, session, 'gnue_class')

    self.modules    = moduleDict
    self.__module   = module

    # Note: this 'predefines' is a RepositoryDefinition
    if predefines is not None:
      for cDict in predefines.classes ():
        cMod     = self.modules.find (cDict ['gnue_module'])
        proplist = predefines.properties (cDict ['gnue_id'])

        aClass = Class (session, self, cMod, None, cDict, proplist)
        self._items [aClass.fullName] = aClass

    # if this class dictionary is for a special module, we populate it with
    # all classes, either from a global class dictionary or a fresh reload
    if module is not None:
      if self.modules.classdict is not None:
        # copy the predefined class dictionary 
        for item in self.modules.classdict.values ():
          if item.module.fullName == module.fullName:
            self._items [item.fullName] = item
      else:
        # fetch a fresh copy from the datasource
        #
        # NOTE: this won't happen very often, since it is only called if a
        # module requests a list of all classes without having it's own
        # ClassDict-instance.
        self.reload ()


  # ---------------------------------------------------------------------------
  # Reload the dictionary and run nested reloads for all classes 
  # ---------------------------------------------------------------------------

  def reload (self):
    """
    This function reloads all available classes of a module, and all their
    properties and procedures.
    """
    BaseDictionary.reload (self)
    for aClass in self.values ():
      aClass.properties.reload ()
      aClass.procedures.reload ()


  # ---------------------------------------------------------------------------
  # Create a new instance for a dictionary-item
  # ---------------------------------------------------------------------------

  def _getNewItem (self, aObject):
    """
    Create a new instance of a class and reload it's properties and procedures.
    """
    module = self.modules.find (aObject.gnue_module.objectId)
    c = Class (self._session, self, module, aObject,
               {"gnue_id": aObject.objectId})
    c.properties.reload ()
    c.procedures.reload ()
    return c


  # ---------------------------------------------------------------------------
  # Get an apropriate reload ()-condition
  # ---------------------------------------------------------------------------

  def _getReloadCondition (self):
    """
    If this class dictionary is bound to a module, the reload condition matches
    all classes of the bound module, otherwise no condition is given.
    """
    if self.__module is not None:
      return [["eq", ""], ["field", u"gnue_module"], 
             ["const", self.__module.gnue_id]]
    else:
      return []


  # ---------------------------------------------------------------------------
  # Create a condition to retrieve the klass specified by 'key'
  # ---------------------------------------------------------------------------

  def _getSingleCondition (self, key):
    """
    A single class is identified by it's modules gnue_id and the classname.
    """
    (moduleName, className) = splitName (key)
    module = self.modules [moduleName]

    return [["eq", ""], ["field", u"gnue_module"], ["const", module.gnue_id],
            ["eq", ""], ["field", u"gnue_name"], ["const", className]]


  # ---------------------------------------------------------------------------
  # Create a key-error exception
  # ---------------------------------------------------------------------------

  def _itemNotFoundError (self, key):
    """
    Return a ClassNotFoundError if an item cannot be found.
    """
    return ClassNotFoundError (key)


  # ---------------------------------------------------------------------------
  # On find () this dictionary will prepare the following columns ()
  # ---------------------------------------------------------------------------

  def _getColumns (self):
    """
    Fields to be fetched on reloads.
    """
    return [u"gnue_module", u"gnue_name"]



# =============================================================================
# An implementation of a single class
# =============================================================================
class Class (BaseObject):
  
  # ---------------------------------------------------------------------------
  # Wrapping a business object class
  # ---------------------------------------------------------------------------

  def __init__ (self, session, classDict, module, object, predefs = None, 
                propDefList = None):
    BaseObject.__init__ (self, session, 'gnue_class', object, predefs)

    self.module     = module
    self.fullName   = createName (module.gnue_name, self.gnue_name)
    self.table      = self.fullName
    self.classes    = classDict

    self.properties = PropertyDict (session, self, propDefList)
    self.procedures = ProcedureDict (session, self)


  # ---------------------------------------------------------------------------
  # Return a property by name
  # ---------------------------------------------------------------------------

  def findProp (self, name):
    """
    Find a property @name.
    """
    return self.properties [name]


  # ---------------------------------------------------------------------------
  # Return a procedure by name
  # ---------------------------------------------------------------------------

  def findProc (self, name):
    """
    Find a procedure @name.
    """
    return self.procedures [name]
	    

  # ---------------------------------------------------------------------------
  # Replace references
  # ---------------------------------------------------------------------------

  def replaceReferences (self, module):
    """
    This class updates all references of a class to it's module object and
    requests all properties and procedures to do update their references too.
    """
    BaseObject.replaceReferences (self, module)

    me = self._getObject ()
    for prop in self.properties.values ():
      prop.replaceReferences (module)
      prop.replaceReferences (me)

    for proc in self.procedures.values ():
      proc.replaceReferences (module)
      proc.replaceReferences (me)
