#include <glib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include "test-stuff.h"
#include "test-engine-stuff.h"
#include "test-file-stuff.h"

#include "core-utils.h"
#include "gnc-book.h"
#include "gnc-engine.h"
#include "TransLog.h"

#include "io-gncxml-v2.h"

const char *possible_envs[] = {
    "C", "af", "ar", "az", "be", "bg", "bg_BG.cp1251", "br", "ca", "cs",
    "cy", "cz", "da", "de", "de_AT", "el", "en", "en_AU", "en_CA", "en_GB",
    "eo", "es", "es_DO", "es_ES", "es_GT", "es_HN", "es_MX", "es_PA",
    "es_PE", "es_SV", "et", "et_EE", "eu", "fi", "fi_FI", "fr", "ga",
    "gd", "gl", "gr", "gv", "he", "hr", "hu", "id", "is", "it", "ja",
    "ja_JP", "ja_JP.SJIS", "ko", "ko_KR", "ko_KR.eucKR", "kw", "l10n",
    "locale.alias", "lt", "nl", "nn", "no", "no@nynorsk", "no_NO", "no_NY",
    "pl", "pl_PL", "pt", "pt_BR", "pt_PT", "ro", "ru", "ru_RU", "rudos",
    "rukoi8", "ruwin", "sk", "sl", "sl_SI", "sp", "sr", "sr_YU", "sv", "ta",
    "tr", "uk", "vi", "vi_VN.VISCII", "wa", "zh", "zh_CN", "zh_CN.EUC",
    "zh_CN.GB2312", "zh_TW", "zh_TW.Big5",
    NULL
};

const char *possible_vars[] = {
    "LANG", "LC_CTYPE", "LC_COLLATE", "LC_TIME", "LC_NUMERIC",
    "LC_MONETARY", "LC_MESSAGES",
    NULL
};

const char *diff_command = "cmp %s %s";
const char *test_dir = "test-files/xml2";
const char *base_env = "C";

static char*
gen_new_file_name(const char *filename, const char *env)
{
    char *ret;

    ret = g_new(char, strlen(filename) + strlen(env) + 2);
    strcpy(ret, filename);
    strcat(ret, "-");
    strcat(ret, env);

    return ret;
}

static int
run_command_get_return(const char *command)
{
    return system(command);
}

static char *
test_file(const char *filename)
{
    int i;

    for(i = 0; possible_envs[i] != NULL; i++)
    {
        GNCBackendError err;
        GNCSession *session;
        char *cmd;
        char *new_file = gen_new_file_name(filename, possible_envs[i]);
        GNCSession *new_session;
        
        session = gnc_session_new();

        gnc_session_begin(session, filename, TRUE, FALSE);
        err = gnc_session_pop_error (session);
        if (err)
        {
            gnc_session_destroy(session);
            return g_strdup_printf("gnc_session_begin errorid %d", err);
        }

        gnc_session_load(session, NULL);
        err = gnc_session_pop_error (session);
        if(err)
        {
            gnc_session_destroy(session);
            return g_strdup_printf("gnc_session_load errorid %d", err);
        }

        if(gnc_setenv("LANG", possible_envs[i], 1) != 0)
          return g_strdup_printf("gnc_setenv for LANG");

        new_session = gnc_session_new();
        
        gnc_session_begin(new_session, new_file, FALSE, FALSE);
        err = gnc_session_pop_error (new_session);
        if(err)
        {
            g_free(new_file);
            gnc_session_destroy(session);
            gnc_session_destroy(new_session);
            return g_strdup_printf("gnc_session_begin 2 with LANG=%s",
                                   possible_envs[i]);
        }

        gnc_session_save(new_session, NULL);

        cmd = g_strdup_printf(diff_command, filename, new_file);

        if(run_command_get_return(cmd) != 0)
        {
            g_free(cmd);
            g_free(new_file);
            gnc_session_destroy(session);
            gnc_session_destroy(new_session);
            return g_strdup_printf("run_command_get_return with LANG=%s",
                                   possible_envs[i]);
        }

        g_free(new_file);
        g_free(cmd);
        gnc_session_destroy(session);
        gnc_session_destroy(new_session);
    }

    return NULL;
}

int
main(int argc, char **argv)
{
    DIR *adir;

    gnc_engine_init(argc, argv);
    xaccLogDisable();
    
    if((adir = opendir(test_dir)) == NULL)
    {
        failure_args("opendir", __FILE__, __LINE__,
                     "couldn't open dir %s", test_dir);
    }
    else
    {
        struct dirent *next_file;

        while((next_file = readdir(adir)) != NULL)
        {
            struct stat file_info;
            char* filename;

            filename = g_strdup_printf("%s/%s", test_dir, next_file->d_name);
            
            if(stat(filename, &file_info) != 0)
            {
                failure_args("stat", __FILE__, __LINE__,
                             "couldn't stat file %s: %s", filename,
                             strerror(errno));
                g_free(filename);
                break;
            }

            if(gnc_setenv("LANG", base_env, 1) != 0)
            {
              failure_args("gnc_setenv", __FILE__, __LINE__,
                           "gnc_setenv of LANG failed");
              g_free(filename);
              break;
            }

            if(!S_ISDIR(file_info.st_mode))
            {
                char *msg = test_file(filename);
                
                if(msg != NULL)
                {
                    failure_args("test_file", __FILE__, __LINE__,
                                 "failure testing file %s with msg %s",
                                 filename, msg);
                }
                g_free(msg);
            }

            g_free(filename);
        }
    }
    
    print_test_results();
    exit(get_rv());
}
