/* vi:set ts=8 sts=0 sw=8:
 * $Id: file.c,v 1.50 2000/03/31 04:39:14 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * file locking/unlocking code borrowed from mutt (www.mutt.org).
 */
#include <stdio.h>
#include <dirent.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/file.h>
#include "main.h"
#include "win.h"
#include "doc.h"
#include "dialog.h"
#include "msgbar.h"
#include "msgbox.h"
#include "file.h"
#include "prefs.h"
#include "misc.h"
#include "undo.h"
#include "gnpintl.h"

#define MAXLOCKATTEMPT 5


/*** external functions ***/
extern void win_set_title(doc_t *d);


#ifdef USE_BACKUP
static void file_backup(char *fname);
#endif	/* USE_BACKUP */
static bool_t file_text_xmhtml_read(doc_t *d);


/*** global function definitions ***/
#ifndef USE_GNOME
/*
 * PUBLIC: file_exist
 *
 * returns true if file exists
 */
bool_t
file_exist(char *fname)
{
	struct stat sb;

	if (stat(fname, &sb) == -1)
		return FALSE;

	return TRUE;
} /* file_exist */
#endif


/*
 * PUBLIC: file_is_readonly
 *
 * returns true/false if file is readonly.  requires filename.
 */
bool_t
file_is_readonly(doc_t *d)
{
#ifdef USE_GTKXMHTML
	if (GTK_IS_XMHTML(d->data))
		return TRUE;
#endif
	if (d->force_ro)
		return TRUE;

	if (!d->fname || access(d->fname, F_OK))
		return FALSE;

	return (access(d->fname, W_OK) != 0);
} /* file_is_readonly */


/*
 * PUBLIC: file_full_pathname_make
 *
 * given a filename, tries to construct the full pathname to the file.  if
 * successful, returns a newly allocated buffer with the full pathname.
 */
char *
file_full_pathname_make(char *fname)
{
	char *full, *cwd;

	if (!fname)
		return NULL;

	if (fname[0] == '/')
		return g_strdup(fname);

	if ((cwd = getcwd(NULL, MAXPATH)) == NULL)
		return g_strdup(fname);

	if (cwd[strlen(cwd) - 1] == '/')
		cwd[strlen(cwd) - 1] = '\0';

	if ((strlen(fname) > 2) && fname[0] == '.' && fname[1] == '/')
		full = my_concat_dir_and_file(cwd, fname + 2);
	else
		full = my_concat_dir_and_file(cwd, fname);

	/* use free() here, not g_free(), since getcwd() uses malloc() */
	free(cwd);

	return full;
} /* file_full_pathname_make */


struct stat *
file_get_stats(char *fname)
{
	struct stat *sb = NULL;

	/* get file stats (e.g., file size is used by files list window) */
	if (fname) {
		sb = g_new(struct stat, 1);
		if (stat(fname, sb) == -1) {
			g_free(sb);
			return NULL;
		}
	}

	return sb;
} /* file_get_stats */


/*
 * PUBLIC: file_open_fp
 *
 * common routine for opening a file and returning a FILE *.  If the fopen()
 * fails, print an error message.  If successful, the file is locked and fp is
 * returned.
 */
FILE *
file_open_fp(const char *fname, const char *fmode, int flags,
	     const char *funcname)
{
	FILE *fp;
	char *buf, *errstr;
#ifndef GTK_HAVE_FEATURES_1_1_0
	int len;
#endif

	if ((fp = fopen(fname, fmode)) == NULL) {
		errstr = g_strerror(errno);
#ifdef GTK_HAVE_FEATURES_1_1_0
		buf = g_strdup_printf("%s: open failed ('%s', err=%d (%s))",
				      funcname, fname, errno, errstr);
#else
		len = strlen(fname) + strlen(funcname) + strlen(errstr) + 64;
		buf = g_new(char, len);
		g_snprintf(buf, len, "%s: open failed ('%s', err=%d (%s))",
			   funcname, fname, errno, errstr);
#endif
		msgbox_printf("%s\n", buf);
		if (!(flags & OPEN_NO_ERROR))
			(void)do_dialog_error(" Error opening file! ", buf);
		g_free(buf);
		return NULL;
	}

	if (!(flags & OPEN_NO_LOCK))
		(void)file_lock(fname, fp, (flags & OPEN_EXCL), FALSE, TRUE);

	return fp;
} /* file_open_fp */


void
file_close_fp(const char *fname, FILE *fp)
{
	(void)fflush(fp);
	(void)file_unlock(fname, fp);
	(void)fclose(fp);
} /* file_close_fp */


/*
 * PUBLIC: file_open_execute
 *
 * actually do the low-level file opening and stuff it into the document's
 * text widget.  note that the GTK text widget is rather primitive, so this
 * routine ends up reading the file text into a buffer first, and then
 * inserting that buffer using gtk_text_insert(), which basically does another
 * strcpy().  a better way would have been to mmap() the file, and then read it
 * directly.  but oh well.
 *
 * returns 0 on success, 1 on failure.
 */
int
file_open_execute(doc_t *d)
{
	if (!d->sb)
		d->sb = file_get_stats(d->fname);

	if (d->sb && d->sb->st_size > 0) {
		if (d->sb && S_ISDIR(d->sb->st_mode)) {
			msgbar_printf(d->w, _("'%s' is a directory"), d->fname);
			msgbox_printf(_("file_open_execute: '%s' is a directory"),
				      d->fname);
			goto out;
		}

#ifdef USE_GTKXMHTML
		if (GTK_IS_TEXT(d->data) || GTK_IS_XMHTML(d->data)) {
			if (!file_text_xmhtml_read(d))
				return 1;
		}
#else
		if (GTK_IS_TEXT(d->data)) {
			if (!file_text_xmhtml_read(d))
				return 1;
		}
#endif
#ifdef USE_GTKHTML
		else if (GTK_IS_HTML(d->data)) {
			gtk_html_begin(GTK_HTML(d->data), d->fname);
			gtk_html_parse(GTK_HTML(d->data));
		}
#endif
	} /* filesize > 0 */

out:
	/* misc settings */
	if (GTK_IS_TEXT(d->data))
		gtk_text_set_point(GTK_TEXT(d->data), 0);

	return 0;
} /* file_open_execute */


/*
 * returns TRUE on success, FALSE otherwise
 */
static bool_t
file_text_xmhtml_read(doc_t *d)
{
	long num;
	FILE *fp;
	size_t size;
	off_t bytesread;
	char *buf;
#ifdef USE_GTKXMHTML
	GString *htmlsrc = NULL;
#endif

	/* open file */
	fp = file_open_fp(d->fname, "r", 0, "file_text_xmhtml_read");
	if (fp == NULL)
		return FALSE;

	if (GTK_IS_TEXT(d->data))
		gtk_text_freeze(GTK_TEXT(d->data));
#ifdef USE_GTKXMHTML
	else if (GTK_IS_XMHTML(d->data))
		htmlsrc = g_string_new(NULL);
#endif

#if 0
	/*
	 * the more aggressive/intelligent method is to try to allocate a
	 * buffer to fit all the contents of the file; that way, we only read
	 * the file once.  basically, we try to allocate a buffer that is twice
	 * the number of bytes that is needed.  this is because
	 * gtk_text_insert() does a memcpy() of the read buffer into the text
	 * widget's buffer, so we must ensure that there is enough memory for
	 * both the read and the memcpy().  since we'd like to read the entire
	 * file at once, so we start with a buffer that is twice the filesize.
	 * if this fails (e.g., we're reading a really large file), we keep
	 * reducing the size by half until we are able to get a buffer.
	 *
	 * Note: this causes memory fragmentation on some systems, so just use
	 * the simplistic method, which isn't all that bad.
	 */
	size = 2 * (d->sb->st_size + 1);
	while ((buf = g_new(char, size)) == NULL)
		size /= 2;

	GNPDBG_FILE(("size %lu is ok (using half)\n", (gulong)size));
	g_free(buf);
	size /= 2;
	if ((buf = g_new(char, size + 1)) == NULL) {
		perror("file_open_execute: unable to malloc read buf");
		fclose(fp);
		return FALSE;
	}
	GNPDBG_FILE(("malloc'd %lu bytes for read buf\n", (gulong)(size+1)));
#endif
	size = (size_t)prefs.read_size;
	buf = g_new(char, size + 1);

	/* buffer allocated, now actually read the file */
	bytesread = 0;
	while (bytesread < d->sb->st_size) {
		gtk_events_flush();

		num = fread(buf, 1, size, fp);
		if (num > 0) {
			if (GTK_IS_TEXT(d->data))
				gtk_text_insert(GTK_TEXT(d->data), NULL, NULL,
						NULL, buf, num);
#ifdef USE_GTKXMHTML
			else if (GTK_IS_XMHTML(d->data))
				htmlsrc = g_string_append(htmlsrc, buf);
#endif
			bytesread += num;
		} else if (ferror(fp)) {
			perror("read error");
			(void)do_dialog_error(
					_("Read error!"),
					_(" file_open_execute: read error! "));
			break;
		} else if (feof(fp)) {
			break;
		} else /* if (num == 0) */ {
			g_warning("file_open_execute: read 0 bytes\n");
			break;
		}
	} /* while more bytes to read */
	g_free(buf);

	file_close_fp(d->fname, fp);
	if (GTK_IS_TEXT(d->data))
		gtk_text_thaw(GTK_TEXT(d->data));
#ifdef USE_GTKXMHTML
	else if (GTK_IS_XMHTML(d->data)) {
		g_assert(htmlsrc);
		gtk_xmhtml_source(GTK_XMHTML(d->data), htmlsrc->str);
		g_string_free(htmlsrc, TRUE);
	}
#endif

	return TRUE;
} /* file_text_xmhtml_read */


/*
 * PUBLIC: file_perm_string
 *
 * returns the rwx permissions of a file in a string.  non-reentrant.  does not
 * look at setgid, setuid, or sticky bit settings.
 */
char *
file_perm_string(struct stat *sb)
{
	static char perm[10];

	g_snprintf(perm, 10, "%c%c%c%c%c%c%c%c%c",
		(sb->st_mode & S_IRUSR) ? 'r' : '-',
		(sb->st_mode & S_IWUSR) ? 'w' : '-',
		(sb->st_mode & S_IXUSR) ? 'x' : '-',
		(sb->st_mode & S_IRGRP) ? 'r' : '-',
		(sb->st_mode & S_IWGRP) ? 'w' : '-',
		(sb->st_mode & S_IXGRP) ? 'x' : '-',
		(sb->st_mode & S_IROTH) ? 'r' : '-',
		(sb->st_mode & S_IWOTH) ? 'w' : '-',
		(sb->st_mode & S_IXOTH) ? 'x' : '-'
		);
	return perm;
} /* file_perm_string */


#ifdef APP_GNP
/*
 * PUBLIC: file_save_execute
 *
 * actually dump the contents of the document's text widget and do the
 * low-level file save.
 */
bool_t
file_save_execute(doc_t *d, bool_t saveas)
{
	FILE *fp;
	char *buf;
	long oldperm = -1;
	uid_t  olduid = (uid_t)-1;
	gid_t  oldgid = (gid_t)-1;
#if defined(USE_UNDOREDO) && defined(GTK_HAVE_FEATURES_1_1_0)
	GList *lp;
#endif

	g_assert(d->fname);
	if (d->sb == NULL)
		d->sb = g_new(struct stat, 1);
	else {
		oldperm = d->sb->st_mode;
		olduid = d->sb->st_uid;
		oldgid = d->sb->st_gid;
	}
	if (stat(d->fname, d->sb) == -1) {
		g_free(d->sb);
		d->sb = NULL;
	}

	if (d->sb && S_ISDIR(d->sb->st_mode)) {
		(void)do_dialog_error( _("File is a directory!"),
				      _(" Cannot save.  File is a directory! "));
		return FALSE;
	}

	if (!saveas && file_is_readonly(d)) {
		(void)do_dialog_error( _("Read only file!"),
				      _(" Cannot save file. File is read only! "));
		return FALSE;
	}

#ifdef USE_GTKXMHTML
	if (GTK_IS_XMHTML(d->data)) {
		(void)do_dialog_error( _("Can't save HTML!"),
				      _(" Can't save HTML files! "));
		return TRUE;
	}
#endif

	/* if the file has no changes, then there's no need to save anything */
	if (!d->changed) {
		msgbar_printf(d->w, _("%s: no changes to save"), doc_basefname(d));
		msgbox_printf(_("%s: no changes to save"), doc_basefname(d));
		return TRUE;
	}

#ifdef USE_BACKUP
	if (IS_DO_BACKUP())
		file_backup(d->fname);
#endif	/* USE_BACKUP */

	fp = file_open_fp(d->fname, "w", OPEN_EXCL, "file_save_execute");
	if (fp == NULL)
		return FALSE;

	/* try to set permissions to original. maybe use fchmod() and fchown? */
	if (oldperm >= 0)
		(void)chmod(d->fname, (mode_t)oldperm);
	(void)chown((char *)d->fname, olduid, oldgid);

	/*
	 * this isn't very memory efficient; gtk_editable_get_chars()
	 * (actually, it's gtk_text_get_chars()), returns the text in a newly
	 * malloc'd buffer.  we're basically making another copy of the
	 * document just to save it.  if the document is large, we might run
	 * into problems.
	 */
	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data), 0,
					gtk_text_get_length(GTK_TEXT(d->data)));
	if (fputs(buf, fp) == EOF) {
		perror("Error saving file");
		g_free(buf);
		file_close_fp(d->fname, fp);
		return FALSE;
	}
	g_free(buf);
	file_close_fp(d->fname, fp);

	gtk_label_set(GTK_LABEL(d->tablabel), doc_basefname(d));
	d->changed = FALSE;
	if (d->changed_id)
		gtk_signal_disconnect(GTK_OBJECT(d->data),  d->changed_id);
	d->changed_id = gtk_signal_connect(
				GTK_OBJECT(d->data), "changed",
				GTK_SIGNAL_FUNC(doc_changed_cb), d);

#if defined(USE_UNDOREDO) && defined(GTK_HAVE_FEATURES_1_1_0)
	/*
	 * update the undo and redo lists.  since we have successfully saved
	 * the file at this point, any further undo's or redo's always results
	 * in the document being "changed".
	 */
	for (lp = d->undolist; lp; lp = g_list_next(lp)) {
		undo_t *undo = (undo_t *)lp->data;
		undo->changed = TRUE;
	}
	for (lp = d->redolist; lp; lp = g_list_next(lp)) {
		undo_t *undo = (undo_t *)lp->data;
		undo->changed = TRUE;
	}
#endif	/* USE_UNDOREDO */

	win_set_title(d);

	msgbar_printf(d->w, _("Saved %s"), doc_basefname(d));
	msgbox_printf(_("saved %s"), doc_basefname(d));
	return TRUE;
} /* file_save_execute */
#endif	/* APP_GNP */



/*
 * PUBLIC: file_lock
 *
 * this routine was taken from mutt (www.mutt.org).  uses either fcntl() or
 * flock() to perform file locking of a file.  dot-locking code is currently
 * not supported.
 *
 * Args:
 *	excl	if excl != 0, request an exclusive lock
 *	dot	if dot != 0, try to dotlock the file
 *	timeout	should retry locking?
 */
int 
file_lock(const char *path, FILE *fp, bool_t excl, bool_t dot, bool_t timeout)
{
	int r = 0;
#if defined (USE_FCNTL) || defined (USE_FLOCK)
	int fd = fileno(fp);
	int count;
	int attempt;
	struct stat prev_sb;
#endif
#ifdef USE_FCNTL
	struct flock lck;

	memset(&lck, 0, sizeof(struct flock));
	lck.l_type = excl ? F_WRLCK : F_RDLCK;
	lck.l_whence = SEEK_SET;

	count = 0;
	attempt = 0;
	while (fcntl(fd, F_SETLK, &lck) == -1) {
		struct stat sb;

		if (errno != EAGAIN && errno != EACCES) {
			perror("file_lock: fcntl error");
			return -1;
		}

		if (fstat(fd, &sb) != 0)
			sb.st_size = 0;

		if (count == 0)
			prev_sb = sb;

		/* only unlock file if it is unchanged */
		if (prev_sb.st_size == sb.st_size && ++count >=
			(timeout ? MAXLOCKATTEMPT : 0)) {

			if (timeout)
				g_warning("Timeout exceeded while attempting "
					  "fcntl lock!");
			return -1;
		}
		prev_sb = sb;

		printf("Waiting for fcntl lock... %d", ++attempt);
		sleep(1);
	}
#endif /* USE_FCNTL */

#ifdef USE_FLOCK
	count = 0;
	attempt = 0;
	while (flock(fd, (excl ? LOCK_EX : LOCK_SH) | LOCK_NB) == -1) {
		struct stat sb;
		if (errno != EWOULDBLOCK) {
			perror("flock");
			r = -1;
			break;
		}
		if (fstat(fd, &sb) != 0)
			sb.st_size = 0;

		if (count == 0)
			prev_sb = sb;

		/* only unlock file if it is unchanged */
		if (prev_sb.st_size == sb.st_size && ++count >=
			(timeout ? MAXLOCKATTEMPT : 0)) {
			if (timeout)
				g_warning("Timeout exceeded while attempting "
					  "flock lock!");
			r = -1;
			break;
		}
		prev_sb = sb;

		printf("Waiting for flock attempt... %d", ++attempt);
		sleep(1);
	}
#endif /* USE_FLOCK */

#ifdef USE_DOTLOCK
	if (r == 0 && dot)
		r = dotlock_file(path, timeout);
#endif /* USE_DOTLOCK */

	if (r == -1) {
		/* release any other locks obtained in this routine */

#ifdef USE_FCNTL
		lck.l_type = F_UNLCK;
		fcntl(fd, F_SETLK, &lck);
#endif /* USE_FCNTL */

#ifdef USE_FLOCK
		flock(fd, LOCK_UN);
#endif /* USE_FLOCK */

		return (-1);
	}
	return 0;
} /* file_lock */


/*
 * PUBLIC: file_unlock
 *
 * this routine was taken from mutt (www.mutt.org).  uses either fcntl() or
 * flock() to unlock a file previously locked by lock_file().  dot-locking code
 * is currently not supported.
 */
int 
file_unlock(const char *path, FILE *fp)
{
#if defined (USE_FCNTL) || defined (USE_FLOCK)
	int fd = fileno(fp);
#endif
#ifdef USE_FCNTL
	struct flock unlockit = {F_UNLCK, 0, 0, 0};

	memset(&unlockit, 0, sizeof(struct flock));
	unlockit.l_type = F_UNLCK;
	unlockit.l_whence = SEEK_SET;
	fcntl(fd, F_SETLK, &unlockit);
#endif

#ifdef USE_FLOCK
	flock(fd, LOCK_UN);
#endif

#ifdef USE_DOTLOCK
	undotlock_file(path);
#endif

	return 0;
} /* file_unlock */


/*** local function defintions ***/
#ifdef USE_BACKUP
/*
 * PRIVATE: file_backup
 *
 * creates a backup file.  first tries to remove the old backup file, if one
 * exists.  if successful or if one never existed, renames the current file to
 * the backup name.
 */
static void
file_backup(char *fname)
{
	char *backupname;
	char *backupdir;
	char *dname = NULL;
#ifndef GTK_HAVE_FEATURES_1_1_0
	int len;
#endif

	if (fname == NULL || !file_exist(fname))
		return;

	GNPDBG_FILE(("file_backup: orig file = '%s'\n", fname));
	if (prefs.backupdir && prefs.backupdir != "" &&
	    strcmp(prefs.backupdir, "(null)")) {
		backupdir = prefs.backupdir;
	} else {
		if (fname[0] == '/')
			dname = my_dirname((const gchar *)fname);
		else {
			char *tmp = file_full_pathname_make(fname);
			dname = my_dirname((const gchar *)tmp);
			g_free(tmp);
		}
		backupdir = dname;
	}
#ifdef GTK_HAVE_FEATURES_1_1_0
	backupname = g_strdup_printf("%s/%s%s", backupdir, my_basename(fname),
				     (prefs.backupsuffix) ?
				     		prefs.backupsuffix : NULL);
#else
	len = 2 + strlen(backupdir) + strlen(my_basename(fname)) +
	      (prefs.backupsuffix ? strlen(prefs.backupsuffix) : 1);
	backupname = g_new(char, len);

	g_snprintf(backupname, len, "%s/%s%s",
		   backupdir, my_basename(fname),
		   (prefs.backupsuffix) ? prefs.backupsuffix : NULL);
#endif
	if (dname)
		free(dname);	/* don't use g_free() */
	GNPDBG_FILE(("file_backup: backup file = '%s'\n", backupname));

	if (file_exist(backupname)) {
		if (unlink(backupname) == -1) {
			perror("file_backup: unlink() error");
			(void)do_dialog_error("unlink() failed!",
					      " Could not remove old backup! ");
			g_free(backupname);
			return;
		}
	}
	
	if (file_exist(fname) && rename(fname, backupname) == -1) {
		perror("file_backup: rename() error");
		(void)do_dialog_error("rename() failed!",
				      " Could not create backup! ");
	}

	g_free(backupname);
} /* file_backup */
#endif	/* USE_BACKUP */


/****** dotlocking is currently not supported.  eventually, it probably will
	be, which is why the code is still here *****/
#if defined(USE_DOTLOCK) && defined(FALSE)
/* HP-UX and ConvexOS don't have this macro */
#ifndef S_ISLNK
#define S_ISLNK(x) (((x) & S_IFMT) == S_IFLNK ? 1 : 0)
#endif

/* parameters: 
 * path - file to lock
 * retry - should retry if unable to lock?
 */
static int 
dotlock_file(const char *path, int retry)
{
	const char *pathptr = path;
	char lockfile[MAXPATH];
	char nfslockfile[MAXPATH];
	char realpath[MAXPATH];
	struct stat sb;
	size_t prev_size = 0;
	int count = 0;
	int attempt = 0;
	int fd;

#if 0
	/* if the file is a symlink, find the real file to which it refers */
	for (;;) {
		if (lstat(pathptr, &sb) != 0) {
			perror(pathptr);
			return -1;
		}

		if (S_ISLNK(sb.st_mode)) {
			char linkfile[MAXPATH];
			char linkpath[MAXPATH];

			if ((count = readlink(pathptr, linkfile,
				sizeof(linkfile))) == -1) {
				perror(path);
				return -1;
			}

			linkfile[count] = 0;	/* readlink() does not NUL terminate the string! */
			mutt_expand_link(linkpath, pathptr, linkfile);
			strfcpy(realpath, linkpath, sizeof(realpath));
			pathptr = realpath;
		} else
			break;
	}
#endif

	snprintf(nfslockfile, sizeof(nfslockfile),
		"%s.%s.%d", pathptr, Hostname, (int)getpid());
	snprintf(lockfile, sizeof(lockfile), "%s.lock", pathptr);
	unlink(nfslockfile);

	while ((fd = open(nfslockfile, O_WRONLY | O_EXCL | O_CREAT, 0)) < 0) {
		if (errno != EAGAIN) {
			mutt_perror( _("cannot open NFS lock file!"));
			return (-1);
		}
	}
	close(fd);

	count = 0;
	for (;;) {
		link(nfslockfile, lockfile);
		if (stat(nfslockfile, &sb) != 0) {
			perror("stat");
			return -1;
		}

		if (sb.st_nlink == 2)
			break;

		if (stat(path, &sb) != 0)
			sb.st_size = 0;

		if (count == 0)
			prev_size = sb.st_size;

		/* only try to remove the lock if the file is not changing */
		if (prev_size == sb.st_size && ++count >=
			(retry ? MAXLOCKATTEMPT : 0)) {
			if (retry && mutt_yesorno( _("Lock count exceeded, remove lock?"), 1) == 1) {
				unlink(lockfile);
				count = 0;
				attempt = 0;
				continue;
			} else
				return (-1);
		}
		prev_size = sb.st_size;

		printf( _("Waiting for lock attempt #%d..."), ++attempt);
		sleep(1);
	}

	unlink(nfslockfile);

	return 0;
} /* dotlock_file */


static int 
undotlock_file(const char *path)
{
	const char *pathptr = path;
	char lockfile[MAXPATH];
	char realpath[MAXPATH];
	struct stat sb;
	int n;

	FOREVER
	{
		dprint(2, (debugfile, "undotlock: unlocking %s\n", path));

		if (lstat(pathptr, &sb) != 0) {
			mutt_perror(pathptr);
			return (-1);
		}
		if (S_ISLNK(sb.st_mode)) {
			char linkfile[MAXPATH];
			char linkpath[MAXPATH];

			if ((n = readlink(pathptr, linkfile, sizeof(linkfile))) == -1) {
				mutt_perror(pathptr);
				return (-1);
			}
			linkfile[n] = 0;	/* readlink() does not NUL terminate the string! */
			mutt_expand_link(linkpath, pathptr, linkfile);
			strfcpy(realpath, linkpath, sizeof(realpath));
			pathptr = realpath;
			continue;
		} else
			break;
	}

	snprintf(lockfile, sizeof(lockfile), "%s.lock", pathptr);
	unlink(lockfile);
	return 0;
}
#endif /* USE_DOTLOCK */


/* the end */
