//  Gnomoradio - gnomoradio/playlist-view.cc
//  Copyright (C) 2003-2004  Jim Garrison, Matt Gerginski
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "main-window.h"
#include "playlist-view.h"
#include "browser-link.h"
#include "song-info.h"
#include <memory>
#include <sstream>

using namespace Gtk;
using namespace Glib;
using namespace sigc;
using namespace std;

Gnomoradio::PlaylistView::PlaylistView (Wraporadio::Init *wrap,
					Wraporadio::SongList list)
	: wr(wrap)
{
	set_shadow_type(SHADOW_ETCHED_IN);

	set(list);
	view.append_column(_("Status"), current_store->status_col);
	view.append_column(_("Title"), current_store->title_col);
	view.append_column(_("Artist"), current_store->artist_col);
	view.append_column(_("Album"), current_store->album_col);

	view.get_column(1)->set_resizable();
	view.get_column(2)->set_resizable();
	view.get_column(3)->set_resizable();

	view.get_column(1)->set_sizing(Gtk::TREE_VIEW_COLUMN_FIXED);
	view.get_column(2)->set_sizing(Gtk::TREE_VIEW_COLUMN_FIXED);
	view.get_column(3)->set_sizing(Gtk::TREE_VIEW_COLUMN_FIXED);

	static std::vector<TargetEntry> t;
	if (t.empty())
		t.push_back(Gnomoradio::MainWindow::song_target);
	view.enable_model_drag_source(t);
	if (current_store->get_song_list().get_mutable())
		view.enable_model_drag_dest(t, Gdk::ACTION_COPY);

	set_policy(POLICY_AUTOMATIC, POLICY_AUTOMATIC);
	add(view);

	view.signal_row_activated().connect(mem_fun(*this, &PlaylistView::on_double_clicked));
	view.signal_button_press_event().connect(mem_fun(*this, &PlaylistView::on_clicked), false);
	view.signal_drag_data_received().connect(mem_fun(*this, &PlaylistView::on_drag_data_received));
	view.signal_drag_data_get().connect(mem_fun(*this, &PlaylistView::on_drag_data_get));
}

Gnomoradio::PlaylistView::~PlaylistView ()
{
}

void Gnomoradio::PlaylistView::set (Wraporadio::SongList list)
{
	PlaylistStore *store;
	_base_list = list;
	if (list.get_shuffle_list().exists())
		store = PlaylistStore::lookup(list.get_shuffle_list());
	else
		store = PlaylistStore::lookup(list);

	view.set_model(store->store);
	current_store = store;
	signal_different_list(list);

	shuffle_con.disconnect();
	repeat_con.disconnect();
	shuffle_con = store->get_song_list().signal_shuffle_changed().connect(mem_fun(*this, &PlaylistView::on_shuffle_or_repeat_changed));
	repeat_con = store->get_song_list().signal_repeat_changed().connect(mem_fun(*this, &PlaylistView::on_shuffle_or_repeat_changed));
}

void Gnomoradio::PlaylistView::scroll_to_song (Wraporadio::SongList::iterator song)
{
	view.scroll_to_row(current_store->store->get_path(current_store->find_row_from_data(song)));
}

Wraporadio::SongList::iterator Gnomoradio::PlaylistView::get_selected_song ()
{
	return view.get_selection()->get_selected()->get_value(current_store->data_col);
}

void Gnomoradio::PlaylistView::on_double_clicked (const TreeModel::Path &p, TreeViewColumn *c)
{
	Wraporadio::SongList::iterator s = view.get_selection()->get_selected()->get_value(current_store->data_col);
	if ((*s)->get_status().ready)
		wr->player()->play(current_list().sl, s);
	else if ((*s)->importable())
		(*s)->import();
}

void Gnomoradio::PlaylistView::on_right_click_menu_play_activate (Wraporadio::SongList::iterator song)
{
	wr->player()->play(current_list().sl, song);
}

void Gnomoradio::PlaylistView::on_right_click_menu_import_activate (Wraporadio::SongList::iterator song)
{
	(*song)->import();
}

void Gnomoradio::PlaylistView::on_right_click_menu_refresh_activate (Wraporadio::SongList::iterator song)
{
	(*song)->obtain_available_info();
}

void Gnomoradio::PlaylistView::on_right_click_menu_rating_activate (Wraporadio::SongList::iterator song, int rating)
{
	(*song)->set_rating(rating);
}

void Gnomoradio::PlaylistView::on_right_click_menu_remove_activate (Wraporadio::SongList::iterator song)
{
	current_list().remove(song);
}

void Gnomoradio::PlaylistView::on_right_click_menu_properties_activate (Wraporadio::SongList::iterator song)
{
	(new SongInfoBox(*song))->show_all();
}

bool Gnomoradio::PlaylistView::on_clicked (GdkEventButton *event)
{
	if (event->type == GDK_BUTTON_PRESS
	    && event->button == 3) { // right mouse button
		int cell_x, cell_y;
		Gtk::TreeModel::Path path;
		Gtk::TreeViewColumn *column;
		view.get_path_at_pos(int(event->x), int(event->y), path, column, cell_x, cell_y);
		if (!path.gobj() || path.empty())
			return true;

		static auto_ptr<Gtk::Menu> menu;
		menu.reset(new Gtk::Menu);

		view.get_selection()->unselect_all();
		view.get_selection()->select(path, path);
		Wraporadio::SongList::iterator song = current_store->store->get_iter(path)->get_value(current_store->data_col);

		Gtk::MenuItem *play = manage(new MenuItem(_("Play")));
		if (!(*song)->get_status().ready)
			play->set_sensitive(false);
		play->signal_activate().connect(bind(mem_fun(*this, &PlaylistView::on_right_click_menu_play_activate), song));
		menu->append(*play);

		if ((*song)->importable()) {
			Gtk::MenuItem *import = manage(new MenuItem(_("Import")));
			if ((*song)->get_status().ready)
				import->set_sensitive(false);
			import->signal_activate().connect(bind(mem_fun(*this, &PlaylistView::on_right_click_menu_import_activate), song));
			menu->append(*import);

			Gtk::MenuItem *refresh = manage(new MenuItem(_("Refresh Info")));
			refresh->signal_activate().connect(bind(mem_fun(*this, &PlaylistView::on_right_click_menu_refresh_activate), song));
			menu->append(*refresh);
		}

		ustring artistsite_url = (*song)->get_info("artistsite");
		if (artistsite_url.size()) {
			Gtk::MenuItem *artistsite = manage(new MenuItem(_("Visit Artist's Web Site...")));
			artistsite->signal_activate().connect(bind(sigc::ptr_fun(&BrowserLink::open_url), artistsite_url));
			menu->append(*artistsite);
		}

		ustring license_url = (*song)->get_info("license");
		if (license_url.substr(0, 7) == "http://") {
			Gtk::MenuItem *license = manage(new MenuItem(_("View Song License...")));
			license->signal_activate().connect(bind(sigc::ptr_fun(&BrowserLink::open_url), license_url));
			menu->append(*license);
		}

		ustring retail_url = (*song)->get_info("retail");
		if (retail_url.size()) {
			Gtk::MenuItem *retail = manage(new MenuItem(_("Buy Album...")));
			retail->signal_activate().connect(bind(sigc::ptr_fun(&BrowserLink::open_url), retail_url));
			menu->append(*retail);
		}

		Gtk::MenuItem *rating = manage(new MenuItem(_("Rate")));
		Gtk::Menu *r_menu = manage(new Menu);
		for (int r = Roboradio::Song::max_rating; r >= Roboradio::Song::min_rating; --r) {
			ostringstream n;
			n << r;
			if (r == (*song)->get_rating())
				n << " *";
			Gtk::MenuItem *num = manage(new MenuItem(n.str()));
			num->signal_activate().connect(bind(mem_fun(*this, &PlaylistView::on_right_click_menu_rating_activate), song, r));
			r_menu->append(*num);
		}
		rating->set_submenu(*r_menu);
		menu->append(*rating);

		Gtk::MenuItem *remove = manage(new MenuItem(_("Remove")));
		remove->signal_activate().connect(bind(mem_fun(*this, &PlaylistView::on_right_click_menu_remove_activate), song));
		if (!current_list().get_mutable())
			remove->set_sensitive(false);
		menu->append(*remove);

		Gtk::MenuItem *prop = manage(new MenuItem(_("Properties...")));
		prop->signal_activate().connect(bind(mem_fun(*this, &PlaylistView::on_right_click_menu_properties_activate), song));
		menu->append(*prop);

		menu->show_all();
		menu->popup(event->button, event->time);

		return true;
	}

	return false; // I have no idea what this does
}

void Gnomoradio::PlaylistView::on_shuffle_or_repeat_changed (bool tmp)
{
	signal_different_list(current_store->get_song_list());
}

void Gnomoradio::PlaylistView::on_drag_data_received (const Glib::RefPtr<Gdk::DragContext> &context, int x, int y, const Gtk::SelectionData &selection_data, guint info, guint time)
{
	ustring url(selection_data.get_text());
	if (url == "")
		return;

	int cell_x, cell_y;
	TreeModel::Path path;
	TreeViewColumn *column;
	view.get_path_at_pos(x, y, path, column, cell_x, cell_y);

	Wraporadio::SongList::iterator current_song = get_selected_song();
	Wraporadio::SongList::iterator dest_song = current_store->get_song_list().end();
	if (path.gobj() && !path.empty())
		dest_song = current_store->store->get_iter(path)->get_value(current_store->data_col);

	if (&*current_song && (*current_song)->get_url() == url)
		current_store->get_song_list().move(current_song, dest_song);
	else
		current_store->get_song_list().insert(dest_song, Wraporadio::Song(url));
}

void Gnomoradio::PlaylistView::on_drag_data_get (const Glib::RefPtr<Gdk::DragContext> &context, Gtk::SelectionData &selection_data, guint info, guint time)
{
	ustring url((*view.get_selection()->get_selected()->get_value(current_store->data_col))->get_url());
	selection_data.set_text(url);
}
