/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2005 Imendio AB
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 *  lb-utils.c
 */

#include "config.h"

#include <string.h>
#include <stdlib.h>

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomeui/gnome-icon-lookup.h>
#include <gtk/gtkicontheme.h>

#include "lb-utils.h"

static GtkIconTheme *icon_theme = NULL;
static GnomeThumbnailFactory *thumbnail_factory = NULL;

static void
ensure_icon_theme (void)
{
	if (icon_theme)
		return;

	icon_theme = gtk_icon_theme_get_default ();
	thumbnail_factory = gnome_thumbnail_factory_new (GNOME_THUMBNAIL_SIZE_NORMAL);
}

gboolean
lb_string_has_substring (const gchar *str,
			 const gchar *substr)
{
	gchar    *norm_str, *norm_substr, *tmp;
	gboolean  retval;

	/* Case-fold and normalize */
	tmp = g_utf8_normalize (str, -1, G_NORMALIZE_ALL);
	norm_str = g_utf8_casefold (tmp, -1);
	g_free (tmp);

	tmp = g_utf8_normalize (substr, -1, G_NORMALIZE_ALL);
	norm_substr = g_utf8_casefold (tmp, -1);
	g_free (tmp);

	retval = strstr (norm_str, norm_substr) != NULL;

	g_free (norm_str);
	g_free (norm_substr);

	return retval;
}

gchar *
lb_string_markup_substring (const gchar *str,
			    const gchar *substr,
			    const gchar *tag)
{
	gchar *norm_str, *norm_substr;
	gchar *fold_str, *fold_substr;
	gchar *found;
	gchar *markup = NULL;

	if (! substr) {
		return g_strdup (str);
	}

	/* Case-fold and normalize */
	norm_str = g_utf8_normalize (str, -1, G_NORMALIZE_ALL);
	fold_str = g_utf8_casefold (norm_str, -1);

	norm_substr = g_utf8_normalize (substr, -1, G_NORMALIZE_ALL);
	fold_substr = g_utf8_casefold (norm_substr, -1);

	found = strstr (fold_str, fold_substr);

	g_free (fold_str);
	g_free (fold_substr);

	if (found) {
		gchar *tmp;
		gchar *pre;
		gchar *bold;
		gchar *post;

		tmp = g_strndup (norm_str, found - fold_str);
		pre = g_markup_escape_text (tmp, -1);
		g_free (tmp);

		tmp  = g_strndup (norm_str + (found - fold_str),
				  strlen (norm_substr));
		bold = g_markup_escape_text (tmp, -1);
		g_free (tmp);

		tmp  = g_strdup (norm_str + (found - fold_str)
				 + strlen (norm_substr));
		post = g_markup_escape_text (tmp, -1);
		g_free (tmp);


		markup = g_strconcat (pre,
				      "<", tag, ">", bold, "</", tag, ">",
				      post,
				      NULL);

		g_free (pre);
		g_free (bold);
		g_free (post);
	}

	g_free (norm_str);
	g_free (norm_substr);

	return markup;
}

gchar *
lb_get_icon_name_for_uri (const gchar      *uri,
			  GnomeVFSFileInfo *info)
{
	gchar                      *name;
	GnomeIconLookupResultFlags  result;
	GnomeVFSFileInfo           *real_info;

	ensure_icon_theme ();

	real_info = info;

	if (!info) {
		info = gnome_vfs_file_info_new ();

		if (gnome_vfs_get_file_info (uri, info,
					     GNOME_VFS_FILE_INFO_GET_MIME_TYPE) != GNOME_VFS_OK) {
			gnome_vfs_file_info_unref (info);
			return NULL;
		}
	}

	name = gnome_icon_lookup (icon_theme, thumbnail_factory,
				  uri, NULL, info, info->mime_type,
				  GNOME_ICON_LOOKUP_FLAGS_EMBEDDING_TEXT |
				  GNOME_ICON_LOOKUP_FLAGS_SHOW_SMALL_IMAGES_AS_THEMSELVES |
				  GNOME_ICON_LOOKUP_FLAGS_ALLOW_SVG_AS_THEMSELVES,
				  &result);

	if (real_info != info) {
		gnome_vfs_file_info_unref (info);
	}

	return name;
}

GdkPixbuf *
lb_get_pixbuf_from_icon_name (const gchar *icon_name)
{
	GdkPixbuf *pixbuf;
	gchar *tmp, *dot;
	GError *error = NULL;
	
	/* return null if we get a null or empty string */
        if (icon_name == NULL || icon_name[0] == '\0') {
                return NULL;
	}

	/* Handle absolute paths. */
	if (icon_name[0] == '/') {
		return gdk_pixbuf_new_from_file_at_size (icon_name,
							 64, 64, NULL);
	}

	ensure_icon_theme ();

	tmp = g_strdup (icon_name);
	dot = strrchr (tmp, '.');
	if (dot)
		*dot = '\0';

	pixbuf = gtk_icon_theme_load_icon (icon_theme,
					   tmp,
					   64,
					   GTK_ICON_LOOKUP_USE_BUILTIN,
					   &error);

	if (!pixbuf) {
		g_print ("Could not load icon \"%s\": %s\n", tmp, error->message);
		g_error_free (error);
	}

	g_free (tmp);


	return pixbuf;
}

gchar *
lb_get_firefox_path (void)
{
	gchar *profile_path;
	GKeyFile *key_file;
	GError *error = NULL;
	gchar **groups;
	gchar *default_group = NULL;
	gchar *tmp, *path = NULL;
	int i;

	profile_path = g_build_filename (g_get_home_dir (), ".mozilla/firefox/profiles.ini", NULL);

	key_file = g_key_file_new ();

	if (!g_key_file_load_from_file (key_file, profile_path, 0,
					&error)) {
		g_free (profile_path);

		g_warning ("Could not load profiles.ini: %s\n", error->message);
		return NULL;
	}

	/* Try to find the default group */
	groups = g_key_file_get_groups (key_file, NULL);
	for (i = 0; groups[i] != NULL; i++) {
		if (!g_str_has_prefix (groups[i], "Profile"))
			continue;

		if (!default_group)
			default_group = g_strdup (groups[i]);
		else {
			tmp = g_key_file_get_string (key_file, groups[i],
						     "Default", NULL);

			if (tmp && strcmp (tmp, "1") == 0) {
				g_free (default_group);
				default_group = g_strdup (groups[i]);
			}

			g_free (tmp);
		}
	}

	g_strfreev (groups);
	g_free (profile_path);

	if (!default_group) {
		g_key_file_free (key_file);
		return NULL;
	}

	tmp = g_key_file_get_string (key_file, default_group,
				     "Path", NULL);

	if (tmp) {
		path = g_build_filename (g_get_home_dir (),
					 ".mozilla/firefox",
					 tmp, NULL);
		g_free (tmp);
	}

	return path;
}

gboolean
lb_file_lock (gint fd)
{
	gint	i;

	/* Attempt to lock the file 5 times,
	 * waiting a random interval (< 1 second)
	 * in between attempts.
	 * We should really be doing asynchronous
	 * locking, but requires substantially larger
	 * changes.
	 */

	lseek (fd, 0, SEEK_SET);

	for (i = 0; i < 5; i++) {
		gint rand_interval;

		if (lockf (fd, F_TLOCK, 0) == 0)
			return TRUE;

		rand_interval = 1 + (gint) (10.0 * rand () / (RAND_MAX + 1.0));

		g_usleep (100000 * rand_interval);
	}

	return FALSE;
}

gboolean
lb_file_unlock (gint fd)
{
	lseek (fd, 0, SEEK_SET);

	return (lockf (fd, F_ULOCK, 0) == 0) ? TRUE : FALSE;
}
