/* 
   Copyright (C) 2004 Fernando Herrera <fherrera@onirica.com>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <gnome-keyring.h>

#include <string.h>
#include <stdlib.h>

#include "gnome-keyring-manager.h"
#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-util.h"
#include "gnome-keyring-manager-attribute-display.h"
#include "gnome-keyring-manager-attribute-editor.h"

enum
{
  ATTRIBUTE_COLUMN_NAME = 0,
  ATTRIBUTE_COLUMN_TYPE,
  ATTRIBUTE_COLUMN_VALUE,
  NUM_ATTRIBUTE_COLUMNS
};

enum
{
  AC_COLUMN_NAME = 0,
  AC_COLUMN_PATH,
  AC_COLUMN_READ,
  AC_COLUMN_WRITE,
  AC_COLUMN_REMOVE,
  NUM_AC_COLUMNS
};


struct _GKMAttributeEditorPrivate
{
  GtkWidget *attributes_tree_view;
  GtkWidget *attributes_remove_button;
  GtkWidget *acl_tree_view;
  GtkWidget *acl_remove_button;

  GtkListStore *item_attributes;
  GtkListStore *item_acl;

  char *keyring_name;
  guint item_id;

};

static void gkm_attribute_editor_class_init (GKMAttributeEditor *class);
static void gkm_attribute_editor_init       (GKMAttributeEditor *window);
static void gkm_attribute_editor_finalize   (GObject *object);
static void gkm_attribute_editor_destroy    (GtkObject *object);

static void gkm_attribute_editor_setup_item_attributes_tree (GKMAttributeEditor *editor);
static void gkm_attribute_editor_update_item_attributes (GKMAttributeEditor *editor);

static void attribute_name_changed_callback (GtkCellRendererText *cellrenderertext, gchar *path, gchar *value, GKMAttributeEditor *editor);
static void attribute_type_changed_callback (GtkCellRendererText *cellrenderertext, gchar *path, gchar *value, GKMAttributeEditor *editor);
static void attribute_value_changed_callback (GtkCellRendererText *cellrenderertext, gchar *path, gchar *value, GKMAttributeEditor *editor);
static void add_item_attribute_callback (GtkWidget *widget, GKMAttributeEditor *editor);
static void remove_item_attribute_callback (GtkWidget *widget, GKMAttributeEditor *editor);
static void attribute_tree_selection_changed (GtkTreeSelection *selection, GKMAttributeEditor *editor);

static void gkm_attribute_editor_setup_item_acl_tree (GKMAttributeEditor *editor);
static void gkm_attribute_editor_update_item_acl (GKMAttributeEditor *editor);
static void ac_read_toggled_callback (GtkCellRendererToggle *cellrenderertoggle, gchar *path, GKMAttributeEditor *editor);
static void ac_write_toggled_callback (GtkCellRendererToggle *cellrenderertoggle, gchar *path, GKMAttributeEditor *editor);
static void ac_remove_toggled_callback (GtkCellRendererToggle *cellrenderertoggle, gchar *path, GKMAttributeEditor *editor);
static void remove_item_ac_callback (GtkWidget *widget G_GNUC_UNUSED, GKMAttributeEditor *editor);
static void ac_tree_selection_changed (GtkTreeSelection *selection, GKMAttributeEditor *editor);

static GtkWindowClass *parent_class = NULL;

GType
gkm_attribute_editor_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (GKMAttributeEditorClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
        (GClassInitFunc) gkm_attribute_editor_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */ 
	sizeof (GKMAttributeEditor),
	0,		/* n_preallocs */
	(GInstanceInitFunc) gkm_attribute_editor_init,
	0
      };

      type = g_type_register_static (GTK_TYPE_WINDOW, "GKMAttributeEditor", &info, 0);
    }

  return type;
}

static void
gkm_attribute_editor_class_init (GKMAttributeEditor *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  gobject_class->finalize = gkm_attribute_editor_finalize;

  object_class->destroy = gkm_attribute_editor_destroy;

}

static void
gkm_attribute_editor_init (GKMAttributeEditor *editor)
{
  GtkWidget *vpaned;
  GtkWidget *attr_vbox;
  GtkWidget *acl_vbox;
  GtkWidget *hbox;
  GtkWidget *vbuttonbox;
  GtkWidget *label;
  GtkWidget *scrolledwindow;
  GtkSizeGroup *button_sizegroup;
  GtkWidget *button;
  GtkTreeSelection *acl_tree_selection;
  GtkTreeSelection *attr_tree_selection;
  gchar *text;

  editor->priv = g_new0 (GKMAttributeEditorPrivate, 1);

  gtk_container_set_border_width (GTK_CONTAINER (editor), 6);

  gtk_window_set_default_size (GTK_WINDOW (editor), 450, 400);

  button_sizegroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

  vpaned = gtk_vpaned_new ();
  gtk_paned_set_position (GTK_PANED (vpaned), 200);
  gtk_container_add (GTK_CONTAINER (editor), vpaned);

  attr_vbox = gtk_vbox_new (FALSE, 12);
  gtk_paned_pack1 (GTK_PANED (vpaned), attr_vbox, FALSE, FALSE);
  
  label = gtk_label_new (NULL);
  text = g_strdup_printf ("<span weight='bold'>%s</span>", _("Attributes"));
  gtk_label_set_markup (GTK_LABEL (label), text);
  g_free (text);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (attr_vbox), label, FALSE, FALSE, 0);
  
  hbox = gtk_hbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX (attr_vbox), hbox, TRUE, TRUE, 0);
  
  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_SHADOW_ETCHED_IN);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (hbox), scrolledwindow, TRUE, TRUE, 0);
 
  editor->priv->attributes_tree_view = gtk_tree_view_new ();
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (editor->priv->attributes_tree_view), TRUE);
  gtk_container_add (GTK_CONTAINER (scrolledwindow), editor->priv->attributes_tree_view);
  g_object_set_data (G_OBJECT (editor), "item-attributes", editor->priv->attributes_tree_view);

  attr_tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->priv->attributes_tree_view));
  g_signal_connect( attr_tree_selection, "changed",
                    G_CALLBACK (attribute_tree_selection_changed), editor);
  
  gkm_attribute_editor_setup_item_attributes_tree (editor);

  vbuttonbox = gtk_vbutton_box_new ();
  gtk_box_set_spacing (GTK_BOX (vbuttonbox), 12);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (vbuttonbox), GTK_BUTTONBOX_START);
  gtk_box_pack_start (GTK_BOX (hbox), vbuttonbox, FALSE, FALSE, 0);

  button = gtk_button_new_from_stock (GTK_STOCK_ADD);
  gtk_size_group_add_widget (button_sizegroup, button);
  gtk_box_pack_start (GTK_BOX (vbuttonbox), button, FALSE, FALSE, 0);
  g_signal_connect (button, "clicked", G_CALLBACK (add_item_attribute_callback), editor);

  editor->priv->attributes_remove_button = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
  gtk_size_group_add_widget (button_sizegroup, editor->priv->attributes_remove_button);
  gtk_box_pack_start (GTK_BOX (vbuttonbox), editor->priv->attributes_remove_button, FALSE, FALSE, 0);
  g_signal_connect (editor->priv->attributes_remove_button, "clicked", G_CALLBACK (remove_item_attribute_callback), editor);
  gtk_widget_set_sensitive( editor->priv->attributes_remove_button, FALSE);

  acl_vbox = gtk_vbox_new (FALSE, 12);
  gtk_paned_pack2 (GTK_PANED (vpaned), acl_vbox, FALSE, FALSE);

  label = gtk_label_new (NULL);
  text = g_strdup_printf ("<span weight='bold'>%s</span>", _("Application Access Rights"));
  gtk_label_set_markup (GTK_LABEL (label), text);
  g_free (text);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (acl_vbox), label, FALSE, FALSE, 0);
  
  hbox = gtk_hbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX (acl_vbox), hbox, TRUE, TRUE, 0);
  
  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_SHADOW_ETCHED_IN);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (hbox), scrolledwindow, TRUE, TRUE, 0);
 
  editor->priv->acl_tree_view = gtk_tree_view_new ();
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (editor->priv->acl_tree_view), TRUE);
  gtk_container_add (GTK_CONTAINER (scrolledwindow), editor->priv->acl_tree_view);
  g_object_set_data (G_OBJECT (editor), "item-acl", editor->priv->acl_tree_view);

  acl_tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->priv->acl_tree_view));
  g_signal_connect( acl_tree_selection, "changed",
                    G_CALLBACK (ac_tree_selection_changed), editor);
  
  gkm_attribute_editor_setup_item_acl_tree (editor);

  vbuttonbox = gtk_vbutton_box_new ();
  gtk_box_set_spacing (GTK_BOX (vbuttonbox), 12);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (vbuttonbox), GTK_BUTTONBOX_START);
  gtk_box_pack_start (GTK_BOX (hbox), vbuttonbox, FALSE, FALSE, 0);

  editor->priv->acl_remove_button = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
  gtk_size_group_add_widget (button_sizegroup, editor->priv->acl_remove_button);
  gtk_box_pack_start (GTK_BOX (vbuttonbox), editor->priv->acl_remove_button, FALSE, FALSE, 0);
  g_signal_connect (editor->priv->acl_remove_button, "clicked", G_CALLBACK (remove_item_ac_callback), editor);
  gtk_widget_set_sensitive( editor->priv->acl_remove_button, FALSE);


  gtk_widget_show_all (vpaned);
}

static void
gkm_attribute_editor_finalize (GObject *object)
{
  GKMAttributeEditor *editor;

  g_return_if_fail (GKM_IS_ATTRIBUTE_EDITOR (object));

  editor = GKM_ATTRIBUTE_EDITOR (object);

  g_free (editor->priv->keyring_name);

  g_free (editor->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gkm_attribute_editor_destroy (GtkObject *object)
{
  GKMAttributeEditor *editor;

  g_return_if_fail (GKM_IS_ATTRIBUTE_EDITOR (object));

  editor = GKM_ATTRIBUTE_EDITOR (object);

  GTK_OBJECT_CLASS (parent_class)->destroy (GTK_OBJECT (editor));
}

GtkWidget *
gkm_attribute_editor_new (const char  *keyring_name,
			  guint        item_id,
			  const gchar *item_name)
{
  GKMAttributeEditor *editor;
  gchar *window_title;

  editor = g_object_new (GKM_TYPE_ATTRIBUTE_EDITOR, NULL);

  window_title = g_strdup_printf (_("Advanced Editor: %s"), item_name);
  gtk_window_set_title (GTK_WINDOW (editor), window_title);

  editor->priv->keyring_name = g_strdup (keyring_name);
  editor->priv->item_id = item_id;
  editor->attributes_modified = FALSE;
  editor->acl_modified = FALSE;
  
  gkm_attribute_editor_update_item_attributes (GKM_ATTRIBUTE_EDITOR (editor));
  gkm_attribute_editor_update_item_acl (GKM_ATTRIBUTE_EDITOR (editor));
  
  return GTK_WIDGET (editor);
}

static void
gkm_attribute_editor_setup_item_attributes_tree (GKMAttributeEditor *editor)
{
  GtkTreeViewColumn   *column;
  GtkCellRenderer     *renderer;
  
  g_return_if_fail (GKM_IS_ATTRIBUTE_EDITOR (editor));
  g_assert (editor->priv->item_attributes == NULL);
  
  editor->priv->item_attributes = gtk_list_store_new (NUM_ATTRIBUTE_COLUMNS, 
                                                      /* ATTRIBUTE_COLUMN_NAME  */   G_TYPE_STRING, 
                                                      /* ATTRIBUTE_COLUMN_TYPE  */   G_TYPE_STRING, 
                                                      /* ATTRIBUTE_COLUMN_VALUE */   G_TYPE_STRING);
  
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Name"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->attributes_tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", ATTRIBUTE_COLUMN_NAME);
  gtk_tree_view_column_set_sort_column_id (column, ATTRIBUTE_COLUMN_NAME);
  g_object_set (G_OBJECT (renderer), "editable", TRUE, NULL);
  g_signal_connect (renderer, "edited", G_CALLBACK (attribute_name_changed_callback), editor);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Type"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->attributes_tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", ATTRIBUTE_COLUMN_TYPE);
  g_object_set (G_OBJECT (renderer), "editable", TRUE, NULL);
  g_signal_connect (renderer, "edited", G_CALLBACK (attribute_type_changed_callback), editor);
  /*gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (editor->priv->keyring_items), ITEM_COLUMN_TYPE, 
                                   (GtkTreeIterCompareFunc) compare_keyring_item_types, GINT_TO_POINTER (ITEM_COLUMN_TYPE), NULL);
*/
  gtk_tree_view_column_set_sort_column_id (column, ATTRIBUTE_COLUMN_TYPE);

 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Value"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->attributes_tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", ATTRIBUTE_COLUMN_VALUE);
  gtk_tree_view_column_set_sort_column_id (column, ATTRIBUTE_COLUMN_VALUE);
  g_object_set (G_OBJECT (renderer), "editable", TRUE, NULL);
  g_signal_connect (renderer, "edited", G_CALLBACK (attribute_value_changed_callback), editor);
  gtk_tree_view_column_set_visible (column, TRUE);

  gtk_tree_view_set_model (GTK_TREE_VIEW (editor->priv->attributes_tree_view), GTK_TREE_MODEL (editor->priv->item_attributes));
  g_object_unref (G_OBJECT (editor->priv->item_attributes));
  gtk_window_set_focus (GTK_WINDOW (editor), NULL);
}


static void
gkm_attribute_editor_setup_item_acl_tree (GKMAttributeEditor *editor)
{
  GtkTreeViewColumn   *column;
  GtkCellRenderer     *renderer;
  
  g_return_if_fail (GKM_IS_ATTRIBUTE_EDITOR (editor));
  g_assert (editor->priv->item_acl == NULL);
  
  editor->priv->item_acl = gtk_list_store_new (NUM_AC_COLUMNS, 
                                               /* AC_COLUMN_NAME  */    G_TYPE_STRING, 
                                               /* AC_COLUMN_PATH */     G_TYPE_STRING,
                                               /* AC_COLUMN_READ */     G_TYPE_BOOLEAN,
                                               /* AC_COLUMN_WRITE */    G_TYPE_BOOLEAN,
                                               /* AC_COLUMN_REMOVE */   G_TYPE_BOOLEAN);
  
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Name"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->acl_tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", AC_COLUMN_NAME);
  gtk_tree_view_column_set_sort_column_id (column, AC_COLUMN_NAME);
  g_object_set (G_OBJECT (renderer), "editable", FALSE, NULL);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Path"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->acl_tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", AC_COLUMN_PATH);
  gtk_tree_view_column_set_sort_column_id (column, AC_COLUMN_PATH);
  g_object_set (G_OBJECT (renderer), "editable", FALSE, NULL);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Read"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->acl_tree_view), column);
  renderer = gtk_cell_renderer_toggle_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "active", AC_COLUMN_READ);
  g_signal_connect (renderer, "toggled", G_CALLBACK (ac_read_toggled_callback), editor);
  gtk_tree_view_column_set_sort_column_id (column, AC_COLUMN_READ);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Write"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->acl_tree_view), column);
  renderer = gtk_cell_renderer_toggle_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "active", AC_COLUMN_WRITE);
  g_signal_connect (renderer, "toggled", G_CALLBACK (ac_write_toggled_callback), editor);
  gtk_tree_view_column_set_sort_column_id (column, AC_COLUMN_WRITE);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Remove"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->acl_tree_view), column);
  renderer = gtk_cell_renderer_toggle_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "active", AC_COLUMN_REMOVE);
  g_signal_connect (renderer, "toggled", G_CALLBACK (ac_remove_toggled_callback), editor);
  gtk_tree_view_column_set_sort_column_id (column, AC_COLUMN_REMOVE);

  gtk_tree_view_set_model (GTK_TREE_VIEW (editor->priv->acl_tree_view), GTK_TREE_MODEL (editor->priv->item_acl));
  g_object_unref (G_OBJECT (editor->priv->item_acl));
  gtk_window_set_focus (GTK_WINDOW (editor), NULL);
}



/***************************************************************************
 * Fill these in.
 */


static void
attribute_name_changed_callback (GtkCellRendererText *cellrenderertext G_GNUC_UNUSED, 
				 gchar *path_str, gchar *new_value,
				 GKMAttributeEditor *editor)
{
  GtkTreeIter iter;
  GtkTreePath *path;

  path = gtk_tree_path_new_from_string (path_str);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (editor->priv->item_attributes), &iter, path);
  gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_NAME, new_value, -1);

  editor->attributes_modified = TRUE;

  gtk_tree_path_free (path);


}

static void
attribute_type_changed_callback (GtkCellRendererText *cellrenderertext G_GNUC_UNUSED,
				 gchar *path_str, gchar *new_value,
				 GKMAttributeEditor *editor)
{
  GtkTreeIter iter;
  GtkTreePath *path;
  gchar *type;

  path = gtk_tree_path_new_from_string (path_str);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (editor->priv->item_attributes), &iter, path);
  gtk_tree_model_get (GTK_TREE_MODEL (editor->priv->item_attributes), &iter, ATTRIBUTE_COLUMN_TYPE, &type, -1);
  if (strcmp (new_value, "string") == 0)
    {
      if (strcmp (new_value, type) != 0)
        {
          gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_TYPE, "string", -1);
          gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_VALUE, "", -1);
          editor->attributes_modified = TRUE;
        }
    }
  else if (strcmp (new_value, "integer") == 0)
    {
      if (strcmp (new_value, type) != 0)
        {
          gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_TYPE, "integer", -1);
          gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_VALUE, "0", -1);
          editor->attributes_modified = TRUE;
        }
    }
  g_free (type);
  gtk_tree_path_free (path);
}



static void
attribute_value_changed_callback (GtkCellRendererText *cellrenderertext G_GNUC_UNUSED,
				  gchar *path_str, gchar *new_value,
				  GKMAttributeEditor *editor)
{
  GtkTreeIter iter;
  GtkTreePath *path;
  gchar *type;

  path = gtk_tree_path_new_from_string (path_str);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (editor->priv->item_attributes), &iter, path);
  gtk_tree_model_get (GTK_TREE_MODEL (editor->priv->item_attributes), &iter, ATTRIBUTE_COLUMN_TYPE, &type, -1);
  if (strcmp (type, "string") == 0)
    {
      gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_VALUE, new_value, -1);
    }
  else if (strcmp (type, "integer") == 0)
    {
      guint32 integer;
      gchar *int_str;
      integer = (guint32)g_ascii_strtod (new_value, NULL);
      int_str = g_strdup_printf ("%d", integer);
      gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_VALUE, int_str, -1);
      g_free (int_str);
    }

  editor->attributes_modified = TRUE;
  g_free (type);
  gtk_tree_path_free (path);
}

static void
add_item_attribute_callback (GtkWidget *widget G_GNUC_UNUSED, GKMAttributeEditor *editor)
{
  GtkTreeIter iter;
  gtk_list_store_append (editor->priv->item_attributes, &iter);
  gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_NAME, "New Attibute", -1);
  gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_TYPE, "string", -1);
}

static void
remove_item_attribute_callback (GtkWidget *widget G_GNUC_UNUSED, GKMAttributeEditor *editor)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->priv->attributes_tree_view));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      if (gtk_list_store_remove (editor->priv->item_attributes, &iter))
	{
	  gtk_tree_selection_select_iter (selection, &iter);
	}
      else
	{
	  if (gtk_tree_model_get_iter_first (model, &iter))
	    {
	      gtk_tree_selection_select_iter (selection, &iter);
	    }
	}
      editor->attributes_modified = TRUE;
    }
}

static void
change_displayed_attributes_callback (GnomeKeyringResult             result,
				      GnomeKeyringAttributeList     *attributes,
				      GKMAttributeEditor *editor)
{
  unsigned int i;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (editor), result);
      return;
    }
  gtk_list_store_clear (editor->priv->item_attributes);
  for (i = 0; i < attributes->len; i++)
    {
      GnomeKeyringAttribute attribute;
      GtkTreeIter iter;

      attribute = gnome_keyring_attribute_list_index (attributes, i);


      gtk_list_store_append (editor->priv->item_attributes, &iter);
      gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_NAME, attribute.name, -1);
      if (attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
        {
          gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_TYPE, "string", -1);
          gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_VALUE, attribute.value.string, -1);
        }
      else
        {
          char *value = g_strdup_printf ("%d", attribute.value.integer);
          gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_TYPE, "integer", -1);
          gtk_list_store_set (editor->priv->item_attributes, &iter, ATTRIBUTE_COLUMN_VALUE, value, -1);
	  g_free (value);
        }
    }
  //FIXME: Attributes FREE!
}

static void 
attribute_tree_selection_changed (GtkTreeSelection *selection, 
                                  GKMAttributeEditor *editor)
{
  GtkTreeModel *model;
  GtkTreeIter  iter;

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gtk_widget_set_sensitive (editor->priv->attributes_remove_button, TRUE);
    }
  else
    {
      gtk_widget_set_sensitive (editor->priv->attributes_remove_button, FALSE);
    }
}

static void 
ac_tree_selection_changed (GtkTreeSelection *selection, 
                           GKMAttributeEditor *editor)
{
  GtkTreeModel *model;
  GtkTreeIter  iter;

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gtk_widget_set_sensitive (editor->priv->acl_remove_button, TRUE);
    }
  else
    {
      gtk_widget_set_sensitive (editor->priv->acl_remove_button, FALSE);
    }
}

static void
ac_read_toggled_callback (GtkCellRendererToggle *cellrenderertoggle G_GNUC_UNUSED,
			  gchar 		*path_str,
			  GKMAttributeEditor    *editor)
{
  GtkTreeModel *model = (GtkTreeModel *)editor->priv->item_acl;
  GtkTreeIter  iter;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
  gboolean active;
  
  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter, AC_COLUMN_READ, &active, -1);

  gtk_list_store_set (GTK_LIST_STORE (model), &iter, AC_COLUMN_READ, !active, -1);
  editor->acl_modified = TRUE;

  gtk_tree_path_free (path);
}

static void
ac_write_toggled_callback (GtkCellRendererToggle *cellrenderertoggle G_GNUC_UNUSED,
			   gchar 	 	 *path_str,
			   GKMAttributeEditor    *editor)
{
  GtkTreeModel *model = (GtkTreeModel *)editor->priv->item_acl;
  GtkTreeIter  iter;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
  gboolean active;
  
  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter, AC_COLUMN_WRITE, &active, -1);

  gtk_list_store_set (GTK_LIST_STORE (model), &iter, AC_COLUMN_WRITE, !active, -1);
  editor->acl_modified = TRUE;

  gtk_tree_path_free (path);
}

static void
ac_remove_toggled_callback (GtkCellRendererToggle *cellrenderertoggle G_GNUC_UNUSED,
			    gchar 		  *path_str,
			    GKMAttributeEditor    *editor)
{
  GtkTreeModel *model = (GtkTreeModel *)editor->priv->item_acl;
  GtkTreeIter  iter;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
  gboolean active;
  
  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter, AC_COLUMN_REMOVE, &active, -1);

  gtk_list_store_set (GTK_LIST_STORE (model), &iter, AC_COLUMN_REMOVE, !active, -1);
  editor->acl_modified = TRUE;

  gtk_tree_path_free (path);
}

static void
remove_item_ac_callback (GtkWidget *widget G_GNUC_UNUSED, GKMAttributeEditor *editor)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->priv->acl_tree_view));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      if (gtk_list_store_remove (editor->priv->item_acl, &iter))
	{
	  gtk_tree_selection_select_iter (selection, &iter);
	}
      else
	{
	  if (gtk_tree_model_get_iter_first (model, &iter))
	    {
	      gtk_tree_selection_select_iter (selection, &iter);
	    }
	}
      editor->acl_modified = TRUE;
    }
}

static void
gkm_attribute_editor_update_item_attributes (GKMAttributeEditor *editor)
{
  gnome_keyring_item_get_attributes (editor->priv->keyring_name,
				     editor->priv->item_id,
				     (GnomeKeyringOperationGetAttributesCallback) change_displayed_attributes_callback,
				     editor, NULL);
}

static gboolean
add_attribute_to_list (GtkTreeModel              *model,
		       GtkTreePath               *path G_GNUC_UNUSED,
		       GtkTreeIter               *iter,
		       GnomeKeyringAttributeList *attributes)
{
  char *name, *type, *str_value;

  gtk_tree_model_get (model, iter,
		      ATTRIBUTE_COLUMN_NAME, &name,
		      ATTRIBUTE_COLUMN_TYPE, &type,
		      ATTRIBUTE_COLUMN_VALUE, &str_value,
		      -1);
  if (strcmp (type, "string") == 0)
    {
      gnome_keyring_attribute_list_append_string (attributes, name, str_value);
    }
  else if (strcmp (type, "integer") == 0)
    {
      guint32 int_value = (guint32)g_ascii_strtod (str_value, NULL);
      gnome_keyring_attribute_list_append_uint32 (attributes, name, int_value);
    }
  
  return FALSE;
}

GnomeKeyringAttributeList*
gkm_attribute_editor_get_attribute_list (GKMAttributeEditor *editor)
{
  GnomeKeyringAttributeList *attribute_list;

  attribute_list = gnome_keyring_attribute_list_new ();
  gtk_tree_model_foreach (GTK_TREE_MODEL (editor->priv->item_attributes), 
			  (GtkTreeModelForeachFunc)add_attribute_to_list, 
			  attribute_list);

  return attribute_list;
}

static gboolean
add_ac_to_list (GtkTreeModel              *model,
		GtkTreePath               *treepath G_GNUC_UNUSED,
		GtkTreeIter               *iter,
		GList 			  **acl)
{
  char *name, *path;
  gboolean read, write, remove;
  GnomeKeyringApplicationRef *ref;
  GnomeKeyringAccessControl *ac;
  GnomeKeyringAccessType perms = 0;

  gtk_tree_model_get (model, iter,
		      AC_COLUMN_NAME, &name,
		      AC_COLUMN_PATH, &path,
		      AC_COLUMN_READ, &read,
		      AC_COLUMN_WRITE, &write,
		      AC_COLUMN_REMOVE, &remove,
		      -1);
  ref = gnome_keyring_application_ref_new ();
  if (read)
    {
      perms |= GNOME_KEYRING_ACCESS_READ;
    }
  if (write)
    {
      perms |= GNOME_KEYRING_ACCESS_WRITE;
    }
  if (remove)
    {
      perms |= GNOME_KEYRING_ACCESS_REMOVE;
    }
 
  ac = gnome_keyring_access_control_new (ref, perms);
  gnome_keyring_item_ac_set_display_name (ac, name);
  gnome_keyring_item_ac_set_path_name (ac, path);

  *acl = g_list_prepend (*acl, ac);

  return FALSE;
}

GList*
gkm_attribute_editor_get_acl (GKMAttributeEditor *editor)
{
  GList *acl = NULL;

  gtk_tree_model_foreach (GTK_TREE_MODEL (editor->priv->item_acl), 
			  (GtkTreeModelForeachFunc)add_ac_to_list, 
			  &acl);

  return acl;
}


static void
change_displayed_acl_callback (GnomeKeyringResult result,
			       GList             *acl,
			       GKMAttributeEditor *editor)
{
  GList *tmp;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (editor), result);
      return;
    }
  gtk_list_store_clear (editor->priv->item_acl);
  for (tmp = acl; tmp; tmp = tmp->next)
    {
      GnomeKeyringAccessControl *ac;
      GnomeKeyringAccessType at;
      GtkTreeIter iter;
      char *name, *path;

      ac = (GnomeKeyringAccessControl*) tmp->data;
      name = gnome_keyring_item_ac_get_display_name (ac);
      path = gnome_keyring_item_ac_get_path_name (ac);

      gtk_list_store_append (editor->priv->item_acl, &iter);
      gtk_list_store_set (editor->priv->item_acl, &iter, AC_COLUMN_NAME, name, -1);
      gtk_list_store_set (editor->priv->item_acl, &iter, AC_COLUMN_PATH, path, -1);
      g_free (name);
      g_free (path);

      at = gnome_keyring_item_ac_get_access_type (ac);
      gtk_list_store_set (editor->priv->item_acl, &iter, AC_COLUMN_READ, (at & GNOME_KEYRING_ACCESS_READ), -1);
      gtk_list_store_set (editor->priv->item_acl, &iter, AC_COLUMN_WRITE, (at & GNOME_KEYRING_ACCESS_WRITE), -1);
      gtk_list_store_set (editor->priv->item_acl, &iter, AC_COLUMN_REMOVE, (at & GNOME_KEYRING_ACCESS_REMOVE), -1);

    }
  /* FIXME: free acl! */
}



static void
gkm_attribute_editor_update_item_acl (GKMAttributeEditor *editor)
{
  gnome_keyring_item_get_acl (editor->priv->keyring_name,
			      editor->priv->item_id,
			      (GnomeKeyringOperationGetListCallback) change_displayed_acl_callback,
			      editor, NULL);
}


