/**
 * gnome-gmail-notifier: the gnome gmail notifier.
 * Copyright (C) 2007 Bradley A. Worley.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * 
 *     Free Software Foundation, Inc.
 *     59 Temple Place, Suite 330
 *     Boston, MA 02111-1307 USA
 **/

/*
 * include our application header.
 */
#include <main.h>

/*
 * private object definition.
 */
struct _GgnAboutDialogPrivate {
    /* our dialog object. */
    GladeXML* xml;
    GtkAboutDialog* dlg;
};

/*
 * forward function definitions.
 */
static void ggn_about_dialog_init (GgnAboutDialog* self);
static void ggn_about_dialog_class_init (GgnAboutDialogClass* klass);
static void ggn_about_dialog_finalize (GObject* obj);

/*
 * define the gobject type and its basic functions.
 */
G_DEFINE_TYPE (GgnAboutDialog, ggn_about_dialog, G_TYPE_OBJECT);

/*
 * ggn_about_dialog_link:
 *
 * Callback function that is executed when the user clicks a
 * hyperlink inside the about dialog.
 *
 * Return value: void.
 */
static void ggn_about_dialog_link (GtkAboutDialog* about,
                                   const gchar* link,
                                   gpointer data) {
    /* the user clicked a link! */
    ggn_client_execute (link);
}

/*
 * ggn_about_dialog_init:
 *
 * This function is used by the gobject library to
 * generate a new instance of our object.
 */
static void ggn_about_dialog_init (GgnAboutDialog* self) {
    /* set up the private data structure. */
    self->priv = g_new0 (GgnAboutDialogPrivate, 1);
    
    /* setup the about dialog hyperlink handler. */
    gtk_about_dialog_set_url_hook (ggn_about_dialog_link, self, NULL);
    gtk_about_dialog_set_email_hook (ggn_about_dialog_link, self, NULL);
    
    /* locate the glade xml file. */
    gchar* filename = ggn_glade_file (GGN_GLADE_ABOUT);
    
    /* make the dialog. */
    self->priv->xml = glade_xml_new (filename, NULL, NULL);
    self->priv->dlg = (GtkAboutDialog*) glade_xml_get_widget (self->priv->xml,
                                                              "GgnAboutDialog");
    
    /* link the response callback to our dialog. */
    g_signal_connect (G_OBJECT (self->priv->dlg),
                      "response",
                      G_CALLBACK (gtk_widget_hide),
                      self);
    
    /* free the file string. */
    g_free (filename);
}

/*
 * ggn_about_dialog_class_init:
 *
 * This function is used by the gobject library to
 * generate a new class object of our object.
 */
static void ggn_about_dialog_class_init (GgnAboutDialogClass* klass) {
    /* setup a gobject class. */
    GObjectClass* gobj_class = G_OBJECT_CLASS (klass);
    
    /* set the locations of our destruction function. */
    gobj_class->finalize = ggn_about_dialog_finalize;
    
    /* setup the default signal handlers. */
}

/*
 * ggn_about_dialog_finalize:
 *
 * This function is used by the gobject library to cleanly finish
 * the destruction process started by the dispose function.
 */
static void ggn_about_dialog_finalize (GObject* obj) {
    /* make a reference to ourself. */
    GgnAboutDialog* self = GGN_ABOUT_DIALOG (obj);
    
    /* free the glade xml and the dialog. */
    g_object_unref (self->priv->xml);
    gtk_widget_destroy (GTK_WIDGET (self->priv->dlg));
    
    /* destroy the private object. */
    g_free (self->priv);
    self->priv = NULL;
    
    /* chain up to the parent class. */
    G_OBJECT_CLASS (ggn_about_dialog_parent_class)->finalize (obj);
}

/*
 * ggn_about_dialog_new:
 *
 * Creates a new GgnPresence object for use in the application.
 * It maintains a presence on the panel, with sound effects,
 * and with libnotify notifications.
 *
 * Return value: the new about dialog object.
 */
GgnAboutDialog* ggn_about_dialog_new (void) {
    /* make a newly created gobject. */
    GgnAboutDialog* dlg = g_object_new (GGN_TYPE_ABOUT_DIALOG, NULL);
    
    /* return the new object. */
    return dlg;
}

/*
 * ggn_about_dialog_free:
 *
 * Frees the given about dialog by decreasing its reference count.
 *
 * Return value: void.
 */
void ggn_about_dialog_free (GgnAboutDialog* dlg) {
    /* unreference the object. */
    while (G_IS_OBJECT (dlg)) {
        /* unreference this object. */
        g_object_unref (G_OBJECT (dlg));
    }
}

/*
 * ggn_about_dialog_show:
 *
 * Makes the provided about dialog visible.
 *
 * Return value: void.
 */
void ggn_about_dialog_show (GgnAboutDialog* dlg) {
    /* show the widget. */
    gtk_widget_show (GTK_WIDGET (dlg->priv->dlg));
}

/*
 * ggn_about_dialog_hide:
 *
 * Makes the provided about dialog invisible.
 *
 * Return value: void.
 */
void ggn_about_dialog_hide (GgnAboutDialog* dlg) {
    /* hide the widget. */
    gtk_widget_hide (GTK_WIDGET (dlg->priv->dlg));
}
