/* vi:set ts=8 sts=0 sw=8:
 * $Id: results.c,v 1.5 2001/01/26 08:35:38 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include "win.h"
#include "main.h"
#include "misc.h"
#include "file.h"
#include "msgbar.h"
#include "prefs.h"
#include "search.h"
#include "results.h"
#include "dir.xpm"
#include "file.xpm"


/*** PRIVATE FUNCTION PROTOTYPES ***/
static void results_ok_cb(GtkWidget *wgt, gpointer cbdata);
static void results_err_cb(GtkWidget *wgt, gpointer cbdata);
static void results_clist_cb(GtkCTree *ctree, GtkCTreeNode *node, int column,
			     gpointer cbdata);
static gboolean results_key_press(GtkWidget *wgt, GdkEventKey *ev, win_t *w);


enum ef_results_id_e {
	Ef_results_dialog,
	Ef_results_clist,	/* clist */
	Ef_results_scrwin,	/* scrolled window */
	Ef_results_detailed,	/* clist */
	Ef_results_scrwin_detailed,/* scrolled window */
	Ef_results_errors,	/* clist */
	Ef_results_cont_errors,	/* scrolled window */
	Ef_results_num_found,	/* gnome app bar */
	Ef_results_toolbar,	/* toolbar */

	Ef_results_ok,		/* button */
	Ef_results_err,		/* button */

	Ef_results_open,	/* button */
	Ef_results_delete,	/* button */
	Ef_results_copy,	/* button */
	Ef_results_print,	/* button */
	Ef_results_execute,	/* button */
	Ef_results_shell,	/* button */
	Ef_results_info,	/* button */

	Ef_results_add,		/* button */
	Ef_results_remove,	/* button */
	Ef_results_list,	/* button */

	Ef_results_pcmd,	/* button */
	Ef_results_MAX
};
typedef enum ef_results_id_e ef_results_id_t;


static ef_t ef_results_widgets[] = {
	{
		"Search Results",
		Ef_results_dialog,
		Ef_wgt_dialog,
	},
	{
		"clist_results",
		Ef_results_clist,
		Ef_wgt_ctree,
		GTK_SIGNAL_FUNC(results_clist_cb)
	},
	{
		"scrwin_results",
		Ef_results_scrwin,
		Ef_wgt_dont_care,
	},
	{
		"clist_detailed",
		Ef_results_detailed,
		Ef_wgt_ctree,
		GTK_SIGNAL_FUNC(results_clist_cb)
	},
	{
		"scrwin_detailed",
		Ef_results_scrwin_detailed,
		Ef_wgt_dont_care,
	},
	{
		"clist_errors",
		Ef_results_errors,
		Ef_wgt_clist,
	},
	{
		"scrwin_errors",
		Ef_results_cont_errors,
		Ef_wgt_dont_care,
	},
	{
		"appbar_results",
		Ef_results_num_found,
		Ef_wgt_dont_care,
	},
	{
		"tb_results",
		Ef_results_toolbar,
		Ef_wgt_toolbar,
	},
	{
		"but_results_ok",
		Ef_results_ok,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(results_ok_cb)
	},
	{
		"but_results_err",
		Ef_results_err,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(results_err_cb)
	},
	{
		"but_file_open",
		Ef_results_open,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_open_cb)
	},
	{
		"but_file_delete",
		Ef_results_delete,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_delete_cb)
	},
	{
		"but_file_copy",
		Ef_results_copy,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_copy_cb)
	},
	{
		"but_file_print",
		Ef_results_print,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_print_cb)
	},
	{
		"but_file_execute",
		Ef_results_execute,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_execute_cb)
	},
	{
		"but_file_shell",
		Ef_results_shell,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_shell_cb)
	},
	{
		"but_file_info",
		Ef_results_info,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_info_cb)
	},

	{
		"but_file_add",
		Ef_results_add,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_add_cb)
	},
	{
		"but_file_remove",
		Ef_results_remove,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_remove_cb)
	},
	{
		"but_file_list",
		Ef_results_list,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(file_list_cb)
	},

	{
		"but_search_prev",
		Ef_results_pcmd,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(search_cmd_prev_cb)
	},
	{ NULL }
}; /* ef_results_widgets[] */


static GtkWidget *pixmap_dir = NULL;
static GtkWidget *pixmap_file = NULL;


static int
compare_fnames(GtkCList *clist, gconstpointer rowp1, gconstpointer rowp2)
{
	GtkCListRow *r1 = (GtkCListRow *)rowp1;
	GtkCListRow *r2 = (GtkCListRow *)rowp2;
	int column = 0;

	return g_strcasecmp(GTK_CELL_TEXT(r1->cell[column])->text,
			    GTK_CELL_TEXT(r2->cell[column])->text);
} /* compare_fnames */


static int
compare_modes(GtkCList *clist, gconstpointer rowp1, gconstpointer rowp2)
{
	GtkCListRow *r1 = (GtkCListRow *)rowp1;
	GtkCListRow *r2 = (GtkCListRow *)rowp2;
	int column = 1;

	return g_strcasecmp(GTK_CELL_TEXT(r1->cell[column])->text,
			    GTK_CELL_TEXT(r2->cell[column])->text);
} /* compare_modes */


static int
compare_sizes(GtkCList *clist, gconstpointer rowp1, gconstpointer rowp2)
{
	GtkCListRow *r1 = (GtkCListRow *)rowp1;
	GtkCListRow *r2 = (GtkCListRow *)rowp2;
	int column = 2;

	return g_strcasecmp(GTK_CELL_TEXT(r1->cell[column])->text,
			    GTK_CELL_TEXT(r2->cell[column])->text);
} /* compare_sizes */


static int
compare_dates(GtkCList *clist, gconstpointer rowp1, gconstpointer rowp2)
{
	time_t time1, time2;
	GtkCListRow *r1 = (GtkCListRow *)rowp1;
	GtkCListRow *r2 = (GtkCListRow *)rowp2;

	/*
	 * row data is a pointer which stores the mtime value of the file.
	 * This was set in results_add().
	 */
	time1 = (time_t)r1->data;
	time2 = (time_t)r2->data;

	if (time1 < time2)
		return -1;
	
	if (time1 == time2)
		return 0;

	return 1;
} /* compare_dates */


static int
compare_owners(GtkCList *clist, gconstpointer rowp1, gconstpointer rowp2)
{
	GtkCListRow *r1 = (GtkCListRow *)rowp1;
	GtkCListRow *r2 = (GtkCListRow *)rowp2;
	int column = 4;

	return g_strcasecmp(GTK_CELL_TEXT(r1->cell[column])->text,
			    GTK_CELL_TEXT(r2->cell[column])->text);
} /* compare_owners */


static int
compare_groups(GtkCList *clist, gconstpointer rowp1, gconstpointer rowp2)
{
	GtkCListRow *r1 = (GtkCListRow *)rowp1;
	GtkCListRow *r2 = (GtkCListRow *)rowp2;
	int column = 5;

	return g_strcasecmp(GTK_CELL_TEXT(r1->cell[column])->text,
			    GTK_CELL_TEXT(r2->cell[column])->text);
} /* compare_sizes */


static GtkCListCompareFunc compare_funcs[] = {
	compare_fnames,
	compare_modes,
	compare_sizes,
	compare_dates,
	compare_owners,
	compare_groups
};

static void
results_detailed_cb(GtkWidget *clist_wgt, int column)
{
	GtkCList *clist = GTK_CLIST(clist_wgt);
	g_assert(column < 6);
	gtk_clist_set_compare_func(clist, compare_funcs[column]);
	if (clist->sort_column == column) {
		clist->sort_type = (clist->sort_type == GTK_SORT_ASCENDING) ?
				   GTK_SORT_DESCENDING : GTK_SORT_ASCENDING;
	}

	gtk_clist_set_sort_column(clist, column);
	gtk_clist_sort(clist);
} /* results_detailed_cb */


static void
results_doc_data_get(GtkWidget *wgt, GdkDragContext *context,
		     GtkSelectionData *selection_data, guint info,
		     guint time, gpointer cbdata)
{
	GList *l;
	GtkCList *clist;
	char *uri, *full, *fname;
	win_t *w = (win_t *)cbdata;

	clist = GTK_CLIST(results_clist_widget(w));
	if (clist->selection) {
		GString *s = g_string_new(NULL);
		for (l = clist->selection; l; l = l->next) {
			GtkCTreeNode *cnode = (GtkCTreeNode *)l->data;
			GtkCListRow *clrow = (GtkCListRow *)cnode->list.data;
			fname = clrow->cell->u.text;
			full = file_full_pathname_make(fname);
			uri = g_strdup_printf("file:%s\r\n", full);
			s = g_string_append(s, uri);
			g_free(uri);
			g_free(full);
		}
		gtk_selection_data_set(selection_data, selection_data->target,
				       8, (guchar *)s->str, strlen(s->str));
		g_string_free(s, TRUE);
	} else {
		gtk_selection_data_set(selection_data, selection_data->target,
				       8, NULL, 0);
	}
} /* results_doc_data_get */


void
results_widgets_setup(win_t *w)
{
	GtkStyle *style;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	char *fname;
	int num;
#include <gtk/gtkselection.h>
	GtkTargetEntry dragtypes[] = {
		{ "STRING",     0, 0 },
		{ "text/uri-list", 0, 0 },
		{ "text/plain", 0, 0 }
	};

	if (pixmap_dir == NULL)
		pixmap_dir = gnome_pixmap_new_from_xpm_d(dir_xpm);
	if (pixmap_file == NULL)
		pixmap_file = gnome_pixmap_new_from_xpm_d(file_xpm);

	num = (sizeof(ef_results_widgets) / sizeof(ef_results_widgets[0])) - 1;
	w->reswgts = my_widgets_setup(w, w->reswgts, num,
				      ef_results_widgets, "Search Results",
				      Ef_results_dialog, Ef_results_MAX, TRUE);
	results_prefs_toolbar_set(w);
	gtk_signal_connect(GTK_OBJECT(w->reswgts[Ef_results_dialog]),
			   "key_press_event",
			   GTK_SIGNAL_FUNC(results_key_press), w);
	gtk_signal_connect(GTK_OBJECT(w->reswgts[Ef_results_dialog]),
			   "delete_event",
			   GTK_SIGNAL_FUNC(gtk_true), NULL);
	my_prefs_window_set_size(w->reswgts[Ef_results_dialog],
				 Prefs_save_results_win,
				 Prefs_res_win_w,
				 Prefs_res_win_h);
	gtk_clist_column_titles_active(
				GTK_CLIST(w->reswgts[Ef_results_detailed]));
	gtk_signal_connect(GTK_OBJECT(w->reswgts[Ef_results_detailed]),
			   "click-column",
			   GTK_SIGNAL_FUNC(results_detailed_cb), w);

	if (prefs_bool_get(Prefs_dnd)) {
		gtk_drag_source_set(w->reswgts[Ef_results_clist],
				    GDK_BUTTON1_MASK, dragtypes, 3,
				    (GdkDragAction)(GDK_ACTION_COPY |
				    		    GDK_ACTION_MOVE));
		gtk_signal_connect(GTK_OBJECT(w->reswgts[Ef_results_clist]),
				   "drag_data_get",
				   GTK_SIGNAL_FUNC(results_doc_data_get), w);

		gtk_drag_source_set(w->reswgts[Ef_results_detailed],
				    GDK_BUTTON1_MASK, dragtypes, 3,
				    (GdkDragAction)(GDK_ACTION_COPY |
				    		    GDK_ACTION_MOVE));
		gtk_signal_connect(GTK_OBJECT(w->reswgts[Ef_results_detailed]),
				   "drag_data_get",
				   GTK_SIGNAL_FUNC(results_doc_data_get), w);
	}

	gtk_widget_realize(w->reswgts[Ef_results_dialog]);
	style = gtk_widget_get_style(w->reswgts[Ef_results_dialog]);

	/*
	 * At one point, glade did this for me.  But suddenly, it one day
	 * decided not to.  So for now, create a pixmap widget and put it into
	 * the buttons ourselves.
	 */
	fname = g_strdup("./shell.xpm");
	if (!g_file_exists(fname)) {
		g_free(fname);
		fname = g_concat_dir_and_file(DATADIR, "shell.xpm");
	}
	if (g_file_exists(fname)) {
		GtkWidget *pixmapwgt;
		pixmap = gdk_pixmap_create_from_xpm(
					w->reswgts[Ef_results_dialog]->window,
					&mask,
					&style->bg[GTK_STATE_NORMAL],
					fname);
		if (pixmap) {
			pixmapwgt = gtk_pixmap_new(pixmap, mask);
			gtk_widget_show(pixmapwgt);
			gtk_container_add(
				GTK_CONTAINER(w->reswgts[Ef_results_shell]),
				pixmapwgt);
		}
	}
	g_free(fname);
	gtk_button_set_relief(GTK_BUTTON(w->reswgts[Ef_results_shell]),
			      GTK_RELIEF_NONE);
} /* results_widgets_setup */


GnomeAppBar *
results_appbar_widget(win_t *w)
{
	return (w->reswgts) ? GNOME_APPBAR(w->reswgts[Ef_results_num_found])
			    : NULL;
} /* results_appbar_widget */


GtkProgress *
results_progbar_widget(win_t *w)
{
	return (w->reswgts) ?
			gnome_appbar_get_progress(results_appbar_widget(w))
			: NULL;
} /* results_progbar_widget */


void
results_found_set(win_t *w)
{
	char *buf;

	buf = g_strdup_printf(_("Files found: %d"), w->num_found);
	gnome_appbar_set_status(GNOME_APPBAR(w->reswgts[Ef_results_num_found]),
				buf);
	g_free(buf);
} /* results_found_set */


void
results_dialog_show(win_t *w)
{
	gtk_widget_set_sensitive(w->reswgts[Ef_results_err], w->num_errs);
	gtk_clist_columns_autosize(GTK_CLIST(w->reswgts[Ef_results_detailed]));
	gtk_widget_show(w->reswgts[Ef_results_dialog]);
} /* results_dialog_show */


void
results_prev_show(win_t *w)
{
	if (w->reswgts && w->reswgts[Ef_results_dialog] &&
	    GTK_IS_WIDGET(w->reswgts[Ef_results_dialog])) {

	    	gdk_window_raise(w->reswgts[Ef_results_dialog]->window);
		gtk_widget_show(w->reswgts[Ef_results_dialog]);
	} else
		msgbar_printf(w, _("No previous search results."));
} /* results_prev_show */


void
results_dialog_destroy(win_t *w)
{
	if (w->reswgts && w->reswgts[Ef_results_dialog])
		gtk_widget_destroy(w->reswgts[Ef_results_dialog]);
} /* results_dialog_destroy */


void
results_add(win_t *w, char *line, char *clist_text[], time_t my_mtime,
	    struct stat *sb)
{
	struct stat statb;
	GnomePixmap *gpmap;
	GtkWidget *ctree;
	char *buf[2];
	int row, err;

	buf[0] = line;
	buf[1] = NULL;
	g_assert(w->reswgts[Ef_results_clist]);

	if (sb == NULL) {
		if (clist_text)
			err = stat((const char *)clist_text[0], &statb);
		else
			err = stat((const char *)line, &statb);
		if (err)
			bzero(&statb, sizeof(struct stat));
		sb = &statb;
	}

	gpmap = (S_ISDIR(sb->st_mode)) ? GNOME_PIXMAP(pixmap_dir) :
					 GNOME_PIXMAP(pixmap_file);

	if (clist_text) {
		GtkCTreeNode *node;
		GList *l;
		ctree = w->reswgts[Ef_results_detailed];
		node = gtk_ctree_insert_node(GTK_CTREE(ctree), NULL, NULL,
					     clist_text, 4,
					     gpmap->pixmap,
					     gpmap->mask,
					     gpmap->pixmap,
					     gpmap->mask,
					     FALSE, FALSE);
		for (row = 0, l = GTK_CLIST(ctree)->row_list;
		     l;
		     row++, l = g_list_next(l)) {
			/*
			 * GTK+ is TOTALLY FUCKED UP.  gtk_ctree_insert_node()
			 * inserts a new node by doing this nonsense:
			 *
			 *	new_row = row_new (ctree);
			 *	list = g_list_alloc();
			 *	list->data = new_row;
			 *	node = GTK_CTREE_NODE (list);
			 *
			 * This means that the address of "list" is the actual
			 * node, not list->data, even though list->data is what
			 * is pointing to the real contents of the row.
			 *
			 * What this ultimately means is that given a
			 * GtkCTreeNode residing in a corresponding GtkCTree,
			 * there's no simple way of finding out the numerical
			 * value of the row that node sits in!
			 */
			GtkCTreeNode *list_node = (GtkCTreeNode *)l;
			if (list_node == node)
				break;
		}
		/* g_print("row = %d\n", row); */
		gtk_clist_set_row_data(GTK_CLIST(ctree), row,
				       (gpointer)my_mtime);
		gtk_widget_show(w->reswgts[Ef_results_scrwin_detailed]);
		gtk_widget_hide(w->reswgts[Ef_results_scrwin]);
	} else {
		ctree = w->reswgts[Ef_results_clist];
		(void)gtk_ctree_insert_node(GTK_CTREE(ctree), NULL, NULL,
					    buf, 4,
					    gpmap->pixmap,
					    gpmap->mask,
					    gpmap->pixmap,
					    gpmap->mask,
					    FALSE, FALSE);
		gtk_widget_show(w->reswgts[Ef_results_scrwin]);
		gtk_widget_hide(w->reswgts[Ef_results_scrwin_detailed]);
	}

	w->num_found++;
	if (w->num_found % 25 == 0)
		msgbar_printf(w, _("Total files found: %d"), w->num_found);
} /* results_add */


void
results_errors_add(win_t *w, char *text)
{
	char *buf[2];
	
	buf[0] = text;
	buf[1] = NULL;
	g_assert(w->reswgts[Ef_results_errors]);
	(void)gtk_clist_append(GTK_CLIST(w->reswgts[Ef_results_errors]), buf);
	w->num_errs++;
} /* results_add */


GtkWidget *
results_clist_widget(win_t *w)
{
	if (!w->reswgts)
		return NULL;

	if (GTK_WIDGET_VISIBLE(w->reswgts[Ef_results_scrwin]))
		return w->reswgts[Ef_results_clist];

	g_assert(GTK_WIDGET_VISIBLE(w->reswgts[Ef_results_scrwin_detailed]));
	return w->reswgts[Ef_results_detailed];
} /* results_clist_widget */


void
results_prefs_toolbar_set(win_t *w)
{
	if (!w->reswgts)
		return;
	prefs_toolbar_set(Prefs_fops_tbstyle, w->reswgts[Ef_results_toolbar]);
} /* results_prefs_toolbar_set */


void
results_prefs_save(win_t *w)
{
	int width, height;

	if (!w->reswgts)
		return;

	gdk_window_get_size(w->reswgts[Ef_results_dialog]->window,
			    &width, &height);
	prefs_int_set(Prefs_res_win_w, width);
	prefs_int_set(Prefs_res_win_h, height);
} /* results_prefs_save */


/*** PRIVATE FUNCTION DEFINTIONS ***/
static void
results_clist_cb(GtkCTree *ctree, GtkCTreeNode *node, int column,
		 gpointer cbdata)
{
	ef_results_id_t id;
	win_t *w = (win_t *)cbdata;

	for (id = Ef_results_open; id <= Ef_results_remove; id++)
		gtk_widget_set_sensitive(w->reswgts[id], TRUE);
#if 0
	gtk_widget_set_sensitive(w->reswgts[Ef_results_open], TRUE);
	gtk_widget_set_sensitive(w->reswgts[Ef_results_delete], TRUE);
	gtk_widget_set_sensitive(w->reswgts[Ef_results_copy], TRUE);
	gtk_widget_set_sensitive(w->reswgts[Ef_results_print], TRUE);
	gtk_widget_set_sensitive(w->reswgts[Ef_results_execute], TRUE);
	gtk_widget_set_sensitive(w->reswgts[Ef_results_shell], TRUE);
	gtk_widget_set_sensitive(w->reswgts[Ef_results_info], TRUE);
	gtk_widget_set_sensitive(w->reswgts[Ef_results_add], TRUE);
	gtk_widget_set_sensitive(w->reswgts[Ef_results_remove], TRUE);
#endif
} /* results_clist_cb */


static void
results_ok_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_widget_hide(w->reswgts[Ef_results_dialog]);
} /* results_ok_cb */


static void
results_err_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (GTK_WIDGET_VISIBLE(w->reswgts[Ef_results_cont_errors]))
		gtk_widget_hide(w->reswgts[Ef_results_cont_errors]);
	else
		gtk_widget_show(w->reswgts[Ef_results_cont_errors]);
} /* results_err_cb */


static gboolean
results_key_press(GtkWidget *wgt, GdkEventKey *ev, win_t *w)
{
	if (ev->keyval == GDK_Escape) {
		results_ok_cb(NULL, w);
		return TRUE;
	}

	return FALSE;
} /* results_key_press */


/* the end */
