/**
 * vim: sw=4 ts=4:
 *
 * GKeyFile based GnomeConfig substitute
 *
 *	(C) 2005 Filip Van Raemdonck <mechanix@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * 	$Id$
 *
 **/

#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <glib.h>
#include <glib/gstdio.h>
#include "conffile.h"

#ifndef _
#if ENABLE_NLS
#define _(x) gettext(x)
#else
#define _(x) (x)
#endif
#endif

ConfFile::ConfFile (const gchar* domain, gboolean isprivate) {
	gboolean loaded;
	const gchar* base = ".gnome2";
	GError* err = NULL;

	if (isprivate) {
		base = ".gnome2_private";
	}
	loadpath = g_build_filename (g_get_home_dir(), base, domain, NULL);

	kf = g_key_file_new();
	loaded = g_key_file_load_from_file (kf, loadpath, GKeyFileFlags (G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS), &err);
	if (err) {
		g_message (err->message);
		g_error_free (err);
	}

	if (!loaded) {
		g_key_file_free (kf);
		kf = g_key_file_new();
	}
}

ConfFile::~ConfFile (void) {
	g_free (loadpath);
	g_key_file_free (kf);
}

void
ConfFile::save (const gchar* path) const {
	gsize len;
	GError* err = NULL;
	gint fd;

	gchar* buf = g_key_file_to_data (kf, &len, &err);
	if (err) {
		g_message (err->message);
		g_error_free (err);
		g_free (buf);
		return;
	}

	fd = g_open (path, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
	if (fd < 0) {
		g_message ("Failed to open temporary config file \"%s\"\n changes not saved", path);
		g_free (buf);
		return;
	}

	if (write (fd, buf, len) < (gssize) len) {
		g_message ("Failed to open temporary config file \"%s\"\n changes not saved", path);
		close (fd);
		g_unlink (path);
		g_free (buf);
		return;
	}
	close (fd);
	g_free (buf);
}

void
ConfFile::sync (void) const {
	save (loadpath);
}

void
ConfFile::clean_section (const gchar* sect) const {
	GError* err = NULL;

	g_key_file_remove_group (kf, sect, &err);
	if (err) {
		g_message (err->message);
		g_error_free (err);
	}
}

gboolean
ConfFile::get_bool (const gchar* sect, const gchar* key, gboolean dflt) const {
	GError* err = NULL;

	gboolean val = g_key_file_get_boolean (kf, sect, key, &err);
	if (err) {
		g_message (err->message);
		g_error_free (err);
		if (dflt) {
			val = dflt;
		}
	}

	return val;
}

gint
ConfFile::get_integer (const gchar* sect, const gchar* key, gint dflt) const {
	GError* err = NULL;

	gint val = g_key_file_get_integer (kf, sect, key, &err);
	if (err) {
		g_message (err->message);
		g_error_free (err);
		if (dflt) {
			val = dflt;
		}
	}

	return val;
}

gchar**
ConfFile::get_keys (const gchar* sect, gsize* len) const {
	GError* err = NULL;

	gchar** strv = g_key_file_get_keys (kf, sect, len, &err);
	if (err) {
		g_message (err->message);
		g_error_free (err);
	}

	return strv;
}

gchar*
ConfFile::get_string (const gchar* sect, const gchar* key, const gchar* dflt) const {
	GError* err = NULL;

	gchar* str = g_key_file_get_string (kf, sect, key, &err);
	if (err) {
		g_message (err->message);
		g_error_free (err);
		if (dflt) {
			str = g_strdup (dflt);
		}
	}

	return str;
}

void
ConfFile::set_bool (const gchar* sect, const gchar* key, const gboolean val) {
	g_key_file_set_boolean (kf, sect, key, val);
}

void
ConfFile::set_integer (const gchar* sect, const gchar* key, const gint val) {
	g_key_file_set_integer (kf, sect, key, val);
}

void
ConfFile::set_string (const gchar* sect, const gchar* key, const gchar* val) {
	g_key_file_set_string (kf, sect, key, val);
}
