with Ada.Command_Line; use Ada.Command_Line;
with Ada.Text_IO;      use Ada.Text_IO;
with GNAT.OS_Lib;      use GNAT.OS_Lib;
with GNAT.Directory_Operations; use GNAT.Directory_Operations;

with Projects.Registry;    use Projects, Projects.Registry;
with Src_Info;             use Src_Info;
with Src_Info.Queries;     use Src_Info.Queries;
with String_Utils; use String_Utils;

with Language_Handlers;       use Language_Handlers;
with Test_Utils;              use Test_Utils;
with VFS;                     use VFS;

procedure Find_Local_Ref is

   procedure Dump (Entity : Entity_Information);
   procedure Dump (Ref    : E_Reference);

   ----------
   -- Dump --
   ----------

   procedure Dump (Ref    : E_Reference) is
      Loc : constant File_Location := Get_Location (Ref);
   begin
      Put (Base_Name (Get_File (Loc)) & ':'
           & Image (Get_Line (Loc)) & ':'
           & Image (Get_Column (Loc)));
   end Dump;

   ----------
   -- Dump --
   ----------

   procedure Dump (Entity : Entity_Information) is
   begin
      Put ('(' & Get_Name (Entity) & ':'
           & Base_Name (Get_Declaration_File_Of (Entity)) & ':'
           & Image (Get_Declaration_Line_Of (Entity)) & ':'
           & Image (Get_Declaration_Column_Of (Entity)) & ")");
   end Dump;


   Prj_Filename     : String_Access;
   Source_Filename  : Virtual_File;
   Registry         : aliased Project_Registry;
   Project          : Project_Type;
   Source_Info_List : Src_Info.LI_File_List;
   LI_Unit          : LI_File_Ptr;
   Iter             : Local_Entities_Iterator;
   Ref              : E_Reference;
   Info             : Entity_Information;
   Handler          : Language_Handler;

begin
   Handler := Create_Lang_Handler (Registry'Unrestricted_Access);

   if Argument_Count /= 2 then
      Put_Line ("invalid number of arguments");
      Put_Line ("find_ref prj_file src_file");
      Put_Line ("  Display all references to all entities found in src_file");
      return;
   end if;

   Prj_Filename := new String'(Argument (1));
   if not Is_Regular_File (Prj_Filename.all) then
      Put_Line ("No such file: '" & Prj_Filename.all & "'");
      return;
   end if;

   Source_Filename := Create
     (Normalize_Pathname (Argument (2), Get_Current_Dir));
   if not Is_Regular_File (Source_Filename) then
      Put_Line ("No such file: '" & Full_Name (Source_Filename).all & "'");
      return;
   end if;

   Reset (Source_Info_List);
   Load_Project (Prj_Filename.all, Handler, Registry, Project);

   Load_LI_File
     (Source_Info_List, Handler, Registry, Source_Filename, LI_Unit);

   if LI_Unit /= No_LI_File then
      Iter := Find_All_References_In_File (LI_Unit, Source_Filename);

      loop
         Ref := Get (Iter);
         if Ref = No_Reference then
            New_Line;
            Destroy (Info);
            Info := Get (Iter);
            exit when Info = No_Entity_Information;
            Put ("New entity: ");
            Dump (Info);
         else
            Put (' ');
            Dump (Ref);
         end if;

         Next (Iter);
      end loop;

      New_Line;

      Destroy (Info);
   else
      Put_Line ("LI file not found");
   end if;
end Find_Local_Ref;
