
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <glib.h>

#include "gmyth_backendinfo.h"
#include "gmyth_scheduler.h"
#include "gmyth_util.h"
#include "gmyth_epg.h"

typedef struct {
    GMythBackendInfo *b_info;
    gboolean        list_channels;
} ls_options_t;

static ls_options_t *
_ls_options_new()
{
    ls_options_t   *options = g_new0(ls_options_t, 1);
    options->b_info = gmyth_backend_info_new();

    return options;
}

static void
_ls_options_free(ls_options_t * options)
{
    g_return_if_fail(options != NULL);

    if (options->b_info)
        g_object_unref(options->b_info);
}

static          gboolean
_parse_args(int argc, char *argv[], ls_options_t * options)
{
    GError         *error = NULL;
    GOptionContext *context;

    gchar          *host_ip = NULL;
    gint            host_port = 0;
    gint            db_port = 0;
    gboolean        list_channels = FALSE;
    gchar          *username = NULL;
    gchar          *password = NULL;

    GOptionEntry    entries[] = {
        {"hostname", 'h', 0, G_OPTION_ARG_STRING, &host_ip,
         "Mythtv backend hostname or " "IP address", "IP_ADDRESS"},

        {"port", 'p', 0, G_OPTION_ARG_INT, &host_port,
         "Mythtv backend port",
         "PORT"},

        {"list_channels", 'c', 0, G_OPTION_ARG_NONE, &list_channels,
         "List channels",
         "LIST_CHANNELS"},

        {"username", 'u', 0, G_OPTION_ARG_STRING, &username,
         "Mysql database username. Default: mythtv",
         "MYSQL_USER"},

        {"password", 'w', 0, G_OPTION_ARG_STRING, &password,
         "Mysql database password. Default: mythtv",
         "MYSQL_PASSWD"},

        {"db_port", 'P', 0, G_OPTION_ARG_STRING, &db_port,
         "Mysql database port. Default: 0",
         "MYSQL_PORT"},

        {NULL}
    };

    g_return_val_if_fail(options != NULL, FALSE);

    context =
        g_option_context_new
        ("- list recorded programs and channels from a mythtv backend\n");
    g_option_context_add_main_entries(context, entries, NULL);
    g_option_context_parse(context, &argc, &argv, &error);
    g_option_context_set_help_enabled(context, TRUE);

    g_option_context_free(context);

    if ((!host_ip) || (host_port == 0)) {
        g_free(host_ip);
        return FALSE;
    }

    gmyth_backend_info_set_hostname(options->b_info, host_ip);
    gmyth_backend_info_set_port(options->b_info, host_port);

    if (username)
        gmyth_backend_info_set_username(options->b_info, username);
    else
        gmyth_backend_info_set_username(options->b_info, "mythtv");

    if (password)
        gmyth_backend_info_set_password(options->b_info, password);
    else
        gmyth_backend_info_set_password(options->b_info, "mythtv");

    gmyth_backend_info_set_db_name(options->b_info, "mythconverg");

    options->list_channels = list_channels;

    g_free(host_ip);

    return TRUE;
}

static          gboolean
_ls_recorded_files(ls_options_t * options)
{
    GMythScheduler *scheduler;
    GList          *list,
                   *iter;
    gint            res = 0;

    g_return_val_if_fail(options != NULL, FALSE);
    g_return_val_if_fail(options->b_info != NULL, FALSE);

    scheduler = gmyth_scheduler_new();

    if (gmyth_scheduler_connect_with_timeout(scheduler,
                                             options->b_info, 10) == FALSE)
    {
        g_warning("Could not connect to backend db");
        g_object_unref(scheduler);
        return FALSE;
    }

    res = gmyth_scheduler_get_recorded_list(scheduler, &list);
    if (res < 0) {
        gmyth_scheduler_disconnect(scheduler);
        g_object_unref(scheduler);
        g_warning("Could not retrieve recorded list");
        return FALSE;
    }

    gmyth_scheduler_disconnect(scheduler);

    if (res == 0) {
        g_print("None file was found in the backend.\n");
        gmyth_scheduler_disconnect(scheduler);
        g_object_unref(scheduler);
        return TRUE;
    }

    iter = list;
    while (iter) {
        RecordedInfo   *recorded_info = (RecordedInfo *) iter->data;

        if (gmyth_util_file_exists
            (options->b_info, recorded_info->basename->str)) {
            g_print("%s\n", recorded_info->basename->str);
        }
        gmyth_recorded_info_free(recorded_info);
        iter = g_list_next(iter);
    }

    g_list_free(list);

    gmyth_scheduler_disconnect(scheduler);
    g_object_unref(scheduler);

    return TRUE;
}

static          gboolean
_ls_channels(ls_options_t * options)
{
    GMythEPG       *epg;
    gint            length;
    GList          *clist,
                   *ch;

    g_return_val_if_fail(options != NULL, FALSE);
    g_return_val_if_fail(options->b_info != NULL, FALSE);


    epg = gmyth_epg_new();
    if (!gmyth_epg_connect(epg, options->b_info)) {
        g_object_unref(epg);
        return FALSE;
    }

    length = gmyth_epg_get_channel_list(epg, &clist);
    for (ch = clist; ch != NULL; ch = ch->next) {
        GMythChannelInfo *info = (GMythChannelInfo *) ch->data;

        if ((info->channel_name == NULL) || (info->channel_num == NULL)) {
            continue;
        }

        g_print("%s\t\t%s\n", info->channel_num->str,
                info->channel_name->str);
    }

    gmyth_free_channel_list(clist);
    gmyth_epg_disconnect(epg);
    g_object_unref(epg);

    return TRUE;
}

int
main(int argc, char *argv[])
{
    gboolean        res = FALSE;
    ls_options_t   *options;

    g_type_init();
    g_thread_init(NULL);

    options = _ls_options_new();
    res = _parse_args(argc, argv, options);
    if (!res) {
        g_printerr("Argument invalid. Type --help\n");
        return 1;
    }

    if (options->list_channels)
        res = _ls_channels(options);
    else
        res = _ls_recorded_files(options);

    _ls_options_free(options);

    return 0;
}
