/*
 * Portions of this file Copyright 1999-2005 University of Chicago
 * Portions of this file Copyright 1999-2005 The University of Southern California.
 *
 * This file or a portion of this file is licensed under the
 * terms of the Globus Toolkit Public License, found at
 * http://www.globus.org/toolkit/download/license.html.
 * If you redistribute this file, with or without
 * modifications, you must include this notice in the file.
 */


/********************************************************************
 *
 * This file implements the DUROC rsl manipulation routines
 *
 ********************************************************************/

#include <assert.h>

#include "globus_common.h"
#include "nexus.h"
#include "globus_rsl.h"

#include "globus_i_duroc_common_utils.h"
#include "globus_i_duroc_common.h"



#define s_is_globus_gram_contact_node(spec) \
( globus_rsl_is_relation_eq (spec) \
  && (utils_streq_ci (globus_rsl_relation_get_attribute (spec), \
		   "resourceManagerContact" )) )

#define s_is_starttype_node(spec) \
     ( globus_rsl_is_relation_eq (spec) \
       && (utils_streq_ci (globus_rsl_relation_get_attribute (spec), \
			"subjobStartType")) )

#define s_is_commstype_node(spec) \
     ( globus_rsl_is_relation_eq (spec) \
       && (utils_streq_ci (globus_rsl_relation_get_attribute (spec), \
			"subjobCommsType")) )

#define s_is_stdout_node(spec) \
     ( globus_rsl_is_relation_eq (spec) \
       && (utils_streq_ci (globus_rsl_relation_get_attribute (spec), \
			"stdout")) )

#define s_is_stderr_node(spec) \
     ( globus_rsl_is_relation_eq (spec) \
       && (utils_streq_ci (globus_rsl_relation_get_attribute (spec), \
			"stderr")) )

#define s_is_and_node(spec) \
     ( globus_rsl_is_boolean_and (spec) ) 

char *globus_duroc_rsl_globus_gram_contact (globus_rsl_t *request)
{
  /* search conjunction for resourceManagerContact field 
   * thus &...(resourceManagerContact=xxx)... is the only
   * recognized RSL form for duroc gram contacts at this time
   */
  if ( globus_rsl_is_boolean_and (request) ) {
    globus_list_t * subclauses;

    subclauses = globus_rsl_boolean_get_operand_list (request);
    while ( ! globus_list_empty (subclauses) ) {
      globus_rsl_t * subclause;

      subclause = ((globus_rsl_t *)
		   globus_list_first (subclauses));

      if ( s_is_globus_gram_contact_node(subclause) ) {
	return globus_rsl_value_literal_get_string (
		    globus_rsl_relation_get_single_value (subclause));
      }
      
      subclauses = globus_list_rest (subclauses);
    }

    return NULL;
  }
  else return NULL;
}

globus_rsl_t *
globus_duroc_rsl_strip_globus_duroc_fields (globus_rsl_t *request)
{
  /* we no longer need to filter out fields, so just
   * maintain the copy semantics until we eliminate all calls to this
   * function
   */
  return globus_rsl_copy_recursive (request);
}


globus_rsl_t *
globus_duroc_rsl_copy (globus_rsl_t *request)
{
  return globus_rsl_copy_recursive (request);
}

int globus_duroc_rsl_free (globus_rsl_t *request)
{
  return globus_rsl_free_recursive (request);
}

#define s_is_env_node(spec) \
( globus_rsl_is_relation_eq (spec) \
  && (utils_streq_ci (globus_rsl_relation_get_attribute (spec), \
		   "environment")) )

static
int s_update_env (globus_rsl_t *request, const char *name, const char *value)
{
  /* search conjunction for environment field 
   * thus &...(environment=xxx)... is the only
   * recognized RSL form for duroc environment data at this time
   */
  if ( globus_rsl_is_boolean_and (request) ) {
    globus_list_t * subclauses;
    globus_rsl_t * subclause;

    subclause = NULL;

    subclauses = globus_rsl_boolean_get_operand_list (request);
    while ( ! globus_list_empty (subclauses) ) {
      subclause = ((globus_rsl_t *)
		   globus_list_first (subclauses));

      if ( s_is_env_node(subclause) ) break;
      
      subclauses = globus_list_rest (subclauses);
    }

    if ( subclauses == NULL ) {
      /* create an environment node since none exists */
      subclause = globus_rsl_make_relation (
				    GLOBUS_RSL_EQ,
				    (void *) utils_strdup ("environment"),
				    globus_rsl_value_make_sequence (NULL));
      globus_list_insert (globus_rsl_boolean_get_operand_list_ref (request),
			  subclause);
    }

    {
      /* update the environment node in place with the new binding */
      globus_list_t      ** binding_listp;
      globus_rsl_value_t  * new_binding;

      new_binding = globus_rsl_value_make_sequence (
			    globus_list_cons ((void *) globus_rsl_value_make_literal(utils_strdup (name)),
			      globus_list_cons ((void *) globus_rsl_value_make_literal(utils_strdup (value)),
						NULL)));

      binding_listp 
	= globus_rsl_value_sequence_get_list_ref (
			  globus_rsl_relation_get_value_sequence (subclause));

      if ( globus_list_empty (*binding_listp) ) {
	/* insert node into previously empty list */
	globus_list_insert (binding_listp,
			    (void *) new_binding);
      }
      else {
	/* append new node to existing list */
	globus_list_t * last;
	
	last = (*binding_listp);
	while ( ! globus_list_empty (globus_list_rest (last)))
	  last = globus_list_rest (last);

	globus_list_insert (globus_list_rest_ref (last),
			    (void *) new_binding);
      }
    }

    return 1;
  }
  else return 0;
}

int globus_duroc_rsl_setenv (globus_rsl_t *request, 
			     const char *name, 
			     const char * value)
{
  if ( s_update_env (request, name, value) )
    return GLOBUS_DUROC_SUCCESS;
  else
    /* malformed request */
    return GLOBUS_DUROC_ERROR_INVALID_REQUEST;
}

int globus_duroc_rsl_starttype (globus_rsl_t *request, char ** strp)
{
  /* search conjunction for resourceManagerContact field 
   * thus &...(subjobStartType=xxx)... is the only
   * recognized RSL form for this field
   */
  if ( globus_rsl_is_boolean_and (request) ) {
    globus_list_t * subclauses;
    utils_debug (GLOBUS_DUROC_DEBUG_FLAG, "searching conjunction\n");
    subclauses = globus_rsl_boolean_get_operand_list (request);
    while ( ! globus_list_empty (subclauses) ) {
      globus_rsl_t * subclause;

      subclause = ((globus_rsl_t *)
		   globus_list_first (subclauses));

      if ( s_is_starttype_node(subclause) ) {
	(*strp) = utils_strdup (globus_rsl_value_literal_get_string (
		    globus_rsl_relation_get_single_value (subclause)));
	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "RSL subjobStartType is >>%s<<\n",
		     (*strp));
	return GLOBUS_DUROC_SUCCESS;
      }
      
      subclauses = globus_list_rest (subclauses);
    }

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "RSL subjobStartType defaulting to strict\n");
    *strp = utils_strdup ("strict-barrier");
    return GLOBUS_DUROC_SUCCESS;
  }
  else return GLOBUS_DUROC_ERROR_INVALID_REQUEST;
}

int globus_duroc_rsl_commstype (globus_rsl_t *request, char ** strp)
{  
  /* search conjunction for resourceManagerContact field 
   * thus &...(subjobCpmmsType=xxx)... is the only
   * recognized RSL form for this field
   */
  if ( globus_rsl_is_boolean_and (request) ) {
    globus_list_t * subclauses;
    utils_debug (GLOBUS_DUROC_DEBUG_FLAG, "searching conjunction\n");
    subclauses = globus_rsl_boolean_get_operand_list (request);
    while ( ! globus_list_empty (subclauses) ) {
      globus_rsl_t * subclause;

      subclause = ((globus_rsl_t *)
		   globus_list_first (subclauses));

      if ( s_is_commstype_node(subclause) ) {
	(*strp) = utils_strdup (globus_rsl_value_literal_get_string (
		    globus_rsl_relation_get_single_value (subclause)));
	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "RSL subjobCommsType is >>%s<<\n",
		     (*strp));
	return GLOBUS_DUROC_SUCCESS;
      }
      
      subclauses = globus_list_rest (subclauses);
    }

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "RSL subjobCommsType defaulting to blocking\n");
    *strp = utils_strdup ("blocking-join");
    return GLOBUS_DUROC_SUCCESS;
  }
  else return GLOBUS_DUROC_ERROR_INVALID_REQUEST;
}

int globus_duroc_rsl_ismulti (globus_rsl_t *request)
{
  return globus_rsl_is_boolean_multi (request);
}

int globus_duroc_rsl_multicount (globus_rsl_t *request)
{
  if ( request == NULL ) 
    return -1;
  else if ( globus_rsl_is_boolean_multi (request) )
    return globus_list_size (globus_rsl_boolean_get_operand_list (request));
  else
    return -2;
}

#define s_is_label_node(spec) \
( globus_rsl_is_relation_eq (spec) \
  && (utils_streq_ci (globus_rsl_relation_get_attribute (spec), \
		   "label")) )

int
globus_duroc_rsl_label (globus_rsl_t *request, char ** strp)
{
  /* search conjunction for resourceManagerContact field 
   * thus &...(label=xxx)... is the only
   * recognized RSL form for this field
   */
  if ( globus_rsl_is_boolean_and (request) ) {
    globus_list_t * subclauses;
    utils_debug (GLOBUS_DUROC_DEBUG_FLAG, "searching conjunction\n");
    subclauses = globus_rsl_boolean_get_operand_list (request);
    while ( ! globus_list_empty (subclauses) ) {
      globus_rsl_t * subclause;

      subclause = ((globus_rsl_t *)
		   globus_list_first (subclauses));

      if ( s_is_label_node(subclause) ) {
	(*strp) = utils_strdup (globus_rsl_value_literal_get_string (
		    globus_rsl_relation_get_single_value (subclause)));
	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "RSL label is >>%s<<\n",
		     (*strp));
	return GLOBUS_DUROC_SUCCESS;
      }
      
      subclauses = globus_list_rest (subclauses);
    }

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "RSL label is NULL\n");
    *strp = NULL;
    return GLOBUS_DUROC_SUCCESS;
  }
  else return GLOBUS_DUROC_ERROR_INVALID_REQUEST;
}

#define s_is_count_node(spec) \
( globus_rsl_is_relation_eq (spec) \
  && (utils_streq_ci (globus_rsl_relation_get_attribute (spec), \
		   "count")) )

int globus_duroc_rsl_num_nodes (globus_rsl_t *request, int *num_nodes)
{
  /* search conjunction for count field 
   * thus &...(count=xxx)... is the only
   * recognized RSL form for this field
   */
  if ( globus_rsl_is_boolean_and (request) ) {
    globus_list_t * subclauses;
    utils_debug (GLOBUS_DUROC_DEBUG_FLAG, "searching conjunction\n");
    subclauses = globus_rsl_boolean_get_operand_list (request);
    while ( ! globus_list_empty (subclauses) ) {
      globus_rsl_t * subclause;

      subclause = ((globus_rsl_t *)
		   globus_list_first (subclauses));

      if ( s_is_count_node(subclause) ) {
	char * str;
	str = utils_strdup (globus_rsl_value_literal_get_string (
		    globus_rsl_relation_get_single_value (subclause)));
	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "RSL count is >>%s<<\n",
		     str);
	(*num_nodes) = (int) strtol (str, NULL, 0);
	return GLOBUS_DUROC_SUCCESS;
      }
      
      subclauses = globus_list_rest (subclauses);
    }

    return GLOBUS_DUROC_ERROR_INVALID_REQUEST;
  }
  else return GLOBUS_DUROC_ERROR_INVALID_REQUEST;
}

globus_list_t *globus_duroc_rsl_subrequests_list (globus_rsl_t *request)
{
  globus_list_t * subrequest_list;
  globus_rsl_t  * tree_copy;

  if ( ! globus_rsl_is_boolean_multi (request) ) {
    utils_debug (GLOBUS_DUROC_DEBUG_FLAG, "can't split non-multireq\n");
    return NULL;
  }

  tree_copy = globus_rsl_copy_recursive (request);
  subrequest_list = globus_rsl_boolean_get_operand_list (tree_copy);

  /* free extra top-level multirequest node we didn't need to copy */
  globus_rsl_free (tree_copy);

  return subrequest_list;
}

int globus_duroc_rsl_subrequests_list_free (globus_list_t *subrequests)
{
  globus_rsl_t *subreq;

  while ( subrequests != NULL ) {
    subreq = (globus_rsl_t *) globus_list_remove (&subrequests, subrequests);
    assert (subreq!=NULL);
    globus_rsl_free (subreq);
  }

  return GLOBUS_DUROC_SUCCESS;
}

int globus_duroc_rsl_stdout (globus_rsl_t *request, char ** strp)
{  
  /* search conjunction for resourceManagerContact field 
   * thus &...(label=xxx)... is the only
   * recognized RSL form for this field
   */
  if ( globus_rsl_is_boolean_and (request) ) {
    globus_list_t * subclauses;
    utils_debug (GLOBUS_DUROC_DEBUG_FLAG, "searching conjunction\n");
    subclauses = globus_rsl_boolean_get_operand_list (request);
    while ( ! globus_list_empty (subclauses) ) {
      globus_rsl_t * subclause;

      subclause = ((globus_rsl_t *)
		   globus_list_first (subclauses));

      if ( s_is_stdout_node(subclause) ) {
	(*strp) = utils_strdup (globus_rsl_value_literal_get_string (
		    globus_rsl_relation_get_single_value (subclause)));
	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "RSL stdout is >>%s<<\n",
		     (*strp));
	return GLOBUS_DUROC_SUCCESS;
      }
      
      subclauses = globus_list_rest (subclauses);
    }

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "RSL stdout is NULL\n");
    *strp = NULL;
    return GLOBUS_DUROC_SUCCESS;
  }
  else return GLOBUS_DUROC_ERROR_INVALID_REQUEST;
}

int globus_duroc_rsl_stderr (globus_rsl_t *request, char ** strp)
{  
  /* search conjunction for resourceManagerContact field 
   * thus &...(label=xxx)... is the only
   * recognized RSL form for this field
   */
  if ( globus_rsl_is_boolean_and (request) ) {
    globus_list_t * subclauses;
    utils_debug (GLOBUS_DUROC_DEBUG_FLAG, "searching conjunction\n");
    subclauses = globus_rsl_boolean_get_operand_list (request);
    while ( ! globus_list_empty (subclauses) ) {
      globus_rsl_t * subclause;

      subclause = ((globus_rsl_t *)
		   globus_list_first (subclauses));

      if ( s_is_stderr_node(subclause) ) {
	(*strp) = utils_strdup (globus_rsl_value_literal_get_string (
		    globus_rsl_relation_get_single_value (subclause)));
	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "RSL stderr is >>%s<<\n",
		     (*strp));
	return GLOBUS_DUROC_SUCCESS;
      }
      
      subclauses = globus_list_rest (subclauses);
    }

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "RSL stderr is NULL\n");
    *strp = NULL;
    return GLOBUS_DUROC_SUCCESS;
  }
  else return GLOBUS_DUROC_ERROR_INVALID_REQUEST;
}

void s_add_subrequest_output_defaults (globus_duroc_rsl_ast_t subrequest, 
				       char * gass_url)
{
  char * existing;

  existing = NULL;
  globus_duroc_rsl_stdout (subrequest, &existing);
  if (existing) {
    globus_free (existing);
  }
  else {
    /* add gass /dev/stdout */
    if ( s_is_and_node (subrequest) ) {
      globus_rsl_t *stdout_node;
      char * url;

      url = globus_malloc (sizeof (char)
			   * (utils_strlen (gass_url)
			      + utils_strlen ("/dev/stdout")
			      + 1));
      assert (url!=NULL);

      utils_sprintf (url, "%s%s", gass_url, "/dev/stdout");

      stdout_node = globus_rsl_make_relation (GLOBUS_RSL_EQ,
					      "stdout",
		      globus_rsl_value_make_sequence (
			      globus_list_cons (((void *) globus_rsl_value_make_literal (url)), 
						NULL)));
      assert (stdout_node!=NULL);

      globus_list_insert (globus_rsl_boolean_get_operand_list_ref (subrequest),
			  (void *) stdout_node);
    }
  }

  existing = NULL;
  globus_duroc_rsl_stderr (subrequest, &existing);
  if (existing) {
    globus_free (existing);
  }
  else {
    /* add gass /dev/stderr */
    if ( s_is_and_node (subrequest) ) {
      globus_rsl_t *stderr_node;
      char * url;

      url = globus_malloc (sizeof (char)
			   * (utils_strlen (gass_url)
			      + utils_strlen ("/dev/stderr")
			      + 1));
      assert (url!=NULL);

      utils_sprintf (url, "%s%s", gass_url, "/dev/stderr");

      stderr_node = globus_rsl_make_relation (GLOBUS_RSL_EQ,
					      "stderr",
		      globus_rsl_value_make_sequence (
			      globus_list_cons (((void *) globus_rsl_value_make_literal (url)), 
						NULL)));
      assert (stderr_node!=NULL);

      globus_list_insert (globus_rsl_boolean_get_operand_list_ref (subrequest),
			  (void *) stderr_node);
    }
  }
}

void globus_duroc_rsl_add_output_defaults (globus_duroc_rsl_ast_t request, 
					   char * gass_url)
{
  if ( globus_duroc_rsl_ismulti (request) ) {
    globus_list_t *child_list;

    child_list = globus_rsl_boolean_get_operand_list (request);


    while (! globus_list_empty (child_list)) {
      s_add_subrequest_output_defaults (globus_list_first (child_list),
					gass_url);
      child_list = globus_list_rest (child_list);
    }
  }
}

