/* GKrellmdk: GKrellM MLDonkey Plugin
 * Copyright (c) 2004 Christophe Badoit
 *
 * Original Author: Christophe Badoit <tof-at-tof2k-dot-com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <gkrellm2/gkrellm.h>
#include <pthread.h>

/*FIXME: should include the .h, but does not work ?*/
#include "mldonkeytools.c"

#define	CONFIG_NAME	"Gkremldk"
#define MONITOR_CONFIG_KEYWORD  "gkremldk"
#define	STYLE_NAME	"gkremldk"

static GkrellmMonitor  *monitor;
static GkrellmPanel    *panel;
static GkrellmDecal    *dl_decal;
static GkrellmDecal    *ul_decal;
static GkrellmKrell    *dl_krell;
static GkrellmKrell    *ul_krell;
static GkrellmKrell    *dl_krell_max;
static GkrellmKrell    *ul_krell_max;
static gint            style_id;

static gfloat          dl_rate = 20.4;
static gfloat          ul_rate = 3.6;

static gfloat          dl_rate_max = 50.0;
static gfloat          ul_rate_max = 8.0;

static gfloat          dl_rate_max_previous;
static gfloat          ul_rate_max_previous;

/*status flags*/
static gboolean        isPaused = FALSE;
static gboolean        isConnected = FALSE;
static gboolean        haveToReconnect = FALSE;

/*thread, connecting to mldonkey and getting rates*/
pthread_t			   donkey_thread;

/*Socket for mldonkey*/
int                     sockfd;

/*Configurations values*/
static gfloat          dl_bandwith_max = 64.0;
static gfloat          ul_bandwith_max = 16.0;
static gfloat          dl_rate_max_paused = 1;
static gfloat          ul_rate_max_paused = 1;

static gchar           *login = "admin";
static gchar           *password = "admin";
static gchar           *host = "localhost";
static gint            port = 4001;

//Configurations Widgets
static GtkWidget	*dl_bandwith_max_widget;
static GtkWidget	*ul_bandwith_max_widget;
static GtkWidget	*dl_rate_max_paused_widget;
static GtkWidget	*ul_rate_max_paused_widget;

static GtkWidget	*host_widget;
static GtkWidget	*port_widget;
static GtkWidget	*login_widget;
static GtkWidget	*password_widget;

/*Set the max_hard_download_rate value*/
void setMaxHardDownloadRate(gfloat value) {
	donkeyMsg msg;

	char buf[16];
	sprintf(buf, "%d", (int)value);
	prepareMsg(&msg, 28);
	writeString(&msg, "max_hard_download_rate");
	writeString(&msg, buf);
	sendMsg(sockfd, &msg);
	freeMsg(&msg);
}

/*Set the max_hard_upload_rate value*/
void setMaxHardUploadRate(gfloat value) {
	donkeyMsg msg;

	char buf[16];
	sprintf(buf, "%d", (int)value);
	prepareMsg(&msg, 28);
	writeString(&msg, "max_hard_upload_rate");
	writeString(&msg, buf);
	sendMsg(sockfd, &msg);
	freeMsg(&msg);
}


/*Donkey Thread*/
void *donkey_update(void *arg) {

	donkeyMsg msg;


	int n;

	while (1) {

		//Main Loop, try to connect permanently

		close(sockfd);
		isConnected = FALSE;
		haveToReconnect = FALSE;

/*		printf("Tempting Connection mldonkey://%s:%s@%s:%d\n",*/
/*				login, password, host, port);*/
		if (!donkeyConnect(&sockfd, host, port, login, password)) {
			sleep(3);
			continue;
		}


		while (!haveToReconnect) {

			// Read donkey message

			if ((n = readMsg(sockfd, &msg)) <= 0) {
/*				printf("Oooops: readMsg returned %d, either server is"*/
/*					" disconnected, or login/password are incorrect.\n",n);*/
				sleep(3);
				break;
			}

			// manage message, depending on opcode
			// See http://www.g2gui.net/wiki/index.php/GuiProtocol 

			switch(msg.opcode) {
				case 49:
					// clientStats
					readLong(&msg); // 1
					readLong(&msg); // 2
					readLong(&msg); // 3
					readLong(&msg); // 4
					readLong(&msg); // 5
					readLong(&msg); // 6
					readLong(&msg); // 7
					ul_rate = (readLong(&msg) / 1000.0); // 8
					dl_rate = (readLong(&msg) / 1000.0); // 9

					// consider we are connected when this kind of msg is received
					isConnected = TRUE;
					break;
				
				case 1:
					// optionsInfos
					{
						short int nb = readInt(&msg);
						short int i;
						for (i = 0; i < nb; i++) {
							char *key = readString(&msg);
							char *value = readString(&msg);
							if (key != NULL && value != NULL) {
								if (strncmp(key, "max_hard_upload_rate", 20) == 0)
									ul_rate_max = atof(value);
								else if (strncmp(key, "max_hard_download_rate", 22) == 0)
									dl_rate_max = atof(value);
							}
							free(key); free(value);
						}
					}
					break;

				default:
					break;
			}
			
			freeMsg(&msg);

		}

	}

}


/*Button is pressed on the panel*/
static void panel_button_press(GtkWidget *widget, GdkEvent *event) {

	gint x;
	gfloat value;
	GkrellmKrell *krell;
	void (*setValue)(gfloat);
	GdkEventButton *ev;

	ev = (GdkEventButton *)event;

	if (event->type == GDK_SCROLL) {
		GdkEventScroll *scrollEvent = (GdkEventScroll *)event;
		if (scrollEvent->direction == GDK_SCROLL_UP)
			ev->button = 4;
		else if (scrollEvent->direction == GDK_SCROLL_DOWN)
			ev->button = 5;
	}

    if (ev->button ==3 ) 
        gkrellm_open_config_window(monitor);

	// can only unpause when in paused mode
	if (isPaused && ev->button != 2)
		return;

	// which krell is clicked ?
	if (ev->y >= dl_decal->y 
			&& ev->y <= dl_krell_max->y0 + dl_krell_max->h_frame) {
		krell = dl_krell;
		setValue = &setMaxHardDownloadRate;
		value = dl_rate_max;
	} else if (ev->y >= ul_decal->y
			&& ev->y <= ul_krell_max->y0 + ul_krell_max->h_frame) {
		krell = ul_krell;
		setValue = &setMaxHardUploadRate;
		value = ul_rate_max;
	} else if (ev->button != 2)
		return;

	switch(ev->button) {
		case 1:
			// set values
			x = ev->x * krell->full_scale / (gkrellm_chart_width() - 1);
			if(x < 1) x = 1;
			if(x > 100) x = 100;
			setValue(x);
			break;

		case 2:
			// pause/unpause
			if (isPaused) {
				dl_rate_max = dl_rate_max_previous;
				setMaxHardDownloadRate(dl_rate_max);
				ul_rate_max = ul_rate_max_previous;
				setMaxHardUploadRate(ul_rate_max);
				isPaused = FALSE;
			} else {
				dl_rate_max_previous = dl_rate_max;
				ul_rate_max_previous = ul_rate_max;
				dl_rate_max = dl_rate_max_paused;
				setMaxHardDownloadRate(dl_rate_max);
				ul_rate_max = ul_rate_max_paused;
				setMaxHardUploadRate(ul_rate_max);
				isPaused = TRUE;
			}
			break;
		case 4:
			if (value < krell->full_scale)
				setValue(value + 1);
			break;
		case 5:
			if (value > 0)
				setValue(value - 1);
			break;
		default:
			return;
			break;
	}

}


/*Update krell displays*/
static void update_plugin() {

	char *buf;

	if (!isConnected) {
		// show connecting message
		gkrellm_update_krell(panel, dl_krell, 0);
		gkrellm_update_krell(panel, ul_krell, 0);
		gkrellm_update_krell(panel, dl_krell_max, 0);
		gkrellm_update_krell(panel, ul_krell_max, 0);
		gkrellm_draw_decal_text(panel, dl_decal, "Connecting ...", -1);
		gkrellm_draw_decal_text(panel, ul_decal, "", -1);

	} else {
		// show values
		gkrellm_update_krell(panel, dl_krell, dl_rate);
		gkrellm_update_krell(panel, ul_krell, ul_rate);
		gkrellm_update_krell(panel, dl_krell_max, dl_rate_max);
		gkrellm_update_krell(panel, ul_krell_max, ul_rate_max);

		buf = g_strdup_printf("%1.1f / %d%s", dl_rate, (int)dl_rate_max,
			(isPaused ? "(P)":""));
		gkrellm_draw_decal_text(panel, dl_decal, buf, -1);
		g_free(buf);

		buf = g_strdup_printf("%1.1f / %d%s", ul_rate, (int)ul_rate_max,
			(isPaused ? "(P)":""));
		gkrellm_draw_decal_text(panel, ul_decal, buf, -1);
		g_free(buf);

	}

	gkrellm_draw_panel_layers(panel);
}

static gint panel_expose_event(GtkWidget *widget, GdkEventExpose *ev) {
    gdk_draw_pixmap(widget->window,
            widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
            panel->pixmap, ev->area.x, ev->area.y, ev->area.x, ev->area.y,
            ev->area.width, ev->area.height);
    return FALSE;
}

static void create_plugin(GtkWidget *vbox, gint first_create) {
	GkrellmStyle    *style;
	GkrellmPiximage *krell_image;

	if (first_create)
		panel = gkrellm_panel_new0();

	// Create Decal & Krells
	dl_decal = gkrellm_create_decal_text(panel, "0/0",
			gkrellm_meter_textstyle(style_id),
			gkrellm_meter_style(style_id),
			0, 2, -1);
	style = gkrellm_meter_style(style_id);
	gkrellm_set_krell_expand(style, "left");
	krell_image = gkrellm_krell_meter_piximage(style_id);

	dl_krell = gkrellm_create_krell(panel, krell_image, style);
	gkrellm_set_krell_full_scale(dl_krell, dl_bandwith_max, 1);
	gkrellm_move_krell_yoff(panel, dl_krell, dl_decal->y + dl_decal->h + 4);

	dl_krell_max = gkrellm_create_krell(panel, krell_image, style);
	gkrellm_set_krell_full_scale(dl_krell_max, dl_bandwith_max, 1);
	gkrellm_move_krell_yoff(panel, dl_krell_max, dl_krell->y0 + dl_krell->h_frame + 4);


	ul_decal = gkrellm_create_decal_text(panel, "0/0",
			gkrellm_meter_textstyle(style_id),
			gkrellm_meter_style(style_id),
			0, dl_krell_max->y0 + dl_krell_max->h_frame + 4, -1);


	ul_krell = gkrellm_create_krell(panel, krell_image, style);
	gkrellm_set_krell_full_scale(ul_krell, ul_bandwith_max, 1);
	gkrellm_move_krell_yoff(panel, ul_krell, ul_decal->y + ul_decal->h + 4);

	ul_krell_max = gkrellm_create_krell(panel, krell_image, style);
	gkrellm_set_krell_full_scale(ul_krell_max, ul_bandwith_max, 1);
	gkrellm_move_krell_yoff(panel, ul_krell_max, ul_krell->y0 + ul_krell->h_frame + 4);

	gkrellm_monotonic_krell_values(dl_krell, FALSE);
	gkrellm_monotonic_krell_values(ul_krell, FALSE);
	gkrellm_monotonic_krell_values(dl_krell_max, FALSE);
	gkrellm_monotonic_krell_values(ul_krell_max, FALSE);

	gkrellm_panel_configure(panel, NULL, style);

	// Create Panel
	gkrellm_panel_create(vbox, monitor, panel);

	if (first_create) {
	    g_signal_connect(G_OBJECT (panel->drawing_area), "expose_event",
    	        G_CALLBACK(panel_expose_event), NULL);
		gtk_signal_connect(GTK_OBJECT(panel->drawing_area), "button_press_event",
		       (GtkSignalFunc) panel_button_press, NULL);
		gtk_signal_connect(GTK_OBJECT(panel->drawing_area), "scroll-event",
		       (GtkSignalFunc) panel_button_press, NULL);


		// Launch Donkey Thread
		pthread_attr_t ThreadAttr;
		pthread_attr_init(&ThreadAttr);
		//pthread_attr_setdetachstate(&ThreadAttr, PTHREAD_CREATE_DETACHED);
		pthread_create(&(donkey_thread), &ThreadAttr, donkey_update, NULL);
	}
}

/***** User Config *****/



/*Save values to file*/
static void save_plugin_config(FILE *f) {
	fprintf(f, "%s dl_bandwith_max %d\n", MONITOR_CONFIG_KEYWORD, (int)dl_bandwith_max);
	fprintf(f, "%s ul_bandwith_max %d\n", MONITOR_CONFIG_KEYWORD, (int)ul_bandwith_max);
	fprintf(f, "%s dl_rate_max_paused %d\n", MONITOR_CONFIG_KEYWORD, (int)dl_rate_max_paused);
	fprintf(f, "%s ul_rate_max_paused %d\n", MONITOR_CONFIG_KEYWORD, (int)ul_rate_max_paused);
	fprintf(f, "%s host %s\n", MONITOR_CONFIG_KEYWORD, host);
	fprintf(f, "%s port %d\n", MONITOR_CONFIG_KEYWORD, port);
	fprintf(f, "%s login %s\n", MONITOR_CONFIG_KEYWORD, login);
	fprintf(f, "%s password %s\n", MONITOR_CONFIG_KEYWORD, password);
}

/*Load values from file*/
static void load_plugin_config(gchar *config_line) {
	gchar	config_keyword[32], config_data[CFG_BUFSIZE];
	gint	n;

	if ((n = sscanf(config_line, "%31s %[^\n]",
			config_keyword, config_data)) < 1)
		return;

	if (n == 1) config_data[0] = '\0';

	if (!strcmp(config_keyword, "dl_bandwith_max"))
		sscanf(config_data, "%f", &dl_bandwith_max);
	else if (!strcmp(config_keyword, "ul_bandwith_max"))
		sscanf(config_data, "%f", &ul_bandwith_max);
	else if (!strcmp(config_keyword, "dl_rate_max_paused"))
		sscanf(config_data, "%f", &dl_rate_max_paused);
	else if (!strcmp(config_keyword, "ul_rate_max_paused"))
		sscanf(config_data, "%f", &ul_rate_max_paused);
	else if (!strcmp(config_keyword, "host"))
		host = g_strdup(config_data);
	else if (!strcmp(config_keyword, "port"))
		sscanf(config_data, "%d", &port);
	else if (!strcmp(config_keyword, "login"))
		login = g_strdup(config_data);
	else if (!strcmp(config_keyword, "password"))
		password = g_strdup(config_data);
}

/*Get values from config panel*/
static void apply_plugin_config(void) {
    GtkSpinButton *spin;
	gchar *buf;
	gint i;
	gboolean changed = FALSE;

    spin = GTK_SPIN_BUTTON(dl_bandwith_max_widget);
    dl_bandwith_max = gtk_spin_button_get_value_as_int(spin);
    spin = GTK_SPIN_BUTTON(ul_bandwith_max_widget);
    ul_bandwith_max = gtk_spin_button_get_value_as_int(spin);

    spin = GTK_SPIN_BUTTON(dl_rate_max_paused_widget);
    dl_rate_max_paused = gtk_spin_button_get_value_as_int(spin);
    spin = GTK_SPIN_BUTTON(ul_rate_max_paused_widget);
    ul_rate_max_paused = gtk_spin_button_get_value_as_int(spin);

	gkrellm_set_krell_full_scale(dl_krell, dl_bandwith_max, 1);
	gkrellm_set_krell_full_scale(dl_krell_max, dl_bandwith_max, 1);
	gkrellm_set_krell_full_scale(ul_krell, ul_bandwith_max, 1);
	gkrellm_set_krell_full_scale(ul_krell_max, ul_bandwith_max, 1);

	buf = gkrellm_gtk_entry_get_text(&host_widget);
	if (strcmp(buf, host) != 0) {
		host = strdup(buf);
		changed = TRUE;
	}

	buf = gkrellm_gtk_entry_get_text(&port_widget);
	i = atof(buf);
	if (i != port) {
		port = i;
		changed = TRUE;
	}

	buf = gkrellm_gtk_entry_get_text(&login_widget);
	if (strcmp(buf, login) != 0) {
		login = strdup(buf);
		changed = TRUE;
	}

	buf = gkrellm_gtk_entry_get_text(&password_widget);
	if (strcmp(buf, password) != 0) {
		password = strdup(buf);
		changed = TRUE;
	}

	haveToReconnect = changed;
}

static gchar	*plugin_info_text[] =
{
_("<h>Setup Notes\n"),
_("This plugin shows the current download/upload rates of mldonkey.\n"),
_("It also shows the current maximum upload/downlad rate\n"),
_("(max_hard_download_rate and max_hard_upload_rate options\n"),
_("in mldonkey), and permit you to change them on the fly by\n"),
_("left-clicking on the krells.\n"),
_("You can also use the mouse wheel to change the values.\n"),
_("It's useful to set up the "),
_("<b>Your Bandwidth "),
_("options, since\n"),
_("they describe the maximum size of the krell (and you won't be\n"),
_("able to change the maximum rate beyond these options).\n"),
_("If you middle-click on the krells, it will set rates in "),
_("<b>paused\n"),
_("mode: it means set the values you specify in the\n"),
_("<b>Values when Paused "),
_("options panel. You won't be able\n"),
_("to change krells values until you unset the pause (by\n"),
_("middle-clicking again).\n"),
"\n\n\n",
"Gkremldk - MLDonkey Gkrellm Plugin - version 0.9.4\n",
"Copyright (c) 2004 Christophe Badoit\n",
"tof-at-tof2k-dot-com\n",
"http://www.tof2k.com\n",
"\n",
"Released under the GNU Public License\n"
};

/*Create the plugin config/help panel */
static void create_plugin_tab(GtkWidget *tab_vbox) {
	GtkWidget	*tabs, *text, *label;
	GtkWidget	*table, *vbox, *vbox1, *vbox2, *hbox;
	gint		i;
	gchar       *buf;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	// Plugin Options
	vbox = gkrellm_gtk_framed_notebook_page(tabs, "Options");
	vbox1 = gkrellm_gtk_framed_vbox(vbox,
			_("Your Bandwidth"), 4, FALSE, 0, 2);
	gkrellm_gtk_spin_button(vbox1, &dl_bandwith_max_widget, (gfloat)
			dl_bandwith_max, 5, 2000.0, 1.0, 5.0, 0, 60, NULL, NULL, FALSE,
			_("kB download (i.e. 64)"));
	gkrellm_gtk_spin_button(vbox1, &ul_bandwith_max_widget, (gfloat)
			ul_bandwith_max, 5, 2000.0, 1.0, 5.0, 0, 60, NULL, NULL, FALSE,
			_("kB upload (i.e. 16)"));

	vbox2 = gkrellm_gtk_framed_vbox(vbox,
			_("Values when Paused"), 4, FALSE, 0, 2);
	gkrellm_gtk_spin_button(vbox2, &dl_rate_max_paused_widget, (gfloat)
			dl_rate_max_paused, 1, 2000.0, 1.0, 5.0, 0, 60, NULL, NULL, FALSE,
			_("kB download (i.e. 1)"));
	gkrellm_gtk_spin_button(vbox2, &ul_rate_max_paused_widget, (gfloat)
			ul_rate_max_paused, 1, 2000.0, 1.0, 5.0, 0, 60, NULL, NULL, FALSE,
			_("kB upload (i.e. 1)"));


	// Server Options
	
	vbox = gkrellm_gtk_framed_notebook_page(tabs, _("Server Setup"));

	vbox1 = gkrellm_gtk_framed_vbox_end(vbox, NULL, 4, FALSE, 0, 2);
	table = gtk_table_new(5, 2, FALSE /*non-homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 2);
	gtk_box_pack_start(GTK_BOX(vbox1), table, FALSE, FALSE, 2);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 0, 1,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Server host name : "));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	host_widget = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(host_widget), 255);
	gtk_table_attach_defaults(GTK_TABLE(table), host_widget,
				1, 2, 0, 1);
	gtk_entry_set_text(GTK_ENTRY(host_widget),
				host);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 1, 2,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Server port number : "));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	port_widget = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(port_widget), 6);
	gtk_table_attach_defaults(GTK_TABLE(table), port_widget, 1, 2, 1, 2);
	buf = g_strdup_printf("%d", port);
	gtk_entry_set_text(GTK_ENTRY(port_widget), buf);
	free(buf);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 2, 3,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Server login (user name) : "));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	login_widget = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(login_widget), 20);
	gtk_table_attach_defaults(GTK_TABLE(table), login_widget, 1, 2, 2, 3);
	gtk_entry_set_text(GTK_ENTRY(login_widget), login);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 3, 4,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Server password : "));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	password_widget = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(password_widget), 20);
	gtk_table_attach_defaults(GTK_TABLE(table), password_widget, 1, 2, 3, 4);
	gtk_entry_set_text(GTK_ENTRY(password_widget), password);

	// Help/About
	vbox = gkrellm_gtk_framed_notebook_page(tabs, "Info");
	text = gkrellm_gtk_scrolled_text_view(vbox, NULL,
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(plugin_info_text)/sizeof(gchar *); ++i)
		gkrellm_gtk_text_view_append(text, plugin_info_text[i]);
	}


/*Monitor Infos*/
static GkrellmMonitor	plugin_mon	=
	{
	CONFIG_NAME,        /* Title for config clist.   */
	0,                  /* Id,  0 if a plugin       */
	create_plugin,      /* The create function      */
	update_plugin,      /* The update function      */
	create_plugin_tab,      /* The create_plugin_tab() config function */
	apply_plugin_config,    /* The apply_plugin_config() function      */

	save_plugin_config,     /* The save_plugin_config() function  */
	load_plugin_config,     /* The load_plugin_config() function  */
	MONITOR_CONFIG_KEYWORD, /* config keyword                     */

	NULL,               /* Undefined 2	*/
	NULL,               /* Undefined 1	*/
	NULL,               /* private	*/

	MON_MAIL,           /* Insert plugin before this monitor			*/

	NULL,               /* Handle if a plugin, filled in by GKrellM     */
	NULL                /* path if a plugin, filled in by GKrellM       */
	};

/*Initialize Plugin*/
GkrellmMonitor * gkrellm_init_plugin(void)
{
	style_id = gkrellm_add_meter_style(&plugin_mon, STYLE_NAME);
	monitor = &plugin_mon;

	return &plugin_mon;
}

