/*
 *  
 *  $Id: controladorextensiones.cpp 3893 2011-06-21 13:01:56Z tovar $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 *
 */
#include <main/entorno.h>
#include <main/controllers/controladorextensiones.h>
#include "controladorlog.h"

#include <string>
#include <map>
#include <cctype>
#include <algorithm>
#include <exception>
#include <wx/dir.h>
#include <wx/file.h>
#include <wx/filename.h>
#include <main/controllers/configurationcontroller.h>
#include <wx/dynlib.h>
#include <algorithm>
#include <set>
#include <prvext/prvext.h>

#if defined(__WXOSX__)
	#include <dlfcn.h>
	#define DLLEXT wxT("*.dylib")
#elif defined(__WXGTK__)
	#include <dlfcn.h>
	#define DLLEXT wxT("*.so")
#elif defined(_WINDOWS)
	#define DLLEXT wxT("*.dll")
#else
	#error Plataforma no soportada
#endif

GNC::GCS::IControladorExtensiones::IControladorExtensiones() {}
GNC::GCS::IControladorExtensiones::~IControladorExtensiones() {}

GNC::ControladorExtensiones* GNC::ControladorExtensiones::m_pInstance = 0;

GNC::ControladorExtensiones::ControladorExtensiones()
{
	m_Manager.Load();
	for (GADVAPI::PrivateExtensionManager::iterator it= m_Manager.begin(); it != m_Manager.end(); ++it)
	{
		RegistrarModulo((*it).second->InitializeLibrary(GNC::Entorno::Instance()));
	}
}

GNC::ControladorExtensiones::~ControladorExtensiones()
{
	DesRegistrarModulos();
}

GNC::ControladorExtensiones* GNC::ControladorExtensiones::Instance()
{
	if (m_pInstance == NULL) {
		m_pInstance = new GNC::ControladorExtensiones();
	}
	return m_pInstance;
}

void GNC::ControladorExtensiones::FreeInstance()
{
	if (m_pInstance != NULL) {
		delete m_pInstance;
	}
	m_pInstance = 0;
}

void GNC::ControladorExtensiones::RegistrarModulo(GNC::GCS::IControladorModulo* pCtrlModulo)
{

	if (pCtrlModulo == NULL) {
		return;
	}

	pCtrlModulo->RegistrarConfiguracion();

	const std::string uid = pCtrlModulo->GetUID();
	m_Modulos[uid] = pCtrlModulo;
	NotificarRegistro(pCtrlModulo);
}

void GNC::ControladorExtensiones::DesRegistrarModulo(GNC::GCS::IControladorModulo* pCtrlModulo)
{
	if (pCtrlModulo == NULL) {
		return;
	}

	NotificarDesRegistro(pCtrlModulo);

	m_Modulos.erase(pCtrlModulo->GetUID());
	delete pCtrlModulo;
}

void GNC::ControladorExtensiones::DesRegistrarModulos()
{
	GNC::ControladorExtensiones::IteradorListaModulos it;

	for (it = m_Modulos.begin(); it != m_Modulos.end(); it++) {
		GNC::GCS::IControladorModulo* item = (*it).second;
		NotificarDesRegistro(item);
		delete item;
	}
	m_Modulos.clear();
}

GNC::GCS::IControladorModulo* GNC::ControladorExtensiones::ObtenerModulo(const std::string& idModulo)
{
	GNC::GCS::IControladorModulo* cm = NULL;
	IteradorListaModulos it = m_Modulos.find(idModulo);
	if (it != m_Modulos.end()) {
		cm = (*it).second;
	}
	return cm;
}


  
bool compareListaModulos(GNC::GCS::IControladorModulo* s1, GNC::GCS::IControladorModulo* s2)
{
	return s1->GetPriority()< s2->GetPriority();
}

std::list<GNC::GCS::IControladorModulo*> GNC::ControladorExtensiones::ModulosOrdenados()
{
	std::list<GNC::GCS::IControladorModulo*> listaModulosPriorizados;
	for(ListaModulos::iterator it = m_Modulos.begin(); it!= m_Modulos.end(); ++it)
	{
		listaModulosPriorizados.push_back((*it).second);
	}

	listaModulosPriorizados.sort(compareListaModulos);	

	return listaModulosPriorizados;
}

const GNC::ControladorExtensiones::ListaModulos& GNC::ControladorExtensiones::Modulos() const
{
	return m_Modulos;
}

// Realización de la interfaz IControladorExtensiones
GADVAPI::PrivateExtensionManager& GNC::ControladorExtensiones::GetPrivateExtensionsManager()
{
	return m_Manager;
}

void GNC::ControladorExtensiones::NotificarRegistro(GNC::GCS::IControladorModulo* pModulo)
{
	GNC::Entorno* pEntorno = GNC::Entorno::Instance();
	Entorno::ListaObservadoresExtensiones::const_iterator it;
	for (it = pEntorno->ObservadoresExtensiones.begin(); it != pEntorno->ObservadoresExtensiones.end(); it++) {
		GNC::GCS::IObservadorExtensiones* ne = *it;
		ne->OnModuloCargado(pModulo);
	}
}

void GNC::ControladorExtensiones::NotificarDesRegistro(GNC::GCS::IControladorModulo* pModulo)
{
	GNC::Entorno* pEntorno = GNC::Entorno::Instance();
	Entorno::ListaObservadoresExtensiones::const_iterator it;
	for (it = pEntorno->ObservadoresExtensiones.begin(); it != pEntorno->ObservadoresExtensiones.end(); it++) {
		GNC::GCS::IObservadorExtensiones* ne = *it;
		ne->OnModuloDescargado(pModulo);
	}
}

//-------------------------------------------------------------------------------------------
// Helpers

inline std::wstring StringToWString(const std::string& str) {
	std::wstring temp(str.length(),L' ');
	std::copy(str.begin(), str.end(), temp.begin());
	return temp;
}

//-------------------------------------------------------------------------------------------
// PrivateExtensionManager

GADVAPI::PrivateExtensionManager::PrivateExtensionManager() : GADVAPI::PrivateExtensionManager::Base()
{
	m_Loaded = false;
}

GADVAPI::PrivateExtensionManager::~PrivateExtensionManager()
{
	UnLoad();
}

PrivateExtension* GADVAPI::PrivateExtensionManager::GetExtension(std::string nombre)
{
	Base& base = *this;
	if (base.find(nombre) != base.end()) {
		return base[nombre];
	} else {
		return NULL;
	}
}

void GADVAPI::PrivateExtensionManager::Load()
{
	if (!m_Loaded) {
		
		wxDir dir;
		wxString rutas[1] = {
			FROMPATH(GNC::Entorno::Instance()->GetPluginsPath())
		};

		wxString fileName;
		wxString filePath;
		for (int i = 0; i < 1; i++) {
			if (dir.Exists(rutas[i]) && dir.Open(rutas[i])) {
				bool cont = dir.GetFirst(&fileName, DLLEXT, wxDIR_FILES);
				while (cont) {
					filePath = rutas[i] + wxFileName::GetPathSeparator() + fileName;

					//se busca el punto de entrada...
					wxDynamicLibrary* dll = new wxDynamicLibrary();
					//std::cout << "Trying to load: " << std::string(TOPATH(filePath)).c_str() << std::endl;
					if (dll->Load(filePath)) {
						//std::cout << "Library loaded" << std::endl;
					}
					else {
						//std::cout << "Fallo al cargar" << std::endl;
					}
					if(!dll->IsLoaded())
					{
						//std::cout << "Lib not loaded!" << std::endl;
						delete dll;
					}
					else {
#if defined(_DEBUG)
						//Create a valid function pointer using the function pointer type in plugin.h
						wxDYNLIB_FUNCTION(CreatePrivateExtension_function,CreatePrivateExtensionDebug,*dll);
						//check if the function is found
						if(pfnCreatePrivateExtensionDebug)
						{
							PrivateExtension* pExtension = pfnCreatePrivateExtensionDebug();
#else
						//Create a valid function pointer using the function pointer type in plugin.h
						wxDYNLIB_FUNCTION(CreatePrivateExtension_function,CreatePrivateExtension,*dll);
						//check if the function is found
						if(pfnCreatePrivateExtension)
						{
							PrivateExtension* pExtension = pfnCreatePrivateExtension();
#endif
							if (Registrar(pExtension)) {
								pExtension->AsignPath(std::string(TOPATH(filePath)));
								m_instancias.push_back(dll);
							} else {
								delete dll;
							}
						} else {
							delete dll;
						}
					}

					cont = dir.GetNext(&fileName);
				}
			}
		}

		m_Loaded = true;
	}
}

void GADVAPI::PrivateExtensionManager::UnLoad()
{
	if (m_Loaded) {
		//delete private extensions
		for (Base::iterator it = Base::begin(); it != Base::end(); ++it) {
			delete (*it).second;
		}
		//delete dlls
		for(ListaInstancias::iterator it = m_instancias.begin(); it != m_instancias.end(); it++)
		{
			delete (*it);			
		}
		m_instancias.clear();
		
		m_Loaded = false;
	}
}

void GADVAPI::PrivateExtensionManager::Reload()
{
	UnLoad();
	Load();
}

GADVAPI::PrivateExtensionManager::iterator GADVAPI::PrivateExtensionManager::begin() {
	return Base::begin();
}

GADVAPI::PrivateExtensionManager::iterator GADVAPI::PrivateExtensionManager::end() {
	return Base::end();
}

bool GADVAPI::PrivateExtensionManager::Registrar(PrivateExtension* ext)
{
	if (ext->GetCoreVersion() != GINKGO_VERSION || ext->GetCoreSubVersion() != GINKGO_SUB_VERSION) 
	{
		LOG_WARN("Extensions", _Std("Extension ") << ext->GetSID() << "(" << ext->GetPath() << ")"<< _Std(" is compatible with Ginkgo CADx ") << ext->GetCoreVersion() << "." << ext->GetCoreSubVersion());
		return false;
	}
	Base& base = *this;
	const std::string& sid = ext->GetSID();
	Base::iterator it = base.find(sid);
	if (it == base.end()) {
		base[sid] = ext;
		return true;
	} else {
		//error extension registrada dos veces
		return false;
	}
}
