/*
 *  
 *  $Id$
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 */
#pragma once
#include <api/api.h>
#include <list>
#include <string>
#include <ostream>

#define STRINGIFY(x) #x
#define TOSTRING(x) STRINGIFY(x)
#define GLOC() (__FILE__ ":" TOSTRING(__LINE__))

//----------------------------------------------------------------------------------------------------
//region Forward Declarations
class wxCriticalSection;
class wxSemaphore;

namespace GNC {
	namespace GCS {
		class ILocker;
		class WaitQueueTask;
	}
}
//endregion


namespace GNC {
	namespace GCS {

		//----------------------------------------------------------------------------------------------------
		//region Interfaz de soporte de cerrojos
		//----------------------------------------------------------------------------------------------------
		class EXTAPI ILockable
		{
			//----------------------------------------------------------------------------------------------------
			//region Constructor y destructor
			public:

				ILockable();
				~ILockable();
			//endregion

			//----------------------------------------------------------------------------------------------------
			//region Interfaz de exclusion mutua
			public:

				void Lock(const std::string& loc );

				void UnLock(const std::string& loc );

				bool IsLocked() const;
			//enregion

			//----------------------------------------------------------------------------------------------------
			//region Interfaz de exclusion mutua desde automatico
			private:

				void AutoLock(ILocker* pLocker, const std::string& loc = "");

				void AutoUnLock(ILocker* pLocker);
			//enregion

			//----------------------------------------------------------------------------------------------------
			//region Interfaz de subscripcion
			private:

				ILocker* m_pLocker; // ILocker que ha bloqueado el cerrojo.
			//endregion

			//----------------------------------------------------------------------------------------------------
			//region Atributos
			private:

				bool               m_IsLocked;
				std::string        m_LocBloqueo;
				wxCriticalSection* m_pCS;             // El cerrojo real.
				wxCriticalSection* m_pCSRegistro;     // Cerrojo para registrar el ocasionador del bloqueo.
			//endregion

			friend class ILocker;
		};
		//endregion

		//----------------------------------------------------------------------------------------------------
		//region Helper de bloqueo automatico
		//----------------------------------------------------------------------------------------------------
		class EXTAPI ILocker
		{
			//----------------------------------------------------------------------------------------------------
			//region Construccion y destruccion
			public:

				ILocker( GNC::GCS::ILockable& pLockable, const std::string& loc = "" );
				ILocker( GNC::GCS::ILockable* pLockable, const std::string& loc = "" );
				~ILocker();
			//endregion

			//----------------------------------------------------------------------------------------------------
			//region Atributos
			private:

				ILockable* m_pLockable;
				std::string m_LocInstanciacion;
			//endregion

			friend class ILockable;
		};
		//endregion

		class EXTAPI TipoListaEspera : public std::list<WaitQueueTask*, std::allocator<WaitQueueTask*> > {};

		//----------------------------------------------------------------------------------------------------
		//region Cola de espera
		//----------------------------------------------------------------------------------------------------
		class EXTAPI WaitQueue : public GNC::GCS::ILockable {

			//----------------------------------------------------------------------------------------------------
			//region Definiciones
			public:

			//endregion

			//----------------------------------------------------------------------------------------------------
			//region Constructor y destructor
			public:

				WaitQueue();
				~WaitQueue();
			//endregion

			//----------------------------------------------------------------------------------------------------
			//region Interfaz general
			public:

				TipoListaEspera::size_type Size(const std::string& loc = "");

				bool Empty(const std::string& loc = "");

				bool NotEmpty(const std::string& loc = "");

				TipoListaEspera::size_type Pending(const std::string& loc = "");

				/* Espera bloqueante hasta que alguna de las tareas registradas termine */
				bool Wait(const std::string& loc);

				/* Espera bloqueante con limite de tiempo hasta que alguna de las tareas registradas termine */
				bool Wait(long msecs, const std::string& loc = "");

				/* Registra una tarea para su espera */
				void RegistrarEspera(WaitQueueTask* task, const std::string& loc = "");

				/* Itera sobre todas las tareas pendientes y si tienen el flag de terminacion activo, vuelve a enviarles una senyal de terminacion */
				void TerminarPendientes(const std::string& loc = "");
			//endregion

			//----------------------------------------------------------------------------------------------------
			//region Interfaz privada


			//endregion

			//----------------------------------------------------------------------------------------------------
			//region Atributos
			private:

				wxSemaphore*               m_pSignaler;

				TipoListaEspera m_ListaEspera;
				TipoListaEspera::size_type m_Size;

			//endregion

			friend class WaitQueueTask;
		};
		//endregion
	}
}

namespace GNC {
	namespace GCS {

		class EXTAPI TipoListaNotificaciones : public std::list<WaitQueue*, std::allocator<WaitQueue*> > {};

		//----------------------------------------------------------------------------------------------------
		// Tarea de la cola de espera
		//----------------------------------------------------------------------------------------------------
		class EXTAPI WaitQueueTask : public GNC::GCS::ILockable {

			//----------------------------------------------------------------------------------------------------
			//region Definiciones
			public:

			//endregion

			//region Interfaz general
			public:
				/* Senala todas las waitqueue a las que se ha registrado esta tarea */
				void Signal(const std::string& loc = "");

				/* Metodo que activa el flag de terminacion e invoca a DoTerminar() */
				void Terminar();
			//endregion

			//region Interfaz privada
			protected:
				/* Metodo privado a implementar en la tarea. Implementa la funcionalidad necesaria para que la tarea se detenga (de manera activa o pasiva, segun corresponda) */
				virtual void DoTerminar() = 0;

			//endregion

			//----------------------------------------------------------------------------------------------------
			//region Constructor y destructor
			public:

				WaitQueueTask();

				~WaitQueueTask();

			//endregion

			//region Atributos
			protected:

				TipoListaNotificaciones            m_ListaNotificacion;
				bool                               m_NotificarTerminacion;
			//endregion

			friend class WaitQueue;
		};
		//endregion
	}
}
