--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.AL.Buffer
-- Copyright   :  (c) Sven Panne 2003-2005
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven.panne@aedion.de
-- Stability   :  provisional
-- Portability :  portable
--
--------------------------------------------------------------------------------

module Sound.OpenAL.AL.Buffer (
   Format(..), BufferSize, Frequency, BufferData(..),
   Buffer, bufferData
) where

import Control.Monad ( liftM )
import Foreign.Marshal.Alloc ( alloca )
import Foreign.Marshal.Array ( withArrayLen, peekArray, allocaArray )
import Foreign.Ptr ( Ptr, nullPtr )
import Foreign.Storable ( Storable(peek) )
import Graphics.Rendering.OpenGL.GL.BufferObjects ( ObjectName(..) )
import Graphics.Rendering.OpenGL.GL.StateVar ( StateVar, makeStateVar )
import Sound.OpenAL.AL.BasicTypes
import Sound.OpenAL.AL.Format ( Format(..), marshalFormat )
import Sound.OpenAL.AL.ALboolean ( unmarshalALboolean )

--------------------------------------------------------------------------------

data BufferData a = BufferData Format (Ptr a) BufferSize Frequency
#ifdef __HADDOCK__
-- Help Haddock a bit, because it doesn't do any instance inference.
instance Eq (BufferData a)
instance Ord (BufferData a)
instance Show (BufferData a)
#else
   deriving ( Eq, Ord, Show )
#endif

type BufferSize = ALsizei

type Frequency = ALsizei   -- or Float???

--------------------------------------------------------------------------------

newtype Buffer = Buffer { bufferID :: ALuint }
   deriving ( Eq, Ord, Show )

instance ObjectName Buffer where
   genObjectNames n =
      allocaArray n $ \buf -> do
        alGenBuffers (fromIntegral n) buf
        liftM (map Buffer) $ peekArray n buf

   deleteObjectNames buffers =
      withArrayLen (map bufferID buffers) $
         alDeleteBuffers . fromIntegral

   isObjectName = liftM unmarshalALboolean . alIsBuffer . bufferID

foreign import CALLCONV unsafe "alGenBuffers"
   alGenBuffers :: ALsizei -> Ptr ALuint -> IO ()

foreign import CALLCONV unsafe "alDeleteBuffers"
   alDeleteBuffers :: ALsizei -> Ptr ALuint -> IO ()

foreign import CALLCONV unsafe "alIsBuffer"
   alIsBuffer :: ALuint -> IO ALboolean

--------------------------------------------------------------------------------

bufferData :: Buffer -> StateVar (BufferData a)
bufferData buffer = makeStateVar (getBufferData buffer) (setBufferData buffer)

getBufferData :: Buffer -> IO (BufferData a)
getBufferData buffer = do
   channels <- getBufferi buffer Channels
   bits <- getBufferi buffer Bits
   raw <- return nullPtr -- ToDo: AL_DATA query missing!!!
   size <- getBufferi buffer Size
   frequency <- getBufferi buffer Frequency
   return $ BufferData (makeFormat channels bits) raw size frequency

setBufferData :: Buffer -> BufferData a -> IO ()
setBufferData buffer (BufferData format raw size frequency) =
      alBufferData buffer (marshalFormat format) raw size frequency

foreign import CALLCONV unsafe "alBufferData"
   alBufferData :: Buffer -> ALenum -> Ptr a -> ALsizei -> ALsizei -> IO ()

--------------------------------------------------------------------------------

-- ToDo: What about IMAADPCMMono16, IMAADPCMStereo16, Vorbis...?
makeFormat :: ALint -> ALint -> Format
makeFormat 1  8 = Mono8
makeFormat 2  8 = Stereo8
makeFormat 4  8 = Quad8
makeFormat 1 16 = Mono16
makeFormat 2 16 = Stereo16
makeFormat 4 16 = Quad16
makeFormat channels bits =
   error ("makeFormat: illegal values " ++ show (channels, bits))

--------------------------------------------------------------------------------

data BufferQuery =
     Frequency
   | Bits
   | Channels
   | Size
   -- AL_DATA ???

marshalBufferQuery :: BufferQuery -> ALenum
marshalBufferQuery x = case x of
   Frequency -> 0x2001
   Bits -> 0x2002
   Channels -> 0x2003
   Size -> 0x2004

--------------------------------------------------------------------------------

getBufferi :: Num a => Buffer -> BufferQuery -> IO a
getBufferi buffer query =
   alloca $ \buf -> do
      alGetBufferi buffer (marshalBufferQuery query) buf
      liftM fromIntegral $ peek buf

foreign import CALLCONV unsafe "alGetBufferi"
   alGetBufferi :: Buffer -> ALenum -> Ptr ALint -> IO ()
