--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.ALUT.Initialization
-- Copyright   :  (c) Sven Panne 2003
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven_panne@yahoo.com
-- Stability   :  provisional
-- Portability :  portable
--
--------------------------------------------------------------------------------

module Sound.OpenAL.ALUT.Initialization (
   runOpenAL, getArgsAndRunOpenAL
) where

import Data.List ( genericLength )
import Foreign.C.String ( CString, withCString, peekCString )
import Foreign.C.Types ( CInt )
import Foreign.Marshal.Array ( withArray0, peekArray )
import Foreign.Marshal.Utils ( with, withMany )
import Foreign.Storable ( Storable(peek) )
import Foreign.Ptr ( Ptr, nullPtr )
import System.Environment ( getProgName, getArgs )
import Sound.OpenAL.ALUT.Exception ( finally )

--------------------------------------------------------------------------------

initialize :: String
           -> [String]
           -> IO [String]
initialize prog args =
   with (1 + genericLength args) $ \argcBuf ->
   withMany withCString (prog : args) $ \argvPtrs ->
   withArray0 nullPtr argvPtrs $ \argvBuf -> do
   alutInit argcBuf argvBuf
   newArgc <- peek argcBuf
   newArgvPtrs <- peekArray (fromIntegral newArgc) argvBuf
   newArgv <- mapM peekCString newArgvPtrs
   return $ tail newArgv

foreign import CALLCONV unsafe "alutInit"
   alutInit :: Ptr CInt -> Ptr CString -> IO  ()

foreign import CALLCONV unsafe "alutExit"
   alutExit :: IO  ()

runOpenAL :: String              -- ^ The program name.
          -> [String]            -- ^ The command line arguments
          -> ([String] -> IO a)  -- ^ Action expecting non-OpenAL command line arguments
          -> IO a
runOpenAL prog args action =
   (initialize prog args >>= action) `finally` alutExit

getArgsAndRunOpenAL :: ([String] -> IO a) -> IO a
getArgsAndRunOpenAL action = do
   prog <- getProgName
   args <- getArgs
   runOpenAL prog args action
