/*
 * ggcov - A GTK frontend for exploring gcov coverage data
 * Copyright (c) 2002-2004 Greg Banks <gnb@alphalink.com.au>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _ggcov_cov_types_H_
#define _ggcov_cov_types_H_ 1

#include "common.h"
#include "list.H"

#if HAVE_STDINT_H
typedef uint64_t  	    	count_t;
#define COV_COUNT_MAX	    	UINT64_MAX
#else
typedef unsigned long long int  count_t;
#define COV_COUNT_MAX	    	18446744073709551615ULL
#endif

class cov_file_t;
class cov_function_t;
class cov_block_t;
class cov_arc_t;
struct cov_callnode_t;
struct cov_callarc_t;
struct cov_read_state_t;

struct cov_location_t
{
    char *filename;
    unsigned long lineno;
};

struct cov
{
    enum status_t
    {
	COVERED,
	PARTCOVERED,
	UNCOVERED,
	UNINSTRUMENTED,
	SUPPRESSED
    };
    enum _constants
    {
	NUM_STATUS=5
    };

private:
    cov() { }    /* prevent instantiation */
};

class cov_stats_t
{
public:
    /*
     * c'tor zeroes out counters, rather than relying on
     * global operator new, because cov_stats_t is mostly
     * used as an auto or member variable, not allocated.
     */
    cov_stats_t()
    {
    	memset(this, 0, sizeof(*this));
    }
    
    void
    clear()
    {
    	memset(this, 0, sizeof(*this));
    }

    void
    accumulate(const cov_stats_t *st)
    {
    	unsigned int i;
	unsigned long *uthis = (unsigned long *)this;
	unsigned long *ust = (unsigned long *)st;
	
	for (i = 0 ; i < sizeof(*this)/sizeof(unsigned long) ; i++)
	    *uthis++ += *ust++;
    }
    
    /* there are no partial blocks; blocks are the unit of coverage */
    unsigned long blocks_executed() const
    {
    	return blocks_[cov::COVERED];
    }
    unsigned long blocks_total() const
    {
	return blocks_[cov::COVERED] +
	       blocks_[cov::UNCOVERED];
    }
    unsigned long blocks_suppressed() const
    {
    	return blocks_[cov::SUPPRESSED];
    }
    const unsigned long *blocks_by_status() const
    {
    	return blocks_;
    }
    double blocks_fraction() const
    {
    	unsigned long n = blocks_executed();
	unsigned long d = blocks_total();
	return (d == 0 ? 0.0 : (double)n / (double)d);
    }
    /* for sorting objects: -1 forces uninstrumented last */
    double blocks_sort_fraction() const
    {
    	unsigned long n = blocks_executed();
	unsigned long d = blocks_total();
	return (d == 0 ? -1.0 : (double)n / (double)d);
    }

    unsigned long lines_executed() const
    {
	return lines_[cov::COVERED] +
	       lines_[cov::PARTCOVERED];
    }
    unsigned long lines_full() const
    {
    	return lines_[cov::COVERED];
    }
    unsigned long lines_partial() const
    {
    	return lines_[cov::PARTCOVERED];
    }
    unsigned long lines_total() const
    {
	return lines_[cov::COVERED] +
	       lines_[cov::PARTCOVERED] +
	       lines_[cov::UNCOVERED];
    }
    unsigned long lines_suppressed() const
    {
    	return lines_[cov::SUPPRESSED];
    }
    const unsigned long *lines_by_status() const
    {
    	return lines_;
    }
    double lines_fraction() const
    {
    	unsigned long n = lines_executed();
	unsigned long d = lines_total();
	return (d == 0 ? 0.0 : (double)n / (double)d);
    }
    /* for sorting objects: -1 forces uninstrumented last */
    double lines_sort_fraction() const
    {
    	unsigned long n = lines_executed();
	unsigned long d = lines_total();
	return (d == 0 ? -1.0 : (double)n / (double)d);
    }

    unsigned long functions_executed() const
    {
	return functions_[cov::COVERED] +
	       functions_[cov::PARTCOVERED];
    }
    unsigned long functions_full() const
    {
    	return functions_[cov::COVERED];
    }
    unsigned long functions_partial() const
    {
    	return functions_[cov::PARTCOVERED];
    }
    unsigned long functions_total() const
    {
	return functions_[cov::COVERED] +
	       functions_[cov::PARTCOVERED] +
	       functions_[cov::UNCOVERED];
    }
    unsigned long functions_suppressed() const
    {
    	return functions_[cov::SUPPRESSED];
    }
    const unsigned long *functions_by_status() const
    {
    	return functions_;
    }
    double functions_fraction() const
    {
    	unsigned long n = functions_executed();
	unsigned long d = functions_total();
	return (d == 0 ? 0.0 : (double)n / (double)d);
    }
    /* for sorting objects: -1 forces uninstrumented last */
    double functions_sort_fraction() const
    {
    	unsigned long n = functions_executed();
	unsigned long d = functions_total();
	return (d == 0 ? -1.0 : (double)n / (double)d);
    }

    /* calls happen on block boundaries hence cannot be partial */
    unsigned long calls_executed() const
    {
    	return calls_[cov::COVERED];
    }
    unsigned long calls_total() const
    {
	return calls_[cov::COVERED] +
	       calls_[cov::UNCOVERED];
    }
    unsigned long calls_suppressed() const
    {
    	return calls_[cov::SUPPRESSED];
    }
    const unsigned long *calls_by_status() const
    {
    	return calls_;
    }
    double calls_fraction() const
    {
    	unsigned long n = calls_executed();
	unsigned long d = calls_total();
	return (d == 0 ? 0.0 : (double)n / (double)d);
    }
    /* for sorting objects: -1 forces uninstrumented last */
    double calls_sort_fraction() const
    {
    	unsigned long n = calls_executed();
	unsigned long d = calls_total();
	return (d == 0 ? -1.0 : (double)n / (double)d);
    }


    /*
     * Branches happen on block boundaries hence cannot be partial,
     * but for storage purposes we use [PARTCOVERED]=test executed
     * but branch not taken and [COVERED]=branch taken.
     */
    unsigned long branches_executed() const
    {
    	return branches_[cov::COVERED] +
	       branches_[cov::PARTCOVERED];
    }
    unsigned long branches_taken() const
    {
    	return branches_[cov::COVERED];
    }
    unsigned long branches_total() const
    {
	return branches_[cov::COVERED] +
	       branches_[cov::PARTCOVERED] +
	       branches_[cov::UNCOVERED];
    }
    unsigned long branches_suppressed() const
    {
    	return branches_[cov::SUPPRESSED];
    }
    const unsigned long *branches_by_status() const
    {
    	return branches_;
    }
    double branches_fraction() const
    {
    	unsigned long n = branches_executed();
	unsigned long d = branches_total();
	return (d == 0 ? 0.0 : (double)n / (double)d);
    }
    /* for sorting objects: -1 forces uninstrumented last */
    double branches_sort_fraction() const
    {
    	unsigned long n = branches_executed();
	unsigned long d = branches_total();
	return (d == 0 ? -1.0 : (double)n / (double)d);
    }

    /* Used in calculating status for aggregated objects like functions */
    cov::status_t status_by_blocks()
    {
	unsigned long ntot = blocks_total();

    	if (!ntot)
	    return (blocks_[cov::SUPPRESSED] ? cov::SUPPRESSED : cov::UNINSTRUMENTED);
	else if (blocks_[cov::COVERED] == ntot)
    	    return cov::COVERED;
	else if (blocks_[cov::UNCOVERED] == ntot)
    	    return cov::UNCOVERED;
	else
    	    return cov::PARTCOVERED;
    }
    cov::status_t status_by_lines()
    {
	unsigned long ntot = lines_total();

    	if (!ntot)
	    return (lines_[cov::SUPPRESSED] ? cov::SUPPRESSED : cov::UNINSTRUMENTED);
	else if (lines_[cov::COVERED] == ntot)
    	    return cov::COVERED;
	else if (lines_[cov::UNCOVERED] == ntot)
    	    return cov::UNCOVERED;
	else
    	    return cov::PARTCOVERED;
    }

private:
    unsigned long blocks_[cov::NUM_STATUS];
    unsigned long lines_[cov::NUM_STATUS];
    unsigned long functions_[cov::NUM_STATUS];
    unsigned long calls_[cov::NUM_STATUS];
    unsigned long branches_[cov::NUM_STATUS];
    
    friend class cov_block_t;
    friend class cov_function_t;
    friend class cov_file_t;
};


#endif /* _ggcov_cov_types_H_ */
