/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <stdlib.h>
#include <math.h>
#include "output.h"
#include "graphic.h"

/* GfsOutput: object */

typedef struct _Format Format;

typedef enum {
  ITER,
  TIME,
  PID,
  NONE
} FormatType;

struct _Format {
  gchar * s;
  FormatType t;
};

static Format * format_new (gchar * s, guint len, 
			    FormatType t)
{
  Format * f = g_malloc (sizeof (Format));
  
  f->s = g_strndup (s, len);
  f->t = t;

  return f;
}

static void format_destroy (Format * f)
{
  g_free (f->s);
  g_free (f);
}

static gchar * format_string (GSList * list, 
			      gint pid, 
			      guint niter,
			      gdouble time)
{
  gchar * s = g_strdup ("");

  while (list) {
    Format * f = list->data;
    gchar * s1, * s2 = NULL;

    switch (f->t) {
    case NONE:
      s2 = g_strconcat (s, f->s, NULL);
      break;
    case PID:
      s1 = g_strdup_printf (f->s, pid);
      s2 = g_strconcat (s, s1, NULL);
      g_free (s1);
      break;
    case ITER:
      s1 = g_strdup_printf (f->s, niter);
      s2 = g_strconcat (s, s1, NULL);
      g_free (s1);
      break;
    case TIME:
      s1 = g_strdup_printf (f->s, time);
      s2 = g_strconcat (s, s1, NULL);
      g_free (s1);
      break;
    default:
      g_assert_not_reached ();
    }
    g_free (s);
    s = s2;
    list = list->next;
  }

  return s;
}

static void output_free (GfsOutput * output)
{
  if (output->format)
    g_free (output->format);
  output->format = NULL;
  g_slist_foreach (output->formats, (GFunc) format_destroy, NULL);
  g_slist_free (output->formats);
  output->formats = NULL;
}

static void gfs_output_destroy (GtsObject * object)
{
  GfsOutput * output = GFS_OUTPUT (object);

  if (output->file)
    gfs_output_file_close (output->file);
  output_free (output);

  (* GTS_OBJECT_CLASS (gfs_output_class ())->parent_class->destroy) 
    (object);
}

static gboolean gfs_output_event (GfsEvent * event, GfsSimulation * sim)
{
  if ((* gfs_event_class ()->event) (event, sim)) {
    GfsOutput * output = GFS_OUTPUT (event);
    gchar * fname;

    if (!output->dynamic) {
      if (!output->file) {
	fname = format_string (output->formats,
			       GFS_DOMAIN (sim)->pid,
			       sim->time.i,
			       sim->time.t);
	output->file = gfs_output_file_open (fname);
	if (output->file == NULL)
	  g_warning ("could not open file `%s'", fname);
      }
      else
	fflush (output->file->fp);
      return TRUE;
    }

    if (output->file)
      gfs_output_file_close (output->file);
    fname = format_string (output->formats, 
			   GFS_DOMAIN (sim)->pid,
			   sim->time.i,
			   sim->time.t);
    output->file = gfs_output_file_open (fname);
    if (output->file == NULL)
      g_warning ("could not open file `%s'", fname);
    g_free (fname);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_write (GtsObject * o, FILE * fp)
{
  GfsOutput * output = GFS_OUTPUT (o);

  (* GTS_OBJECT_CLASS (gfs_output_class ())->parent_class->write) (o, fp);

  if (output->format)
    fprintf (fp, " %s", output->format);
}

static gboolean char_in_string (char c, const char * s)
{
  while (*s != '\0')
    if (*(s++) == c)
      return TRUE;
  return FALSE;
}

static void gfs_output_read (GtsObject ** o, GtsFile * fp)
{
  GfsOutput * output;
  gchar * c, * start, * fname, * fnamebak;
  FILE * fptr;
  guint len;

  (* GTS_OBJECT_CLASS (gfs_output_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  output = GFS_OUTPUT (*o);
  if (output->file)
    gfs_output_file_close (output->file);
  output->file = NULL;
  if (output->format)
    g_free (output->format);
  output->format = NULL;
  output->dynamic = FALSE;

  if (fp->type != GTS_STRING) {
    gts_file_error (fp, "expecting a string (format)");
    return;
  }
  output->format = g_strdup (fp->token->str);
  gts_file_next_token (fp);

  if (!strcmp (output->format, "stderr")) {
    output->file = gfs_output_file_open ("stderr");
    return;
  }

  if (!strcmp (output->format, "stdout")) {
    output->file = gfs_output_file_open ("stdout");
    return;
  }

  start = c = output->format;
  while (*c != '\0') {
    if (*c == '%') {
      gchar * startf = c, * prev = c;

      len = GPOINTER_TO_UINT (startf) -  GPOINTER_TO_UINT (start);
      if (len > 0)
	output->formats = g_slist_prepend (output->formats,
					   format_new (start, len, NONE));

      len = 1;
      c++;
      while (*c != '\0' && !char_in_string (*c, "diouxXeEfFgGaAcsCSpn%")) {
	prev = c;
	c++;
	len++;
      }
      len++;
      if (*c == '%')
	output->formats = g_slist_prepend (output->formats,
					   format_new ("%", 1, NONE));
      else if (char_in_string (*c, "diouxXc")) {
	if (*prev == 'l') {
	  output->formats = g_slist_prepend (output->formats,
					     format_new (startf, len, ITER));
	  output->dynamic = TRUE;
	}
	else
	  output->formats = g_slist_prepend (output->formats,
					     format_new (startf, len, PID));
      }
      else if (char_in_string (*c, "eEfFgGaA")) {
	output->formats = g_slist_prepend (output->formats,
					   format_new (startf, len, TIME));
	output->dynamic = TRUE;
      }
      else {
	gts_file_error (fp, 
			"unknown conversion specifier `%c' of format `%s'",
			*c, output->format);
	output_free (output);
	return;
      }
      start = c;
      start++;
    }
    c++;
  }
  len = GPOINTER_TO_UINT (c) -  GPOINTER_TO_UINT (start);
  if (len > 0)
    output->formats = g_slist_prepend (output->formats,
				       format_new (start, len, NONE));
  output->formats = g_slist_reverse (output->formats);
  
  fname = format_string (output->formats, -1, 0, 0.);
  fnamebak = g_strconcat (fname, "~", NULL);
  g_free (fname);
  fptr = fopen (fnamebak, "w");
  if (fptr == NULL) {
    gts_file_error (fp, "cannot open file specified by format `%s'",
		    output->format);
    g_free (fnamebak);
    output_free (output);
    return;
  }
  fclose (fptr);
  remove (fnamebak);
  g_free (fnamebak);
}

static void gfs_output_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_event;

  GTS_OBJECT_CLASS (klass)->write = gfs_output_write;
  GTS_OBJECT_CLASS (klass)->read = gfs_output_read;
  GTS_OBJECT_CLASS (klass)->destroy = gfs_output_destroy;
}

static void gfs_output_init (GfsOutput * object)
{
  object->file = NULL;
  object->format = NULL;
  object->formats = NULL;
  object->dynamic = FALSE;
}

GfsOutputClass * gfs_output_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_info = {
      "GfsOutput",
      sizeof (GfsOutput),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_class_init,
      (GtsObjectInitFunc) gfs_output_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_event_class ()),
				  &gfs_output_info);
  }

  return klass;
}

/**
 * gfs_output_mute:
 * @output: a #GfsOutput.
 *
 * "Mutes" the output defined by @output, the event associated with
 * @output still takes place but the output itself is redirected to
 * /dev/null.
 */
void gfs_output_mute (GfsOutput * output)
{
  g_return_if_fail (output != NULL);

  output->dynamic = FALSE;
  if (output->file)
    gfs_output_file_close (output->file);
  output->file = gfs_output_file_open ("/dev/null");
}

static GHashTable * gfs_output_files = NULL;

/**
 * gfs_output_file_open:
 * @name: the name of the file to open.
 *
 * Checks whether @name has already been opened. If it has, its
 * reference count is incremented and the corresponding #GfsOutputFile
 * is returned. If it has not, it is created and opened for writing.
 *
 * Returns: the #GfsOutputFile of file @name.  
 */
GfsOutputFile * gfs_output_file_open (const gchar * name)
{
  GfsOutputFile * file;
  FILE * fp;

  g_return_val_if_fail (name != NULL, NULL);

  if (gfs_output_files == NULL) {
    gfs_output_files = g_hash_table_new (g_str_hash, g_str_equal);
    file = g_malloc (sizeof (GfsOutputFile));
    file->refcount = 2;
    file->name = g_strdup ("stderr");
    file->fp = stderr;
    g_hash_table_insert (gfs_output_files, file->name, file);
    file = g_malloc (sizeof (GfsOutputFile));
    file->refcount = 2;
    file->name = g_strdup ("stdout");
    file->fp = stdout;
    g_hash_table_insert (gfs_output_files, file->name, file);
  }

  if ((file = g_hash_table_lookup (gfs_output_files, name))) {
    file->refcount++;
    return file;
  }

  fp = fopen (name, "a");
  if (fp == NULL)
    return NULL;

  file = g_malloc (sizeof (GfsOutputFile));
  file->refcount = 1;
  file->name = g_strdup (name);
  file->fp = fp;
  g_hash_table_insert (gfs_output_files, file->name, file);

  return file;  
}

/**
 * gfs_output_file_close:
 * @file: a #GfsOutputFile.
 * 
 * Decreases the reference count of @file. If it reaches zero the file
 * corresponding to @file is closed and @file is freed.
 */
void gfs_output_file_close (GfsOutputFile * file)
{
  g_return_if_fail (file);

  file->refcount--;
  if (file->refcount == 0) {
    g_hash_table_remove (gfs_output_files, file->name);
    fclose (file->fp);
    g_free (file->name);
    g_free (file);
  }
}

/* GfsOutputTime: Object */

static gboolean time_event (GfsEvent * event, GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (gfs_output_class())->event) (event, sim)) {
    fprintf (GFS_OUTPUT (event)->file->fp,
	     "step: %7u t: %15.8f dt: %13.6e\n",
	     sim->time.i, sim->time.t, 
	     sim->advection_params.dt);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_time_class_init (GfsEventClass * klass)
{
  klass->event = time_event;
}

GfsOutputClass * gfs_output_time_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_time_info = {
      "GfsOutputTime",
      sizeof (GfsOutput),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_time_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_time_info);
  }

  return klass;
}

/* GfsOutputProjectionStats: Object */

static gdouble rate (gdouble a, gdouble b, guint n)
{
  if (a > 0. && b > 0. && n > 0)
    return exp (log (b/a)/n);
  return 0.;
}

static gboolean projection_stats_event (GfsEvent * event, GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (gfs_output_class())->event) (event, sim)) {
    fprintf (GFS_OUTPUT (event)->file->fp,
	     "MAC projection        before     after       rate\n"
	     "    niter: %4d\n"
	     "    residual.bias:   % 10.3e % 10.3e\n"
	     "    residual.first:  % 10.3e % 10.3e %5.1f\n"
	     "    residual.second: % 10.3e % 10.3e %5.1f\n"
	     "    residual.infty:  % 10.3e % 10.3e %5.1f\n"
	     "Approximate projection\n"
	     "    niter: %4d\n"
	     "    residual.bias:   % 10.3e % 10.3e\n"
	     "    residual.first:  % 10.3e % 10.3e %5.1f\n"
	     "    residual.second: % 10.3e % 10.3e %5.1f\n"
	     "    residual.infty:  % 10.3e % 10.3e %5.1f\n",
	     sim->projection_params.niter,
	     sim->projection_params.residual_before.bias,
	     sim->projection_params.residual.bias,
	     sim->projection_params.residual_before.first,
	     sim->projection_params.residual.first,
	     rate (sim->projection_params.residual.first,
		   sim->projection_params.residual_before.first,
		   sim->projection_params.niter),
	     sim->projection_params.residual_before.second,
	     sim->projection_params.residual.second,
	     rate (sim->projection_params.residual.second,
		   sim->projection_params.residual_before.second,
		   sim->projection_params.niter),
	     sim->projection_params.residual_before.infty,
	     sim->projection_params.residual.infty,
	     rate (sim->projection_params.residual.infty,
		   sim->projection_params.residual_before.infty,
		   sim->projection_params.niter),
	     sim->approx_projection_params.niter,
	     sim->approx_projection_params.residual_before.bias,
	     sim->approx_projection_params.residual.bias,
	     sim->approx_projection_params.residual_before.first,
	     sim->approx_projection_params.residual.first,
	     rate (sim->approx_projection_params.residual.first,
		   sim->approx_projection_params.residual_before.first,
		   sim->approx_projection_params.niter),
	     sim->approx_projection_params.residual_before.second,
	     sim->approx_projection_params.residual.second,
	     rate (sim->approx_projection_params.residual.second,
		   sim->approx_projection_params.residual_before.second,
		   sim->approx_projection_params.niter),
	     sim->approx_projection_params.residual_before.infty,
	     sim->approx_projection_params.residual.infty,
	     rate (sim->approx_projection_params.residual.infty,
		   sim->approx_projection_params.residual_before.infty,
		   sim->approx_projection_params.niter));
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_projection_stats_class_init (GfsEventClass * klass)
{
  klass->event = projection_stats_event;
}

GfsOutputClass * gfs_output_projection_stats_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_projection_stats_info = {
      "GfsOutputProjectionStats",
      sizeof (GfsOutput),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_projection_stats_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_projection_stats_info);
  }

  return klass;
}

/* GfsOutputSolidStats: Object */

static gboolean gfs_output_solid_stats_event (GfsEvent * event, 
					     GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_solid_stats_class ())->parent_class)->event) (event, sim)) {
    GtsRange stats = gfs_domain_stats_solid (GFS_DOMAIN (sim));
    GtsRange ma, mn;

    gfs_domain_stats_merged (GFS_DOMAIN (sim), &ma, &mn);
    fprintf (GFS_OUTPUT (event)->file->fp,
	     "Solid volume fraction\n"
	     "    min: %10.3e avg: %10.3e | %10.3e max: %10.3e n: %10d\n"
	     "Total merged solid volume fraction\n"
	     "    min: %10.3e avg: %10.3e | %10.3e max: %10.3e n: %10d\n"
	     "Number of cells merged per merged cell\n"
	     "    min: %10.0f avg: %10.3f | %10.3f max: %10.0f n: %10d\n",
	     stats.min, stats.mean, stats.stddev, stats.max, stats.n,
	     ma.min, ma.mean, ma.stddev, ma.max, ma.n,
	     mn.min, mn.mean, mn.stddev, mn.max, mn.n);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_solid_stats_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_solid_stats_event;
}

GfsOutputClass * gfs_output_solid_stats_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_solid_stats_info = {
      "GfsOutputSolidStats",
      sizeof (GfsOutput),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_solid_stats_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_solid_stats_info);
  }

  return klass;
}

/* GfsOutputTiming: Object */

static void timing_print (GtsRange * r, gdouble total, FILE * fp)
{
  fprintf (fp, 
	   "      min: %9.3f avg: %9.3f (%4.1f%%) | %7.3f max: %9.3f\n",
	   r->min,
	   r->mean, total > 0. ? 100.*r->mean/total : 0.,
	   r->stddev, 
	   r->max);	   
}

static void timing_bc_print (GtsRange * r, gdouble total, guint n, FILE * fp)
{
  fprintf (fp,
	   "      calls: %7d avg: %9.3f (%4.1f%%)\n",
	   r->n,
	   n > 0 ? r->sum/n : 0.,
	   total > 0. ? 100.*r->sum/(n*total) : 0.);
}

static gboolean timing_event (GfsEvent * event, GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (gfs_output_class())->event) (event, sim)) {
    GfsDomain * domain = GFS_DOMAIN (sim);
    FILE * fp = GFS_OUTPUT (event)->file->fp;

    if (domain->timestep.mean > 0.) {
      fprintf (fp,
	       "Timing summary: %u timesteps %.0f node.timestep/s\n"
	       "  timestep:\n"
	       "      min: %9.3f avg: %9.3f         | %7.3f max: %9.3f\n"
               "  domain size:\n"
	       "      min: %9.0f avg: %9.0f         | %7.0f max: %9.0f\n",
	       domain->timestep.n,
	       domain->size.mean/domain->timestep.mean,
	       domain->timestep.min,
	       domain->timestep.mean,
	       domain->timestep.stddev, 
	       domain->timestep.max,
	       domain->size.min,
	       domain->size.mean,
	       domain->size.stddev, 
	       domain->size.max);
      fputs ("  predicted_face_velocities:\n", fp);
      timing_print (&domain->predicted_face_velocities, 
		    domain->timestep.mean, fp);
      fputs ("  mac_projection:\n", fp);
      timing_print (&domain->mac_projection, 
		    domain->timestep.mean, fp);
      fputs ("  tracer_advection:\n", fp);
      timing_print (&domain->tracer_advection, 
		    domain->timestep.mean, fp);
      fputs ("  centered_velocity_advection:\n", fp);
      timing_print (&domain->centered_velocity_advection, 
		    domain->timestep.mean, fp);
      fputs ("  approximate_projection:\n", fp);
      timing_print (&domain->approximate_projection, 
		    domain->timestep.mean, fp);
      if (domain->adapt.n > 0) {
	fputs ("  adapt:\n", fp);
	timing_bc_print (&domain->adapt, 
			 domain->timestep.mean,
			 domain->timestep.n, fp);
      }
      if (domain->pressure_bc.n > 0) {
	fputs ("  pressure_bc:\n", fp);
	timing_bc_print (&domain->pressure_bc, 
			 domain->timestep.mean,
			 domain->timestep.n, fp);
      }
      if (domain->center_bc.n > 0) {
	fputs ("  center_bc:\n", fp);
	timing_bc_print (&domain->center_bc, 
			 domain->timestep.mean,
			 domain->timestep.n, fp);      
      }
      if (domain->face_bc.n > 0) {
	fputs ("  face_bc:\n", fp);
	timing_bc_print (&domain->face_bc, 
			 domain->timestep.mean,
			 domain->timestep.n, fp);      
      }
      if (domain->match.n > 0) {
	fputs ("  match:\n", fp);
	timing_bc_print (&domain->match, 
			 domain->timestep.mean,
			 domain->timestep.n, fp);
      }
      if (domain->mpi_messages.n > 0)
	fprintf (fp,
		 "Message passing summary\n"
		 "  n: %10d size: %10.0f bytes\n",
		 domain->mpi_messages.n,
		 domain->mpi_messages.sum);
    }
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_timing_class_init (GfsEventClass * klass)
{
  klass->event = timing_event;
}

GfsOutputClass * gfs_output_timing_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_timing_info = {
      "GfsOutputTiming",
      sizeof (GfsOutput),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_timing_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_timing_info);
  }

  return klass;
}

/* GfsOutputBalance: Object */

static gboolean gfs_output_balance_event (GfsEvent * event, 
					  GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_balance_class ())->parent_class)->event) (event, sim)) {
    GfsDomain * domain = GFS_DOMAIN (sim);
    FILE * fp = GFS_OUTPUT (event)->file->fp;
    GtsRange size, boundary, mpiwait;
    
    gfs_domain_stats_balance (domain, &size, &boundary, &mpiwait);
    fprintf (fp, 
	     "Balance summary: %u PE\n"
	     "  domain size:\n"
	     "      min: %9.0f avg: %9.0f         | %7.0f max: %9.0f\n",
	     size.n,
	     size.min, size.mean, size.stddev, size.max);
    if (boundary.max > 0)
      fprintf (fp, 
	       "  parallel boundary size:\n"
	       "      min: %9.0f avg: %9.0f         | %7.0f max: %9.0f\n"
	       "  average timestep MPI wait time:\n"
	       "      min: %9.3f avg: %9.3f         | %7.3f max: %9.3f\n",
	       boundary.min, boundary.mean, boundary.stddev, boundary.max,
	       mpiwait.min, mpiwait.mean, mpiwait.stddev, mpiwait.max);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_balance_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_balance_event;
}

GfsOutputClass * gfs_output_balance_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_balance_info = {
      "GfsOutputBalance",
      sizeof (GfsOutput),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_balance_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_balance_info);
  }

  return klass;
}

/* GfsOutputLocation: Object */

static void gfs_output_location_read (GtsObject ** o, GtsFile * fp)
{
  GfsOutputLocation * l = GFS_OUTPUT_LOCATION (*o);

  if (GTS_OBJECT_CLASS (gfs_output_location_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_output_location_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.x)");
    return;
  }
  l->p.x = atof (fp->token->str);
  gts_file_next_token (fp);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.y)");
    return;
  }
  l->p.y = atof (fp->token->str);
  gts_file_next_token (fp);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.z)");
    return;
  }
  l->p.z = atof (fp->token->str);
  gts_file_next_token (fp);  
}

static void gfs_output_location_write (GtsObject * o, FILE * fp)
{
  GfsOutputLocation * l = GFS_OUTPUT_LOCATION (o);

  if (GTS_OBJECT_CLASS (gfs_output_location_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_output_location_class ())->parent_class->write) 
      (o, fp);
  fprintf (fp, " %g %g %g", l->p.x, l->p.y, l->p.z);
}

static gboolean gfs_output_location_event (GfsEvent * event, 
					   GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_location_class ())->parent_class)->event) (event, sim)) {
    GfsDomain * domain = GFS_DOMAIN (sim);
    GfsOutputLocation * location = GFS_OUTPUT_LOCATION (event);
    FttCell * cell = gfs_domain_locate (domain, location->p, -1);

    if (cell != NULL) {
      FILE * fp = GFS_OUTPUT (event)->file->fp;
      GfsVariable * v = domain->variables;
   
      fprintf (fp, "%g %g %g %g", 
	       sim->time.t,
	       location->p.x, location->p.y, location->p.z);
      while (v) {
	fprintf (fp, " %g", gfs_interpolate (cell, location->p, v->i));
	v = v->next;
      }
      fputc ('\n', fp);
    }
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_location_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_location_event;
  GTS_OBJECT_CLASS (klass)->read = gfs_output_location_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_output_location_write;
}

static void gfs_output_location_init (GfsOutputLocation * object)
{
  object->p.x = object->p.y = object->p.z = 0.;
}

GfsOutputClass * gfs_output_location_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_location_info = {
      "GfsOutputLocation",
      sizeof (GfsOutputLocation),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_location_class_init,
      (GtsObjectInitFunc) gfs_output_location_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_location_info);
  }

  return klass;
}

/* GfsOutputSimulation: Object */

static void output_simulation_destroy (GtsObject * object)
{
  GfsOutputSimulation * output = GFS_OUTPUT_SIMULATION (object);

  gfs_variable_list_destroy (output->var);

  (* GTS_OBJECT_CLASS (gfs_output_simulation_class ())->parent_class->destroy) 
    (object);
}

static gboolean output_simulation_event (GfsEvent * event, GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (gfs_output_class())->event) (event, sim)) {
    GfsDomain * domain = GFS_DOMAIN (sim);
    GfsVariable * var = domain->variables_io;

    domain->variables_io = GFS_OUTPUT_SIMULATION (event)->var;
    gfs_simulation_write (sim, 
			 GFS_OUTPUT_SIMULATION (event)->max_depth,
			 GFS_OUTPUT (event)->file->fp);
    domain->variables_io = var;
    fflush (GFS_OUTPUT (event)->file->fp);
    return TRUE;
  }
  return FALSE;
}

static void output_simulation_write (GtsObject * o, FILE * fp)
{
  GfsOutputSimulation * output = GFS_OUTPUT_SIMULATION (o);
  GfsVariable * v = output->var;

  (* GTS_OBJECT_CLASS (gfs_output_simulation_class ())->parent_class->write) 
    (o, fp);

  fputs (" {", fp);
  if (output->max_depth != -1)
    fprintf (fp, " depth = %d", output->max_depth);
  if (v != NULL) {
    fprintf (fp, " variables = %s", v->name);
    v = v->next;
    while (v) {
      fprintf (fp, ",%s", v->name);
      v = v->next;
    }
  }
  fputs (" }", fp);
}

static void output_simulation_read (GtsObject ** o, GtsFile * fp)
{
  GtsFileVariable var[] = {
    {GTS_INT,    "depth", TRUE},
    {GTS_STRING, "variables", TRUE},
    {GTS_NONE}
  };
  gchar * variables = NULL;
  GfsOutputSimulation * output = GFS_OUTPUT_SIMULATION (*o);
  GfsDomain * domain = gfs_object_simulation (output);

  (* GTS_OBJECT_CLASS (gfs_output_simulation_class ())->parent_class->read) 
    (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  var[0].data = &output->max_depth;
  var[1].data = &variables;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR) {
    g_free (variables);
    return;
  }

  if (variables != NULL) {
    gchar * error = NULL;
    GfsVariable * vars = gfs_variables_from_list (domain->variables, 
						  variables, &error);

    if (vars == NULL) {
      gts_file_variable_error (fp, var, "variables",
			       "unknown variable `%s'", error);
      g_free (variables);
      return;
    }
    if (output->var)
      gfs_variable_list_destroy (output->var);
    output->var = vars;
    g_free (variables);
  }
  else if (output->var == NULL)
    output->var = gfs_variable_list_copy (domain->variables, 
					  GTS_OBJECT (domain));
}

static void gfs_output_simulation_class_init (GfsEventClass * klass)
{
  klass->event = output_simulation_event;
  GTS_OBJECT_CLASS (klass)->destroy = output_simulation_destroy;
  GTS_OBJECT_CLASS (klass)->read = output_simulation_read;
  GTS_OBJECT_CLASS (klass)->write = output_simulation_write;
}

static void gfs_output_simulation_init (GfsOutputSimulation * object)
{
  object->max_depth = -1;
  object->var = NULL;
}

GfsOutputClass * gfs_output_simulation_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_simulation_info = {
      "GfsOutputSimulation",
      sizeof (GfsOutputSimulation),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_simulation_class_init,
      (GtsObjectInitFunc) gfs_output_simulation_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_simulation_info);
  }

  return klass;
}

/* GfsOutputBoundaries: Object */

static gboolean output_boundaries_event (GfsEvent * event, GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (gfs_output_class())->event) (event, sim)) {
    GfsDomain * domain = GFS_DOMAIN (sim);
    FILE * fp = GFS_OUTPUT (event)->file->fp;
    
    gfs_draw_refined_boundaries (domain, fp);
    gfs_draw_solid_boundaries (domain, fp);
    gfs_draw_boundary_conditions (domain, fp);
    fflush (fp);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_boundaries_class_init (GfsEventClass * klass)
{
  klass->event = output_boundaries_event;
}

GfsOutputClass * gfs_output_boundaries_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_boundaries_info = {
      "GfsOutputBoundaries",
      sizeof (GfsOutput),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_boundaries_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_boundaries_info);
  }

  return klass;
}

/* GfsOutputScalar: Object */

static void gfs_output_scalar_read (GtsObject ** o, GtsFile * fp)
{
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "min",      TRUE},
    {GTS_DOUBLE, "max",      TRUE},
    {GTS_STRING, "v",        TRUE},
    {GTS_INT,    "maxlevel", TRUE},
    {GTS_NONE}
  };
  GfsOutputScalar * output;
  GfsDomain * domain;
  gchar * vname = NULL;

  if (GTS_OBJECT_CLASS (gfs_output_scalar_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_output_scalar_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  output = GFS_OUTPUT_SCALAR (*o);
  domain = gfs_object_simulation (output);
  output->autoscale = TRUE;

  var[0].data = &output->min;
  var[1].data = &output->max;
  var[2].data = &vname;
  var[3].data = &output->maxlevel;
  
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;

  if (vname != NULL) {
    gfs_derived_last->next = domain->variables;
    output->v = gfs_variable_from_name (gfs_derived_first, vname);
    if (output->v == NULL) {
      gts_file_variable_error (fp, var, "v", "unknown scalar `%s'", 
			       vname);
      g_free (vname);
      return;
    }
    g_free (vname);
  }
  
  if (var[0].set && output->min > output->max) {
    gts_file_variable_error (fp, var, "min", 
	     "min `%g' must be smaller than or equal to max `%g'", 
			     output->min, output->max);
    return;
  }

  if (var[1].set && output->max < output->min) {
    gts_file_variable_error (fp, var, "max", 
	     "max `%g' must be larger than or equal to min `%g'", 
			     output->max, output->min);
    return;
  }

  if (var[0].set || var[1].set)
    output->autoscale = FALSE;
}

static void gfs_output_scalar_write (GtsObject * o, FILE * fp)
{
  GfsOutputScalar * output = GFS_OUTPUT_SCALAR (o);

  if (GTS_OBJECT_CLASS (gfs_output_scalar_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_output_scalar_class ())->parent_class->write) 
      (o, fp);

  fprintf (fp, " { v = %s", output->v->name);
  if (output->maxlevel >= 0)
    fprintf (fp, " maxlevel = %d", output->maxlevel);
  if (!output->autoscale)
    fprintf (fp, " min = %g max = %g }", output->min, output->max);
  else
    fputs (" }", fp);
}

static gboolean gfs_output_scalar_event (GfsEvent * event,
					 GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_scalar_class ())->parent_class)->event) (event, sim)) {
    GfsOutputScalar * output = GFS_OUTPUT_SCALAR (event);
    GfsDomain * domain = GFS_DOMAIN (sim);
    
    if (output->v->derived) {
      gfs_variable_set_parent (output->v, domain);
      gfs_domain_cell_traverse (domain,
				FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
				(FttCellTraverseFunc) output->v->derived, 
				output->v);
    }
    if (output->maxlevel >= 0)
        gfs_domain_cell_traverse (domain,
				  FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
				  output->v->fine_coarse ? 
	    (FttCellTraverseFunc) output->v->fine_coarse : 
	    (FttCellTraverseFunc) gfs_get_from_below_intensive,
				  output->v);
    if (output->autoscale) {
      GtsRange stats = gfs_domain_stats_variable (domain, output->v, 
	     FTT_TRAVERSE_LEAFS|FTT_TRAVERSE_LEVEL, output->maxlevel);

      output->min = stats.min;
      output->max = stats.max;
    }
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_scalar_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_scalar_event;
  GTS_OBJECT_CLASS (klass)->read = gfs_output_scalar_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_output_scalar_write;
}

static void gfs_output_scalar_init (GfsOutputScalar * object)
{
  object->v = NULL;
  object->min = object->max = 0.;
  object->autoscale = TRUE;
  object->maxlevel = -1;
}

GfsOutputClass * gfs_output_scalar_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_scalar_info = {
      "GfsOutputScalar",
      sizeof (GfsOutputScalar),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_scalar_class_init,
      (GtsObjectInitFunc) gfs_output_scalar_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_class ()),
				  &gfs_output_scalar_info);
  }

  return klass;
}

/* GfsOutputScalarNorm: Object */

static gboolean gfs_output_scalar_norm_event (GfsEvent * event, 
					      GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_scalar_norm_class ())->parent_class)->event) (event, sim)) {
    GfsOutputScalar * output = GFS_OUTPUT_SCALAR (event);
    GfsNorm norm = gfs_domain_norm_variable (GFS_DOMAIN (sim), 
					     output->v,
		 FTT_TRAVERSE_LEAFS|FTT_TRAVERSE_LEVEL, output->maxlevel);
    
    fprintf (GFS_OUTPUT (event)->file->fp,
	     "%s first: % 10.3e second: % 10.3e infty: % 10.3e\n",
	     output->v->name,
	     norm.first, norm.second, norm.infty);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_scalar_norm_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_scalar_norm_event;
}

GfsOutputClass * gfs_output_scalar_norm_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_scalar_norm_info = {
      "GfsOutputScalarNorm",
      sizeof (GfsOutputScalar),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_scalar_norm_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_scalar_class ()),
				  &gfs_output_scalar_norm_info);
  }

  return klass;
}

/* GfsOutputScalarStats: Object */

static gboolean gfs_output_scalar_stats_event (GfsEvent * event, 
					     GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_scalar_stats_class ())->parent_class)->event) (event, sim)) {
    GfsOutputScalar * output = GFS_OUTPUT_SCALAR (event);
    GtsRange stats = gfs_domain_stats_variable (GFS_DOMAIN (sim), 
						output->v,
			 FTT_TRAVERSE_LEAFS|FTT_TRAVERSE_LEVEL, 
						output->maxlevel);
    
    fprintf (GFS_OUTPUT (event)->file->fp,
	     "%s min: %10.3e avg: %10.3e | %10.3e max: %10.3e\n",
	     output->v->name,
	     stats.min, stats.mean, stats.stddev, stats.max);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_scalar_stats_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_scalar_stats_event;
}

GfsOutputClass * gfs_output_scalar_stats_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_scalar_stats_info = {
      "GfsOutputScalarStats",
      sizeof (GfsOutputScalar),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_scalar_stats_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_scalar_class ()),
				  &gfs_output_scalar_stats_info);
  }

  return klass;
}

/* GfsOutputScalarSum: Object */

static void add (FttCell * cell, gpointer * data)
{
  GfsSolidVector * solid = GFS_STATE (cell)->solid;
  gdouble vol = (solid ? solid->a : 1.)*ftt_cell_volume (cell);
  GfsVariable * v = data[0];
  gdouble * sum = data[1];

  *sum += vol*GFS_VARIABLE (cell, v->i);
}

static gboolean gfs_output_scalar_sum_event (GfsEvent * event, 
					     GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_scalar_sum_class ())->parent_class)->event) (event, sim)) {
    GfsOutputScalar * output = GFS_OUTPUT_SCALAR (event);
    gpointer data[2];
    gdouble sum = 0.;

    data[0] = output->v;
    data[1] = &sum;
    gfs_domain_cell_traverse (GFS_DOMAIN (sim),
			      FTT_PRE_ORDER, 
			      FTT_TRAVERSE_LEAFS|FTT_TRAVERSE_LEVEL,
			      output->maxlevel,
			      (FttCellTraverseFunc) add, data);
    fprintf (GFS_OUTPUT (event)->file->fp,
	     "%s sum: % 15.6e\n",
	     output->v->name, sum);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_scalar_sum_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_scalar_sum_event;
}

GfsOutputClass * gfs_output_scalar_sum_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_scalar_sum_info = {
      "GfsOutputScalarSum",
      sizeof (GfsOutputScalar),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_scalar_sum_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_scalar_class ()),
				  &gfs_output_scalar_sum_info);
  }

  return klass;
}

/* GfsOutputSquares: Object */

static gboolean gfs_output_squares_event (GfsEvent * event, GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_squares_class ())->parent_class)->event) (event, sim)) {
    GfsOutputScalar * output = GFS_OUTPUT_SCALAR (event);
    
    gfs_write_squares (GFS_DOMAIN (sim), 
		       output->v, output->min, output->max,
		       FTT_TRAVERSE_LEAFS|FTT_TRAVERSE_LEVEL,
		       output->maxlevel, NULL, 
		       GFS_OUTPUT (event)->file->fp);
    fflush (GFS_OUTPUT (event)->file->fp);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_squares_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_squares_event;
}

GfsOutputClass * gfs_output_squares_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_squares_info = {
      "GfsOutputSquares",
      sizeof (GfsOutputScalar),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_squares_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_scalar_class ()),
				  &gfs_output_squares_info);
  }

  return klass;
}

/* GfsOutputStreamline: Object */

static void gfs_output_streamline_read (GtsObject ** o, GtsFile * fp)
{
  GfsOutputStreamline * l = GFS_OUTPUT_STREAMLINE (*o);

  if (GTS_OBJECT_CLASS (gfs_output_streamline_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_output_streamline_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.x)");
    return;
  }
  l->p.x = atof (fp->token->str);
  gts_file_next_token (fp);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.y)");
    return;
  }
  l->p.y = atof (fp->token->str);
  gts_file_next_token (fp);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.z)");
    return;
  }
  l->p.z = atof (fp->token->str);
  gts_file_next_token (fp);
}

static void gfs_output_streamline_write (GtsObject * o, FILE * fp)
{
  GfsOutputStreamline * l = GFS_OUTPUT_STREAMLINE (o);

  if (GTS_OBJECT_CLASS (gfs_output_streamline_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_output_streamline_class ())->parent_class->write) 
      (o, fp);
  fprintf (fp, " %g %g %g", l->p.x, l->p.y, l->p.z);
}

static gboolean gfs_output_streamline_event (GfsEvent * event, 
					    GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_streamline_class ())->parent_class)->event) (event,sim)) {
    GSList * stream = gfs_streamline_new (GFS_DOMAIN (sim),
					  GFS_OUTPUT_STREAMLINE (event)->p,
					  GFS_OUTPUT_SCALAR (event)->v,
					  0., 0.,
					  TRUE);

    gfs_streamline_write (stream, GFS_OUTPUT (event)->file->fp);
    fflush (GFS_OUTPUT (event)->file->fp);
    gfs_streamline_destroy (stream);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_streamline_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_streamline_event;
  GTS_OBJECT_CLASS (klass)->read = gfs_output_streamline_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_output_streamline_write;
}

GfsOutputClass * gfs_output_streamline_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_streamline_info = {
      "GfsOutputStreamline",
      sizeof (GfsOutputStreamline),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_streamline_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_scalar_class ()),
				  &gfs_output_streamline_info);
  }

  return klass;
}

/* GfsOutputStreakline: Object */

static void gfs_output_streakline_destroy (GtsObject * o)
{
  GfsOutputStreakline * l = GFS_OUTPUT_STREAKLINE (o);

  gts_object_destroy (GTS_OBJECT (l->p));
  g_slist_foreach (l->streak, (GFunc) gts_object_destroy, NULL);
  g_slist_free (l->streak);
  
  (* GTS_OBJECT_CLASS (gfs_output_streakline_class ())->parent_class->destroy) 
    (o);
}

static void gfs_output_streakline_read (GtsObject ** o, GtsFile * fp)
{
  GfsOutputStreakline * l = GFS_OUTPUT_STREAKLINE (*o);

  if (GTS_OBJECT_CLASS (gfs_output_streakline_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_output_streakline_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.x)");
    return;
  }
  l->p->x = atof (fp->token->str);
  gts_file_next_token (fp);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.y)");
    return;
  }
  l->p->y = atof (fp->token->str);
  gts_file_next_token (fp);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (p.z)");
    return;
  }
  l->p->z = atof (fp->token->str);
  gts_file_next_token (fp);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (ds)");
    return;
  }
  l->ds = atof (fp->token->str);
  gts_file_next_token (fp);

  l->streak = g_slist_prepend (l->streak, 
			       gts_point_new (gts_point_class (), 
					      l->p->x, l->p->y, l->p->z));
}

static void gfs_output_streakline_write (GtsObject * o, FILE * fp)
{
  GfsOutputStreakline * l = GFS_OUTPUT_STREAKLINE (o);

  if (GTS_OBJECT_CLASS (gfs_output_streakline_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_output_streakline_class ())->parent_class->write) 
      (o, fp);
  fprintf (fp, " %g %g %g %g", l->p->x, l->p->y, l->p->z, l->ds);
}

static void advect_point (GtsPoint * p, GfsSimulation * sim)
{
  gfs_domain_advect_point (GFS_DOMAIN (sim), p, sim->advection_params.dt);
}

static gboolean gfs_output_streakline_event (GfsEvent * event, 
					    GfsSimulation * sim)
{
  GfsOutputStreakline * l = GFS_OUTPUT_STREAKLINE (event);
  gboolean ret = FALSE;

  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_streakline_class ())->parent_class)->event) (event,sim)) {
    FILE * fp = GFS_OUTPUT (event)->file->fp;
    GSList * i = l->streak;

    fprintf (fp, "%s %u\n", 
	     GTS_OBJECT (event)->klass->info.name, 
	     g_slist_length (i));
    while (i) {
      fprintf (fp, "%g %g %g\n", 
	       GTS_POINT (i->data)->x, 
	       GTS_POINT (i->data)->y, 
	       GTS_POINT (i->data)->z);
      i = i->next;
    }
    l->started = ret = TRUE;
  }
  if (l->started) {
    if (gts_point_distance2 (l->p, l->streak->data) >= l->ds*l->ds)
      l->streak = g_slist_prepend (l->streak, 
				   gts_point_new (gts_point_class (), 
						  l->p->x, l->p->y, l->p->z));
    g_slist_foreach (l->streak, (GFunc) advect_point, sim);
  }
  return ret;
}

static void gfs_output_streakline_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_streakline_event;
  GTS_OBJECT_CLASS (klass)->read = gfs_output_streakline_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_output_streakline_write;
  GTS_OBJECT_CLASS (klass)->destroy = gfs_output_streakline_destroy;
}

static void gfs_output_streakline_init (GfsOutputStreakline * l)
{
  l->started = FALSE;
  l->ds = G_MAXDOUBLE;
  l->p = gts_point_new (gts_point_class (), 0., 0., 0.);
  l->streak = NULL;
}

GfsOutputClass * gfs_output_streakline_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_streakline_info = {
      "GfsOutputStreakline",
      sizeof (GfsOutputStreakline),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_streakline_class_init,
      (GtsObjectInitFunc) gfs_output_streakline_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS 
				  (gfs_output_scalar_class ()),
				  &gfs_output_streakline_info);
  }

  return klass;
}

#ifdef FTT_2D

/* GfsOutputPPM: Object */

static gboolean gfs_output_ppm_event (GfsEvent * event, GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_output_ppm_class ())->parent_class)->event) (event, sim)) {
    GfsOutputScalar * output = GFS_OUTPUT_SCALAR (event);

    gfs_write_ppm (GFS_DOMAIN (sim), 
		   output->v, output->min, output->max,
		   FTT_TRAVERSE_LEAFS|FTT_TRAVERSE_LEVEL, output->maxlevel,
		   GFS_OUTPUT (event)->file->fp);
    fflush (GFS_OUTPUT (event)->file->fp);
    return TRUE;
  }
  return FALSE;
}

static void gfs_output_ppm_class_init (GfsOutputClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_output_ppm_event;
}

GfsOutputClass * gfs_output_ppm_class (void)
{
  static GfsOutputClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_output_ppm_info = {
      "GfsOutputPPM",
      sizeof (GfsOutputScalar),
      sizeof (GfsOutputClass),
      (GtsObjectClassInitFunc) gfs_output_ppm_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_output_scalar_class ()),
				  &gfs_output_ppm_info);
  }

  return klass;
}

#endif /* 2D */
