/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <math.h>

#include "fluid.h"
#include "domain.h"

typedef struct _Gradient Gradient;

/* grad(p) = -a*p(cell) + b*p(neighbor) + c */
struct _Gradient {
  gdouble a, b, c;
};

static gdouble average_neighbor_value (const FttCellFace * face,
				       guint v,
				       gdouble * x)
{
  /* check for corner refinement violation (topology.fig) */
  g_assert (ftt_cell_level (face->neighbor) == ftt_cell_level (face->cell));
  
  if (FTT_CELL_IS_LEAF (face->neighbor))
    return GFS_VARIABLE (face->neighbor, v);
  else {
    FttCellChildren children;
    gdouble av = 0.;
    guint i;
    
    ftt_cell_children_direction (face->neighbor, 
				 FTT_OPPOSITE_DIRECTION (face->d), 
				 &children);
    for (i = 0; i < FTT_CELLS/2; i++) {
      /* check for mixed cell refinement violation (topology.fig) */
      g_assert (children.c[i]);
      av += GFS_VARIABLE (children.c[i], v);
    }
    *x = 3./4.;
    return av/(FTT_CELLS/2);
  }
}

#ifdef FTT_2D

/* v = a*v(cell) + b 
 * 
 * Second order 1D interpolation.
 */
static GfsGradient interpolate_1D2 (FttCell * cell,
				    FttDirection d,
				    gdouble x,
				    guint v)
{
  GfsGradient p;
  FttCellFace f1, f2;
  gdouble p1 = 0., p2 = 0.;
  gdouble x1 = 1., x2 = 1.;
  gdouble a1, a2;

  g_return_val_if_fail (cell != NULL, p);
  g_return_val_if_fail (!GFS_IS_MIXED (cell), p);
  
  f1 = ftt_cell_face (cell, FTT_OPPOSITE_DIRECTION (d));
  if (f1.neighbor)
    p1 = average_neighbor_value (&f1, v, &x1);
  f2 = ftt_cell_face (cell, d);
  if (f2.neighbor)
    p2 = average_neighbor_value (&f2, v, &x2);

  a1 = x*(x - x2)/(x1*(x1 + x2));
  a2 = x*(x + x1)/(x2*(x1 + x2));

  p.a = 1. - a1 - a2;
  p.b = 0.;
  if (f1.neighbor)
    p.b += a1*p1;
  else
    p.a += a1;
  if (f2.neighbor)
    p.b += a2*p2;
  else
    p.a += a2;

  return p;
}

/* v = a*v(cell) + b 
 * 
 * First order 1D interpolation.
 */
static GfsGradient interpolate_1D1 (FttCell * cell,
				    FttDirection d,
				    gdouble x,
				    guint v)
{
  GfsGradient p;
  FttCellFace f;

  g_return_val_if_fail (cell != NULL, p);
  g_return_val_if_fail (!GFS_IS_MIXED (cell), p);

  f = ftt_cell_face (cell, d);
  if (f.neighbor) {
    gdouble p2;
    gdouble x2 = 1.;

    p2 = average_neighbor_value (&f, v, &x2);
    p.a = 1. - x/x2;
    p.b = p2*x/x2;
  }
  else {
    p.a = 1.;
    p.b = 0.;
  }

  return p;
}

#else /* not FTT_2D */

/* v = a*v(cell) + b 
 * 
 * First order 2D interpolation.
 */
static GfsGradient interpolate_2D1 (FttCell * cell,
				   FttDirection d1, FttDirection d2,
				   gdouble x, gdouble y,
				   guint v)
{
  GfsGradient p;
  gdouble y1 = 1.;
  gdouble x2 = 1.;
  gdouble p1 = 0., p2 = 0.;
  gdouble a1, a2;
  FttCellFace f1, f2;

  g_return_val_if_fail (cell != NULL, p);
  g_return_val_if_fail (!GFS_IS_MIXED (cell), p);
  
  f1 = ftt_cell_face (cell, d1);
  if (f1.neighbor)
    p1 = average_neighbor_value (&f1, v, &y1);
  f2 = ftt_cell_face (cell, d2);
  if (f2.neighbor)
    p2 = average_neighbor_value (&f2, v, &x2);

  a1 = y/y1;
  a2 = x/x2;

  p.a = 1. - a1 - a2;
  p.b = 0.;
  if (f1.neighbor)
    p.b += a1*p1;
  else
    p.a += a1;
  if (f2.neighbor)
    p.b += a2*p2;
  else
    p.a += a2;
  
  return p;
}

#endif /* not FTT_2D */

#ifdef FTT_2D
static gint perpendicular[FTT_NEIGHBORS][FTT_CELLS] = 
  {{-1,  2, -1,  3},
   { 2, -1,  3, -1},
   { 1,  0, -1, -1},
   {-1, -1,  1,  0}};
#else  /* FTT_3D */
static gint perpendicular[FTT_NEIGHBORS][FTT_CELLS][2] = 
  {{{-1,-1},{2,4},{-1,-1},{3,4},{-1,-1},{2,5},{-1,-1},{3,5}},
   {{2,4},{-1,-1},{3,4},{-1,-1},{2,5},{-1,-1},{3,5},{-1,-1}},
   {{4,1},{4,0},{-1,-1},{-1,-1},{5,1},{5,0},{-1,-1},{-1,-1}},
   {{-1,-1},{-1,-1},{4,1},{4,0},{-1,-1},{-1,-1},{5,1},{5,0}},
   {{1,2},{0,2},{1,3},{0,3},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{1,2},{0,2},{1,3},{0,3}}};
#endif /* FTT_3D */

static Gradient gradient_fine_coarse (const FttCellFace * face,
				      guint v,
				      gint max_level)
{
  Gradient g;
  guint level;
  FttCell * n;
  GfsGradient p;
#ifdef FTT_2D
  gint dp;
#else  /* FTT_3D */
  gint * dp;
#endif /* FTT_3D */

  g_return_val_if_fail (face != NULL, g);
  g_return_val_if_fail (ftt_face_type (face) == FTT_FINE_COARSE, g);

  /* check for mixed cell refinement violation (topology.fig) */
  g_return_val_if_fail (!GFS_IS_MIXED (face->neighbor), g);

  level = ftt_cell_level (face->cell);
  n = ftt_cell_neighbor (face->cell, FTT_OPPOSITE_DIRECTION (face->d));
  g_assert (n == NULL || ftt_cell_level (n) == level);

  dp = perpendicular[face->d][FTT_CELL_ID (face->cell)];
#ifdef FTT_2D
  g_assert (dp >= 0);
  p = interpolate_1D1 (face->neighbor, dp, 1./4., v);
#else  /* FTT_3D */
  g_assert (dp[0] >= 0 && dp[1] >= 0);
  p = interpolate_2D1 (face->neighbor, dp[0], dp[1], 1./4., 1./4., v);
#endif /* FTT_3D */

  if (n == NULL) {
    g.a = 2./3.;
    g.b = 2.*p.a/3.;
    g.c = 2.*p.b/3.;
  }
  else if (level == max_level || FTT_CELL_IS_LEAF (n)) {
    g.a = 1./3.;
    g.b = 8.*p.a/15.;
    g.c = 8.*p.b/15. - GFS_VARIABLE (n, v)/5.;
  }
  else {
    FttCellChildren children;
    guint i;

    ftt_cell_children_direction (n, face->d, &children);
    g.a = 2./9.;
    g.b = 14.*p.a/27.;
    g.c = 0.;
    for (i = 0; i < FTT_CELLS/2; i++) {
      /* check for mixed cell refinement violation (topology.fig) */
      g_assert (children.c[i]);
      g.c += GFS_VARIABLE (children.c[i], v);
    }
    g.c = (14.*p.b - 16.*g.c/FTT_CELLS)/27.;
  }

  return g;
}

#define REFINE_CORNER(cell) {if (cell && FTT_CELL_IS_LEAF (cell) && \
                              ftt_cell_level (cell) < level - 1) \
	                    ftt_cell_refine_single (cell, init, data);}

void ftt_cell_refine_corners (FttCell * cell,
			      FttCellInitFunc init,
			      gpointer data)
{
  FttDirection d;
  FttCellNeighbors neighbor;
  guint level;

  g_return_if_fail (cell != NULL);

  level = ftt_cell_level (cell);
  ftt_cell_neighbors (cell, &neighbor);
  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (neighbor.c[d] && ftt_cell_level (neighbor.c[d]) < level) {
      if (GFS_CELL_IS_BOUNDARY (neighbor.c[d]))
	ftt_cell_refine_single (neighbor.c[d], init, data);
      else {
	FttCell * n;
#ifdef FTT_2D
	gint dp;
#else  /* FTT_3D */
	gint * dp;
#endif /* FTT_3D */
	
	dp = perpendicular[d][FTT_CELL_ID (cell)];
#ifdef FTT_2D
	g_assert (dp >= 0);
	n = ftt_cell_neighbor (neighbor.c[d], dp);
	REFINE_CORNER (n)
#else  /* FTT_3D */
	g_assert (dp[0] >= 0 && dp[1] >= 0);
	n = ftt_cell_neighbor (neighbor.c[d], dp[0]);
	REFINE_CORNER (n)
	n = ftt_cell_neighbor (neighbor.c[d], dp[1]);
	REFINE_CORNER (n)
#endif /* FTT_3D */
      }
    }
}

static gdouble neighbor_value (const FttCellFace * face,
			       guint v,
			       gdouble * x)
{
  GfsGradient vc;
#ifdef FTT_2D
  gint dp;
#else  /* FTT_3D */
  gint * dp;
#endif /* FTT_3D */

  if (ftt_cell_level (face->neighbor) == ftt_cell_level (face->cell))
    /* neighbor at same level */
    return average_neighbor_value (face, v, x);
  else {
    /* neighbor at coarser level */
    dp = perpendicular[face->d][FTT_CELL_ID (face->cell)];
#ifdef FTT_2D
    g_assert (dp >= 0);
    vc = interpolate_1D1 (face->neighbor, dp, 1./4., v);
#else  /* FTT_3D */
    g_assert (dp[0] >= 0 && dp[1] >= 0);
    vc = interpolate_2D1 (face->neighbor, dp[0], dp[1], 1./4., 1./4., v);
#endif /* FTT_3D */
    *x = 3./2.;
    return vc.a*GFS_VARIABLE (face->neighbor, v) + vc.b;
  }
}

/**
 * gfs_center_gradient:
 * @cell: a #FttCell.
 * @c: a component.
 * @v: a #GfsVariable index.
 *
 * The gradient is normalized by the size of the cell.
 *
 * Returns: the value of the @c component of the gradient of variable @v
 * at the center of the cell.  
 */
gdouble gfs_center_gradient (FttCell * cell,
			     FttComponent c,
			     guint v)
{
  FttDirection d = 2*c;
  FttCellFace f1;
  gdouble v0;

  g_return_val_if_fail (cell != NULL, 0.);
  g_return_val_if_fail (c < FTT_DIMENSION, 0.);

  f1 = ftt_cell_face (cell, FTT_OPPOSITE_DIRECTION (d));
  if (f1.neighbor == cell) /* periodic */
    return 0.;
  v0 = GFS_VARIABLE (cell, v);
  if (f1.neighbor) {
    FttCellFace f2 = ftt_cell_face (cell, d);
    gdouble x1 = 1., v1;
    
    v1 = neighbor_value (&f1, v, &x1);
    if (f2.neighbor) {
      /* two neighbors: second-order differencing (parabola) */
      gdouble x2 = 1., v2;
      
      v2 = neighbor_value (&f2, v, &x2);
      return (x1*x1*(v2 - v0) + x2*x2*(v0 - v1))/(x1*x2*(x2 + x1));
    }
    else
      /* one neighbor: first-order differencing */
      return (v0 - v1)/x1;
  }
  else {
    FttCellFace f2 = ftt_cell_face (cell, d);

    if (f2.neighbor) {
      gdouble x2 = 1.;
      
      /* one neighbor: first-order differencing */
      return (neighbor_value (&f2, v, &x2) - v0)/x2;
    }
  }
  /* no neighbors */
  return 0.;
}

/**
 * gfs_center_van_leer_gradient:
 * @cell: a #FttCell.
 * @c: a component.
 * @v: a #GfsVariable index.
 *
 * The gradient is normalized by the size of the cell and is limited
 * using van Leer's limiter.
 *
 * Returns: the value of the @c component of the gradient of variable @v
 * at the center of the cell.  
 */
gdouble gfs_center_van_leer_gradient (FttCell * cell,
				      FttComponent c,
				      guint v)
{
  FttDirection d = 2*c;
  FttCellFace f1;
  
  g_return_val_if_fail (cell != NULL, 0.);
  g_return_val_if_fail (c < FTT_DIMENSION, 0.);

  f1 = ftt_cell_face (cell, FTT_OPPOSITE_DIRECTION (d));
  if (f1.neighbor == cell) /* periodic */
    return 0.;
  if (f1.neighbor) {
    FttCellFace f2 = ftt_cell_face (cell, d);
    
    if (f2.neighbor) {
      /* two neighbors: second-order differencing (parabola)
	 + van Leer limiter */
      gdouble x1 = 1., x2 = 1., v0, v1, v2;
      gdouble s0, s1, s2;
      
      v0 = GFS_VARIABLE (cell, v);
      v1 = neighbor_value (&f1, v, &x1);
      v2 = neighbor_value (&f2, v, &x2);      
      
      s1 = 2.*(v0 - v1);
      s2 = 2.*(v2 - v0);

      if (s1*s2 <= 0.)
	return 0.;
      s0 = (x1*x1*(v2 - v0) + x2*x2*(v0 - v1))/(x1*x2*(x2 + x1));
      if (ABS (s2) < ABS (s1))
	s1 = s2;
      if (ABS (s0) < ABS (s1))
	return s0;
      return s1;
    }
  }
  /* only one or no neighbors */
  return 0.;
}
					 
/**
 * gfs_face_gradient:
 * @face: a #FttCellFace.
 * @g: the #GfsGradient.
 * @v: a #GfsVariable index.
 * @max_level: the maximum cell level to consider (-1 means no restriction).
 *
 * Set the value of @g as the gradient of variable @v on the
 * @face. The value returned is second order accurate in space and
 * conservative, in the sense that values at a coarse/fine cell
 * boundary are consistent.  
 */
void gfs_face_gradient (const FttCellFace * face,
			GfsGradient * g,
			guint v,
			gint max_level)
{
  guint level;

  g_return_if_fail (face != NULL);

  g->a = g->b = 0.;
  if (face->neighbor == NULL)
    return;

  level = ftt_cell_level (face->cell);
  if (ftt_cell_level (face->neighbor) < level) {
    /* neighbor is at a shallower level */
    Gradient gcf;

    gcf = gradient_fine_coarse (face, v, max_level);
    g->a = gcf.a;
    g->b = gcf.b*GFS_VARIABLE (face->neighbor, v) + gcf.c;
  }
  else {
    if (level == max_level || FTT_CELL_IS_LEAF (face->neighbor)) {
      /* neighbor is at the same level */
      g->a = 1.;
      g->b = GFS_VARIABLE (face->neighbor, v);
    }
    else {
      /* neighbor is at a deeper level */
      FttCellChildren children;
      FttCellFace f;
      guint i;
      
      f.d = FTT_OPPOSITE_DIRECTION (face->d);
      ftt_cell_children_direction (face->neighbor, f.d, &children);
      f.neighbor = face->cell;
      for (i = 0; i < FTT_CELLS/2; i++) {
	Gradient gcf;
	
	f.cell = children.c[i];
	
	/* check for mixed cell refinement violation (topology.fig) */
	g_assert (f.cell);
	
	gcf = gradient_fine_coarse (&f, v, max_level);
	g->a += gcf.b;
	g->b += gcf.a*GFS_VARIABLE (f.cell, v) - gcf.c;
      }
#ifndef FTT_2D
      g->a /= 2.;
      g->b /= 2.;
#endif /* not FTT_2D */
    }
  }
}

/**
 * gfs_face_weighted_gradient:
 * @face: a #FttCellFace.
 * @g: the #GfsGradient.
 * @v: a #GfsVariable index.
 * @max_level: the maximum cell level to consider (-1 means no restriction).
 *
 * Set the value of @g as the gradient of variable @v on the @face
 * weighted by the value of the @v field of the face state vector of the
 * corresponding cell. The value returned is second order accurate in
 * space and conservative, in the sense that values at a coarse/fine
 * cell boundary are consistent.  
 */
void gfs_face_weighted_gradient (const FttCellFace * face,
				 GfsGradient * g,
				 guint v,
				 gint max_level)
{
  guint level;

  g_return_if_fail (face != NULL);

  g->a = g->b = 0.;
  if (face->neighbor == NULL)
    return;

  level = ftt_cell_level (face->cell);
  if (ftt_cell_level (face->neighbor) < level) {
    /* neighbor is at a shallower level */
    Gradient gcf;
    gdouble w = GFS_STATE (face->cell)->f[face->d].v;

    gcf = gradient_fine_coarse (face, v, max_level);
    g->a = w*gcf.a;
    g->b = w*(gcf.b*GFS_VARIABLE (face->neighbor, v) + gcf.c);
  }
  else {
    if (level == max_level || FTT_CELL_IS_LEAF (face->neighbor)) {
      /* neighbor is at the same level */
      gdouble w = GFS_STATE (face->cell)->f[face->d].v;

      g->a = w;
      g->b = w*GFS_VARIABLE (face->neighbor, v);
    }
    else {
      /* neighbor is at a deeper level */
      FttCellChildren children;
      FttCellFace f;
      guint i;
      
      f.d = FTT_OPPOSITE_DIRECTION (face->d);
      ftt_cell_children_direction (face->neighbor, f.d, &children);
      f.neighbor = face->cell;
      for (i = 0; i < FTT_CELLS/2; i++) {
	Gradient gcf;
	gdouble w;

	f.cell = children.c[i];
	w = GFS_STATE (f.cell)->f[f.d].v;
	
	/* check for mixed cell refinement violation (topology.fig) */
	g_assert (f.cell);
	
	gcf = gradient_fine_coarse (&f, v, max_level);
	g->a += w*gcf.b;
	g->b += w*(gcf.a*GFS_VARIABLE (f.cell, v) - gcf.c);
      }
#ifndef FTT_2D
      g->a /= 2.;
      g->b /= 2.;
#endif /* not FTT_2D */
    }
  }
}

/**
 * gfs_get_from_above:
 * @cell: a #FttCell.
 * @v: a #GfsVariable to "get from above".
 *
 * Sets the value of the variable @v of @cell to the value of this
 * variable in its parent cell.
 *
 * This function fails if @cell is the root of the cell tree.
 */
void gfs_get_from_above (FttCell * cell, const GfsVariable * v)
{
  g_return_if_fail (cell != NULL);
  g_return_if_fail (!FTT_CELL_IS_ROOT (cell));
  g_return_if_fail (v != NULL);

  GFS_VARIABLE (cell, v->i) = GFS_VARIABLE (ftt_cell_parent (cell), v->i);
}

/**
 * gfs_get_from_below_intensive:
 * @cell: a #FttCell.
 * @v: a #GfsVariable to "get from below".
 *
 * Sets the value of the "intensive" variable @v of @cell by taking
 * the volume weighted average of the values of its children cells.
 *
 * This functions fails if @cell is a leaf of the cell tree.
 */
void gfs_get_from_below_intensive (FttCell * cell, const GfsVariable * v)
{
  gdouble val = 0., sa = 0.;
  guint i;
  FttCellChildren child;

  g_return_if_fail (cell != NULL);
  g_return_if_fail (!FTT_CELL_IS_LEAF (cell));
  g_return_if_fail (v != NULL);

  ftt_cell_children (cell, &child);
  for (i = 0; i < FTT_CELLS; i++)
    if (child.c[i]) {
      gdouble a = GFS_IS_MIXED (child.c[i]) ? 
	GFS_STATE (child.c[i])->solid->a : 1.;

      val += GFS_VARIABLE (child.c[i], v->i)*a;
      sa += a;
    }
  if (sa > 0.)
    GFS_VARIABLE (cell, v->i) = val/sa;
}

/**
 * gfs_get_from_below_extensive:
 * @cell: a #FttCell.
 * @v: a #GfsVariable to "get from below".
 *
 * Sets the value of the "extensive" variable @v of @cell as (half in
 * 3D) the sum of the values of its children cells.
 *
 * This functions fails if @cell is a leaf of the cell tree.  
 */
void gfs_get_from_below_extensive (FttCell * cell, const GfsVariable * v)
{
  gdouble val = 0.;
  guint i;
  FttCellChildren child;

  g_return_if_fail (cell != NULL);
  g_return_if_fail (!FTT_CELL_IS_LEAF (cell));
  g_return_if_fail (v != NULL);

  ftt_cell_children (cell, &child);
  for (i = 0; i < FTT_CELLS; i++)
    if (child.c[i])
      val += GFS_VARIABLE (child.c[i], v->i);
  GFS_VARIABLE (cell, v->i) = val
#ifndef FTT_2D
    /2.
#endif /* not FTT_2D */
    ;
}

/**
 * gfs_cell_cleanup:
 * @cell: a #FttCell.
 *
 * Frees the memory allocated for extra data associated with @cell.
 *
 * This function must be used as "cleanup function" when using
 * ftt_cell_destroy().
 */
void gfs_cell_cleanup (FttCell * cell)
{
  g_return_if_fail (cell != NULL);
  
  if (cell->data && GFS_STATE (cell)->solid) {
    g_free (GFS_STATE (cell)->solid);
    GFS_STATE (cell)->solid = NULL;
  }
  g_free (cell->data);
}

/**
 * gfs_cell_reset:
 * @cell: a #FttCell.
 * @v: a #GfsVariable to reset.
 *
 * Sets the value of the variable @v of @cell to zero.
 */
void gfs_cell_reset (FttCell * cell, GfsVariable * v)
{
  g_return_if_fail (cell != NULL);
  g_return_if_fail (v != NULL);

  GFS_VARIABLE (cell, v->i) = 0.;
}

static void add_stats (const FttCell * cell, gpointer * data)
{
  GtsRange * s = data[0];
  GfsVariable * v = data[1];

  gts_range_add_value (s, GFS_VARIABLE (cell, v->i));
}

/**
 * gfs_stats_variable:
 * @root: the root #FttCell of the tree to obtain statistics from.
 * @v: the variable to consider for statistics.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 *
 * Traverses the cell tree defined by @root using ftt_cell_traverse()
 * and gathers statistics about variable @v.
 *
 * Returns: a #GtsRange containing the statistics about @v.
 */
GtsRange gfs_stats_variable (FttCell * root,
			     GfsVariable * v,
			     FttTraverseFlags flags,
			     gint max_depth)
{
  GtsRange s;
  gpointer data[2];

  g_return_val_if_fail (root != NULL, s);
  g_return_val_if_fail (v != NULL, s);
  
  gts_range_init (&s);
  data[0] = &s;
  data[1] = v;
  ftt_cell_traverse (root, FTT_PRE_ORDER, flags, max_depth, 
		     (FttCellTraverseFunc) add_stats, data);
  gts_range_update (&s);

  return s;
}

static void add_norm (const FttCell * cell, gpointer * data)
{
  GfsNorm * n = data[0];
  GfsVariable * v = data[1];

  gfs_norm_add (n, GFS_VARIABLE (cell, v->i), 
		ftt_cell_volume (cell)*(GFS_IS_MIXED (cell) ? 
				    GFS_STATE (cell)->solid->a : 1.));
}

/**
 * gfs_norm_variable:
 * @root: the root #FttCell of the tree to obtain norm from.
 * @v: the variable to consider for norm statistics.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 *
 * Traverses the cell tree defined by @root using ftt_cell_traverse()
 * and gathers norm statistics about variable @v.
 *
 * Returns: a #GfsNorm containing the norm statistics about @v.
 */
GfsNorm gfs_norm_variable (FttCell * root,
			   GfsVariable * v,
			   FttTraverseFlags flags,
			   gint max_depth)
{
  GfsNorm n;
  gpointer data[2];

  g_return_val_if_fail (root != NULL, n);
  g_return_val_if_fail (v != NULL, n);
  
  gfs_norm_init (&n);
  data[0] = &n;
  data[1] = v;
  ftt_cell_traverse (root, FTT_PRE_ORDER, flags, max_depth, 
		     (FttCellTraverseFunc) add_norm, data);
  gfs_norm_update (&n);

  return n;
}

/**
 * gfs_norm_init:
 * @n: a #GfsNorm.
 *
 * Initializes a #GfsNorm.
 */
void gfs_norm_init (GfsNorm * n)
{
  g_return_if_fail (n != NULL);

  n->bias = n->first = n->second = 0.;
  n->infty = - G_MAXDOUBLE;
  n->w = 0.;
}

/**
 * gfs_norm_reset:
 * @n: a #GfsNorm.
 *
 * Sets all the fields of @n to 0.
 */
void gfs_norm_reset (GfsNorm * n)
{
  g_return_if_fail (n != NULL);

  n->bias = n->first = n->second = 0.;
  n->infty = 0.;
  n->w = 0.;
}

/**
 * gfs_norm_add:
 * @n: a #GfsNorm.
 * @val: a value to add to @n.
 * @weight: weight of @val.
 *
 * Adds @val to @n.
 */
void gfs_norm_add (GfsNorm * n, gdouble val, gdouble weight)
{
  g_return_if_fail (n != NULL);

  n->bias += weight*val;
  val = fabs (val);
  if (val > n->infty)
    n->infty = val;
  n->first += weight*val;
  n->second += weight*val*val;
  n->w += weight;
}

/**
 * gfs_norm_update:
 * @n: a #GfsNorm.
 * 
 * Updates the fields of @n.
 */
void gfs_norm_update (GfsNorm * n)
{
  g_return_if_fail (n != NULL);

  if (n->w > 0.0) {
    n->bias /= n->w;
    n->first /= n->w;
    n->second = sqrt (n->second/n->w);
  }
  else
    n->infty = 0.0;
}

/**
 * gfs_face_interpolated_value:
 * @face: a #FttFace.
 * @v: a #GfsVariable index.
 *
 * Computes the value of variable @v on the @face using second-order
 * interpolation from the cell-centered values.
 *
 * Returns: the value of variable @v on the face.  
 */
gdouble gfs_face_interpolated_value (const FttCellFace * face,
				     guint v)
{
  gdouble x1 = 1., v1;
#if 1
  g_return_val_if_fail (face != NULL, 0.);

  v1 = neighbor_value (face, v, &x1);
  return ((x1 - 0.5)*GFS_VARIABLE (face->cell, v) + 0.5*v1)/x1;
#else
  gdouble v0;
  FttCellFace f2;

  g_return_val_if_fail (face != NULL, 0.);

  v0 = GFS_VARIABLE (face->cell, v);
  v1 = neighbor_value (face, v, &x1);
  f2 = ftt_cell_face (face->cell, FTT_OPPOSITE_DIRECTION (face->d));
  if (f2.neighbor) {
    gdouble x2 = 1.;
    gdouble v2 = neighbor_value (&f2, v, &x2);

    return v0 + (x2*(v1 - v0)*(1. + 2.*x2) - x1*(v0 - v2)*(1. - 2.*x1))
      /(4.*x1*x2*(x1 + x2));
  }
  else
    return ((x1 - 0.5)*v0 + 0.5*v1)/x1;
#endif
}

/**
 * gfs_normal_divergence:
 * @cell: a #FttCell.
 *
 * Fills variable %GFS_DIV of @cell with the integral of the divergence
 * of the (MAC) velocity field in this cell.  
 */
void gfs_normal_divergence (FttCell * cell)
{
  FttComponent c;
  gdouble div = 0.;

  g_return_if_fail (cell != NULL);

  if (GFS_IS_MIXED (cell)) {
    GfsSolidVector * solid = GFS_STATE (cell)->solid;
    
    for (c = 0; c < FTT_DIMENSION; c++) {
      FttDirection d = 2*c;
      
      div += 
	solid->s[d]*GFS_STATE (cell)->f[d].un - 
	solid->s[d + 1]*GFS_STATE (cell)->f[d + 1].un;
    }
  }
  else
    for (c = 0; c < FTT_DIMENSION; c++) {
      FttDirection d = 2*c;
      
      div += 
	GFS_STATE (cell)->f[d].un - 
	GFS_STATE (cell)->f[d + 1].un;
    }
  GFS_STATE (cell)->div = div*ftt_cell_size (cell);
}

/**
 * gfs_divergence:
 * @cell: a #FttCell.
 *
 * Fills variable %GFS_DIV of @cell with the divergence of the
 * (centered) velocity field in this cell.  
 */
void gfs_divergence (FttCell * cell)
{
  FttComponent c;
  gdouble div = 0.;

  g_return_if_fail (cell != NULL);

  for (c = 0; c < FTT_DIMENSION; c++)
    div += gfs_center_gradient (cell, c, GFS_VELOCITY_INDEX (c));
  GFS_STATE (cell)->div = div/ftt_cell_size (cell);
}

/**
 * gfs_vorticity:
 * @cell: a #FttCell.
 * @v: a #GfsVariable.
 *
 * Fills variable @v of @cell with the vorticity (norm of the
 * vorticity vector in 3D) of the velocity field in this cell.  
 */
void gfs_vorticity (FttCell * cell,
		    GfsVariable * v)
{
  gdouble size;
#ifndef FTT_2D
  FttVector vort;
#endif /* FTT_3D */
  FttVector * l;

  g_return_if_fail (cell != NULL);
  g_return_if_fail (v != NULL);

  size = ftt_cell_size (cell);
  l = &GFS_DOMAIN (gfs_variable_parent (v))->lambda;
#ifdef FTT_2D
  GFS_VARIABLE (cell, v->i) = 
    (l->x*gfs_center_gradient (cell, FTT_X, GFS_V)/l->y -
     l->y*gfs_center_gradient (cell, FTT_Y, GFS_U)/l->x)/size;
#else  /* FTT_3D */
  vort.x = (l->y*gfs_center_gradient (cell, FTT_Y, GFS_W)/l->z -
	    l->z*gfs_center_gradient (cell, FTT_Z, GFS_V)/l->y)/size;
  vort.y = (l->z*gfs_center_gradient (cell, FTT_Z, GFS_U)/l->x -
	    l->x*gfs_center_gradient (cell, FTT_X, GFS_W)/l->z)/size;
  vort.z = (l->x*gfs_center_gradient (cell, FTT_X, GFS_V)/l->y -
	    l->y*gfs_center_gradient (cell, FTT_Y, GFS_U)/l->x)/size;
  GFS_VARIABLE (cell, v->i) = sqrt (vort.x*vort.x + 
				    vort.y*vort.y + 
				    vort.z*vort.z);
#endif /* FTT_3D */
}

/**
 * gfs_velocity_norm:
 * @cell: a #FttCell.
 * @v: a #GfsVariable.
 *
 * Fills variable @v of @cell with the norm of the velocity field in
 * this cell.
 */
void gfs_velocity_norm (FttCell * cell,
			GfsVariable * v)
{
  GfsStateVector * s;
  
  g_return_if_fail (cell != NULL);
  g_return_if_fail (v != NULL);

  s = GFS_STATE (cell);
#ifdef FTT_2D
  GFS_VARIABLE (cell, v->i) = sqrt (s->u*s->u + s->v*s->v);
#else  /* FTT_3D */
  GFS_VARIABLE (cell, v->i) = sqrt (s->u*s->u + s->v*s->v + s->w*s->w);
#endif /* FTT_3D */
}

/**
 * gfs_velocity_norm2:
 * @cell: a #FttCell.
 * @v: a #GfsVariable.
 *
 * Fills variable @v of @cell with the squared norm of the velocity field in
 * this cell.
 */
void gfs_velocity_norm2 (FttCell * cell,
			 GfsVariable * v)
{
  GfsStateVector * s;

  g_return_if_fail (cell != NULL);
  g_return_if_fail (v != NULL);

  s = GFS_STATE (cell);
#ifdef FTT_2D
  GFS_VARIABLE (cell, v->i) = s->u*s->u + s->v*s->v;
#else  /* FTT_3D */
  GFS_VARIABLE (cell, v->i) = s->u*s->u + s->v*s->v + s->w*s->w;
#endif /* FTT_3D */
}

static void cell_traverse_mixed (FttCell * cell,
				 FttCellTraverseFunc func,
				 gpointer data)
{
  if (!GFS_IS_MIXED (cell))
    return;
  if (FTT_CELL_IS_LEAF (cell))
    (* func) (cell, data);
  else {
    struct _FttOct * children = cell->children;
    guint n;

    for (n = 0; n < FTT_CELLS; n++) {
      FttCell * c = &(children->cell[n]);

      if (!FTT_CELL_IS_DESTROYED (c))
	cell_traverse_mixed (c, func, data);
    }
  }
}

/**
 * gfs_cell_traverse_mixed:
 * @root: the root #FttCell of the tree to traverse.
 * @func: the function to call for each visited #FttCell.
 * @data: user data to pass to @func.
 * 
 * Traverses a cell tree starting at the given root #FttCell. Calls
 * the given function for each leaf cell which is also a mixed cell.
 */
void gfs_cell_traverse_mixed (FttCell * root,
			     FttCellTraverseFunc func,
			     gpointer data)
{
  g_return_if_fail (root != NULL);
  g_return_if_fail (func != NULL);

  cell_traverse_mixed (root, func, data);
}

/**
 * gfs_cell_write:
 * @cell: a #FttCell.
 * @fp: a file pointer.
 * @variables: the #GfsVariable to be written.
 *
 * Writes in @fp the fluid data associated with @cell and described by
 * @variables. This function is generally used in association with
 * ftt_cell_write().  
 */
void gfs_cell_write (const FttCell * cell, FILE * fp, 
		     GfsVariable * variables)
{
  g_return_if_fail (cell != NULL);
  g_return_if_fail (fp != NULL);

  if (GFS_IS_MIXED (cell)) {
    GfsStateVector * s = GFS_STATE (cell);
    guint i;

    for (i = 0; i < FTT_NEIGHBORS; i++)
      fprintf (fp, " %g", s->solid->s[i]);
    fprintf (fp, " %g", s->solid->a);
  }
  else
    fputs (" -1", fp);
  
  while (variables) {
    fprintf (fp, " %g", GFS_VARIABLE (cell, variables->i));
    variables = variables->next;
  }
}

/**
 * gfs_interpolate:
 * @cell: a #FttCell containing location @p.
 * @p: the location at which to interpolate.
 * @v: a #GfsVariable index.
 *
 * Interpolates the @v variable of @cell, at location @p. Linear
 * interpolation is used and the boundaries of the domain are treated
 * as planes of symmetry for all variables.
 *
 * Note that to work correctly this function needs a domain where the
 * values of variable @v are defined on all levels.
 * 
 * Returns: the interpolated value of variable @v at location @p.
 */
gdouble gfs_interpolate (FttCell * cell,
			 FttVector p,
			 guint v)
{
  gboolean leveled;
  FttVector o;
  gdouble f[FTT_DIMENSION], size, v0, val;
  FttComponent c;

  g_return_val_if_fail (cell != NULL, 0.);

  do {
    guint level;

    ftt_cell_pos (cell, &o);
    level = ftt_cell_level (cell);
    leveled = TRUE;

    for (c = 0; c < FTT_DIMENSION && leveled; c++) {
      FttCell * neighbor;

      if (((gdouble *) &p)[c] >= ((gdouble *) &o)[c])
	neighbor = ftt_cell_neighbor (cell, 2*c);
      else
	neighbor = ftt_cell_neighbor (cell, 2*c + 1);

      if (neighbor) {
	if (ftt_cell_level (neighbor) != level)
	  leveled = FALSE;
	else
	  f[c] = GFS_VARIABLE (neighbor, v);
      }
      else /* symmetry conditions */
	f[c] = GFS_VARIABLE (cell, v);
    }
    if (!leveled) {
      cell = ftt_cell_parent (cell);
      g_assert (cell);
    }
  } while (!leveled);

  size = ftt_cell_size (cell);
  val = v0 = GFS_VARIABLE (cell, v);
  for (c = 0; c < FTT_DIMENSION; c++)
    val += fabs (((gdouble *) &p)[c] - ((gdouble *) &o)[c])*(f[c] - v0)/size;

  return val;
}

/* GfsVariable: Object */

static void gfs_variable_destroy (GtsObject * object)
{
  GfsVariable * v = GFS_VARIABLE1 (object);

  g_free (v->name);
  if (v->sources)
    gts_object_destroy (GTS_OBJECT (v->sources));

  (* GTS_OBJECT_CLASS (gfs_variable_class ())->parent_class->destroy) 
    (object);
}

static void gfs_variable_clone (GtsObject * clone, GtsObject * object)
{
  GfsVariable * c = GFS_VARIABLE1 (clone);
  GfsVariable * v = GFS_VARIABLE1 (object);

  (* GTS_OBJECT_CLASS (gfs_variable_class ())->parent_class->clone) 
    (clone, object);
  if (v->name)
    c->name = g_strdup (v->name);
  if (v->sources) {
    GSList * i = GTS_SLIST_CONTAINER (v->sources)->items;

    c->sources = 
      gts_container_new (GTS_CONTAINER_CLASS (gts_slist_container_class ()));
    while (i) {
      gts_container_add (c->sources, GTS_CONTAINEE (i->data));
      i = i->next;
    }
  }
}

static void gfs_variable_class_init (GfsVariableClass * klass)
{
  GTS_OBJECT_CLASS (klass)->destroy = gfs_variable_destroy;
  GTS_OBJECT_CLASS (klass)->clone = gfs_variable_clone;
}

GfsVariableClass * gfs_variable_class (void)
{
  static GfsVariableClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_variable_info = {
      "GfsVariable",
      sizeof (GfsVariable),
      sizeof (GfsVariableClass),
      (GtsObjectClassInitFunc) gfs_variable_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gts_object_class ()),
				  &gfs_variable_info);
  }

  return klass;
}

/**
 * gfs_variable_new:
 * @klass: a #GfsVariableClass.
 * @parent: the parent or %NULL.
 * @name: the name of the variable.
 * @i: the variable index.
 *
 * Returns: a newly allocated #GfsVariable,
 */
GfsVariable * gfs_variable_new (GfsVariableClass * klass,
				GtsObject * parent,
				const gchar * name, 
				guint i)
{
  GfsVariable * v;

  v = GFS_VARIABLE1 (gts_object_new (GTS_OBJECT_CLASS (klass)));
  if (name)
    v->name = g_strdup (name);
  v->i = i;
  v->p = parent;

  return v;
}

/**
 * gfs_variable_list_copy:
 * @v: a #GfsVariable.
 * @parent: the parent of the new list or %NULL.
 *
 * Returns: a new variable list copy of @v.
 */
GfsVariable * gfs_variable_list_copy (GfsVariable * v,
				      GtsObject * parent)
{
  GfsVariable * start = NULL, * prev = NULL;

  while (v) {
    GfsVariable * n = GFS_VARIABLE1 (gts_object_clone (GTS_OBJECT (v)));

    n->p = parent;
    if (prev == NULL)
      start = n;
    else
      prev->next = n;
    prev = n;
    v = v->next;
  }
  return start;
}

/**
 * gfs_variable_list_destroy:
 * @v: a #GfsVariable.
 *
 * Free all the memory allocated for the list starting at @v.
 */
void gfs_variable_list_destroy (GfsVariable * v)
{
  while (v) {
    GfsVariable * next = v->next;

    gts_object_destroy (GTS_OBJECT (v));
    v = next;
  }
}

/**
 * gfs_variable_from_name:
 * @variables: the list of available #GfsVariable.
 * @name: the name of the variable to find.
 *
 * Returns: the #GfsVariable @name or %NULL if this variable name does
 * not exist.  
 */
GfsVariable * gfs_variable_from_name (GfsVariable * variables,
				      const gchar * name)
{
  g_return_val_if_fail (name != NULL, NULL);

  while (variables && strcmp (name, variables->name))
    variables = variables->next;
  return variables;
}

/**
 * gfs_variables_from_list:
 * @variables: the list of available #GfsVariable.
 * @list: a malloc'ed string containing comma separated variable names.
 * @error: where to return the variable name in case of error.
 *
 * Returns: a list of variables or %NULL in case of error, in which
 * case *@error points to the name of the unknown variable.  
 */
GfsVariable * gfs_variables_from_list (GfsVariable * variables,
				       gchar * list,
				       gchar ** error)
{
  gchar * s;
  GfsVariable * var = NULL, * prev = NULL;

  g_return_val_if_fail (list != NULL, NULL);
  g_return_val_if_fail (error != NULL, NULL);

  s = strtok (list, ",");
  while (s) {
    GfsVariable * v = gfs_variable_from_name (variables, s), * n;

    if (v == NULL) {
      *error = s;
      gfs_variable_list_destroy (var);
      return NULL;
    }
    n = gfs_variable_new (gfs_variable_class (), v->p, v->name, v->i);
    if (prev)
      prev->next = n;
    else
      var = n;
    prev = n;
    s = strtok (NULL, ",");
  }
  return var;
}
