/*
** 1998-07-12 -	Various little helper routines for building/managing GUIs.
** 1999-03-13 -	Adapted for new dialog module.
*/

#include "gentoo.h"

#include <string.h>

#include "dialog.h"
#include "guiutil.h"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	Dialog		*dlg;
	GtkWidget	*vbox;
	GtkWidget	*label;
	GtkWidget	*obj;
	GtkWidget	*bar;
	gboolean	terminate;
} PBar;

typedef struct
{
	GObject	*object;
	gulong	handler;
} SigHandler;

struct GuiHandlerGroup
{
	GList	*handlers;
};

/* ----------------------------------------------------------------------------------------- */

/* 1998-07-12 -	Moved this out of the dirpane config module and into this brand new GUI
**		utilities module.
*/
GtkWidget * gui_arrow_button_new(GtkArrowType at)
{
	GtkWidget	*but, *arr;

	but = gtk_button_new();
	arr = gtk_arrow_new(at, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(but), arr);

	return but;
}

/* 1999-04-26 -	Create a button with a "details" icon on it. This is typically a magnifying
**		class. This button is a graphical replacement for a button labeled "...".
** 2009-04-23 -	Happy ten year anniversary, gui_details_button_new()! W00t! Rewritten to use
**		GTK+ 2.0 stock image, GTK_STOCK_FIND is what we use. It used to be a magnifying
**		glass, but in current GTK+ it's a pair of binoculars. Oh well.
*/
GtkWidget * gui_details_button_new(GdkWindow *win)
{
	GtkWidget	*btn, *img;

	btn = gtk_button_new();
	/* This is a bit of a hack; even the MENU size is not quite small enough. */
	img = gtk_image_new_from_stock(GTK_STOCK_FIND, GTK_ICON_SIZE_MENU);
	gtk_button_set_image(GTK_BUTTON(btn), img);
	return btn;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-07-30 -	Build a group of radio buttons with the given labels.
** 1999-05-15 -	Now also sets the GTK+ object user data to the index of each button.
*/
GSList * gui_radio_group_new(guint num, const gchar **label, GtkWidget *but[])
{
	GSList	*list = NULL;
	guint	i;

	for(i = 0; i < num; i++)
	{
		but[i] = gtk_radio_button_new_with_label(list, _(label[i]));
		list = gtk_radio_button_group(GTK_RADIO_BUTTON(but[i]));
		gtk_object_set_user_data(GTK_OBJECT(but[i]), GUINT_TO_POINTER(i));
	}
	return list;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-05 -	This gets called by the dialog module if the user hits the button. */
static void callback_done(gint button, gpointer user)
{
	PBar	*pbar = user;

	pbar->terminate = TRUE;
}

/* 1998-09-05 -	Initialize a user-abortable progress "session". Opens up a dialog window
**		with the given <body> text and <button>, as well as a progress bar widget
**		and a "current object name" indicator (a bare label).
**		Returns an "anchor"; a pointer to some private data. You use this pointer
**		in later calls to identify the session. Progress sessions don't nest!
**		Use calls to gui_progress_update() to make the bar move, and also to learn
**		about any button actions.
*/
gpointer gui_progress_begin(const gchar *body, const gchar *button)
{
	static PBar	pbar;

	pbar.vbox  = gtk_vbox_new(FALSE, 0);
	pbar.label = gtk_label_new(body);
	gtk_box_pack_start(GTK_BOX(pbar.vbox), pbar.label, TRUE, TRUE, 0);
	pbar.obj = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(pbar.vbox), pbar.obj, TRUE, TRUE, 0);
	pbar.bar = gtk_progress_bar_new();
	gtk_box_pack_start(GTK_BOX(pbar.vbox), pbar.bar, TRUE, TRUE, 5);
	pbar.terminate = FALSE;

	pbar.dlg = dlg_dialog_async_new(pbar.vbox, _("Progress"), button, callback_done, &pbar);

	return &pbar;
}

/* 1998-09-05 -	Update the progress bar to a new value (which should be between 0.0 and 1.0).
**		Also allows you to change the indication of the current object.
**		Returns TRUE if the user has hit the button in the window, typically indicating
**		that the entire operation should be aborted ASAP.
*/
gboolean gui_progress_update(gpointer anchor, gfloat value, const gchar *obj)
{
	PBar	*pbar = anchor;

	if(!pbar->terminate)
	{
		gtk_progress_bar_update(GTK_PROGRESS_BAR(pbar->bar), value);
		if((obj != NULL) && (pbar->obj != NULL))
			gtk_label_set_text(GTK_LABEL(pbar->obj), obj);
		gui_events_flush();
	}
	return pbar->terminate;
}

/* 1998-09-05 -	Close down the progress session, since the time-consuming operation has been
**		finished.
*/
void gui_progress_end(gpointer anchor)
{
	PBar	*pbar = anchor;

	if(!pbar->terminate)
		dlg_dialog_async_close(pbar->dlg, -1);
	memset(pbar, 0, sizeof *pbar);		/* Makes sure we notice any foul play. */
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-06-22 -	Just a utility function to help the config modules to their thing. This one
**		builds a menu from a NULL-terminated vector of item texts. Very simple, but
**		also very useful. Like so many other good things. :) Each menu item will
**		have the given <func> connected to it, which will be called with <data> as
**		the user data (as usual). Also, to make it possible for the callback to
**		determine which item was selected, each item will have its index (from 0)
**		set as object's user data. Not pretty, but still simple and useful.
*/
GtkWidget * gui_build_menu(const gchar *text[], GtkSignalFunc func, gpointer user)
{
	GtkWidget	*menu, *item;
	guint		i;

	menu = gtk_menu_new();
	for(i = 0; text[i] != NULL; i++)
	{
		item = gtk_menu_item_new_with_label(_(text[i]));
		if(func != NULL)
		{
			gtk_object_set_user_data(GTK_OBJECT(item), GUINT_TO_POINTER(i));
			g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(func), user);
		}
		gtk_menu_append(GTK_MENU(menu), item);
		gtk_widget_show(item);
	}
	return menu;
}

/* ----------------------------------------------------------------------------------------- */

GuiHandlerGroup * gui_handler_group_new(void)
{
	GuiHandlerGroup	*g;

	g = g_slice_new(GuiHandlerGroup);
	if(g != NULL)
	{
		g->handlers = NULL;
	}
	return g;
}

gulong gui_handler_group_connect(GuiHandlerGroup *g, GObject *obj, const gchar *signal, GCallback cb, gpointer user)
{
	SigHandler	*sh;

	if(g == NULL || obj == NULL || signal == NULL || cb == NULL)
		return 0ul;

	if((sh = g_slice_new(SigHandler)) == NULL)
		return 0ul;
	sh->object = obj;
	sh->handler = g_signal_connect(obj, signal, cb, user);
	/* Prepend, for O(1) performance; order doesn't really matter. */
	g->handlers = g_list_prepend(g->handlers, sh);

	/* Return handler ID, if user code wants to play with it. */
	return sh->handler;
}

void gui_handler_group_block(const GuiHandlerGroup *g)
{
	const GList	*iter;

	if(g == NULL)
		return;
	for(iter = g->handlers; iter != NULL; iter = g_list_next(iter))
	{
		const SigHandler *sh = iter->data;
		g_signal_handler_block(sh->object, sh->handler);
	}
}

void gui_handler_group_unblock(const GuiHandlerGroup *g)
{
	const GList	*iter;

	if(g == NULL)
		return;
	for(iter = g->handlers; iter != NULL; iter = g_list_next(iter))
	{
		const SigHandler *sh = iter->data;
		g_signal_handler_unblock(sh->object, sh->handler);
	}
}

/* ----------------------------------------------------------------------------------------- */

/* 2008-07-04 -	Flush pending GTK+ events. This is handy as a separate function, even though
**		it's only two lines of actual code. It seems that on some systems (Ubuntu),
**		GTK+ smashes errno when running the event flush, and this encapsulation
**		gives us a chance of handling that easily.
*/
void gui_events_flush(void)
{
	gint	old_errno = errno;

	/* Due to side-effects, this can clobber errno, so save it. */
	while(gtk_events_pending())
		gtk_main_iteration();

	errno = old_errno;
}
