# Copyright (C) 2009-2011 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the GNU General Public License version 2
# (see the file COPYING).
"""Bazaar integration."""

__metaclass__ = type

import os
from StringIO import StringIO

from gettext import gettext as _

from gi.repository import Gtk

from bzrlib import workingtree
from bzrlib.branch import Branch
from bzrlib.errors import NotBranchError, NoWorkingTree
from bzrlib.revisionspec import RevisionSpec

try:
    from bzrlib.plugin import load_plugins
    load_plugins()
    from bzrlib.plugins.gtk.annotate.config import GAnnotateConfig
    from bzrlib.plugins.gtk.commit import CommitDialog
    from bzrlib.plugins.gtk.conflicts import ConflictsDialog
    from bzrlib.plugins.gtk.dialog import error_dialog
    from bzrlib.plugins.gtk.initialize import InitDialog
    from bzrlib.plugins.gtk.merge import MergeDialog
    from bzrlib.plugins.gtk.push import PushDialog
    from bzrlib.plugins.gtk.status import StatusWindow
    HAS_BZR_GTK = True
except ImportError:
    HAS_BZR_GTK = False

from gdp import ControllerMixin


__all__ = [
    'BzrProject',
    ]


class BzrProject(ControllerMixin):
    """View and manage a bazaar branch."""

    def __init__(self, window, working_tree=None):
        self.window = window
        self.working_tree = working_tree or self.set_working_tree()

    def deactivate(self):
        """Clean up resources before deactivation."""
        self.working_tree = None

    @property
    def has_bzr_gtk(self):
        """Is bzr-gtk available?"""
        return HAS_BZR_GTK

    def set_working_tree(self):
        """Return the working tree for the working directory or document"""
        doc = self.active_document
        if doc is None:
            self.working_tree = None
            return
        file_path = doc.get_uri_for_display()
        if file_path is None:
            cwd = os.getcwd()
        else:
            cwd = os.path.dirname(file_path)
        try:
            working_tree, relpath = workingtree.WorkingTree.open_containing(
                cwd)
            self.working_tree = working_tree
            self.relpath = relpath
        except (NotBranchError, NoWorkingTree):
            self.working_tree = None

    def _get_branch_revision_tree(self, uri):
        """Return a branch tree for a revision."""
        if uri is None:
            return None
        revision = RevisionSpec.from_string('branch:%s' % uri)
        return revision.as_tree(self.working_tree.branch)

    @property
    def _push_tree(self):
        """The push location tree."""
        return self._get_branch_revision_tree(
            self.working_tree.branch.get_push_location())

    @property
    def _parent_tree(self):
        """The parent location tree."""
        return self._get_branch_revision_tree(
            self.working_tree.branch.get_parent())

    def open_changed_files(self, other_tree):
        """Open files in the bzr branch."""
        if other_tree is None:
            return
        try:
            self.working_tree.lock_read()
            other_tree.lock_read()
            changes = self.working_tree.iter_changes(
                other_tree, False, None,
                require_versioned=False, want_unversioned=False)
            for change in changes:
                # change is: (file_id, paths, content_change, versioned,
                #             parent_id, name, kind, executable)
                # paths is (previous_path, current_path)
                tree_file_path = change[1][1]
                if tree_file_path is None:
                    continue
                base_dir = self.working_tree.basedir
                uri = 'file://%s' % os.path.join(base_dir, tree_file_path)
                self.open_doc(uri)
        finally:
            self.working_tree.unlock()
            other_tree.unlock()

    def open_uncommitted_files(self, action):
        """Open modified and added files in the bzr branch."""
        self.open_changed_files(self.working_tree.basis_tree())

    def open_changed_files_to_push(self, action):
        """Open the changed files in the branch that not been pushed."""
        self.open_changed_files(self._push_tree)

    def open_changed_files_from_parent(self, action):
        """Open the changed files that diverged from the parent branch."""
        self.open_changed_files(self._parent_tree)

    @property
    def diff_file_path(self):
        """The path of the diff file."""
        return os.path.join(self.working_tree.basedir, '_diff.diff')

    def _diff_tree(self, another_tree):
        """Diff the working tree against an anoter tree."""
        if another_tree is None:
            return
#        if not HAS_BZR_GTK:
#            return
        # XXX sinzui 2011-08-01: Hack diff on.
        from bzrlib.plugin import load_plugins
        load_plugins()
        from bzrlib.plugins.gtk.diff import DiffWindow
        window = DiffWindow(parent=self.window)
        window.set_diff("Working Tree", self.working_tree, another_tree)
        window.props.title = "Diff branch - gedit"
        window.show()
        with open(self.diff_file_path, 'w') as diff_file:
            diff_buffer = window.diff.diff_view.buffer
            start_iter = diff_buffer.get_start_iter()
            end_iter = diff_buffer.get_end_iter()
            diff_file.write(diff_buffer.get_text(start_iter, end_iter, True))

    def diff_uncommited_changes(self, action):
        """Create a diff of uncommitted changes."""
        self._diff_tree(self.working_tree.basis_tree())

    def diff_changes_from_parent(self, action):
        """Create a diff of changes from the parent tree."""
        self._diff_tree(self._parent_tree)

    def diff_changes_to_push(self, action):
        """Create a diff of changes to the push tree."""
        self._diff_tree(self._push_tree)

    def commit_changes(self, action):
        """Commit the changes in the working tree."""
        try:
            self.working_tree.lock_write()
            dialog = CommitDialog(self.working_tree, self.window)
            dialog.props.title = "Commit changes - gedit"
            response = dialog.run()
            if response != Gtk.ResponseType.NONE:
                dialog.hide()
                dialog.destroy()
        finally:
            self.working_tree.unlock()

    def show_status(self, action):
        """Show the status of the working tree."""
        base_dir = self.working_tree.basedir
        window = StatusWindow(self.working_tree, base_dir, None)
        window.props.title = "Branch status - gedit"
        window._parent = self.window
        window.show()

    def show_conflicts(self, action):
        """Show the merge, revert, or pull conflicts in the working tree."""
        dialog = ConflictsDialog(self.working_tree)
        dialog.props.title = "Branch conflicts - gedit"
        dialog.run()
        dialog.hide()
        dialog.destroy()

    def show_missing(self, action):
        """Show unmerged/unpulled revisions between two branches."""
        if not HAS_BZR_GTK:
            return
        from bzrlib.plugins.gtk.missing import MissingWindow
        parent_location = self.working_tree.branch.get_parent()
        if parent_location is None:
            message = _("Nothing to do; this branch does not have a parent.")
            dialog = Gtk.MessageDialog(
                type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.CLOSE,
                message_format=message)
            dialog.run()
            dialog.destroy()
            return
        parent_branch = Branch.open_containing(parent_location)[0]
        self.working_tree.branch.lock_read()
        try:
            parent_branch.lock_read()
            try:
                window = MissingWindow(
                    self.working_tree.branch, parent_branch)
                window.run()
            finally:
                parent_branch.unlock()
        finally:
            self.working_tree.branch.unlock()

    def show_tags(self, action):
        """Show the tags in the branch."""
        # XXX sinzui 2009-10-03 bug 107169: seahorse dbus often fails to
        # connect on a cold system. This should not kill the module.
        if not HAS_BZR_GTK:
            return
        from bzrlib.plugins.gtk.tags import TagsWindow
        window = TagsWindow(self.working_tree.branch, self.window)
        window.props.title = "Branch tags - gedit"
        window.show()

    def show_annotations(self, action):
        """Show the annotated revisions of the file."""
        document = self.window.get_active_document()
        file_path = document.get_uri_for_display()
        if file_path is None:
            return
        base_dir = self.working_tree.basedir
        file_path = file_path.replace(base_dir, '')
        file_id = self.working_tree.path2id(file_path)
        if file_id is None:
            return
        # XXX sinzui 2009-10-03 bug 107169: seahorse dbus often fails to
        # connect on a cold system. This should not kill the module.
        if not HAS_BZR_GTK:
            return
        from bzrlib.plugins.gtk.annotate.gannotate import GAnnotateWindow
        window = GAnnotateWindow(parent=self.window)
        window.props.title = "Annotate (" + file_path + ") - gedit"
        GAnnotateConfig(window)
        window.show()
        branch = self.working_tree.branch

        def destroy_window(window):
            branch.unlock()
            self.working_tree.unlock()

        window.connect("destroy", destroy_window)
        branch.lock_read()
        self.working_tree.lock_read()
        window.annotate(self.working_tree, branch, file_id)

    def visualise_branch(self, action):
        """Visualise the tree."""
        limit = None
        branch = self.working_tree.branch
        revisions = [branch.last_revision()]
        # XXX sinzui 2009-10-03 bug 107169: seahorse dbus often fails to
        # connect on a cold system. This should not kill the module.
        if not HAS_BZR_GTK:
            return
        from bzrlib.plugins.gtk.viz import BranchWindow
        window = BranchWindow(branch, revisions, limit, parent=self.window)
        window.props.title = "Visualise branch - gedit"
        window.show()

    def merge_changes(self, action):
        """Merge changes from another branch into te working tree."""
        branch = self.working_tree.branch
        old_tree = branch.repository.revision_tree(branch.last_revision())
        delta = self.working_tree.changes_from(old_tree)
        if (len(delta.added) or len(delta.removed)
            or len(delta.renamed) or len(delta.modified)):
            error_dialog(
                _('There are local changes in the branch'),
                 _('Commit or revert the changes before merging.'))
        else:
            parent_branch_path = branch.get_parent()
            dialog = MergeDialog(
                self.working_tree, self.relpath, parent_branch_path)
            dialog.props.title = "Merge changes - gedit"
            dialog.run()
            dialog.destroy()

    def push_changes(self, action):
        """Push the changes in the working tree."""
        branch = self.working_tree.branch
        dialog = PushDialog(branch.repository, branch.last_revision(), branch)
        dialog.run()
        dialog.hide()
        dialog.destroy()

    def send_merge(self, action):
        """Mail or create a merge-directive for submitting changes."""
        from bzrlib.plugins.gtk.mergedirective import SendMergeDirectiveDialog
        if not HAS_BZR_GTK:
            return
        branch = self.working_tree.branch
        dialog = SendMergeDirectiveDialog(branch)
        dialog.props.title = "Send merge directive - gedit"
        response = dialog.run()
        if response == Gtk.ResponseType.OK:
            directive_file = StringIO()
            directive_file.writelines(dialog.get_merge_directive().to_lines())
            mail_client = branch.get_config().get_mail_client()
            mail_client.compose_merge_request(
                dialog.get_mail_to(), "[MERGE]", directive_file.getvalue())
        dialog.hide()
        dialog.destroy()

    def initialise_branch(self, action):
        """Make a directory into a versioned branch."""
        dialog = InitDialog(os.path.abspath(os.path.curdir))
        dialog.props.title = "Initialize branch - gedit"
        dialog.run()
        dialog.hide()
        dialog.destroy()

    def branch_branch(self, action):
        """Create a new branch that is a copy of an existing branch."""
        # XXX sinzui 2009-10-03 bug 107169: seahorse dbus often fails to
        # connect on a cold system. This should not kill the module.
        if not HAS_BZR_GTK:
            return
        from bzrlib.plugins.gtk.branch import BranchDialog
        dialog = BranchDialog('')
        dialog.props.title = "Branch branch - gedit"
        dialog.run()
        dialog.hide()
        dialog.destroy()

    def checkout_branch(self, action):
        """Create a new checkout of an existing branch."""
        # XXX sinzui 2009-10-03 bug 107169: seahorse dbus often fails to
        # connect on a cold system. This should not kill the module.
        if not HAS_BZR_GTK:
            return
        from bzrlib.plugins.gtk.checkout import CheckoutDialog
        dialog = CheckoutDialog('')
        dialog.props.title = "Checkout branch - gedit"
        dialog.run()
        dialog.hide()
        dialog.destroy()

    def preferences(self, action):
        """Bazaar preferences."""
        if not HAS_BZR_GTK:
            return
        from bzrlib.plugins.gtk.preferences import PreferencesWindow
        dialog = PreferencesWindow()
        dialog.props.title = "Bazaar preferences - gedit"
        dialog.run()
        dialog.hide()
        dialog.destroy()
