# Copyright (C) 2011-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the MIT license (see the file COPYING).

import os

from gi.repository import GObject
from gi.repository import Gtk

from gdp import Config
from gdp import find
from gdp.find import (
    find_params,
    Finder,
    FinderWorker,
    )
from testing import GeditTestCase


class TestCallback:

    def __init__(self, func=None):
        self.called = False
        if func is None:
            func = self.on_callback
        self.func = func

    def on_callback(self):
        self.called = True

    def __call__(self):
        self.func()
        self.on_callback()


class FinderWorkerTestCase(GeditTestCase):

    def makeTreeStore(self):
        return Gtk.TreeStore(
            GObject.TYPE_STRING, GObject.TYPE_STRING, GObject.TYPE_INT,
            GObject.TYPE_STRING, GObject.TYPE_STRING)

    def test__init__(self):
        params = find_params('./', 'snark', False, False, None)
        treestore = self.makeTreeStore()
        callback = TestCallback()
        worker = FinderWorker(treestore, params, callback)
        self.assertEqual(treestore, worker.treestore)
        self.assertEqual(params, worker.find_params)
        self.assertEqual(callback, worker.callback)
        self.assertEqual(None, worker.substitution)
        self.assertEqual(Gtk.IconTheme.get_default(), worker.theme)

    def test_pattern_simple(self):
        params = find_params('./', 'Snark(', False, False, None)
        treestore = self.makeTreeStore()
        callback = TestCallback()
        worker = FinderWorker(treestore, params, callback)
        self.assertEqual('(?i)Snark\\(', worker.pattern)

    def test_pattern_re(self):
        params = find_params('./', 'Snark(', True, False, None)
        treestore = self.makeTreeStore()
        callback = TestCallback()
        worker = FinderWorker(treestore, params, callback)
        self.assertEqual('(?i)Snark(', worker.pattern)

    def test_pattern_case(self):
        params = find_params('./', 'Snark(', False, True, None)
        treestore = self.makeTreeStore()
        callback = TestCallback()
        worker = FinderWorker(treestore, params, callback)
        self.assertEqual('Snark\\(', worker.pattern)

    def test_append_match(self):
        params = find_params('./', 'snark', False, False, None)
        treestore = self.makeTreeStore()
        callback = TestCallback()
        worker = FinderWorker(treestore, params, callback)
        result = worker.append_match(
            None, 'hello', 'stock_dialog-info', 0, None, None)
        self.assertIs(False, result)
        self.assertEqual('hello', worker.treestore[0][0])

    def test_run(self):
        params = find_params('./', 'snark', False, False, None)
        treestore = self.makeTreeStore()
        callback = TestCallback()
        worker = FinderWorker(treestore, params, callback)
        worker.run()
        self.assertIs(True, callback.called)

    def test_start(self):
        params = find_params('./', 'snark', False, False, None)
        treestore = self.makeTreeStore()
        callback = TestCallback()
        worker = FinderWorker(treestore, params, callback)
        result = worker.start()
        self.assertIs(False, result)
        worker.join()
        self.assertIs(True, callback.called)


class FinderTestCase(GeditTestCase):

    def test_setup_widgets(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        finder = Finder(window)
        finder.setup_widgets()
        self.assertIsNot(None, finder.path_comboentry)
        self.assertEqual(
            os.getcwd(), finder.path_comboentry.get_active_text())
        self.assertIsNot(None, finder.pattern_comboentry)
        self.assertEqual('', finder.pattern_comboentry.get_active_text())
        self.assertIsNot(None, finder.file_comboentry)
        self.assertEqual(
            '<Any Text File>', finder.file_comboentry.get_active_text())
        self.assertIsNot(None, finder.substitution_comboentry)
        self.assertEqual('', finder.substitution_comboentry.get_active_text())
        self.assertIsNot(None, finder.file_lines_view)

    def test_setup_comboentry(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        finder = Finder(window)
        comboentry = Gtk.ComboBoxText.new_with_entry()
        finder.setup_comboentry(comboentry)
        self.assertEqual(0, comboentry.get_entry_text_column())
        liststore = comboentry.get_model()
        self.assertIsInstance(liststore, Gtk.ListStore)
        self.assertEqual(1, liststore.get_n_columns())
        self.assertEqual(GObject.TYPE_STRING, liststore.get_column_type(0))
        self.assertEqual(
            (0, Gtk.SortType.ASCENDING), liststore.get_sort_column_id())

    def test_setup_comboentry_with_default(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        finder = Finder(window)
        comboentry = Gtk.ComboBoxText.new_with_entry()
        finder.setup_comboentry(comboentry, 'example')
        self.assertEqual(0, comboentry.get_entry_text_column())
        self.assertEqual('example', comboentry.get_active_text())

    def test_setup_comboentry_with_default_and_config_key(self):
        self.make_config(find, Config)
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        finder = Finder(window)
        comboentry = Gtk.ComboBoxText.new_with_entry()
        finder.setup_comboentry(comboentry, 'eg', 'paths')
        self.assertEqual('eg', comboentry.get_active_text())
        words = sorted(row[0] for row in iter(comboentry.get_model()))
        self.assertEqual(
            ['<Current File>', '<Working Directory>', 'eg'], words)

    def test_setup_comboentry_with_default_and_config_key_paths(self):
        self.make_config(find, Config)
        temp_dir = os.path.dirname(self.make_file('').name)
        find.config.setlist(
            'finder', 'paths',
            ['<Current File>', '<Working Directory>', 'gone', temp_dir])
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        finder = Finder(window)
        comboentry = Gtk.ComboBoxText.new_with_entry()
        finder.setup_comboentry(comboentry, '<Working Directory>', 'paths')
        words = sorted(row[0] for row in iter(comboentry.get_model()))
        self.assertEqual(
            [temp_dir, '<Current File>', '<Working Directory>'], words)

    def test_update_comboentry(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        finder = Finder(window)
        comboentry = Gtk.ComboBoxText.new_with_entry()
        finder.update_comboentry(comboentry, 'one', False)
        finder.update_comboentry(comboentry, 'two', True)
        finder.update_comboentry(comboentry, 'one', False)
        finder.update_comboentry(comboentry, 'three', False)
        words = sorted(row[0] for row in iter(comboentry.get_model()))
        self.assertEqual(['one', 'three', 'two'], words)
        self.assertEqual('two', comboentry.get_active_text())

    def test_save_find_data(self):
        # Setup a test config so that the singleton is not mutated.
        file_ = self.make_config(find, Config)
        find.config.setlist(
            'finder', 'matches', ['one', 'two', 'three', 'four', 'five',
                'six', 'seven', 'eight', 'nine', 'ten'])
        find.config.setlist('finder', 'files', ['red', 'green', 'blue'])
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        finder = Finder(window)
        finder.update_comboentry(finder.pattern_comboentry, 'eleven', True)
        finder.update_comboentry(finder.file_comboentry, 'green', True)
        finder.save_find_data()
        self.assertEqual(
            ['eleven', 'one', 'two', 'three', 'four', 'five',
             'six', 'seven', 'eight', 'nine'],
            find.config.getlist('finder', 'matches'))
        self.assertEqual(
            ['green', 'red', 'blue'], find.config.getlist('finder', 'files'))
        self.assertEqual(
            [os.getcwd(), '<Current File>', '<Working Directory>'],
            find.config.getlist('finder', 'paths'))
        self.assertEqual(
            [''], find.config.getlist('finder', 'substitutions'))
        config_data = file_.read()
        self.assertIn('matches = eleven\n', config_data)
        self.assertNotIn('\t.\n', config_data)

    def test_on_find_in_files(self):
        # Setup a test config so that the singleton is not mutated.
        file_ = self.make_config(find, Config)
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        finder = Finder(window)
        finder.update_comboentry(finder.pattern_comboentry, 'snark', True)
        finder.update_comboentry(
            finder.path_comboentry, '<Current File>', True)
        finder.on_find_in_files()
        self.assertEqual(finder.get_find_params(), finder.last_find)
        self.assertEqual(
            'Matches for [snark]',
            finder.file_lines_view.get_column(0).props.title)
        self.assertIn('matches = snark\n', file_.read())

    def test_show_replace(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        panel = window.get_side_panel()
        panel.props.visible == False
        finder = Finder(window)
        finder.show_replace(None)
        self.assertTrue(panel.props.visible)

    def test_get_untested_replacement_dialog(self):
        # If the plugin's last_find is not equal to the current find,
        # a window is presented to confirm the replace.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        panel = window.get_side_panel()
        panel.props.visible == False
        finder = Finder(window)
        finder.pattern_comboentry.get_children()[0].set_text('pting')
        find_params = finder.get_find_params()
        dialog = finder._get_untested_replacement_dialog(find_params)
        self.assertIsNot(None, dialog)
