/*
 *  Main authors:
 *     Patrick Pekczynski <pekczynski@ps.uni-sb.de>
 *
 *  Copyright:
 *     Patrick Pekczynski, 2004
 *
 *  Last modified:
 *     $Date: 2005-12-02 18:22:34 +0100 (Fri, 02 Dec 2005) $ by $Author: pekczynski $
 *     $Revision: 2691 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  See the file "LICENSE" for information on usage and
 *  redistribution of this file, and for a
 *     DISCLAIMER OF ALL WARRANTIES.
 *
 *
 */

#ifndef __GECODE_INT_SORTEDNESS_HH__
#define __GECODE_INT_SORTEDNESS_HH__


#include "int.hh"
#include "int/rel.hh"
#include "int/distinct.hh"

/**
 * \namespace Gecode::Int::Sortedness
 * \brief %Sortedness propagators
 */

namespace Gecode { namespace Int { namespace Sortedness {

  /**
   * \brief Bounds consistent sortedness propagator
   * \par [Reference] 
   *  The algorithm is taken from: \n
   *  Sven Thiel: Efficient Algorithms for Constraint Propagation 
   *  and for Processing Tree Descriptions (pages 39 to 59)
   *  [http://www.mpi-sb.mpg.de/~sthiel/thesis.pdf]
   *
   * Requires \code #include "int/sortedness.hh" \endcode 
   * and \code #include "int/distinct.hh" \endcode.
   * The latter is required for the extended version of sortedness 
   * including permutation views. 
   * \ingroup FuncIntProp
   */

  template<class View, class Tuple, bool Perm, bool shared>
  class Sortedness : public Propagator {
  protected:
    
    /**
     *  \brief Views to be sorted
     *
     *   If Tuple is instantiated to ViewTuple<View,2>, 
     *   \f$xz_i\f$ is a pair \f$(x_i, z_i)\f$, where
     *   \f$x_i\f$ denotes an unsorted view
     *   and \f$z_i\f$ is the explicit 
     *   permutation view mapping \f$x_i\f$ to its unsorted counterpart
     *   \f$y_{z_i}\f$. 
     *   If Tuple is instantiated to ViewTuple<View,1>, 
     *   \f$xz_i\f$ is the unsorted view \f$x_i\f$ lifted to a ViewTuple.
     */
    ViewArray<Tuple> xz;

    /// Views denoting the sorted version of \a x
    ViewArray<View> y;
    /// original y array
    ViewArray<View> w;
    /// connection to dropped view
    int reachable;
    /// Constructor for posting
    Sortedness(Space*, 
	       ViewArray<Tuple>&, 
	       ViewArray<View>&);

    /// Constructor for cloning 
    Sortedness(Space* home, 
	       bool share, 
	       Sortedness<View, Tuple, Perm, shared>& p);

  public:
    /// Destructor
    virtual ~Sortedness(void);
    
    /// Copy propagator during cloning
    virtual Actor* copy(Space* home, bool share);
    
    /**
     * \brief Cost function returning PC_LINEAR_HI
     */
    virtual PropCost cost (void) const;
    
    /// Perform propagation
    virtual ExecStatus propagate(Space* home);
    
    /// Post propagator for the views \a xz and \a y
    static  ExecStatus post(Space*, 
			    ViewArray<Tuple>&, 
			    ViewArray<View>&);

  };

  
}}}

#include "int/sortedness/sortsup.icc"
#include "int/sortedness/order.icc"
#include "int/sortedness/matching.icc"
#include "int/sortedness/narrowing.icc"
#include "int/sortedness/sortedness.icc"

#endif


// STATISTICS: int-prop

