if (HAVE_WIN32):
    from Win32Backend import backend
else:
    from GConfBackend import backend

from ConfigDialog import ConfigDialog
from utils.datatypes import *


class DisplayConfigger(ConfigDialog):
    """
      Configuration Dialog for the displays. This class handles loading / saving
      the configuration.
    """

    def __init__(self, ident):

        # may we save already, or is it too early?
        self.__may_save = False

        # ID for storing the configuration
        self.__ident = ident

        # the scripting environment with which the config interacts
        self.__scripting_environment = None


        ConfigDialog.__init__(self)
        self.set_property("title", _("Configuration"))



    #
    # Sets the scripting environment to be used.
    #
    def set_scripting_environment(self, script):

        self.__scripting_environment = script
        self._set_setter(self.__setter_wrapper)
        self._set_getter(self.__scripting_environment.get_value)
        self._set_caller(self.__scripting_environment.call_function)



    #
    # Wraps the config setter handler to include saving config.
    #
    def __setter_wrapper(self, key, value, datatype):
        assert key

        self.__scripting_environment.set_value(key, value)

        # save config, but not too early
        if (self.__may_save):
            rep = dtype_repr(datatype, value)
            backend.set(self.__ident, key, rep)
            # this hack allows us to store extended type information in GConf;
            # but we're going to ditch the GConf backend anyway, soon ^_^
            backend.set(self.__ident, key + "_TYPE", dtype_name(datatype))



    def build(self, items):

        ConfigDialog.build(self, items)

        # put the elements into the scripting environment
        for c in self.get_config_items():
            ident = c.get_prop("id")
            if (ident):
                self.__scripting_environment.add_element("Prefs", ident, c)
        #end for


    #
    # Loads the initial configuration.
    #
    def load_config(self):

        # read stored configuration
        for key in backend.list(self.__ident):
            if (key.endswith("_TYPE")): continue

            rep = backend.get(self.__ident, key)
            dtype = backend.get(self.__ident, key + "_TYPE")

            if (rep == backend.UNDEF):
                continue

            elif (dtype == backend.UNDEF):
                # legacy case: data saved with previous versions of gdesklets
                # has no type information
                value = rep

            else:
                datatype = dtype_get_type(dtype)
                value = dtype_build(datatype, rep)

            try:
                self.__setter_wrapper(key, value, None)
            except:
                log("Couldn't pass arguments (%s, %s) to setter."
                    % (key, value))
        #end for

        # have the children update themselves
        for c in self.get_config_items(): c.update()



    #
    # Removes the configuration.
    #
    def remove_config(self, *path):

        if (not path): path = [self.__ident]

        entries = backend.list(*path)
        if (not entries):
            backend.clear(*path)
        else:
            for e in entries:
                self.remove_config( * list(path) + [e] )



    def show(self):

        self.__may_save = True
        ConfigDialog.show(self)
