from DisplayTarget import DisplayTarget
from utils import dialog
from main import _

import gtk


#
# Abstract class for targets that can contain others.
#
class ContainerTarget(DisplayTarget):


    def __init__(self, parent, display):

        # the list is necessary for preserving order
        self.__children = []

        # mapping: id -> child
        self.__children_id = {}

        # mapping: child -> id
        self.__ids = {}
        
        self.__settings = []
        

        DisplayTarget.__init__(self, parent, display)
        


    def _register_child(self, child, cid):

        self.__children_id[cid] = child
        self.__ids[child] = cid
        self.__children.append(child)



    def _unregister_child(self, child):

        cid = self.__ids[child]
        if (self.__children_id.has_key(cid)): del self.__children_id[cid]
        if (self.__ids.has_key(child)): del self.__ids[child]
        self.__children.remove(child)



    def add_children(self, childrendata):

        for childname, childclass, childsettings, children in childrendata:
            self.add_child(childname, childclass, childsettings, children)



    def add_child(self, cid, clss, settings, children):

        display = self._get_display()
        child = clss(self, display)
        if (children): child.add_children(children)

        self._register_child(child, cid)

        if (settings):
            for key, value in settings.get_entries():
                self.__settings.append((child, key, value))



    def _init_children(self):

        for child, key, value in self.__settings:
            child.set_config(key, value)
        

        

    def _get_children(self): return self.__children[:]

    def _get_child(self):

        children = self._get_children()
        if (not children): return None
        elif (len(children) > 1):
            name = "???"
            msg = _("The %(name)s container accepts only one child.")
            print msg % vars()
            #dialog.warning(_("Too many children"), msg % vars())
            return children[0]

        else: return children[0]

    def get_child_by_id(self, id): return self.__children_id.get(id)

    def get_id_by_child(self, child): return self.__ids.get(child)



    def delete(self):

        DisplayTarget.delete(self)
        for c in self._get_children():
            c.delete()
            self._unregister_child(c)
        self.__children = []
        self.__children_id.clear()
        self.__ids.clear()




    #
    # Override this method if the container size differs from the target size.
    #
    def get_container_geometry(self):

        return self.get_geometry()


    #
    # Returns the border size of this container. Override this method to return
    # the sizes of the four borders (left, top, right, bottom).
    #
    def get_border_size(self):

        return (0, 0, 0, 0)


    #
    # Returns the bounding box occupied by the children of this container.
    #
    def __get_children_bbox(self):

        bx1 = by1 = 1000000
        bx2 = by2 = 0
        for c in self._get_children():
            cx, cy, cw, ch = c.get_geometry()
            px, py, pw, ph = c.get_percentual_state()
            if (not pw):
                bx1 = min(bx1, cx)
                bx2 = max(bx2, cx + cw)
            if (not ph):
                by1 = min(by1, cy)
                by2 = max(by2, cy + ch)

        bx1 = min(bx1, bx2)
        by1 = min(by1, by2)
        bx2 = max(bx1, bx2)
        by2 = max(by1, by2)

        return (bx1, by1, bx2 - bx1, by2 - by1)




    #
    # Sets the size of this container.
    #
    def set_size(self, width, height):

        DisplayTarget.set_size(self, width, height)


    #
    # Returns the geometry value that should be used for unset values.
    #
    def _get_value_for_unset(self, x = None, y = None,
                             width = None, height = None):

        bx, by, bw, bh = self.__get_children_bbox()
        bw1, bh1, bw2, bh2 = self.get_border_size()
        bw += bw1 + bw2
        bh += bh1 + bh2
        
        if (x != None): return x
        elif (y != None): return y
        elif (width != None): return bx + bw
        elif (height != None): return by + bh



    #
    # Tells other targets about geometry changes.
    #
    def _propagate_geometry(self):

        parent = self._get_parent()
        if (parent): gtk.idle_add(parent.adjust_geometry)

        # tell percentual children about the new size
        for c in self._get_children():
            px, py, pw, ph = c.get_percentual_state()
            if (px or py or pw or ph): c.adjust_geometry()
            
        self.update_observer(self.OBS_GEOMETRY)



    def handle_action(self, action, px, py, path, args):

        DisplayTarget.handle_action(self, action, px, py, path[:], args)

        if (self.get_index() != -1): path.append(self.get_index())

        bw, bn, be, bs = self.get_border_size()
        children = self._get_children()
        px -= bw; py -= bn
        for c in children:
            cx, cy, cw, ch = c.get_geometry()
            if (cx <= px <= cx + cw and cy <= py <= cy + ch):
                c.handle_action(action, px - cx, py - cy, path[:], args)
                c.notify_handle_action(1, path[:])
            else:
                c.notify_handle_action(0, path[:])
        #end for


    #
    # Propagate LEAVE events to all children.
    #
    def notify_handle_action(self, value, path):

        DisplayTarget.notify_handle_action(self, value, path[:])
        if (value == 0):
            if (self.get_index() != -1): path.append(self.get_index())
            for c in self._get_children():
                c.notify_handle_action(0, path[:])
        #end if
