package GCOptions;

###################################################
#
#  Copyright 2005 Tian
#
#  This file is part of GCfilms.
#
#  GCfilms is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCfilms is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCfilms; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;

{
    package GCOptionLoader;

    my $DEFAULT_LANG='EN';

    #use GCLang;

    sub new
    {
        my ($proto, $file) = @_;
        my $class = ref($proto) || $proto;
        my $self  = {};
        
        bless ($self, $class);
        $self->load($file) if $file;
        return $self;
    }

    sub setParent
    {
        my ($self, $parent) = @_;
        $self->{parent} = $parent;
    }

    sub load
    {
        my ($self, $file) = @_;

        $self->{file} = $file;

        $self->{options} = undef;

        if (! -e $file)
        {
            my $tmpLang = $ENV{LANG};
            $tmpLang =~ s/(..)_.*/\U$1\E/ if $tmpLang;
            $tmpLang = $DEFAULT_LANG if ! $tmpLang;
        
            open OPTIONS, ">$file" or $self->{parent}->optionsError('create');
            printf OPTIONS 'images='.$ENV{GCF_DATA_HOME}.'/images/
autosave=1
browser=firefox
player=mplayer
file='.$ENV{GCF_DATA_HOME}.'/films.gcf
order=0
split=300
width=920
height=640
confirm=1
lang='.$tmpLang.'
status=1
splash=1
toolbar=2
toolbarPosition=0
status=1
transform=1
articles=le,la,les,l,un,une,des,a,the,der,die,das,ein,eine,el,los,una
plugin=multiask
askImport=0
searchStop=1
alwaysOriginal=0
proxy=
borrowers=
emails=
view=0
columns=3
listBgPicture=1
mailer=Sendmail
from=
subject=GCfilms email
smtp=
template=Hello %1,<br><br>You borrowed me %2 since %3. I\'d like to get it back shortly.
history=none|none|none|none|none
useRelativePaths=0
';
            close OPTIONS;
        }

        open OPTIONS, $file or $self->{parent}->optionsError('open');
        
        while (<OPTIONS>)
        {
            chomp;
            my @assign =  split m/=/;
            $self->{options}->{$assign[0]} = $assign[1];
        }
                
        close OPTIONS;
        
        $self->lang($DEFAULT_LANG) if (! exists $GCLang::langs{$self->lang});
    }

    sub getFullLang
    {
        my $self = shift;
        
        (my $lang = $self->lang) =~ s/(.*)/\L$1\E_$1/;
        $lang =~ s/_EN/_US/;
        
        $lang .= '.UTF-8';
        
        return $lang;
    }

    sub save
    {
        my $self = shift;

        open OPTIONS, '>'.$self->{file} or $self->{parent}->optionsError('save');
   
        my %options = %{$self->{options}};
     
        my $line = '';
		my $key;
		my $value;
        while (($key,$value) = each(%options))
        {
            $line = $key.'='.$value;

            printf OPTIONS "$line\n";            
        }
        
        close OPTIONS;
    }

    sub exists
    {
        my ($self, $name) = @_;
        
        return exists $self->{options}->{$name};
    }

    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        if (@_)
        {
            $self->{options}->{$name} = shift;
        }
        else
        {
            return $self->{options}->{$name};
        }
    }
}

{
    package GCOptionsDialog;
    use GCPlugins;

    use Gtk2;
    use GCLang;
    use GCStyle;
    use base "Gtk2::Dialog";

    sub on_destroy
    {
        my ($widget, $self) = @_;
        return 1;
    }

    sub initValues
    {
        my $self = shift;
        
        $self->{browser}->set_text($self->{options}->browser);
        $self->{player}->set_text($self->{options}->player);
        $self->{images}->set_text($self->{options}->images);
        $self->{confirm}->set_active($self->{options}->confirm);
        $self->{autosave}->set_active($self->{options}->autosave);
        $self->{splash}->set_active($self->{options}->splash);
        $self->{useRelativePaths}->set_active($self->{options}->useRelativePaths);
        $self->{status}->set_active($self->{options}->status);
        $self->{proxycb}->set_active($self->{options}->proxy);
		$self->{proxyurl}->set_text($self->{options}->proxy);
        $self->{transform}->set_active($self->{options}->transform);
        $self->{articles}->set_text($self->{options}->articles);
        $self->{formats}->set_text($self->{options}->formats);

        $self->{options}->columns(3) if ! $self->{options}->exists('columns');
        $self->{columns} = $self->{options}->columns;

        $self->{options}->toolbar(1) if ! $self->{options}->exists('toolbar');
        $self->{toolbar}->set_active($self->{options}->toolbar);
        $self->{toolbarOption}->remove_menu;
        $self->{toolbarOption}->set_menu($self->{toolbar});

        $self->{options}->toolbarPosition(0) if ! $self->{options}->exists('toolbarPosition');
        $self->{toolbarPosition}->set_active($self->{options}->toolbarPosition);
        $self->{toolbarPositionOption}->remove_menu;
        $self->{toolbarPositionOption}->set_menu($self->{toolbarPosition});

        $self->{options}->view(0) if ! $self->{options}->exists('view');
        $self->{view}->set_active($self->{options}->view);
        $self->{viewOption}->remove_menu;
        $self->{viewOption}->set_menu($self->{view});
       
        my @plugins = @{$self->{plugins}};
		$self->{options}->plugin('ask') if ! $self->{options}->exists('plugin');
        my $i = 0;
        foreach (@plugins)
        {
            last if ($self->{options}->plugin eq 'ask')
                     || ($self->{options}->plugin eq 'multi')
										 || ($self->{options}->plugin eq 'multiask')
                     || ($_ eq $self->{options}->plugin);
            $i++;
        }
        $i++ if $self->{options}->plugin eq 'multiask';
        $i = $i + 2 if $self->{options}->plugin eq 'multi';
        $self->{plugin}->set_active($i);
        $self->{pluginOption}->remove_menu;
        $self->{pluginOption}->set_menu($self->{plugin});

        $self->{askImport}->set_active($self->{options}->askImport);
        $self->{searchStop}->set_active($self->{options}->searchStop);
        $self->{alwaysOriginal}->set_active($self->{options}->alwaysOriginal);
        
        my @langs= @{$self->{langs}};
        #my $langCurrent = $self->{parent}->{lang}->{LangName};
        my $langCurrent = $langs{$self->{options}->lang}->{LangName};
        $i = 0;
        foreach (@langs)
        {
            last if ($_ eq $langCurrent);
            $i++;
        }
        $self->{langMenu}->set_active($i);
        $self->{langOption}->remove_menu;
        $self->{langOption}->set_menu($self->{langMenu});

        my @styles= @{$self->{styles}};
        my $styleCurrent = $self->{parent}->{style}->{name};
        $i = 0;
        foreach (@styles)
        {
            last if ($_ eq $styleCurrent);
            $i++;
        }
        $self->{style}->set_active($i);
        $self->{styleOption}->remove_menu;
        $self->{styleOption}->set_menu($self->{style});

        $self->{order}->set_active($self->{options}->order);
        $self->{orderOption}->remove_menu;
        $self->{orderOption}->set_menu($self->{order});        
        
        $self->{options}->mailer('Sendmail') if ! $self->{options}->exists('mailer');
        my $mailer = ($self->{options}->mailer eq 'Sendmail') ? 0 : 1;
        $self->{mailer}->set_active($mailer);
        $self->{mailerOption}->remove_menu;
        $self->{mailerOption}->set_menu($self->{mailer});

        $self->{from}->set_text($self->{options}->from);
        
        $self->{options}->listImgSize(2) if ! $self->{options}->exists('listImgSize');
        $self->{listImgSize} = $self->{options}->listImgSize;
        $self->{options}->listImgSkin($GCStyle::defaultList) if ! $self->{options}->exists('listImgSkin');
        $self->{listImgSkin} = $self->{options}->listImgSkin;
        $self->{options}->listBgColor('65535,65535,65535') if ! $self->{options}->exists('listBgColor');
        $self->{options}->listFgColor('0,0,0') if ! $self->{options}->exists('listFgColor');
        $self->{mlbg} = $self->{options}->listBgColor;
        $self->{mlfg} = $self->{options}->listFgColor;
        $self->{listBgPicture} = $self->{options}->listBgPicture;
                
        $self->{proxyurl}->set_editable(0) if ! $self->{proxycb}->get_active;
        $self->{proxyurl}->set_editable(1) if $self->{proxycb}->get_active;

        $self->{historysize}->set_value($self->{options}->historysize);
        
        $self->{options}->details('image|title|date|director|type') if ! $self->{options}->exists('details');
        $self->{details} = $self->{options}->details;
        $self->{options}->groupMovies(1) if ! $self->{options}->exists('groupMovies');
        $self->{groupMovies} = $self->{options}->groupMovies;
    }
    
    sub show
    {
        my $self = shift;

        $self->initValues;
        
        $self->show_all;
                
        $self->{hboxImages}->hide;
        $self->{hboxDetails}->hide;
        $self->{hboxImages}->show_all if ($self->{options}->view == 1);
        $self->{hboxDetails}->show_all if ($self->{options}->view == 2);

        $self->{hboxSMTP}->hide if $self->{options}->mailer eq 'Sendmail';
        $self->{pluginList}->hide if (($self->{options}->plugin ne 'multi') && ($self->{options}->plugin ne 'multiask'));

        my $code = $self->run;
        $self->hide;
        if ($code eq 'ok')
        {
            $self->{options}->browser($self->{browser}->get_text);
            $self->{options}->player($self->{player}->get_text);
            $self->{options}->images($self->{images}->get_text);
	        $self->{options}->confirm(($self->{confirm}->get_active) ? 1 : 0);
	        $self->{options}->autosave(($self->{autosave}->get_active) ? 1 : 0);
	        $self->{options}->status(($self->{status}->get_active) ? 1 : 0);
	        $self->{options}->splash(($self->{splash}->get_active) ? 1 : 0);
	        $self->{options}->useRelativePaths(($self->{useRelativePaths}->get_active) ? 1 : 0);
	        $self->{options}->transform(($self->{transform}->get_active) ? 1 : 0);
            $self->{options}->articles($self->{articles}->get_text);
            $self->{options}->formats($self->{formats}->get_text);

            if ($self->{proxycb}->get_active)
		    {
	            $self->{options}->proxy($self->{proxyurl}->get_text);
		    }
		    else
		    {
	            $self->{options}->proxy('');
		    }
	
	        my $lang = $self->{langOption}->child->get_label;
            foreach (keys %GCLang::langs)
            {
                $self->{options}->lang($_);
                last if $GCLang::langs{$_}->{LangName} eq $lang;
            }
            
            my $order = $self->{orderOption}->child->get_label;
            my $i = 0;
            foreach (@{$self->{orders}})
            {
                $self->{options}->order($i);
                last if $_ eq $order;
                $i++;
            }
            
	        my $style = $self->{styleOption}->child->get_label;
            foreach (keys %GCStyle::styles)
            {
                $self->{options}->style($_);
                last if $GCStyle::styles{$_}->{name} eq $style;
            }
            	
	        my %toolbars = %{$self->{parent}->{lang}->{OptionsToolbars}};
	        foreach my $i (0..(scalar(keys %toolbars) - 1))
	        {
	            $self->{options}->toolbar($i);
	            last if $toolbars{$i} eq $self->{toolbarOption}->child->get_label;
	        }

	        my %toolbarPositions = %{$self->{parent}->{lang}->{OptionsToolbarPositions}};
	        foreach my $i (0..(scalar(keys %toolbarPositions) - 1))
	        {
	            $self->{options}->toolbarPosition($i);
	            last if $toolbarPositions{$i} eq $self->{toolbarPositionOption}->child->get_label;
	        }

            my $currentView = $self->{options}->view;
            my %views = %{$self->{parent}->{lang}->{OptionsViews}};
	        foreach my $i (0..(scalar(keys %views) - 1))
	        {
	            $self->{options}->view($i);
	            last if $views{$i} eq $self->{viewOption}->child->get_label;
	        }
            
            my $currentColumns = $self->{options}->columns;
            my $currentImgSize = $self->{options}->listImgSize;
            my $currentImgSkin = $self->{options}->listImgSkin;
            my $currentBg = $self->{options}->listBgColor;
            my $currentFg = $self->{options}->listFgColor;
            my $currentBgPicture = $self->{options}->listBgPicture;
            my $currentDetails = $self->{options}->details;
            my $currentGroupMovies = $self->{options}->groupMovies;
            $self->{options}->columns($self->{columns});
            $self->{options}->listImgSize($self->{listImgSize});
            $self->{options}->listImgSkin($self->{listImgSkin});
            $self->{options}->listBgColor($self->{mlbg});
            $self->{options}->listFgColor($self->{mlfg});
            $self->{options}->listBgPicture($self->{listBgPicture});
            $self->{options}->details($self->{details});
            $self->{options}->groupMovies($self->{groupMovies});
            
            if (
                 ($currentView != $self->{options}->view)
                 ||
                 (
                    ($self->{options}->view == 1)
                    &&
                    (
                      ($currentColumns != $self->{options}->columns)
                      ||
                      ($currentImgSize ne $self->{options}->listImgSize)
                      ||
                      ($currentImgSkin ne $self->{options}->listImgSkin)
                      ||
                      ($currentBg ne $self->{options}->listBgColor)
                      ||
                      ($currentFg ne $self->{options}->listFgColor)
                      ||
                      ($currentBgPicture != $self->{options}->listBgPicture)
                    )
                 )
                 ||
                 (
                    ($self->{options}->view == 2)
                    &&
                    (
                      ($currentImgSize ne $self->{options}->listImgSize)
                      ||
                      ($currentDetails ne $self->{options}->details)
                      ||
                      ($currentGroupMovies != $self->{options}->groupMovies)
                    )
                 )
               )
            {
                $self->{parent}->setMoviesList($self->{options}->view,
                                               $self->{options}->columns);
            }

		    my $selPlugin = $self->{pluginOption}->child->get_label;
		    my $plugin = 'ask';
		    foreach (@{$self->{plugins}})
		    {
				last if $selPlugin eq $self->{parent}->{lang}->{OptionsAskPlugins};
                last if $selPlugin eq $self->{parent}->{lang}->{OptionsPluginsMulti};
                last if $selPlugin eq $self->{parent}->{lang}->{OptionsPluginsMultiAsk};
				$plugin = $_;
				last if $plugin eq $selPlugin;
		    }
		    $plugin = 'multi' if $selPlugin eq $self->{parent}->{lang}->{OptionsPluginsMulti};
		    $plugin = 'multiask' if $selPlugin eq $self->{parent}->{lang}->{OptionsPluginsMultiAsk};
		    $self->{options}->plugin($plugin);
       
            $self->{options}->askImport(($self->{askImport}->get_active) ? 1 : 0);
            $self->{options}->searchStop(($self->{searchStop}->get_active) ? 1 : 0);
            $self->{options}->alwaysOriginal(($self->{alwaysOriginal}->get_active) ? 1 : 0);

            $self->{options}->mailer($self->{mailerOption}->child->get_label);
            $self->{options}->smtp($self->{smtp}->get_text);
	        $self->{options}->from($self->{from}->get_text);
            
	        $self->{parent}->checkImagesDirectory(1);	       
            $self->{options}->historysize($self->{historysize}->get_value);
            $self->{parent}->{menubar}->{menuHistoryItem}->remove_submenu();
            $self->{parent}->{menubar}->{menuHistory} = Gtk2::Menu->new();
            $self->{parent}->{menubar}->addHistoryMenu();
            
            $self->{options}->details($self->{details});
            
	        $self->{options}->save;
	    }
    }

    sub browserBrowse
    {
        my ($widget, $self) = @_;
        
        $self->{browserDialog} = new GCFileChooserDialog($self->{parent}->{lang}->{OptionsBrowserTitle}, $self->{parent}, 'open')
            if ! $self->{browserDialog};
        
        $self->{browserDialog}->set_filename($self->{browser}->get_text);
		my $response = $self->{browserDialog}->run;
		if ($response eq 'ok')
		{
	            $self->{browser}->set_text($self->{browserDialog}->get_filename);
		}
        $self->{browserDialog}->hide;
    }

    sub playerBrowse
    {
        my ($widget, $self) = @_;

        $self->{playerDialog} = new GCFileChooserDialog($self->{parent}->{lang}->{OptionsPlayerTitle}, $self->{parent}, 'open')
            if ! $self->{playerDialog};

        $self->{playerDialog}->set_filename($self->{player}->get_text);
		my $response = $self->{playerDialog}->run;
		if ($response eq 'ok')
		{
	            $self->{player}->set_text($self->{playerDialog}->get_filename);
		}
        $self->{playerDialog}->hide;
    }

    sub imagesBrowse
    {
        my ($widget, $self) = @_;
  
        $self->{imagesDialog} = new GCFileChooserDialog($self->{parent}->{lang}->{OptionsImagesTitle}, $self->{parent}, 'select-folder')
            if ! $self->{imagesDialog};  
  
        $self->{imagesDialog}->set_filename($self->{images}->get_text);
        my $response = $self->{imagesDialog}->run;
		if ($response eq 'ok')
		{
	            $self->{images}->set_text($self->{imagesDialog}->get_filename);
		}
        $self->{imagesDialog}->hide;
    }

    sub changeDetails
    {
        my $self = shift;
        my $parent = $self->{parent};
        
        my @tmpOptionsArray = split m/\|/, $self->{details};
        for my $i (0..$#tmpOptionsArray)
        {
            $tmpOptionsArray[$i] = $parent->{lang}->{FieldsList}->{$tmpOptionsArray[$i]}
        }
        
        my $fieldsDialog = new GCDetailedOptionsDialog($self,
                                                       \@tmpOptionsArray);
        
        
        $fieldsDialog->show;
        $self->{details} = join('|', @{$self->{fields}});
        $fieldsDialog->destroy;
    }

    sub new
    {
        my ($proto, $parent, $options) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent->{lang}->{OptionsTitle},
                              $parent,
                              [qw/modal destroy-with-parent/],
                              @GCDialogs::okCancelButtons
                            );

        $self->set_has_separator(0);

        $options = $parent->{options} if !$options;
        $self->{options} = $options;
        
        $self->{parent} = $parent;
        $self->{lang} = $parent->{lang};
        $self->{tooltips} = $parent->{tooltips};

        my $tablePath = new Gtk2::Table(4, 3, 0);
        $tablePath->set_row_spacings(10);
        $tablePath->set_col_spacings(10);
        $tablePath->set_border_width(20);

        my $labelBrowser = new Gtk2::Label($parent->{lang}->{OptionsBrowser});
        $labelBrowser->set_alignment(0,0.5);
        $tablePath->attach($labelBrowser, 0, 1, 0, 1, 'fill', 'fill', 0, 0);
        $self->{browser} = new Gtk2::Entry;
        $self->{browser}->set_width_chars(40);
        $tablePath->attach($self->{browser}, 1, 2, 0, 1, 'expand', 'fill', 0, 0);
        my $browserButton = Gtk2::Button->new_from_stock('gtk-open');
        $browserButton->signal_connect('clicked', \&browserBrowse, $self);
        $tablePath->attach($browserButton, 2, 3, 0, 1, 'fill', 'fill', 0, 0);

        my $labelPlayer = new Gtk2::Label($parent->{lang}->{OptionsPlayer});
        $labelPlayer->set_alignment(0,0.5);
        $tablePath->attach($labelPlayer, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
        $self->{player} = new Gtk2::Entry;
        $self->{player}->set_width_chars(40);
        $tablePath->attach($self->{player}, 1, 2, 1, 2, 'expand', 'fill', 0, 0);
        my $playerButton = Gtk2::Button->new_from_stock('gtk-open');
        $playerButton->signal_connect('clicked', \&playerBrowse, $self);
        $tablePath->attach($playerButton, 2, 3, 1, 2, 'fill', 'fill', 0, 0);

        my $labelImages = new Gtk2::Label($parent->{lang}->{OptionsImages});
        $labelImages->set_alignment(0,0.5);
        $tablePath->attach($labelImages, 0, 1, 2, 3, 'fill', 'fill', 0, 0);
        $self->{images} = new Gtk2::Entry;
        $self->{images}->set_width_chars(40);
        $tablePath->attach($self->{images}, 1, 2, 2, 3, 'expand', 'fill', 0, 0);
        my $imagesButton = Gtk2::Button->new_from_stock('gtk-open');
        $imagesButton->signal_connect('clicked', \&imagesBrowse, $self);
        $tablePath->attach($imagesButton, 2, 3, 2, 3, 'fill', 'fill', 0, 0);
	
	    $self->{useRelativePaths} = new Gtk2::CheckButton($parent->{lang}->{OptionsUseRelativePaths});
        $self->{useRelativePaths}->set_active($options->useRelativePaths);
        $self->{useRelativePaths}->set_active(0) if (! $options->exists('useRelativePaths'));
        $tablePath->attach($self->{useRelativePaths}, 0, 3, 3, 4, 'fill', 'fill', 0, 0);

        my $tableMain = new Gtk2::Table(2,2,0);
        $tableMain->set_row_spacings(10);
        $tableMain->set_col_spacings(20);
        $tableMain->set_border_width(0);
        
        my $labelLang = new Gtk2::Label($parent->{lang}->{OptionsLang});
        $labelLang->set_alignment(0,0.5);
		$self->{langOption} = new Gtk2::OptionMenu();
		$self->{langMenu} = new Gtk2::Menu();
		my @langs = @GCLang::langsNames;
		foreach (@langs)
		{
		    my $item = Gtk2::MenuItem->new_with_label($_);
		    $self->{langMenu}->append($item);
		}
		$self->{langs} = \@langs;
		$self->{langOption}->set_menu($self->{langMenu}) ;
        $tableMain->attach($labelLang, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
        $tableMain->attach($self->{langOption}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);

        my $labelStyle = new Gtk2::Label($parent->{lang}->{OptionsStyle});
        $labelStyle->set_alignment(0,0.5);
		$self->{styleOption} = new Gtk2::OptionMenu();
		$self->{style} = new Gtk2::Menu();
		my @styles = sort keys %GCStyle::styles;
		foreach (@styles)
		{
		    my $item = Gtk2::MenuItem->new_with_label($_);
		    $self->{style}->append($item);
		}
		$self->{styles} = \@styles;
		$self->{styleOption}->set_menu($self->{style}) ;
        $tableMain->attach($labelStyle, 0, 1, 2, 3, 'fill', 'fill', 0, 0);
        $tableMain->attach($self->{styleOption}, 1, 2, 2, 3, 'fill', 'fill', 0, 0);
	
        my $labelWarning = new Gtk2::Label;
        $labelWarning->set_markup('<b>'.$parent->{lang}->{OptionsWarning}.'</b>');
        $labelWarning->set_use_underline(1);
		$labelWarning->set_line_wrap(1);
		$labelWarning->set_justify('center');
        
        my $vboxMain = new Gtk2::VBox(0,0);
        $vboxMain->set_border_width(20);
        $vboxMain->pack_start($tableMain,1,1,0);
        $vboxMain->pack_start(Gtk2::HSeparator->new,1,1,10);
        $vboxMain->pack_start($labelWarning,1,1,0);

        my $vboxConvenience = new Gtk2::VBox(0,0);
        $vboxConvenience->set_border_width(20);

        $self->{confirm} = new Gtk2::CheckButton($parent->{lang}->{OptionsRemoveConfirm});
        $self->{confirm}->set_active($options->confirm);
        $vboxConvenience->pack_start($self->{confirm}, 0, 0, 10);
        $self->{autosave} = new Gtk2::CheckButton($parent->{lang}->{OptionsAutoSave});
        $self->{autosave}->set_active($options->autosave);
        $vboxConvenience->pack_start($self->{autosave}, 0, 0, 10);
        $self->{splash} = new Gtk2::CheckButton($parent->{lang}->{OptionsSplash});
        $self->{splash}->set_active($options->splash);
        $self->{splash}->set_active(1) if (! $options->exists('splash'));
        $vboxConvenience->pack_start($self->{splash}, 0, 0, 10);
        #vboxConvenience->pack_start($self->{useRelativePaths}, 0, 0, 10);

        $vboxConvenience->pack_start(Gtk2::HSeparator->new, 1, 1, 0);
        my $tableHistory = new Gtk2::Table(0,1,0);
        $tableHistory->set_row_spacings(10);
        $tableHistory->set_col_spacings(10);
        $tableHistory->set_border_width(0);
        my $labelHistorysize = new Gtk2::Label($parent->{lang}->{OptionsHistory});
        my $adjHistory = Gtk2::Adjustment->new(0, 1, 20, 1, 1, 0) ;
        $self->{historysize} = Gtk2::SpinButton->new($adjHistory, 0, 0);
        $self->{buttonClearHistory} = new Gtk2::Button($parent->{lang}->{OptionsClearHistory});
        $self->{buttonClearHistory}->signal_connect('clicked' => sub {
            $self->{options}->history("");
        });
        $tableHistory->attach($labelHistorysize, 0, 1, 0, 1, 'fill', 'fill', 0, 10);
        $tableHistory->attach($self->{historysize}, 1, 2, 0, 1, 'fill', 'fill', 0, 10);
        $tableHistory->attach($self->{buttonClearHistory}, 2, 3, 0, 1, 'fill', 'fill', 0, 10);
        $vboxConvenience->pack_start($tableHistory,0,0,0);

        ##
        $vboxConvenience->pack_start(Gtk2::HSeparator->new, 1, 1, 0);
        my $labelFormats = new Gtk2::Label($parent->{lang}->{OptionsFormats});
        my $hboxFormats = new Gtk2::HBox(0,0);
		$self->{formats} = new Gtk2::Entry;
        $hboxFormats->pack_start($labelFormats,0,0,0);
        $hboxFormats->pack_start($self->{formats},1,1,5);
        $vboxConvenience->pack_start($hboxFormats, 1,1,5);
        ##

        my $tableDisplay1 = new Gtk2::Table(3, 4, 0);
        $tableDisplay1->set_row_spacings(10);
        $tableDisplay1->set_col_spacings(10);
        $tableDisplay1->set_border_width(0);
        
        my $labelView = new Gtk2::Label($parent->{lang}->{OptionsView});
        $labelView->set_alignment(0,0.5);
        $self->{viewOption} = new Gtk2::OptionMenu();
		$self->{view} = new Gtk2::Menu();
		my %views = %{$parent->{lang}->{OptionsViews}};
		foreach (0..(scalar(keys %views) - 1))
		{
		    my $item = Gtk2::MenuItem->new_with_label($views{$_});
		    $self->{view}->append($item);
		}
        $self->{viewOption}->set_menu($self->{view});
        $self->{viewOption}->set_alignment(0.0,0.0);
        $tableDisplay1->attach($labelView, 0, 1, 0, 1, 'fill', 'fill', 0, 0);
        $tableDisplay1->attach($self->{viewOption}, 1, 2, 0, 1, 'fill', 'fill', 0, 0);

        $self->{hboxImages} = new Gtk2::HBox(0,0);
        $self->{imagesOptionsDialog} = new GCImagesOptionsDialog($self);
        $self->{imagesButton} = new Gtk2::Button($parent->{lang}->{ImagesOptionsButton});
        $self->{imagesButton}->signal_connect('clicked', sub {
            $self->{imagesOptionsDialog}->show;
        });
        $self->{hboxImages}->pack_start($self->{imagesButton}, 0, 0, 5);

        $self->{hboxDetails} = new Gtk2::HBox(0,0);
        $self->{buttonDetails} = new Gtk2::Button($parent->{lang}->{ImagesOptionsButton});
        $self->{buttonDetails}->signal_connect('clicked' => sub {
            $self->changeDetails;
        });
        $self->{hboxDetails}->pack_start($self->{buttonDetails}, 0, 0, 5);

        $self->{viewOption}->signal_connect('changed' => sub {
                my %views = %{$self->{parent}->{lang}->{OptionsViews}};
                my $i = 0;
                $i++ while ($views{$i} ne $self->{viewOption}->child->get_label);
                $self->{hboxImages}->hide;
                $self->{hboxDetails}->hide;
                $self->{hboxImages}->show_all if ($i == 1);
                $self->{hboxDetails}->show_all if ($i == 2);
	        });
        $tableDisplay1->attach($self->{hboxImages}, 2, 4, 0, 1, 'fill', 'fill', 10, 0);
        $tableDisplay1->attach($self->{hboxDetails}, 2, 4, 0, 1, 'fill', 'fill', 10, 0);

        my $labelOrder = new Gtk2::Label($parent->{lang}->{OptionsOrder});
        $labelOrder->set_alignment(0,0.5);
		$self->{orderOption} = new Gtk2::OptionMenu();
		$self->{order} = new Gtk2::Menu();
        my @orders = ($parent->{lang}->{OptionsTabs},
                      $parent->{lang}->{OptionsAllInOne},
                      $parent->{lang}->{OptionsCompact},
                      $parent->{lang}->{OptionsReadOnly});
                      
        push @orders, $parent->{lang}->{OptionsWebSite}
            if (-d $ENV{GCF_SHARE_DIR}.'/panel/');
        foreach (@orders)
		{
		    my $item = Gtk2::MenuItem->new_with_label($_);
		    $self->{order}->append($item);
		}
		$self->{orders} = \@orders;
		$self->{orderOption}->set_menu($self->{order}) ;
        $tableDisplay1->attach($labelOrder, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
        $tableDisplay1->attach($self->{orderOption}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);

        my $labelToolbar = new Gtk2::Label($parent->{lang}->{OptionsToolbar});
        $labelToolbar->set_alignment(0,0.5);
        $self->{toolbarOption} = new Gtk2::OptionMenu();
		$self->{toolbar} = new Gtk2::Menu();
		my %toolbars = %{$parent->{lang}->{OptionsToolbars}};
		foreach (0..(scalar(keys %toolbars) - 1))
		{
		    my $item = Gtk2::MenuItem->new_with_label($toolbars{$_});
		    $self->{toolbar}->append($item);
		}
        $self->{toolbarOption}->set_menu($self->{toolbar}) ;
        $tableDisplay1->attach($labelToolbar, 0, 1, 2, 3, 'fill', 'fill', 0, 0);
        $tableDisplay1->attach($self->{toolbarOption}, 1, 2, 2, 3, 'fill', 'fill', 0, 0);

        my $labelToolbarPosition = new Gtk2::Label($parent->{lang}->{OptionsToolbarPosition});
        $labelToolbarPosition->set_alignment(0,0.5);
        $labelToolbarPosition->set_padding(10, 0);
        $self->{toolbarPositionOption} = new Gtk2::OptionMenu();
		$self->{toolbarPosition} = new Gtk2::Menu();
		my %toolbarPositions = %{$parent->{lang}->{OptionsToolbarPositions}};
		foreach (0..(scalar(keys %toolbarPositions) - 1))
		{
		    my $item = Gtk2::MenuItem->new_with_label($toolbarPositions{$_});
		    $self->{toolbarPosition}->append($item);
		}
        $self->{toolbarPositionOption}->set_menu($self->{toolbarPosition}) ;
        $tableDisplay1->attach($labelToolbarPosition, 2, 3, 2, 3, 'fill', 'fill', 0, 0);
        $tableDisplay1->attach($self->{toolbarPositionOption}, 3, 4, 2, 3, 'fill', 'fill', 0, 0);

        my $tableDisplay2 = new Gtk2::Table(2, 2, 0);
        $tableDisplay2->set_row_spacings(10);
        $tableDisplay2->set_col_spacings(10);
        $tableDisplay2->set_border_width(0);
        
        my $display2Fill = new Gtk2::Label(' ');
        $display2Fill->set_size_request(-1,1);
        $tableDisplay2->attach($display2Fill, 1, 2, 0, 1, 'expand', 'fill', 0, 0);
        $self->{status} = new Gtk2::CheckButton($parent->{lang}->{OptionsStatus});
        $self->{status}->set_active($options->status);
        $tableDisplay2->attach($self->{status}, 0, 2, 1, 2, 'fill', 'fill', 0, 0);
        $self->{transform} = new Gtk2::CheckButton($parent->{lang}->{OptionsTransform});
        $tableDisplay2->attach($self->{transform}, 0, 2, 2, 3, 'fill', 'fill', 0, 0);
        my $labelArticles = new Gtk2::Label('     '.$parent->{lang}->{OptionsArticles});
        $labelArticles->set_alignment(1,0.5);
        #$labelArticles->set_padding(10,0);
        $tableDisplay2->attach($labelArticles, 0, 1, 3, 4, 'fill', 'fill', 0, 0); 
		$self->{articles} = new Gtk2::Entry;
        $tableDisplay2->attach($self->{articles}, 1, 2, 3, 4, 'fill', 'fill', 5, 0);
        $options->articles('le,la,les,l,un,une,des,a,the') if (! $options->exists('articles'));

        my $vboxDisplay = new Gtk2::VBox(0,0);
        $vboxDisplay->set_border_width(20);
        $vboxDisplay->pack_start($tableDisplay1,1,1,0);
        $vboxDisplay->pack_start(Gtk2::HSeparator->new,0,0,10);
        $vboxDisplay->pack_start($tableDisplay2,1,1,0);
        
        my $tableInternet1 = new Gtk2::Table(4,3,0);
        $tableInternet1->set_row_spacings(10);
        $tableInternet1->set_col_spacings(10);
        $tableInternet1->set_border_width(0);
	
		my $pluginLabel = new Gtk2::Label($parent->{lang}->{OptionsPlugins});
        $pluginLabel->set_alignment(0,0.5);
		$self->{pluginOption} = new Gtk2::OptionMenu();
		$self->{plugin} = new Gtk2::Menu();
		my @plugins = ($parent->{lang}->{OptionsAskPlugins},
		  $parent->{lang}->{OptionsPluginsMultiAsk},
		  $parent->{lang}->{OptionsPluginsMulti},
		  @GCPlugins::pluginsNameArray);
		foreach (@plugins)
		{
		    my $item = Gtk2::MenuItem->new_with_label($_);
		    $self->{plugin}->append( $item) ;
		}
		$self->{plugins} = \@plugins;
		$self->{pluginOption}->set_menu($self->{plugin});		
		$self->{pluginList} = new Gtk2::Button($parent->{lang}->{OptionsPluginsList});
		$self->{pluginList}->signal_connect('clicked' => sub {
		  $parent->selectMultiPluginsList;
		});
		$self->{pluginOption}->signal_connect('changed' => sub {
	       if (($self->{pluginOption}->child->get_label eq $parent->{lang}->{OptionsPluginsMulti}) || ($self->{pluginOption}->child->get_label eq $parent->{lang}->{OptionsPluginsMultiAsk}))
           {
               $self->{pluginList}->show;
           }
           else
           {
               $self->{pluginList}->hide;
            }
	    });		
        $self->{askImport} = new Gtk2::CheckButton($parent->{lang}->{OptionsAskImport});
        #$self->{askImport}->set_size_request(1,-1);
        $tableInternet1->attach($pluginLabel, 0, 1, 0, 1, 'fill', 'fill', 0, 0); 
        $tableInternet1->attach($self->{pluginOption}, 1, 2, 0, 1, 'fill', 'fill', 0, 0); 
        $tableInternet1->attach($self->{pluginList}, 2, 3, 0, 1, 'fill', 'fill', 20, 0);
        $tableInternet1->attach($self->{askImport}, 0, 3, 1, 2, 'fill', 'fill', 0, 0); 

        $self->{alwaysOriginal} = new Gtk2::CheckButton($parent->{lang}->{OptionsAlwaysOriginal});
        #$self->{alwaysOriginal}->set_size_request(1,-1);
        $tableInternet1->attach($self->{alwaysOriginal}, 0, 3, 2, 3, 'fill', 'fill', 0, 0);

        $self->{searchStop} = new Gtk2::CheckButton($parent->{lang}->{OptionsSearchStop});
        #$self->{searchStop}->set_size_request(1,-1);
        $tableInternet1->attach($self->{searchStop}, 0, 3, 3, 4, 'fill', 'fill', 0, 0) if ($^O !~ /win32/i);
        
        
        my $tableInternet2 = new Gtk2::Table(3,3,0);
        $tableInternet2->set_row_spacings(10);
        $tableInternet2->set_col_spacings(10);
        $tableInternet2->set_border_width(0);
	
        $self->{proxycb} = new Gtk2::CheckButton($parent->{lang}->{OptionsProxy});
        $self->{proxycb}->set_active($options->proxy);
		$self->{proxyurl} = new Gtk2::Entry;
		$self->{proxyurl}->set_text($options->proxy);
        $self->{proxycb}->signal_connect('clicked' => sub {
	            if ($self->{proxycb}->get_active)
                {
                    $self->{proxyurl}->set_editable(1);
                }
                else
                {
                    $self->{proxyurl}->set_editable(0);
                }
	        });

        $tableInternet2->attach($self->{proxycb}, 0, 1, 0, 1, 'fill', 'fill', 0, 0);
        $tableInternet2->attach($self->{proxyurl}, 1, 2, 0, 1, 'fill', 'shrink', 0, 0);
        
        my $labelFrom = new Gtk2::Label($self->{parent}->{lang}->{OptionsFrom});
        $labelFrom->set_alignment(0,0.5);
        $self->{from} = new Gtk2::Entry;
        $tableInternet2->attach($labelFrom, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
        $tableInternet2->attach($self->{from}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);

        my $mailerLabel = new Gtk2::Label($parent->{lang}->{OptionsMailer});
        $mailerLabel->set_alignment(0,0.5);
		$self->{mailerOption} = new Gtk2::OptionMenu();
		$self->{mailer} = new Gtk2::Menu();
		my @mailers = ('Sendmail', 'SMTP');
		foreach (@mailers)
		{
		    my $item = Gtk2::MenuItem->new_with_label($_);
		    $self->{mailer}->append( $item) ;
		}
		$self->{mailers} = \@mailers;
		$self->{mailerOption}->set_menu($self->{mailer});
        $tableInternet2->attach($mailerLabel, 0, 1, 2, 3, 'fill', 'fill', 0, 0);
        $tableInternet2->attach($self->{mailerOption}, 1, 2, 2, 3, 'fill', 'fill', 0, 0);
        
        $self->{hboxSMTP} = new Gtk2::HBox(0,0);
        my $SMTPLabel = new Gtk2::Label($parent->{lang}->{OptionsSMTP});
        $self->{smtp} = new Gtk2::Entry;
		$self->{smtp}->set_text($options->smtp);
		$self->{smtp}->set_width_chars(20);
        $self->{hboxSMTP}->pack_start($SMTPLabel,0,0,0);
        $self->{hboxSMTP}->pack_start($self->{smtp},0,0,5);
        $tableInternet2->attach($self->{hboxSMTP}, 2, 3, 2, 3, 'fill', 'fill', 0, 0);

        $self->{mailerOption}->signal_connect('changed' => sub {
	            if ('SMTP' eq $self->{mailerOption}->child->get_label)
                {
                    $self->{hboxSMTP}->show;
                }
                else
                {
                    $self->{hboxSMTP}->hide;
                }
	        });
        
        my $vboxInternet = new Gtk2::VBox(0,0);
        $vboxInternet->set_border_width(20);
        $vboxInternet->pack_start($tableInternet1,0,0,0);
        $vboxInternet->pack_start(Gtk2::HSeparator->new,0,0,10);
        $vboxInternet->pack_start($tableInternet2,0,0,0);

        my $tabs = Gtk2::Notebook->new();
        $tabs->set_name('GCOptionsTabs');
        $tabs->set_tab_pos('left');
        $tabs->set_show_border(0);

        my ($mainButton, $displayButton, $pathButton, $internetButton, $conveniencesButton);

        $mainButton = GCImageBox->new_from_stock('gtk-home',
                                                  $parent->{lang}->{OptionsMain});
        $displayButton = GCImageBox->new_from_stock('gtk-select-color',
                                                     $parent->{lang}->{OptionsDisplay});
        $pathButton = GCImageBox->new_from_stock('gtk-directory',
                                                  $parent->{lang}->{OptionsPaths});
        $internetButton = GCImageBox->new_from_stock('gtk-network',
                                                      $parent->{lang}->{OptionsInternet});
        $conveniencesButton = GCImageBox->new_from_stock('gtk-properties',
                                                          $parent->{lang}->{OptionsConveniences});
        
        $tabs->append_page_menu($vboxMain, $mainButton, Gtk2::Label->new($parent->{lang}->{OptionsMain}));
        $tabs->append_page_menu($vboxDisplay, $displayButton, Gtk2::Label->new($parent->{lang}->{OptionsDisplay}));
        $tabs->append_page_menu($tablePath, $pathButton, Gtk2::Label->new($parent->{lang}->{OptionsPaths}));
        $tabs->append_page_menu($vboxInternet, $internetButton, Gtk2::Label->new($parent->{lang}->{OptionsInternet}));
        $tabs->append_page_menu($vboxConvenience, $conveniencesButton, Gtk2::Label->new($parent->{lang}->{OptionsConveniences}));

        $tabs->set_tab_label_packing ($vboxMain, 1, 0, 'start');
        $tabs->set_tab_label_packing ($vboxDisplay, 1, 0, 'start');
        $tabs->set_tab_label_packing ($tablePath, 1, 0, 'start');
        $tabs->set_tab_label_packing ($vboxInternet, 1, 0, 'start');
        $tabs->set_tab_label_packing ($vboxConvenience, 1, 0, 'start');

        $self->vbox->pack_start($tabs, 1, 1, 0);

        $self->{lang} = $parent->{lang};

        bless ($self, $class);
        return $self;
    }

}

{
    # Class used to let user select images options
    package GCImagesOptionsDialog;
    use base "Gtk2::Dialog";
    
    sub initValues
    {
        my $self = shift;
        
        $self->{columns}->set_value($self->{parent}->{columns});
        
        $self->{imgSize}->set_active($self->{parent}->{listImgSize});
        $self->{imgSizeOption}->remove_menu;
        $self->{imgSizeOption}->set_menu($self->{imgSize});
        
        my $idx = 0;
        foreach (@GCStyle::lists)
        {
            last if $_ eq $self->{parent}->{listImgSkin};
            $idx++;
        }
        $self->{style}->set_active($idx);
        $self->{optionStyle}->remove_menu;
        $self->{optionStyle}->set_menu($self->{style});
        
        $self->{listBgPicture}->set_active($self->{parent}->{listBgPicture});
        $self->activateColors(! $self->{parent}->{listBgPicture});
        $self->{mlbg} = $self->{parent}->{mlbg};
        $self->{mlfg} = $self->{parent}->{mlfg};
    }
    
    sub saveValues
    {
        my $self = shift;
        
        $self->{parent}->{columns} = $self->{columns}->get_value;
        
        my %sizes = %{$self->{parent}->{lang}->{ImagesOptionsSizeList}};
        foreach my $i (0..(scalar(keys %sizes) - 1))
        {
            $self->{parent}->{listImgSize} = $i;
            last if $sizes{$i} eq $self->{imgSizeOption}->child->get_label;
        }

        $self->{parent}->{listImgSkin} = $self->{optionStyle}->child->get_label;
        $self->{parent}->{mlbg} = $self->{mlbg};
        $self->{parent}->{mlfg} = $self->{mlfg};
        $self->{parent}->{listBgPicture} = (($self->{listBgPicture}->get_active) ? 1 : 0);
    }
    
    sub show
    {
        my $self = shift;

        $self->initValues;
        
        $self->show_all;
        my $code = $self->run;
        if ($code eq 'ok')
        {
            $self->saveValues;
        }
        $self->hide;
    }
    
    sub changeColor
    {
        my ($self, $type) = @_;
        
        my $dialog = new Gtk2::ColorSelectionDialog($self->{parent}->{lang}->{ImagesOptionsSelectColor});
        my $vboxPicture = new Gtk2::VBox(0,0);
        my @colors = split m/,/, $self->{'ml'.$type};
        my $previous = new Gtk2::Gdk::Color($colors[0], $colors[1], $colors[2]);
        $dialog->colorsel->set_current_color($previous) if $previous;
        my $response = $dialog->run;
        if ($response eq 'ok')
        {
            my $color = $dialog->colorsel->get_current_color;
            $self->{'ml'.$type} = join ',',$color->red, $color->green, $color->blue;
        }
        $dialog->destroy;
    }
    
    sub activateColors
    {
        my ($self, $value) = @_;
        
        $self->{labelStyle}->set_sensitive(!$value);
        $self->{optionStyle}->set_sensitive(!$value);
        $self->{labelBg}->set_sensitive($value);
        $self->{buttonBg}->set_sensitive($value);
        $self->{labelFg}->set_sensitive($value);
        $self->{buttonFg}->set_sensitive($value);
    }
    
    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        
        my $self  = $class->SUPER::new($parent->{lang}->{ImagesOptionsTitle},
                              $parent,
                              [qw/modal destroy-with-parent/],
                              @GCDialogs::okCancelButtons
                            );

        $self->{parent} = $parent;

        my $tableLayout = new Gtk2::Table(6,3);
        $tableLayout->set_col_spacings(10);
        $tableLayout->set_row_spacings(10);

        my $labelColumns = new Gtk2::Label($parent->{lang}->{OptionsColumns});
        $labelColumns->set_alignment(0,0.5);
        my  $adj = Gtk2::Adjustment->new(0, 1, 10, 1, 1, 0) ;
        $self->{columns} = Gtk2::SpinButton->new($adj, 0, 0);

        my $imgSizeLabel = new Gtk2::Label($parent->{lang}->{ImagesOptionsSizeLabel});
        $imgSizeLabel->set_alignment(0,0.5);
        $self->{imgSizeOption} = new Gtk2::OptionMenu();
		$self->{imgSize} = new Gtk2::Menu();
		my %imgSizes = %{$parent->{lang}->{ImagesOptionsSizeList}};
		foreach (0..(scalar(keys %imgSizes) - 1))
		{
		    my $item = Gtk2::MenuItem->new_with_label($imgSizes{$_});
		    $self->{imgSize}->append($item);
		}
        $self->{imgSizeOption}->set_menu($self->{imgSize}) ;

        $self->{listBgPicture} = new Gtk2::CheckButton($parent->{lang}->{ImagesOptionsBgPicture});
        $self->{listBgPicture}->signal_connect('clicked' => sub {
            $self->activateColors(! $self->{listBgPicture}->get_active);
        });
        
        $self->{labelStyle} = new Gtk2::Label($parent->{lang}->{OptionsStyle});
        $self->{labelStyle}->set_alignment(0,0.5);
		$self->{optionStyle} = new Gtk2::OptionMenu();
		$self->{style} = new Gtk2::Menu();
		foreach (@GCStyle::lists)
		{
		    my $item = Gtk2::MenuItem->new_with_label($_);
		    $self->{style}->append($item);
		}
		$self->{optionStyle}->set_menu($self->{style}) ;

       
        $self->{labelBg} = new Gtk2::Label($parent->{lang}->{ImagesOptionsBg});
        $self->{labelBg}->set_alignment(0,0.5);
        $self->{buttonBg} = new Gtk2::Button($parent->{lang}->{ImagesOptionsSelectColor});
        $parent->{tooltips}->set_tip($self->{buttonBg},
                                     $parent->{lang}->{ImagesOptionsBgTooltip});
        $self->{buttonBg}->signal_connect('clicked' => sub {
            $self->changeColor('bg');
        });
        
        $self->{labelFg} = new Gtk2::Label($parent->{lang}->{ImagesOptionsFg});
        $self->{labelFg}->set_alignment(0,0.5);
        $self->{buttonFg} = new Gtk2::Button($parent->{lang}->{ImagesOptionsSelectColor});
        $self->{buttonFg}->signal_connect('clicked' => sub {
            $self->changeColor('fg');
        });
        $parent->{tooltips}->set_tip($self->{buttonFg},
                                     $parent->{lang}->{ImagesOptionsFgTooltip});

        $tableLayout->attach($labelColumns, 0, 1, 0, 1, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{columns}, 1, 2, 0, 1, 'fill', 'fill', 0, 0);
        $tableLayout->attach($imgSizeLabel, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{imgSizeOption}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{listBgPicture}, 0, 2, 2, 3, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{labelStyle}, 1, 2, 3, 4, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{optionStyle}, 2, 3, 3, 4, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{labelBg}, 1, 2, 4, 5, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{buttonBg}, 2, 3, 4, 5, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{labelFg}, 1, 2, 5, 6, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{buttonFg}, 2, 3, 5, 6, 'fill', 'fill', 0, 0);

        $tableLayout->show_all;

        $self->vbox->pack_start($tableLayout,1,1,0);

        bless ($self, $class);
        return $self;
    }
}

{
    # Class used to let user select images options
    package GCDetailedOptionsDialog;
    use GCDialogs;
    use base "GCFieldsSelectionDialog";
    
    sub initValues
    {
        my $self = shift;
        
        $self->{imgSize}->set_active($self->{parent}->{listImgSize});
        $self->{imgSizeOption}->remove_menu;
        $self->{imgSizeOption}->set_menu($self->{imgSize});
        
        $self->{groupMovies}->set_active($self->{parent}->{groupMovies});
    }
    
    sub saveValues
    {
        my $self = shift;
        
        my %sizes = %{$self->{parent}->{lang}->{ImagesOptionsSizeList}};
        foreach my $i (0..(scalar(keys %sizes) - 1))
        {
            $self->{parent}->{listImgSize} = $i;
            last if $sizes{$i} eq $self->{imgSizeOption}->child->get_label;
        }

        $self->{parent}->{groupMovies} = (($self->{groupMovies}->get_active) ? 1 : 0);
    }
    
    sub show
    {
        my $self = shift;

        $self->initValues;
        
        my $code = $self->SUPER::show;
        if ($code eq 'ok')
        {
            $self->saveValues;
        }
        $self->hide;
    }
    
    sub new
    {
        my ($proto, $parent, $preList) = @_;
        my $class = ref($proto) || $proto;
        
        my $self  = $class->SUPER::new($parent,
                                       $parent->{lang}->{DetailedOptionsTitle},
                                       $preList);

        bless ($self, $class);

        my $tableLayout = new Gtk2::Table(2,2);
        $tableLayout->set_col_spacings(10);
        $tableLayout->set_row_spacings(10);

        $self->{groupMovies} = new Gtk2::CheckButton($parent->{lang}->{DetailedOptionsGroupMovies});

        my $imgSizeLabel = new Gtk2::Label($parent->{lang}->{DetailedOptionsImageSize});
        $imgSizeLabel->set_alignment(0,0.5);
        $self->{imgSizeOption} = new Gtk2::OptionMenu();
		$self->{imgSize} = new Gtk2::Menu();
		my %imgSizes = %{$parent->{lang}->{ImagesOptionsSizeList}};
		foreach (0..(scalar(keys %imgSizes) - 1))
		{
		    my $item = Gtk2::MenuItem->new_with_label($imgSizes{$_});
		    $self->{imgSize}->append($item);
		}
        $self->{imgSizeOption}->set_menu($self->{imgSize}) ;

        $tableLayout->attach($self->{groupMovies}, 0, 2, 0, 1, 'fill', 'fill', 0, 0);
        $tableLayout->attach($imgSizeLabel, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
        $tableLayout->attach($self->{imgSizeOption}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);

        $tableLayout->show_all;
        
        my $fieldsLabel = new Gtk2::Label($parent->{lang}->{DetailedOptionsFields});

        my $separator = new Gtk2::HSeparator;
        $self->vbox->pack_start($tableLayout,0,0,20);
        $self->vbox->pack_start($separator,0,0,0);
        $self->vbox->pack_start($fieldsLabel,0,0,10);

        $self->vbox->reorder_child($fieldsLabel,0);
        $self->vbox->reorder_child($separator,0);
        $self->vbox->reorder_child($tableLayout,0);

        return $self;
    }
}

{
    # Class used to let user select information to be displayed
    package GCDisplayOptionsDialog;
    use base "Gtk2::Dialog";
    
    sub show
    {
        my $self = shift;

        $self->initValues;
        
        $self->show_all;
        my $code = $self->run;
        if ($code eq 'ok')
        {
            $self->saveValues;
        }
        $self->hide;
    }
    
    sub new
    {
        my ($proto, $parent, $optionName) = @_;
        my $class = ref($proto) || $proto;
        
        my $self  = $class->SUPER::new($parent->{lang}->{DisplayOptionsTitle},
                              $parent,
                              [qw/modal destroy-with-parent/],
                              @GCDialogs::okCancelButtons
                            );
        
        $self->{show} = {};
        $self->{options} = $parent->{options};
        $self->{optionName} = $optionName;
        $self->{lang} = $parent->{lang};
        $self->{parent} = $parent;        
        
        $self->{fields} = [
            'search', 'actions',
            'seen', 'rating', 'format', 'number', 'place', 'video', 'collection', 'rank', 'trailer', 'audio', 'subt', 'comment',
            'orig', 'image', 'date', 'director', 'time', 'nat', 'age', 'type', 'actors', 'synopsis',
            'borrower', 'lendDate', 'mail', 'history',
            'id'
        ];
        
        bless($self, $class);
        
        $self->{options}->$optionName(('1|' x (scalar @{$self->{fields}})))
            if ! $self->{options}->exists($optionName);
            
        #Small hack for users migrating from versions < 6.0
        my $tmpOptions = $self->{options}->$optionName;
        if ((length $tmpOptions) < 50)
        {
            $tmpOptions =~ s/^(.{14})(.*)$/${1}1|1|1|1|$2/;
            $self->{options}->$optionName($tmpOptions);
        }
            
        $self->createComponents;
        $self->initValues;
        
        return $self;
    }
    
    sub initValues
    {
        my $self = shift;
        $self->{show} = {};
        
        my $optionName = $self->{optionName};
        my @values = split m/\|/, $self->{options}->$optionName;
        my $i = 0;
        foreach (@{$self->{fields}})
        {
            $self->{show}->{$_} = $values[$i];
            $self->{$_}->set_active($values[$i]);
            $i++;
        }
    }
    
    sub saveValues
    {
        my $self = shift;
        $self->{show} = {};
        
        my $optionName = $self->{optionName};
        my @values;
        foreach (@{$self->{fields}})
        {
            my $value = $self->{$_}->get_active ? 1 : 0;
            $self->{show}->{$_} = $value;
            push @values, $value;
        }
        $self->{options}->$optionName(join '|', @values);
        $self->{options}->save;
    }
    
    sub selectAll
    {
        my $self = shift;
        
        foreach (@{$self->{fields}})
        {
            $self->{$_}->set_active(1);
        }
    }
    
    sub initBox
    {
        my ($self, $box, @values) = @_;
        $box->set_border_width(2);
        foreach (@values)
        {
            #(my $name = $_) =~ s/(.)(.*)/\U$1\E$2/;
            #$self->{$_} = new Gtk2::CheckButton($self->{lang}->{'DisplayOptions'.$name});
            $self->{$_} = new Gtk2::CheckButton($self->{lang}->{Displayed}->{$_});
            $box->pack_start($self->{$_}, 0,0,5);
        }
    }
    
    sub createComponents
    {
        my $self = shift;
        
        my @fields = @{$self->{fields}};
        
        my $frameMain = new Gtk2::Frame();
        $frameMain->set_border_width(2);
        $frameMain->set_label_align(1.0, 0.0);
		$frameMain->set_shadow_type('none');
		my $vboxMain = new Gtk2::VBox(0,0);
        $self->initBox($vboxMain, @fields[0..1], @fields[28..28]);
        $frameMain->add($vboxMain);

        my $framePerso = new Gtk2::Frame();
        $framePerso->set_border_width(2);
        $framePerso->set_label_align(1.0, 0.0);
		$framePerso->set_shadow_type('none');
		my $vboxPerso = new Gtk2::VBox(0,0);
        $self->initBox($vboxPerso, @fields[2..13]);
        $framePerso->add($vboxPerso);

        my $frameInfo = new Gtk2::Frame();
        $frameInfo->set_label_align(1.0, 0.0);
		$frameInfo->set_shadow_type('none');
		$frameInfo->set_border_width(2);
        my $vboxInfo = new Gtk2::VBox(0, 0);
        $self->initBox($vboxInfo, @fields[14..23]);
        $frameInfo->add($vboxInfo);
       
        my $frameLend = new Gtk2::Frame();
		$frameLend->set_shadow_type('none');
        $frameLend->set_border_width(2);
        $frameLend->set_label_align(1.0, 0.0);
        my $vboxLend = new Gtk2::VBox(0,0);
        $self->initBox($vboxLend, @fields[24..27]);
        $frameLend->add($vboxLend);

		my $myTabs = Gtk2::Notebook->new();
		$myTabs->append_page($frameMain, $self->{lang}->{DisplayOptionsMain});
		$myTabs->append_page($frameInfo, $self->{lang}->{Displayed}->{info});  
		$myTabs->append_page($framePerso, $self->{lang}->{Displayed}->{perso});  
		$myTabs->append_page($frameLend, $self->{lang}->{Displayed}->{borrowings});
		
		my $hboxActions = new Gtk2::HBox(0,0);
		my $allButton = new Gtk2::Button($self->{lang}->{DisplayOptionsAll});
		$allButton->signal_connect( clicked => sub {
		  $self->selectAll;
		});
		$hboxActions->pack_end($allButton,1,0,20);
		
		$self->vbox->pack_start($myTabs, 1, 1, 2);
        $self->vbox->pack_start($hboxActions, 0, 0, 10);
    }
}

1;
