""" Python module for properties behind an oblique shock in perfect gases
    
An Oblique shock wave is inclined at an angle to the direction of upstream
flow. Supersonic flow passing through an oblique shock (wave angle = beta)
is deflected through an angle (delta).

Assumptions:
1) Perfect gas with constant specific heats and molecular weights
2) Adiabatic flow
3) Absence of body forces
4) Absence of external work
    
"""
"""
 * Copyright (C) 2006 Varun Hiremath.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * Authors: Varun Hiremath, Venkattraman A
 * Version 0.1
"""

from scipy import optimize
from math import *
from NormalShock import NormalShock

class ObliqueShock:
    def __init__(self, g = 1.4):
        """ g is the value of gamma (ratio of specific heats), default = 1.4
        """
        self.g = g
        self.NS = NormalShock(self.g)
    
    def get_Turn_Angle_from_Mx_and_Wave_Angle(self, Mx, beta):
        """Returns turn angle (degrees) from Mx and wave angle beta (degrees)""" 
        beta = beta * pi/180
        if Mx < 1:
            print 'The value of Mx should be greater than 1'
            return
        else :
            return atan(2/tan(beta)*(Mx**2 * sin(beta)**2 - 1)/(Mx**2 *(self.g + cos(2*beta)) + 2)) * 180/pi
        
    def get_Wave_Angle_from_Mx_and_Turn_Angle(self, Mx, theta, Strong=False):
        """Returns wave angle (degrees) from Mx and turn angle theta (degrees) for a
        weak shock. Pass True as the third argument for strong shock."""
        if Strong:
            return optimize.fsolve(lambda beta:self.get_Turn_Angle_from_Mx_and_Wave_Angle(Mx, beta) - theta, 80.0)
        else:
            return optimize.fsolve(lambda beta:self.get_Turn_Angle_from_Mx_and_Wave_Angle(Mx, beta) - theta, 10.0)

    def get_My_from_Mx_and_Turn_Angle(self, Mx, theta, Strong=False):
        """Returns My from Mx and turn angle theta (degrees) for a
        weak shock. Pass True as the third argument for strong shock."""
        beta = self.get_Wave_Angle_from_Mx_and_Turn_Angle(Mx, theta, Strong)
        beta = beta * pi/180
        theta = theta * pi/180
        M1n = Mx * sin(beta)
        M2n = self.NS.get_My_from_Mx(M1n)
        return  M2n / sin(beta - theta)

    def get_Py_by_Px_from_Mx_and_Turn_Angle(self, Mx, theta, Strong=False):
        """Returns py/px from Mx and turn angle theta (degrees) for a
        weak shock. Pass True as the third argument for strong shock."""
        beta = self.get_Wave_Angle_from_Mx_and_Turn_Angle(Mx , theta, Strong)
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_Py_by_Px_from_Mx(M1n)

    def get_Py_by_Px_from_Mx_and_Wave_Angle(self, Mx, beta):
        """Returns py/px from Mx and wave angle beta (degrees)"""
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_Py_by_Px_from_Mx(M1n)

    def get_Ty_by_Tx_from_Mx_and_Turn_Angle(self, Mx, theta, Strong=False):
        """Returns Ty/Tx from Mx and turn angle theta (degrees) for a
        weak shock. Pass True as the third argument for strong shock."""
        beta = self.get_Wave_Angle_from_Mx_and_Turn_Angle(Mx , theta, Strong)
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_Ty_by_Tx_from_Mx(M1n)

    def get_Ty_by_Tx_from_Mx_and_Wave_Angle(self, Mx, beta):
        """Returns Ty/Tx from Mx and wave angle equal to beta"""
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_Ty_by_Tx_from_Mx(M1n)

    def get_rhoy_by_rhox_from_Mx_and_Turn_Angle(self, Mx, theta, Strong=False):
        """Returns rhoy/rhox from Mx and turn angle equal to theta for a
        weak shock. Pass True as the third argument for strong shock."""
        beta = self.get_Wave_Angle_from_Mx_and_Turn_Angle(Mx , theta, Strong)
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_rhoy_by_rhox_from_Mx(M1n)

    def get_rhoy_by_rhox_from_Mx_and_Wave_Angle(self, Mx, beta):
        """Returns rhoy/rhox from Mx and wave angle equal to beta"""
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_rhoy_by_rhox_from_Mx(M1n)

    def get_Poy_by_Pox_from_Mx_and_Turn_Angle(self, Mx, theta, Strong=False):
        """Returns poy/pox from Mx and turn angle equal to theta for a
        weak shock. Pass True as the third argument for strong shock."""
        beta = self.get_Wave_Angle_from_Mx_and_Turn_Angle(Mx , theta, Strong)
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_Poy_by_Px_from_Mx(M1n)

    def get_Poy_by_Pox_from_Mx_and_Wave_Angle(self, Mx, beta):
        """Returns poy/pox from Mx and wave angle equal to beta"""
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_Poy_by_Pox_from_Mx(M1n)

    def get_Poy_by_Px_from_Mx_and_Turn_Angle(self, Mx, theta, Strong=False):
        """Returns poy/pox from Mx and turn angle equal to theta for a
        weak shock. Pass True as the third argument for strong shock."""
        beta = self.get_Wave_Angle_from_Mx_and_Turn_Angle(Mx , theta, Strong)
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_Poy_by_Px_from_Mx(M1n)

    def get_Poy_by_Px_from_Mx_and_Wave_Angle(self, Mx, beta):
        """Returns poy/px from Mx and wave angle equal to beta"""
        beta = beta * pi/180
        M1n = Mx * sin(beta)
        return self.NS.get_Poy_by_Px_from_Mx(M1n)
