/* sqlqueryedit.c
 *
 * Copyright (C) 1999 - 2001 Vivien Malerba
 * Copyright (C) 2001 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "sqlqueryedit.h"
#include "sqlquery.h"
#include "choicecombo.h"
#include "packedclist.h"

static void sql_query_edit_class_init (SqlQueryEditClass * class);
static void sql_query_edit_init (SqlQueryEdit * qe);
static void sql_query_edit_destroy (GtkObject * object);
static void sql_query_edit_post_init (SqlQueryEdit * qw);

/*
 *
 * Main functions for the object
 *
 */

/* get a pointer to the parents to be able to call their destructor */
static GtkObject *parent_class = NULL;

guint
sql_query_edit_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Query_Edit",
			sizeof (SqlQueryEdit),
			sizeof (SqlQueryEditClass),
			(GtkClassInitFunc) sql_query_edit_class_init,
			(GtkObjectInitFunc) sql_query_edit_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_vbox_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_query_edit_class_init (SqlQueryEditClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
	parent_class = gtk_type_class (gtk_vbox_get_type ());
	object_class->destroy = sql_query_edit_destroy;
}

static void
sql_query_edit_init (SqlQueryEdit * qe)
{
	qe->sel_table = NULL;	/* last selection froml SqlWidDbTree */
	qe->obj_add_dlg = NULL;	/* no dialog out right now */
	qe->depend_sel = -1;
	qe->h_modif_dlg = g_hash_table_new (NULL, NULL);	/* pointers ! */
}

/* main notebook page interface cb */
static void query_name_or_descr_changed_cb (GtkWidget * wid,
					    SqlQueryEdit * qe);
static void tree_table_sel_changed_cb (GtkWidget * wid, SqlMemTable * t,
				       SqlQueryEdit * qe);
static void button_tree_add_table_cb (GtkWidget * wid, SqlQueryEdit * qe);
static void button_tree_del_table_cb (GtkWidget * wid, SqlQueryEdit * qe);
static void objects_swapped_cb (PackedCList *clist, 
				gpointer o1, gpointer o2, SqlQueryEdit * qe);
static void object_clist_select_row_cb (GtkWidget * wid,
					gint row, gint column,
					GdkEventButton * event,
					SqlQueryEdit * qe);
static void object_clist_unselect_row_cb (GtkWidget * wid,
					  gint row, gint column,
					  GdkEventButton * event,
					  SqlQueryEdit * qe);
static void obj_add_button (GtkWidget * widget, SqlQueryEdit * qe);
static void object_edit_cb (GtkWidget * widget, SqlQueryEdit * qe);
static void obj_del_button (GtkWidget * widget, SqlQueryEdit * qe);
static void depend_clist_select_row_cb (GtkWidget * wid,
					gint row, gint column,
					GdkEventButton * event,
					SqlQueryEdit * qe);
/*
static void depend_update_one_entry (GtkWidget * button, SqlQueryEdit * qe);
static void refresh_depend (GtkWidget * bttn, SqlQueryEdit * qe);
static void use_all_depend (GtkWidget * bttn, SqlQueryEdit * qe);
static void use_none_depend (GtkWidget * bttn, SqlQueryEdit * qe);
*/

/* Joins page */ 
static void joins_help(GtkWidget *bttn, SqlQueryEdit *qe);
static void join_options_cb(GtkWidget *bttn, SqlQueryEdit *qe);
static void text_only_text_changed_cb (GtkWidget * widget, SqlQueryEdit * qe);
static void text_only_validate_cb (GtkWidget * widget, SqlQueryEdit * qe);
static void sql_query_edit_elt_created_cb (GtkObject * obj,
					   QueryElement * elt,
					   SqlQueryEdit * qe);
static void depend_add_to_list (GtkObject * obj, QueryLink * ql,
				SqlQueryEdit * qe);
static void where_manage_depend_ql (SqlQueryEdit * qe, QueryLink * ql);
static void sql_query_edit_changed_cb (GtkObject * obj, SqlQueryEdit * qe);
static void text_only_tbutton_cb (GtkWidget * widget, SqlQueryEdit * qe);
static void initial_where_fill (SqlQueryEdit * qe, GNode * nde,
				GtkCTreeNode * tnde);
static void condlist_tree_move_cb (CondList * cdlist, GtkCTreeNode * node,
				   GtkCTreeNode * new_parent,
				   GtkCTreeNode * new_sibling,
				   SqlQueryEdit * qe);
static void condlist_node_created_cb (CondList * cdlist, GtkCTreeNode * node,
				      SqlQueryEdit * qe);
static void condlist_node_dropped_cb (CondList * cdlist, GtkCTreeNode * node,
				      SqlQueryEdit * qe);
static void condlist_selection_changed_cb (CondList * cdlist,
					   SqlQueryEdit * qe);
static void cond_add_clicked_cb (GtkWidget * btn, SqlQueryEdit * qe);
static void cond_modif_clicked_cb (GtkWidget * btn, SqlQueryEdit * qe);
static void cond_del_clicked_cb (GtkWidget * btn, SqlQueryEdit * qe);
static GSList *radio_group_add_option(GSList *group, GtkWidget *bb,
				      JoinType jt,
				      SqlQueryEdit *qe)
{
	GtkWidget *button;
	
	button = gtk_radio_button_new_with_label(group, join_type_get_label(jt));
	gtk_signal_connect(GTK_OBJECT(button), "toggled",
			   GTK_SIGNAL_FUNC(join_options_cb), qe);
	gtk_box_pack_start(GTK_BOX(bb), button, FALSE, FALSE, 0);
	gtk_widget_show(button);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
	qe->join_options[jt] = button;
	
	return group;
}

static void
sql_query_edit_post_init (SqlQueryEdit * qe)
{
	GtkWidget *hb, *tree, *label, *vb, *button, *clist, *bb, *hp, *vp;
	GtkWidget *table, *sw, *te, *text, *rel;
	/*GtkWidget *arrow, *wid, *nb;*/
	GtkWidget *nb;
	GtkWidget *arrow, *wid; 
	GtkWidget *frame;
	GSList *group;

	gchar *titles[] =
		{ N_("Object type"), N_("Object Name"), N_("Object content"),
		N_("Print Name")
	};
	/*gchar *titles2[] = { N_("Used"), N_("Link From"), N_("Link To") };*/
	gchar *titles2[]={N_("Link From"), N_("Join Type"), N_("Link To")};
	gchar *wh_titles[] =
		{ N_("Logic / Operand"), N_("condition"), N_("Operand"),
	   N_("Comments") };

	gchar *relname;
	gint i;
	GSList *list;

	table = gtk_table_new (2, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (qe), table, FALSE, TRUE, 0);
	gtk_widget_show (table);

	/* Label of the query */
	label = gtk_label_new (_("Query name:"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  0, 0, GNOME_PAD, 0);
	gtk_widget_show (label);

	te = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (te), qe->q->name);
	gtk_signal_connect (GTK_OBJECT (te), "changed",
			    GTK_SIGNAL_FUNC (query_name_or_descr_changed_cb),
			    qe);
	gtk_table_attach_defaults (GTK_TABLE (table), te, 1, 2, 0, 1);
	gtk_widget_show (te);
	qe->name = te;

	/* description of the query */
	label = gtk_label_new (_("Query description:"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  0, 0, GNOME_PAD, 0);
	gtk_widget_show (label);

	te = gtk_entry_new ();
	if (qe->q->descr)
		gtk_entry_set_text (GTK_ENTRY (te), qe->q->descr);
	gtk_signal_connect (GTK_OBJECT (te), "changed",
			    GTK_SIGNAL_FUNC (query_name_or_descr_changed_cb),
			    qe);
	gtk_table_attach_defaults (GTK_TABLE (table), te, 1, 2, 1, 2);
	gtk_widget_show (te);
	qe->descr = te;

	/* Main VPaned */
	vp = gtk_vpaned_new ();
	gtk_box_pack_start (GTK_BOX (qe), vp, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vp), GNOME_PAD/2.);
	gtk_widget_show (vp);

	/* Top Hpaned */
	wid = gtk_vbox_new(FALSE, 0);
	gtk_paned_pack1 (GTK_PANED (vp), wid, TRUE, TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (wid), GNOME_PAD/2.);
	gtk_widget_show (wid);

	hp = gtk_hpaned_new ();
	gtk_box_pack_start (GTK_BOX (wid), hp, TRUE, TRUE, 0);
	gtk_widget_show (hp);

	/* SqlWidDbTree */
	hb = gtk_hbox_new (FALSE, 0);
	gtk_box_set_spacing (GTK_BOX (hb), GNOME_PAD/2.);
	gtk_container_set_border_width (GTK_CONTAINER (hb), GNOME_PAD/2.);
	gtk_paned_add1 (GTK_PANED (hp), hb);
	gtk_widget_show (hb);

	tree = sql_wid_db_tree_new (qe->q->conf->db);
	sql_wid_db_tree_set_mode (SQL_WID_DB_TREE (tree),
				  SQL_WID_DB_TREE_TABLES |
				  SQL_WID_DB_TREE_TABLES_SEL);
	gtk_signal_connect (GTK_OBJECT (tree), "table_selected",
			    GTK_SIGNAL_FUNC (tree_table_sel_changed_cb), qe);
	gtk_box_pack_start (GTK_BOX (hb), tree, TRUE, TRUE, 0);
	gtk_widget_show (tree);
	qe->tree = tree;

	bb = gtk_vbutton_box_new ();
	gtk_button_box_set_child_size (GTK_BUTTON_BOX (bb), 15, 15);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (hb), bb, FALSE, TRUE, 0);
	gtk_widget_show (bb);

	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (bb), button);
	gtk_widget_show (button);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (button_tree_add_table_cb), qe);

	arrow = gtk_arrow_new (GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);
	gtk_widget_show (arrow);

	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (bb), button);
	gtk_widget_show (button);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (button_tree_del_table_cb), qe);

	arrow = gtk_arrow_new (GTK_ARROW_LEFT, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);
	gtk_widget_show (arrow);


	/* SqlWidDbRel Widget */
	relname = g_strdup_printf ("Q%d", qe->q->id);
	rel = sql_wid_db_rel_new (qe->q->conf, relname);
	g_free (relname);
	gtk_paned_add2 (GTK_PANED (hp), rel);
	gtk_widget_show (rel);
	qe->relview = rel;
	gtk_container_set_border_width (GTK_CONTAINER (rel), GNOME_PAD/2.);

	/* notebook */
	wid = gtk_vbox_new(FALSE, 0);
	gtk_paned_pack2 (GTK_PANED (vp), wid, TRUE, TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (wid), GNOME_PAD/2.);
	gtk_widget_show (wid);

	nb = gtk_notebook_new ();
	gtk_box_pack_start (GTK_BOX (wid), nb, TRUE, TRUE, 0);
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (nb), GTK_POS_TOP);
	gtk_widget_show (nb);
	qe->notebook = nb;

	/* notebook Objects page */
	wid = gtk_label_new (_("Objects"));
	gtk_widget_show (wid);

	hb = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hb), GNOME_PAD);
	gtk_notebook_append_page (GTK_NOTEBOOK (qe->notebook), hb, wid);
	gtk_widget_show (hb);

	wid = packed_clist_new_with_titles(4, titles);
	gtk_box_pack_start (GTK_BOX (hb), wid, TRUE, TRUE, 0);
	gtk_widget_show(wid);
	qe->clist = wid;
	gtk_signal_connect (GTK_OBJECT (PACKED_CLIST (wid)->clist), "select_row",
			    GTK_SIGNAL_FUNC (object_clist_select_row_cb), qe);
	gtk_signal_connect (GTK_OBJECT (PACKED_CLIST (wid)->clist), "unselect_row",
			    GTK_SIGNAL_FUNC (object_clist_unselect_row_cb), qe);
	gtk_signal_connect (GTK_OBJECT (wid), "objects_swapped",
			    GTK_SIGNAL_FUNC (objects_swapped_cb), qe);

	bb = gtk_vbutton_box_new ();
	gtk_container_set_border_width (GTK_CONTAINER (bb), GNOME_PAD);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (hb), bb, FALSE, TRUE, 0);
	gtk_widget_show (bb);

	wid = gtk_button_new_with_label (_("Add object"));
	gtk_container_add (GTK_CONTAINER (bb), wid);
	gtk_widget_show (wid);
	qe->add_button = wid;
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (obj_add_button), qe);

	wid = gtk_button_new_with_label (_("Edit object"));
	gtk_container_add (GTK_CONTAINER (bb), wid);
	gtk_widget_set_sensitive (wid, FALSE);
	gtk_widget_show (wid);
	qe->edit_button = wid;
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (object_edit_cb), qe);

	wid = gtk_button_new_with_label (_("Remove object"));
	gtk_container_add (GTK_CONTAINER (bb), wid);
	gtk_widget_show (wid);
	qe->del_button = wid;
	gtk_widget_set_sensitive (qe->del_button, FALSE);
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (obj_del_button), qe);

	
	/* Joins page creation */
	/*label = gtk_label_new (_("Automatic dependencies"));*/
	label = gtk_label_new(_("Joins"));
	gtk_widget_show (label);
	
	/* Main horizontal box for the page, hb */
	hb = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hb), GNOME_PAD);
	gtk_notebook_append_page (GTK_NOTEBOOK (qe->notebook), hb, label);
	gtk_widget_show (hb);

	/* List box with available links, sw+wid */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_ALWAYS);
	gtk_box_pack_start (GTK_BOX (hb), sw, TRUE, TRUE, 0);
	gtk_widget_show (sw);

	wid = gtk_clist_new_with_titles (3, titles2);
	gtk_clist_set_selection_mode (GTK_CLIST (wid), GTK_SELECTION_SINGLE);
	gtk_clist_column_titles_passive (GTK_CLIST (wid));
	for (i = 0; i < 3; i++) {
		/*gtk_clist_set_column_title (GTK_CLIST (wid), i,_(titles2[i]));*/
		gtk_clist_set_column_auto_resize (GTK_CLIST (wid), i, TRUE);
	}
	gtk_container_add (GTK_CONTAINER (sw), wid);
	gtk_widget_show (wid);
	gtk_signal_connect (GTK_OBJECT (wid), "select_row",
			    GTK_SIGNAL_FUNC (depend_clist_select_row_cb), qe);
	qe->depend_clist = wid;

	/* Auxiliar vertical box on the right of the page, vb */
	/*vb = gtk_vbutton_box_new ();*/
	vb = gtk_vbox_new(FALSE, 10);

	gtk_container_set_border_width (GTK_CONTAINER (vb), GNOME_PAD);
	/*gtk_button_box_set_layout (GTK_BUTTON_BOX (vb), GTK_BUTTONBOX_SPREAD);*/
	gtk_box_pack_start (GTK_BOX (hb), vb, FALSE, FALSE, 0);
	gtk_widget_show (vb);


	 /* Join types frame = frame box + vertical box (frame + bb) */
	frame = gtk_frame_new("Join type");
	gtk_box_pack_start(GTK_BOX(vb), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);
	bb = gtk_vbox_new(TRUE, 0);
	gtk_container_add(GTK_CONTAINER(frame), bb);
	gtk_widget_show(bb);
	qe->frame_join_type = frame;
	
	/* Addition of radio buttons to the frame */
	group = radio_group_add_option(NULL, bb, JOIN_INNER, qe);
	group = radio_group_add_option(group, bb, JOIN_LEFT_OUTER, qe);
	group = radio_group_add_option(group, bb, JOIN_RIGHT_OUTER, qe);
	group = radio_group_add_option(group, bb, JOIN_FULL_OUTER, qe);
	group = radio_group_add_option(group, bb, JOIN_CROSS, qe);
	
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(qe->join_options[JOIN_INNER]), TRUE);
	qe->group_join_type = group;
	
	/* Button Help */
	button = gtk_button_new_with_label("Help");
	gtk_box_pack_start(GTK_BOX(vb), button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(joins_help), qe);
	gtk_widget_show (button);

	/*
	wid = gtk_check_button_new_with_label (_("Use Link"));
	gtk_container_add (GTK_CONTAINER (vb), wid);
	gtk_widget_show (wid);
	gtk_widget_set_sensitive (wid, FALSE);
	gtk_signal_connect (GTK_OBJECT (wid), "toggled",
			    GTK_SIGNAL_FUNC (depend_update_one_entry), qe);
	qe->use_button = wid;

	wid = gtk_button_new_with_label (_("Refresh"));
	gtk_container_add (GTK_CONTAINER (vb), wid);
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (refresh_depend), qe);
	gtk_widget_show (wid);

	wid = gtk_button_new_with_label (_("Use All"));
	gtk_container_add (GTK_CONTAINER (vb), wid);
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (use_all_depend), qe);
	gtk_widget_show (wid);

	wid = gtk_button_new_with_label (_("Use None"));
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (use_none_depend), qe);
	gtk_container_add (GTK_CONTAINER (vb), wid);
	gtk_widget_show (wid);
	*/

	/* notebook Where page */
	label = gtk_label_new (_("Where clauses"));
	gtk_widget_show (label);

	vb = gtk_vbox_new (FALSE, GNOME_PAD);
	gtk_notebook_append_page (GTK_NOTEBOOK (qe->notebook), vb, label);
	gtk_widget_show (vb);

	wid = cond_list_new (4, 0, wh_titles);
	gtk_box_pack_start (GTK_BOX (vb), wid, TRUE, TRUE, GNOME_PAD);
	gtk_widget_show (wid);
	qe->cdlist = wid;

	hb = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vb), hb, FALSE, FALSE, GNOME_PAD);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hb), GTK_BUTTONBOX_SPREAD);
	gtk_widget_show (hb);

	wid = gtk_button_new_with_label (_("Add condition"));
	gtk_container_add (GTK_CONTAINER (hb), wid);
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (cond_add_clicked_cb), qe);
	gtk_widget_show (wid);

	wid = gtk_button_new_with_label (_("Modify condition"));
	gtk_container_add (GTK_CONTAINER (hb), wid);
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (cond_modif_clicked_cb), qe);
	gtk_widget_show (wid);
	gtk_widget_set_sensitive (wid, FALSE);	/* no default selection */
	qe->wh_modif_button = wid;

	wid = gtk_button_new_with_label (_("Remove condition"));
	gtk_container_add (GTK_CONTAINER (hb), wid);
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (cond_del_clicked_cb), qe);
	gtk_widget_show (wid);
	gtk_widget_set_sensitive (wid, FALSE);	/* no default selection */
	qe->wh_del_button = wid;

	initial_where_fill (qe, NULL, NULL);	/* signals connection after initial filling */
	gtk_signal_connect (GTK_OBJECT (qe->cdlist), "tree_move",
			    GTK_SIGNAL_FUNC (condlist_tree_move_cb), qe);

	gtk_signal_connect (GTK_OBJECT (qe->cdlist), "node_created",
			    GTK_SIGNAL_FUNC (condlist_node_created_cb), qe);

	gtk_signal_connect (GTK_OBJECT (qe->cdlist), "node_dropped",
			    GTK_SIGNAL_FUNC (condlist_node_dropped_cb), qe);

	gtk_signal_connect (GTK_OBJECT (qe->cdlist), "selection_changed",
			    GTK_SIGNAL_FUNC (condlist_selection_changed_cb),
			    qe);

	/* notebook Options page */
	label = gtk_label_new (_("Options"));
	gtk_widget_show (label);

	hb = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hb), GNOME_PAD);
	gtk_box_set_spacing (GTK_BOX (hb), GNOME_PAD);
	gtk_notebook_append_page (GTK_NOTEBOOK (qe->notebook), hb, label);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hb), vb, TRUE, TRUE, 0);

	wid = gtk_label_new (_("Order by:"));
	gtk_box_pack_start (GTK_BOX (vb), wid, FALSE, TRUE, 0);

	wid = gtk_combo_new ();
	gtk_box_pack_start (GTK_BOX (vb), wid, FALSE, TRUE, 0);
	qe->ordersel = wid;

	bb = gtk_hbutton_box_new ();
	gtk_container_set_border_width (GTK_CONTAINER (bb), GNOME_PAD);
	gtk_button_box_set_child_size (GTK_BUTTON_BOX (bb), 15, 15);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (vb), bb, FALSE, TRUE, 0);

	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (bb), button);

	arrow = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);

	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (bb), button);

	arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);

	wid = packed_clist_new (1);
	gtk_box_pack_start (GTK_BOX (vb), wid, TRUE, TRUE, 0);
	qe->orderlist = wid;


	wid = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);


	vb = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hb), vb, TRUE, TRUE, 0);

	wid = gtk_label_new (_("Group by:"));
	gtk_box_pack_start (GTK_BOX (vb), wid, FALSE, TRUE, 0);

	wid = gtk_combo_new ();
	gtk_box_pack_start (GTK_BOX (vb), wid, FALSE, TRUE, 0);
	qe->groupsel = wid;

	bb = gtk_hbutton_box_new ();
	gtk_button_box_set_child_size (GTK_BUTTON_BOX (bb), 15, 15);
	gtk_container_set_border_width (GTK_CONTAINER (bb), GNOME_PAD);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (vb), bb, FALSE, TRUE, 0);

	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (bb), button);

	arrow = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);

	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (bb), button);

	arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);

	wid = packed_clist_new (1);
	gtk_box_pack_start (GTK_BOX (vb), wid, TRUE, TRUE, 0);
	qe->grouplist = wid;

	gtk_widget_show_all (hb);


	/* Text notebook page */
	label = gtk_label_new (_("Text SQL"));
	gtk_widget_show (label);

	hb = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER (hb), GNOME_PAD);
	gtk_notebook_append_page (GTK_NOTEBOOK (qe->notebook), hb, label);
	gtk_widget_show (hb);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_ALWAYS);
	gtk_box_pack_start (GTK_BOX (hb), sw, TRUE, TRUE, 0);
	gtk_widget_show (sw);

	text = gtk_text_new (NULL, NULL);
	gtk_text_set_editable (GTK_TEXT (text), TRUE);
	gtk_text_set_word_wrap (GTK_TEXT (text), TRUE);
	gtk_container_add (GTK_CONTAINER (sw), text);
	gtk_widget_show (text);
	qe->text = text;
	gtk_text_set_editable (GTK_TEXT (qe->text), FALSE);
	gtk_signal_connect (GTK_OBJECT (text), "changed",
			    GTK_SIGNAL_FUNC (text_only_text_changed_cb), qe);

	vb = gtk_vbox_new (FALSE, GNOME_PAD);
	gtk_box_pack_start (GTK_BOX (hb), vb, FALSE, FALSE, 0);
	gtk_widget_show (vb);


	wid = gtk_check_button_new_with_label (_("Text Query Only"));
	gtk_box_pack_start (GTK_BOX (vb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	qe->texttoggle = wid;

	wid = gtk_button_new_with_label (_("Validate"));
	gtk_box_pack_start (GTK_BOX (vb), wid, FALSE, FALSE, 0);
	gtk_widget_set_sensitive (wid, FALSE);
	gtk_widget_show (wid);
	qe->text_valid_btn = wid;
	gtk_signal_connect (GTK_OBJECT (wid), "clicked",
			    GTK_SIGNAL_FUNC (text_only_validate_cb), qe);

	/****************************************************/
	/* To take care of the state of the SqlQuery object */
	/****************************************************/

	/* if the query is text only */
	gtk_text_set_editable (GTK_TEXT (qe->text), qe->q->text_only);
	for (i = 0; i < 4; i++) {
		wid = gtk_notebook_get_nth_page (GTK_NOTEBOOK (qe->notebook),
						 i);
		gtk_widget_set_sensitive (wid, !qe->q->text_only);
	}
	if (qe->q->text_only) {
		gtk_notebook_set_page (GTK_NOTEBOOK (qe->notebook), 4);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
					      (qe->texttoggle), TRUE);
		gtk_text_set_point (GTK_TEXT (qe->text), 0);
		gtk_text_insert (GTK_TEXT (qe->text), NULL, NULL, NULL,
				 qe->q->text_sql, -1);
	}

	/* insertion of all the objects */
	list = qe->q->objects;
	while (list) {
		sql_query_edit_elt_created_cb (GTK_OBJECT (qe->q),
					       (QueryElement *) (list->data),
					       qe);
		list = g_slist_next (list);
	}

	/* putting the automatic links in the right CList */
	list = qe->q->depend_list;
	while (list) {
		depend_add_to_list (NULL, (QueryLink *) (list->data), qe);
		list = g_slist_next (list);
	}

	/* to tell the SqlQueryEdit to write the right SQL text in the text widget */
	sql_query_edit_changed_cb (GTK_OBJECT (qe->q), qe);

	/* late signal connections */
	gtk_signal_connect (GTK_OBJECT (qe->texttoggle), "toggled",
			    GTK_SIGNAL_FUNC (text_only_tbutton_cb), qe);
}


static void sql_query_edit_elt_dropped_cb (GtkObject * obj,
					   QueryElement * elt,
					   SqlQueryEdit * qe);
static void sql_query_edit_elt_modified_cb (GtkObject * obj,
					    QueryElement * elt,
					    SqlQueryEdit * qe);
static void wid_rel_show_table_cb (GtkObject * obj, SqlMemTable * t,
				   SqlQueryEdit * qe);
static void wid_rel_hide_table_cb (GtkObject * obj, SqlMemTable * t,
				   SqlQueryEdit * qe);
static void depend_remove_from_list (GtkObject * obj, QueryLink * ql,
				     SqlQueryEdit * qe);
static void where_cond_created_cb (SqlQuery * q, QueryWhereNode * new_node,
				   SqlQueryEdit * qe);
static void where_cond_dropped_cb (SqlQuery * q, QueryWhereNode * node,
				   SqlQueryEdit * qe);
static void where_cond_modified_cb (SqlQuery * q, QueryWhereNode * node,
				    SqlQueryEdit * qe);
static void where_cond_moved_cb (SqlQuery * q, QueryWhereNode * node,
				 SqlQueryEdit * qe);

GtkWidget *
sql_query_edit_new (SqlQuery * q)
{
	GtkWidget *obj;
	SqlQueryEdit *qe;

	obj = GTK_WIDGET (gtk_type_new (sql_query_edit_get_type ()));
	qe = SQL_QUERY_EDIT (obj);
	qe->q = q;
	sql_query_edit_post_init (qe);
	qe->edited_objects = NULL;
	qe->cond_add_dlg = NULL;
	/* connection to signals from SqlDb */
	gtk_signal_connect_while_alive (GTK_OBJECT (q), "elt_created",
					GTK_SIGNAL_FUNC
					(sql_query_edit_elt_created_cb), qe,
					GTK_OBJECT (qe));
	gtk_signal_connect_while_alive (GTK_OBJECT (q), "elt_dropped",
					GTK_SIGNAL_FUNC
					(sql_query_edit_elt_dropped_cb), qe,
					GTK_OBJECT (qe));
	gtk_signal_connect_while_alive (GTK_OBJECT (q), "elt_modified",
					GTK_SIGNAL_FUNC
					(sql_query_edit_elt_modified_cb), qe,
					GTK_OBJECT (qe));
	gtk_signal_connect_while_alive (GTK_OBJECT (q), "changed",
					GTK_SIGNAL_FUNC
					(sql_query_edit_changed_cb), qe,
					GTK_OBJECT (qe));

	gtk_signal_connect_while_alive (GTK_OBJECT (qe->relview),
					"table_shown",
					GTK_SIGNAL_FUNC
					(wid_rel_show_table_cb), qe,
					GTK_OBJECT (qe));

	gtk_signal_connect_while_alive (GTK_OBJECT (qe->relview),
					"table_hidden",
					GTK_SIGNAL_FUNC
					(wid_rel_hide_table_cb), qe,
					GTK_OBJECT (qe));

	gtk_signal_connect_while_alive (GTK_OBJECT (qe->q), "auto_link_added",
					GTK_SIGNAL_FUNC (depend_add_to_list),
					qe, GTK_OBJECT (qe));

	gtk_signal_connect_while_alive (GTK_OBJECT (qe->q),
					"auto_link_removed",
					GTK_SIGNAL_FUNC
					(depend_remove_from_list), qe,
					GTK_OBJECT (qe));

	gtk_signal_connect_while_alive (GTK_OBJECT (qe->q), "wh_cond_created",
					GTK_SIGNAL_FUNC
					(where_cond_created_cb), qe,
					GTK_OBJECT (qe));

	gtk_signal_connect_while_alive (GTK_OBJECT (qe->q), "wh_cond_dropped",
					GTK_SIGNAL_FUNC
					(where_cond_dropped_cb), qe,
					GTK_OBJECT (qe));

	gtk_signal_connect_while_alive (GTK_OBJECT (qe->q),
					"wh_cond_modified",
					GTK_SIGNAL_FUNC
					(where_cond_modified_cb), qe,
					GTK_OBJECT (qe));

	gtk_signal_connect_while_alive (GTK_OBJECT (qe->q), "wh_cond_moved",
					GTK_SIGNAL_FUNC (where_cond_moved_cb),
					qe, GTK_OBJECT (qe));

	return obj;
}


static void set_clist_entry_text (SqlQueryEdit * qe, QueryElement * elt,
				  gchar * row[]);
static void
sql_query_edit_elt_created_cb (GtkObject * obj,
			       QueryElement * elt, SqlQueryEdit * qe)
{
	gchar *row[4];
	gint i;

	/* clist update */
	set_clist_entry_text (qe, elt, row);
	i = gtk_clist_append (GTK_CLIST (PACKED_CLIST (qe->clist)->clist), row);
	gtk_clist_set_row_data (GTK_CLIST (PACKED_CLIST (qe->clist)->clist), i, elt);
	gtk_clist_select_row (GTK_CLIST (PACKED_CLIST (qe->clist)->clist), i, 0);
	packed_clist_refresh_arrows (PACKED_CLIST (qe->clist));
	for (i = 0; i < 4; i++)
		g_free (row[i]);

	/* Add condition dialog update */
	if (qe->cond_add_dlg) {
		GtkWidget *combo;
		combo = gtk_object_get_data (GTK_OBJECT (qe->cond_add_dlg),
					     "lcombo");
		if (combo && IS_CHOICE_COMBO (combo))
			choice_combo_set_content (CHOICE_COMBO (combo),
						  qe->q->objects,
						  GTK_STRUCT_OFFSET
						  (QueryElement, name));
		combo = gtk_object_get_data (GTK_OBJECT (qe->cond_add_dlg),
					     "rcombo");
		if (combo && IS_CHOICE_COMBO (combo))
			choice_combo_set_content (CHOICE_COMBO (combo),
						  qe->q->objects,
						  GTK_STRUCT_OFFSET
						  (QueryElement, name));
	}
}

static void
set_clist_entry_text (SqlQueryEdit * qe, QueryElement * elt, gchar * row[])
{
	/* WARNING: the allocated strings here will have to be freed */
	GSList *list;
	GString *gstr;
	gboolean first = TRUE;

	switch (elt->type) {
	case QUERY_ELT_FIELD:
		row[0] = g_strdup (_("Table field"));
		row[2] = g_strdup_printf ("%s.%s",
					  sql_db_find_table_from_field (qe->
									q->
									conf->
									db,
									SQL_MEM_FIELD
									(elt->
									 main))->
					  name,
					  SQL_MEM_FIELD (elt->main)->name);
		break;
	case QUERY_ELT_FUNCTION:
		row[0] = g_strdup (_("Function"));
		list = elt->args;
		gstr = g_string_new ("");
		g_string_sprintf (gstr, "%s (",
				  SQL_DATA_FUNCTION (elt->main)->sqlname);
		while (list) {
			if (first) {
				g_string_sprintfa (gstr, "%s",
						   ((QueryElement *) (list->
								      data))->
						   name);
				first = FALSE;
			}
			else
				g_string_sprintfa (gstr, ", %s",
						   ((QueryElement *) (list->
								      data))->
						   name);
			list = g_slist_next (list);
		}
		g_string_sprintfa (gstr, ")");
		row[2] = g_strdup (gstr->str);
		g_string_free (gstr, TRUE);
		break;
	case QUERY_ELT_AGGREGATE:
		row[0] = g_strdup (_("Aggregate"));
		row[2] = g_strdup_printf ("%s (%s)",
					  SQL_DATA_AGGREGATE (elt->main)->
					  sqlname,
					  ((QueryElement *) (elt->args->
							     data))->name);
		break;
	case QUERY_ELT_VALUE:
		row[0] = g_strdup (_("Value"));
		row[2] = g_strdup_printf ("%s",
					  ((QueryElementValue *) (elt->
								  main))->
					  default_val);
		break;
	}
	row[1] = g_strdup (elt->name);
	row[3] = g_strdup (elt->print_name);
}


static void object_edit_dialog_clicked_cb (GnomeDialog * dialog,
					   gint button_number,
					   SqlQueryEdit * qe);
static void
sql_query_edit_elt_dropped_cb (GtkObject * obj,
			       QueryElement * elt, SqlQueryEdit * qe)
{
	gint row;
	GSList *list;
	QueryElement *telt;

	/* in the CList */
	row = gtk_clist_find_row_from_data (GTK_CLIST (PACKED_CLIST (qe->clist)->clist), elt);
	if (row >= 0) {
		gtk_clist_remove (GTK_CLIST (PACKED_CLIST (qe->clist)->clist), row);
		packed_clist_refresh_arrows (PACKED_CLIST (qe->clist));
		if (!PACKED_CLIST (qe->clist)->actual_selection) {
			gtk_widget_set_sensitive (qe->del_button, FALSE);
			gtk_widget_set_sensitive (qe->edit_button, FALSE);
		}
	}

	/* ANY Edit dialog opened for that object is closed */
	list = qe->edited_objects;
	while (list) {
		telt = gtk_object_get_data (GTK_OBJECT (list->data),
					    "QueryElement");
		if (telt == elt) {
			object_edit_dialog_clicked_cb (GNOME_DIALOG
						       (list->data),
						       1 /* CANCEL BUTTON */ ,
						       qe);
			list = qe->edited_objects;
		}
		else
			list = g_slist_next (list);
	}

	/* Add condition dialog update */
	if (qe->cond_add_dlg) {
		GtkWidget *combo;
		combo = gtk_object_get_data (GTK_OBJECT (qe->cond_add_dlg),
					     "lcombo");
		if (combo && IS_CHOICE_COMBO (combo))
			choice_combo_set_content (CHOICE_COMBO (combo),
						  qe->q->objects,
						  GTK_STRUCT_OFFSET
						  (QueryElement, name));
		combo = gtk_object_get_data (GTK_OBJECT (qe->cond_add_dlg),
					     "rcombo");
		if (combo && IS_CHOICE_COMBO (combo))
			choice_combo_set_content (CHOICE_COMBO (combo),
						  qe->q->objects,
						  GTK_STRUCT_OFFSET
						  (QueryElement, name));
	}
}

static void
sql_query_edit_elt_modified_cb (GtkObject * obj,
				QueryElement * elt, SqlQueryEdit * qe)
{
	GSList *elts;
	gint row, i;
	gchar *rows[4];

	/* That element in the clist */
	row = gtk_clist_find_row_from_data (GTK_CLIST (PACKED_CLIST (qe->clist)->clist), elt);
	if (row >= 0) {
		gtk_clist_freeze (GTK_CLIST (PACKED_CLIST (qe->clist)->clist));
		set_clist_entry_text (qe, elt, rows);
		for (i = 0; i < 4; i++) {
			gtk_clist_set_text (GTK_CLIST (PACKED_CLIST (qe->clist)->clist), row, i,
					    rows[i]);
			g_free (rows[i]);
		}
		gtk_clist_thaw (GTK_CLIST (PACKED_CLIST (qe->clist)->clist));
	}

	/* other elements which might reference that elt.
	   NO recursive parsing because of what is displayed is only one leved
	   deep. */
	elts = qe->q->objects;
	while (elts) {
		if (g_slist_find (((QueryElement *) (elts->data))->args, elt)) {
			row = gtk_clist_find_row_from_data (GTK_CLIST (PACKED_CLIST (qe->clist)->clist),
							    (QueryElement
							     *) (elts->data));
			if (row) {
				gtk_clist_freeze (GTK_CLIST (PACKED_CLIST (qe->clist)->clist));
				set_clist_entry_text (qe,
						      (QueryElement *) (elts->
									data),
						      rows);
				for (i = 0; i < 4; i++) {
					gtk_clist_set_text (GTK_CLIST (PACKED_CLIST (qe->clist)->clist), row,
							    i, rows[i]);
					g_free (rows[i]);
				}
				gtk_clist_thaw (GTK_CLIST (PACKED_CLIST (qe->clist)->clist));
			}
		}
		elts = g_slist_next (elts);
	}
}

static void h_modif_dlg_foreach_clear (GNode * key,
				       GtkWidget * dlg, gpointer user_data);
static void
sql_query_edit_destroy (GtkObject * object)
{
	SqlQueryEdit *qe;
	GSList *list;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_QUERY_EDIT (object));

	qe = SQL_QUERY_EDIT (object);

	/* removing the ADD element dialog if opened */
	if (qe->obj_add_dlg)
		gtk_object_destroy (GTK_OBJECT (qe->obj_add_dlg));

	/* removing all the where modif dialogs (in h_modif_dlg) */
	g_hash_table_foreach (qe->h_modif_dlg,
			      (GHFunc) h_modif_dlg_foreach_clear, NULL);
	g_hash_table_destroy (qe->h_modif_dlg);

	/* removing the ADD condition dialog if opened */
	if (qe->cond_add_dlg)
		gtk_object_destroy (GTK_OBJECT (qe->cond_add_dlg));

	/* removing any Element Object modify dialog still opened */
	list = qe->edited_objects;
	while (list) {
		object_edit_dialog_clicked_cb (GNOME_DIALOG (list->data),
					       1 /* CANCEL BUTTON */ , qe);
		list = qe->edited_objects;
	}

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
h_modif_dlg_foreach_clear (GNode * key, GtkWidget * dlg, gpointer user_data)
{
	if (dlg)
		gtk_object_destroy (GTK_OBJECT (dlg));
}


/*
 *
 * Above the Notebook interface CBs
 *
 */
static void
query_name_or_descr_changed_cb (GtkWidget * wid, SqlQueryEdit * qe)
{
	sql_query_set_name (qe->q, gtk_entry_get_text (GTK_ENTRY (qe->name)),
			    gtk_entry_get_text (GTK_ENTRY (qe->descr)));
}

static void
tree_table_sel_changed_cb (GtkWidget * wid, SqlMemTable * t,
			   SqlQueryEdit * qe)
{
	qe->sel_table = t;
}

static void
button_tree_add_table_cb (GtkWidget * wid, SqlQueryEdit * qe)
{
	if (qe->sel_table)
		sql_wid_db_rel_show_table (SQL_WID_DB_REL (qe->relview),
					   qe->sel_table);
}

static void
button_tree_del_table_cb (GtkWidget * wid, SqlQueryEdit * qe)
{
	if (qe->sel_table)
		sql_wid_db_rel_hide_table (SQL_WID_DB_REL (qe->relview),
					   qe->sel_table);
}

static void sql_query_edit_auto_links_add_table (SqlQueryEdit * qe,
						 SqlMemTable * t);
static void
wid_rel_show_table_cb (GtkObject * obj, SqlMemTable * t, SqlQueryEdit * qe)
{
	sql_query_edit_auto_links_add_table (qe, t);
}

static void sql_query_edit_auto_links_del_table (SqlQueryEdit * qe,
						 SqlMemTable * t);
static void
wid_rel_hide_table_cb (GtkObject * obj, SqlMemTable * t, SqlQueryEdit * qe)
{
	sql_query_edit_auto_links_del_table (qe, t);
}

/*
 *
 * Notebook Objects Page CBs
 *
 */
static void 
objects_swapped_cb (PackedCList *clist, gpointer o1, gpointer o2, SqlQueryEdit * qe)
{
	sql_query_swap_objects (qe->q, (QueryElement*) o1, (QueryElement*) o2);	
}

static void
object_clist_select_row_cb (GtkWidget * wid,
			    gint row, gint column,
			    GdkEventButton * event, SqlQueryEdit * qe)
{
	gtk_widget_set_sensitive (GTK_WIDGET (qe->del_button), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (qe->edit_button), TRUE);
}

static void
object_clist_unselect_row_cb (GtkWidget * wid,
			      gint row, gint column,
			      GdkEventButton * event, SqlQueryEdit * qe)
{
	gtk_widget_set_sensitive (GTK_WIDGET (qe->del_button), FALSE);
	gtk_widget_set_sensitive (GTK_WIDGET (qe->edit_button), FALSE);
}

typedef enum
{
	OBJECT_FIELD,
	OBJECT_FUNC,
	OBJECT_AGG,
	OBJECT_VAL
}
ObjectType;

typedef struct
{
	GtkWidget *dlg;
	SqlQueryEdit *qe;

	ObjectType type;
	GtkWidget *name;
	GtkWidget *pname;	/* print name */

	GtkWidget *notebook;
	GtkWidget *page_field;
	GtkWidget *page_func;
	GtkWidget *page_agg;
	GtkWidget *page_val;

	GtkWidget *ok_button;
	gboolean ok_state[4];	/* for the 4 notebook pages */
}
ObjAddDlgInfo;

static void object_add_type_choice_toggle_cb (GtkWidget * wid,
					      GnomeDialog * dlg);
/* widgets to edit different types of objects: empty if qel==NULL
   and ready to edit qel if not NULL */
static GtkWidget *object_get_widget_field (SqlQueryEdit * qe,
					   ObjAddDlgInfo * dlginfo,
					   QueryElement * qel);
static GtkWidget *object_get_widget_func (SqlQueryEdit * qe,
					  ObjAddDlgInfo * dlginfo,
					  QueryElement * qel);
static GtkWidget *object_get_widget_agg (SqlQueryEdit * qe,
					 ObjAddDlgInfo * dlginfo,
					 QueryElement * qel);
static GtkWidget *object_get_widget_val (SqlQueryEdit * qe,
					 ObjAddDlgInfo * dlginfo,
					 QueryElement * qel);
static void object_add_button_dlg_clicked_cb (GnomeDialog * dialog,
					      gint button_number,
					      SqlQueryEdit * qe);
static void object_add_data_aggregates_add_or_del_cb (GtkObject * obj,
						      SqlDataAggregate * agg,
						      ObjAddDlgInfo *
						      dlginfo);
static void object_add_data_functions_add_or_del_cb (GtkObject * obj,
						     SqlDataFunction * func,
						     ObjAddDlgInfo * dlginfo);
static void object_add_data_datatypes_updated_cb (GtkObject * obj,
						  ObjAddDlgInfo * dlginfo);
static void
obj_add_button (GtkWidget * widget, SqlQueryEdit * qe)
{
	GtkWidget *dlg, *wid, *hb, *vb, *sw, *button, *table;
	GtkWidget *frame, *nb;
	gchar *str;
	GSList *grp;
	gint i;

	if (!qe->obj_add_dlg) {
		ObjAddDlgInfo *dlginfo;

		dlginfo = g_new0 (ObjAddDlgInfo, 1);
		dlginfo->qe = qe;

		for (i = 0; i < 4; i++)
			dlginfo->ok_state[i] = FALSE;

		str = g_strdup_printf (_
				       ("Adding an object to the query \"%s\""),
				       qe->q->name);
		dlg = gnome_dialog_new (str, GNOME_STOCK_BUTTON_OK,
					GNOME_STOCK_BUTTON_CANCEL, NULL);
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 0, FALSE);
		gtk_widget_set_usize (dlg, 400, 500);
		gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, FALSE);
		g_free (str);
		gtk_object_set_data (GTK_OBJECT (dlg), "info", dlginfo);
		dlginfo->dlg = dlg;

		qe->obj_add_dlg = dlg;
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 0, FALSE);

		/* top frame */
		hb = gtk_hbox_new (FALSE, GNOME_PAD);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
				    hb, FALSE, TRUE, 0);
		gtk_widget_show (hb);

		/* what kind of objet is it ? */
		vb = gtk_vbox_new (TRUE, 0);
		gtk_box_pack_start (GTK_BOX (hb), vb, FALSE, TRUE, 0);
		gtk_widget_show (vb);

		button = gtk_radio_button_new_with_label (NULL,
							  _
							  ("Add a table field"));
		gtk_box_pack_start (GTK_BOX (vb), button, TRUE, TRUE, 0);
		gtk_widget_show (button);
		grp = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
					      TRUE);
		gtk_signal_connect (GTK_OBJECT (button), "toggled",
				    GTK_SIGNAL_FUNC
				    (object_add_type_choice_toggle_cb), dlg);
		gtk_object_set_data (GTK_OBJECT (button), "order",
				     GINT_TO_POINTER (0));

		button = gtk_radio_button_new_with_label (grp,
							  _
							  ("Add a function"));
		grp = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
		gtk_box_pack_start (GTK_BOX (vb), button, TRUE, TRUE, 0);
		gtk_widget_show (button);
		gtk_signal_connect (GTK_OBJECT (button), "toggled",
				    GTK_SIGNAL_FUNC
				    (object_add_type_choice_toggle_cb), dlg);
		gtk_object_set_data (GTK_OBJECT (button), "order",
				     GINT_TO_POINTER (1));

		button = gtk_radio_button_new_with_label (grp,
							  _
							  ("Add an aggregate"));
		grp = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
		gtk_box_pack_start (GTK_BOX (vb), button, TRUE, TRUE, 0);
		gtk_widget_show (button);
		gtk_signal_connect (GTK_OBJECT (button), "toggled",
				    GTK_SIGNAL_FUNC
				    (object_add_type_choice_toggle_cb), dlg);
		gtk_object_set_data (GTK_OBJECT (button), "order",
				     GINT_TO_POINTER (2));

		button = gtk_radio_button_new_with_label (grp,
							  _("Add a value"));
		grp = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
		gtk_box_pack_start (GTK_BOX (vb), button, TRUE, TRUE, 0);
		gtk_widget_show (button);
		gtk_signal_connect (GTK_OBJECT (button), "toggled",
				    GTK_SIGNAL_FUNC
				    (object_add_type_choice_toggle_cb), dlg);
		gtk_object_set_data (GTK_OBJECT (button), "order",
				     GINT_TO_POINTER (3));

		/* object informations */
		table = gtk_table_new (2, 2, FALSE);
		gtk_container_set_border_width (GTK_CONTAINER (table),
						GNOME_PAD);
		gtk_box_pack_start (GTK_BOX (hb), table, FALSE, FALSE,
				    GNOME_PAD);
		gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);
		gtk_widget_show (table);

		wid = gtk_label_new (_("Object Name:"));
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 0,
					   1);
		gtk_widget_show (wid);

		wid = gtk_entry_new ();
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0,
					   1);
		gtk_widget_show (wid);
		dlginfo->name = wid;

		wid = gtk_label_new (_("Print Name:"));
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 1,
					   2);
		gtk_widget_show (wid);

		wid = gtk_entry_new ();
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 1,
					   2);
		gtk_widget_show (wid);
		dlginfo->pname = wid;

		/* lower part of the DLG in a notebook without any tabs */
		frame = gtk_frame_new (_("Object specifics..."));
		gtk_frame_set_shadow_type (GTK_FRAME (frame),
					   GTK_SHADOW_ETCHED_OUT);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox), frame,
				    TRUE, TRUE, 0);
		gtk_widget_show (frame);
		gtk_container_set_border_width (GTK_CONTAINER (frame),
						GNOME_PAD / 2.);

		nb = gtk_notebook_new ();
		gtk_container_add (GTK_CONTAINER (frame), nb);
		gtk_container_set_border_width (GTK_CONTAINER (nb),
						GNOME_PAD / 2.);
		gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
		gtk_widget_show (nb);
		dlginfo->notebook = nb;

		/* for the field types */
		wid = object_get_widget_field (qe, dlginfo, NULL);
		gtk_notebook_append_page (GTK_NOTEBOOK (nb), wid, NULL);
		gtk_widget_show (wid);

		/* for the functions */
		wid = object_get_widget_func (qe, dlginfo, NULL);
		gtk_notebook_append_page (GTK_NOTEBOOK (nb), wid, NULL);
		gtk_widget_show (wid);

		/* for the aggregates */
		wid = object_get_widget_agg (qe, dlginfo, NULL);
		gtk_notebook_append_page (GTK_NOTEBOOK (nb), wid, NULL);
		gtk_widget_show (wid);

		/* for the values */
		wid = object_get_widget_val (qe, dlginfo, NULL);
		gtk_notebook_append_page (GTK_NOTEBOOK (nb), wid, NULL);
		gtk_widget_show (wid);

		/* DLG signal */
		gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
				    GTK_SIGNAL_FUNC
				    (object_add_button_dlg_clicked_cb), qe);

		/* connection to signals from SqlAccess */
		gtk_signal_connect_while_alive (GTK_OBJECT (qe->q->conf->srv),
						"data_function_added",
						GTK_SIGNAL_FUNC
						(object_add_data_functions_add_or_del_cb),
						dlginfo, GTK_OBJECT (dlg));
		gtk_signal_connect_while_alive (GTK_OBJECT (qe->q->conf->srv),
						"data_function_removed",
						GTK_SIGNAL_FUNC
						(object_add_data_functions_add_or_del_cb),
						dlginfo, GTK_OBJECT (dlg));
		gtk_signal_connect_while_alive (GTK_OBJECT (qe->q->conf->srv),
						"data_aggregate_added",
						GTK_SIGNAL_FUNC
						(object_add_data_aggregates_add_or_del_cb),
						dlginfo, GTK_OBJECT (dlg));
		gtk_signal_connect_while_alive (GTK_OBJECT (qe->q->conf->srv),
						"data_aggregate_removed",
						GTK_SIGNAL_FUNC
						(object_add_data_aggregates_add_or_del_cb),
						dlginfo, GTK_OBJECT (dlg));
		gtk_signal_connect_while_alive (GTK_OBJECT (qe->q->conf->srv),
						"data_types_updated",
						GTK_SIGNAL_FUNC
						(object_add_data_datatypes_updated_cb),
						dlginfo, GTK_OBJECT (dlg));
		gtk_widget_show (dlg);

		gtk_widget_set_sensitive (GTK_WIDGET (qe->add_button), FALSE);
	}
}

static void
object_add_type_choice_toggle_cb (GtkWidget * wid, GnomeDialog * dlg)
{
	gint tab;
	ObjAddDlgInfo *dlginfo;

	dlginfo = gtk_object_get_data (GTK_OBJECT (dlg), "info");

	if (GTK_TOGGLE_BUTTON (wid)->active) {
		tab = GPOINTER_TO_INT (gtk_object_get_data
				       (GTK_OBJECT (wid), "order"));
		gtk_notebook_set_page (GTK_NOTEBOOK (dlginfo->notebook), tab);
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlginfo->dlg), 0,
					    dlginfo->ok_state[tab]);
		dlginfo->type = (ObjectType) tab;
	}
}

/*
 * Adding a Field object 
 */
static void object_add_field_sel_cb (GtkWidget * widget,
				     SqlMemTable * t, SqlMemField * f,
				     GtkWidget * wid);
static GtkWidget *
object_get_widget_field (SqlQueryEdit * qe, ObjAddDlgInfo * dlginfo,
			 QueryElement * qel)
{
	GtkWidget *wid;

	wid = sql_wid_db_tree_new (qe->q->conf->db);
	gtk_widget_set_usize (GTK_WIDGET (wid), 200, 100);
	sql_wid_db_tree_set_mode (SQL_WID_DB_TREE (wid),
				  SQL_WID_DB_TREE_FIELDS |
				  SQL_WID_DB_TREE_FIELDS_SEL);
	gtk_signal_connect (GTK_OBJECT (wid), "field_selected",
			    GTK_SIGNAL_FUNC (object_add_field_sel_cb), wid);

	gtk_object_set_data (GTK_OBJECT (wid), "info", dlginfo);
	dlginfo->page_field = wid;

	return wid;
}

static void
object_add_field_sel_cb (GtkWidget * widget,
			 SqlMemTable * t, SqlMemField * f, GtkWidget * wid)
{
	/* wid is the widget used to edit the field object */
	gchar *str;
	ObjAddDlgInfo *dlginfo;
	GtkWidget *entry;

	dlginfo = gtk_object_get_data (GTK_OBJECT (wid), "info");
	gtk_object_set_data (GTK_OBJECT (wid), "field_selected", f);

	dlginfo->ok_state[0] = f ? TRUE : FALSE;

	/* name and print name if not already selected */
	if (f) {
		entry = dlginfo->name;
		if (*gtk_entry_get_text (GTK_ENTRY (entry)) == '\0')
			gtk_entry_set_text (GTK_ENTRY (entry), f->name);
		entry = dlginfo->pname;
		if (*gtk_entry_get_text (GTK_ENTRY (entry)) == '\0') {
			str = g_strdup_printf ("%s_%s", t->name, f->name);
			gtk_entry_set_text (GTK_ENTRY (entry), str);
			g_free (str);
		}
	}

	gnome_dialog_set_sensitive (GNOME_DIALOG (dlginfo->dlg), 0,
				    dlginfo->ok_state[0]);
}

/*
 * Adding a function object
 */
static void object_add_user_fns_toggled_cb (GtkWidget * tbutton,
					    GtkWidget * wid);
static void object_add_system_fns_toggled_cb (GtkWidget * tbutton,
					      GtkWidget * wid);
static void object_add_func_changed_cb (GtkWidget * choice,
					gpointer listdata, GtkWidget * wid);
static void object_add_set_function_list (GtkWidget * wid);
static GtkWidget *
object_get_widget_func (SqlQueryEdit * qe, ObjAddDlgInfo * dlginfo,
			QueryElement * qel)
{
	GtkWidget *wid, *sw, *vb, *hb, *button, *table;
	gint i;
	GtkWidget **args_tab;
	gchar *str;

	args_tab = g_new0 (GtkWidget *, 16);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_object_set_data (GTK_OBJECT (sw), "args_tab", args_tab);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw), vb);

	hb = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_box_pack_start (GTK_BOX (vb), hb, FALSE, FALSE, 0);

	button = gtk_check_button_new_with_label (_
						  ("User defined functions"));
	gtk_box_pack_start (GTK_BOX (hb), button, FALSE, FALSE, 0);
	gtk_widget_show (button);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);	/*default */
	gtk_signal_connect (GTK_OBJECT (button), "toggled",
			    GTK_SIGNAL_FUNC (object_add_user_fns_toggled_cb),
			    sw);
	gtk_object_set_data (GTK_OBJECT (sw), "func_choice_user_bool",
			     GINT_TO_POINTER (1));

	button = gtk_check_button_new_with_label (_("System functions"));
	gtk_box_pack_start (GTK_BOX (hb), button, FALSE, FALSE, 0);
	gtk_widget_show (button);
	gtk_signal_connect (GTK_OBJECT (button), "toggled",
			    GTK_SIGNAL_FUNC
			    (object_add_system_fns_toggled_cb), sw);
	gtk_object_set_data (GTK_OBJECT (sw), "func_choice_sys_bool",
			     GINT_TO_POINTER (0));

	table = gtk_table_new (9, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (vb), table, FALSE, FALSE, GNOME_PAD);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);

	wid = gtk_label_new (_("Function:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 0, 1);
	gtk_widget_show (wid);

	wid = choice_combo_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
	gtk_widget_show (wid);
	gtk_object_set_data (GTK_OBJECT (sw), "func_choice_combo", wid);
	gtk_signal_connect (GTK_OBJECT (wid), "selection_changed",
			    GTK_SIGNAL_FUNC (object_add_func_changed_cb), sw);

	for (i = 0; i < 8; i++) {
		str = g_strdup_printf (_("Argument #%d:"), i + 1);
		wid = gtk_label_new (str);
		g_free (str);
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1,
					   1 + i, 2 + i);
		args_tab[2 * i] = wid;
		gtk_widget_show (wid);

		wid = choice_combo_new ();
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2,
					   1 + i, 2 + i);
		args_tab[2 * i + 1] = wid;
		gtk_widget_show (wid);
	}

	gtk_widget_show_all (vb);
	gtk_object_set_data (GTK_OBJECT (sw), "info", dlginfo);
	dlginfo->page_func = sw;
	object_add_set_function_list (sw);

	return sw;
}

static void
object_add_user_fns_toggled_cb (GtkWidget * tbutton, GtkWidget * wid)
{
	if (GTK_TOGGLE_BUTTON (tbutton)->active)
		gtk_object_set_data (GTK_OBJECT (wid),
				     "func_choice_user_bool",
				     GINT_TO_POINTER (1));
	else
		gtk_object_set_data (GTK_OBJECT (wid),
				     "func_choice_user_bool", NULL);
	object_add_set_function_list (wid);
}


static void
object_add_system_fns_toggled_cb (GtkWidget * tbutton, GtkWidget * wid)
{
	if (GTK_TOGGLE_BUTTON (tbutton)->active)
		gtk_object_set_data (GTK_OBJECT (wid), "func_choice_sys_bool",
				     GINT_TO_POINTER (1));
	else
		gtk_object_set_data (GTK_OBJECT (wid), "func_choice_sys_bool",
				     NULL);
	object_add_set_function_list (wid);
}

static void
object_add_func_changed_cb (GtkWidget * choice,
			    gpointer listdata, GtkWidget * wid)
{
	SqlDataFunction *f = NULL;
	gint i, limit;
	GtkWidget **args_tab;
	GSList *nlist, *elts;
	gboolean activate_ok = TRUE;
	ObjAddDlgInfo *dlginfo;

	args_tab =
		(GtkWidget **) gtk_object_get_data (GTK_OBJECT (wid),
						    "args_tab");
	dlginfo = gtk_object_get_data (GTK_OBJECT (wid), "info");

	/* showing / hiding of arguments for that function */
	if (listdata) {
		f = SQL_DATA_FUNCTION (listdata);
		limit = g_slist_length (f->args);
	}
	else {
		limit = 0;
		activate_ok = FALSE;
	}

	for (i = 0; i < limit; i++) {
		gtk_widget_set_sensitive (args_tab[2 * i], TRUE);
		gtk_widget_set_sensitive (args_tab[2 * i + 1], TRUE);
		/* possible choices for that argument */
		elts = dlginfo->qe->q->objects;
		nlist = NULL;
		while (elts) {
			SqlDataType *type;

			switch (((QueryElement *) (elts->data))->type) {
			case QUERY_ELT_FIELD:	/* Table Field */
				type = SQL_MEM_FIELD (((QueryElement
							*) (elts->data))->
						      main)->type;
				if (type == (g_slist_nth (f->args, i))->data) {
					nlist = g_slist_append (nlist,
								elts->data);
				}
				break;
			case QUERY_ELT_FUNCTION:	/* Function */
				type = SQL_DATA_FUNCTION (((QueryElement
							    *) (elts->data))->
							  main)->result_type;
				if (type == (g_slist_nth (f->args, i))->data) {
					nlist = g_slist_append (nlist,
								elts->data);
				}
				break;
			case QUERY_ELT_AGGREGATE:
				break;
			case QUERY_ELT_VALUE:
				if (((QueryElementValue *) (((QueryElement *)
							     (elts->data))->
							    main))->
				    data_type ==
				    (g_slist_nth (f->args, i))->data)
					nlist = g_slist_append (nlist,
								elts->data);
				break;
			}
			elts = g_slist_next (elts);
		}
		choice_combo_set_content (CHOICE_COMBO (args_tab[2 * i + 1]),
					  nlist,
					  GTK_STRUCT_OFFSET (QueryElement,
							     name));
		if (!nlist)
			activate_ok = FALSE;
		g_slist_free (nlist);
	}

	for (i = limit; i < 8; i++) {
		gtk_widget_set_sensitive (args_tab[2 * i], FALSE);
		gtk_widget_set_sensitive (args_tab[2 * i + 1], FALSE);
		choice_combo_set_content (CHOICE_COMBO (args_tab[2 * i + 1]),
					  NULL, 0);
	}

	dlginfo->ok_state[1] = activate_ok;
	gnome_dialog_set_sensitive (GNOME_DIALOG (dlginfo->dlg), 0,
				    dlginfo->ok_state[1]);
}

static void
object_add_set_function_list (GtkWidget * wid)
{
	gboolean sys, user;
	GSList *list, *lptr;
	GtkWidget *combo;
	ObjAddDlgInfo *dlginfo;

	dlginfo = gtk_object_get_data (GTK_OBJECT (wid), "info");

	combo = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (wid),
						 "func_choice_combo"));
	sys = gtk_object_get_data (GTK_OBJECT (wid),
				   "func_choice_sys_bool") ? TRUE : FALSE;
	user = gtk_object_get_data (GTK_OBJECT (wid),
				    "func_choice_user_bool") ? TRUE : FALSE;
	if (sys && user)
		choice_combo_set_content (CHOICE_COMBO (combo),
					  dlginfo->qe->q->conf->srv->
					  data_functions,
					  GTK_STRUCT_OFFSET (SqlDataFunction,
							     sqlname));
	else {
		list = NULL;
		lptr = dlginfo->qe->q->conf->srv->data_functions;
		while (lptr) {
			if ((SQL_DATA_FUNCTION (lptr->data)->is_user && user)
			    || (!SQL_DATA_FUNCTION (lptr->data)->is_user
				&& sys))
				list = g_slist_append (list, lptr->data);
			lptr = g_slist_next (lptr);
		}
		choice_combo_set_content (CHOICE_COMBO (combo), list,
					  GTK_STRUCT_OFFSET (SqlDataFunction,
							     sqlname));
		g_slist_free (list);
	}
}


/*
 * Adding an aggregate object
 */
static void object_add_agg_changed_cb (GtkWidget * choice,
				       gpointer listdata, GtkWidget * wid);
static void object_add_set_agg_list (GtkWidget * wid);
static GtkWidget *
object_get_widget_agg (SqlQueryEdit * qe, ObjAddDlgInfo * dlginfo,
		       QueryElement * qel)
{
	GtkWidget *wid, *vb, *table;

	vb = gtk_vbox_new (FALSE, 0);

	table = gtk_table_new (2, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (vb), table, FALSE, FALSE, GNOME_PAD);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);

	wid = gtk_label_new (_("Aggregate:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 0, 1);
	gtk_widget_show (wid);

	wid = choice_combo_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
	gtk_widget_show (wid);
	gtk_object_set_data (GTK_OBJECT (vb), "agg_choice_agg", wid);
	gtk_signal_connect (GTK_OBJECT (wid), "selection_changed",
			    GTK_SIGNAL_FUNC (object_add_agg_changed_cb), vb);

	wid = gtk_label_new (_("Apply to field:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 1, 2);
	gtk_widget_show (wid);

	wid = choice_combo_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 1, 2);
	gtk_object_set_data (GTK_OBJECT (vb), "agg_choice_obj", wid);
	gtk_widget_show (wid);
	gtk_widget_show_all (table);

	gtk_object_set_data (GTK_OBJECT (vb), "info", dlginfo);
	dlginfo->page_agg = vb;
	object_add_set_agg_list (vb);

	return vb;
}

static void
object_add_agg_changed_cb (GtkWidget * choice,
			   gpointer listdata, GtkWidget * wid)
{
	SqlDataAggregate *agg = NULL;
	GtkWidget *arg_wid;
	GSList *nlist = NULL, *elts, *rlist;
	gboolean activate_ok = TRUE;
	gboolean all_types;
	ObjAddDlgInfo *dlginfo;

	dlginfo = gtk_object_get_data (GTK_OBJECT (wid), "info");
	/* showing / hiding of arguments for that function */
	if (listdata)
		agg = SQL_DATA_AGGREGATE (listdata);
	else
		activate_ok = FALSE;

	arg_wid = gtk_object_get_data (GTK_OBJECT (wid), "agg_choice_obj");
	if (activate_ok) {
		all_types = FALSE;
		/* rlist is the list of acceptable types for that aggregate name */
		rlist = NULL;
		nlist = dlginfo->qe->q->conf->srv->data_aggregates;
		while (nlist) {
			if (!g_slist_find
			    (rlist,
			     SQL_DATA_AGGREGATE (nlist->data)->arg_type)
			    && !strcmp (SQL_DATA_AGGREGATE (nlist->data)->
					sqlname, agg->sqlname)) {
				rlist = g_slist_append (rlist,
							SQL_DATA_AGGREGATE
							(nlist->data)->
							arg_type);
				if (!SQL_DATA_AGGREGATE (nlist->data)->
				    arg_type)
					all_types = TRUE;	/* all data types allowed */
			}
			nlist = g_slist_next (nlist);
		}

		/* possible choices for that argument */
		elts = dlginfo->qe->q->objects;
		nlist = NULL;
		while (elts) {
			SqlDataType *type;

			switch (((QueryElement *) (elts->data))->type) {
			case QUERY_ELT_FIELD:	/* Table Field */
				type = SQL_MEM_FIELD (((QueryElement
							*) (elts->data))->
						      main)->type;
				if (all_types) {
					nlist = g_slist_append (nlist,
								elts->data);
					break;
				}
				if (g_slist_find (rlist, type))
					nlist = g_slist_append (nlist,
								elts->data);
				break;
			case QUERY_ELT_FUNCTION:	/* Function */
				type = SQL_DATA_FUNCTION (((QueryElement
							    *) (elts->data))->
							  main)->result_type;
				if (all_types) {
					nlist = g_slist_append (nlist,
								elts->data);
					break;
				}
				if (g_slist_find (rlist, type))
					nlist = g_slist_append (nlist,
								elts->data);
				break;
			case QUERY_ELT_AGGREGATE:
				break;
			case QUERY_ELT_VALUE:
				if (all_types) {
					nlist = g_slist_append (nlist,
								elts->data);
					break;
				}
				if (g_slist_find
				    (rlist,
				     ((QueryElementValue *) (((QueryElement *)
							      (elts->data))->
							     main))->
				     data_type))
					nlist = g_slist_append (nlist,
								elts->data);
				break;
			}
			elts = g_slist_next (elts);
		}
		g_slist_free (rlist);
	}
	choice_combo_set_content (CHOICE_COMBO (arg_wid), nlist,
				  GTK_STRUCT_OFFSET (QueryElement, name));
	if (!nlist)
		activate_ok = FALSE;
	else
		g_slist_free (nlist);

	dlginfo->ok_state[2] = activate_ok;
	gnome_dialog_set_sensitive (GNOME_DIALOG (dlginfo->dlg), 0,
				    dlginfo->ok_state[2]);
}

static void
object_add_set_agg_list (GtkWidget * wid)
{
	GtkWidget *caggs;
	GSList *list, *nlist;
	ObjAddDlgInfo *dlginfo;

	dlginfo = gtk_object_get_data (GTK_OBJECT (wid), "info");
	caggs = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (wid),
						 "agg_choice_agg"));
	/* aggregates lists (one per aggregate name) */
	list = dlginfo->qe->q->conf->srv->data_aggregates;
	nlist = NULL;
	while (list) {
		if (!nlist ||
		    (strcmp
		     (SQL_DATA_AGGREGATE (g_slist_last (nlist)->data)->
		      sqlname, SQL_DATA_AGGREGATE (list->data)->sqlname)))
			nlist = g_slist_append (nlist, list->data);
		list = g_slist_next (list);
	}
	choice_combo_set_content (CHOICE_COMBO (caggs), nlist,
				  GTK_STRUCT_OFFSET (SqlDataAggregate,
						     sqlname));
	g_slist_free (nlist);
}

/*
 * Adding a value object 
 */
static void object_add_datatype_changed_cb (GtkWidget * choice,
					    gpointer listdata,
					    GtkWidget * wid);
static void object_add_set_datatypes_list (GtkWidget * wid);
static GtkWidget *
object_get_widget_val (SqlQueryEdit * qe, ObjAddDlgInfo * dlginfo,
		       QueryElement * qel)
{
	GtkWidget *wid, *vb, *table;

	vb = gtk_vbox_new (FALSE, 0);

	table = gtk_table_new (3, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (vb), table, FALSE, FALSE, GNOME_PAD);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);
	gtk_object_set_data (GTK_OBJECT (vb), "val_table", table);

	wid = gtk_label_new (_("Data type:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 0, 1);
	gtk_widget_show (wid);

	wid = choice_combo_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
	gtk_object_set_data (GTK_OBJECT (vb), "val_choice_dt", wid);
	gtk_signal_connect (GTK_OBJECT (wid), "selection_changed",
			    GTK_SIGNAL_FUNC (object_add_datatype_changed_cb),
			    vb);
	gtk_widget_show (wid);

	wid = gtk_label_new (_("Value defaults to:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 1, 2);
	gtk_widget_show (wid);

	wid = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 1, 2);
	gtk_widget_show (wid);
	gtk_object_set_data (GTK_OBJECT (vb), "val_default", wid);

	wid = gtk_label_new (_("Can be modified:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 2, 3);
	gtk_widget_show (wid);

	wid = gtk_check_button_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 2, 3);
	gtk_widget_show (wid);
	gtk_object_set_data (GTK_OBJECT (vb), "val_mod", wid);

	gtk_widget_show_all (table);

	gtk_object_set_data (GTK_OBJECT (vb), "info", dlginfo);
	dlginfo->page_val = vb;
	object_add_set_datatypes_list (vb);

	return vb;
}

static void
object_add_datatype_changed_cb (GtkWidget * choice,
				gpointer listdata, GtkWidget * wid)
{
	SqlDataType *dt = NULL;
	gboolean activate_ok = TRUE;
	GtkWidget *table, *def_val;
	ObjAddDlgInfo *dlginfo;

	dlginfo = gtk_object_get_data (GTK_OBJECT (wid), "info");
	/* showing / hiding of arguments for that function */
	if (listdata)
		dt = SQL_DATA_TYPE (listdata);
	if (!dt)
		activate_ok = FALSE;

	if (activate_ok) {
		SqlDataDisplayFns *fns;

		/* replaces the default widget with a data type entry widget */
		table = gtk_object_get_data (GTK_OBJECT (wid), "val_table");
		def_val =
			gtk_object_get_data (GTK_OBJECT (wid), "val_default");
		gtk_container_remove (GTK_CONTAINER (table), def_val);
		fns = sql_access_get_object_display_fns (dlginfo->qe->q->
							 conf->srv,
							 GTK_OBJECT (dt));
		def_val = GTK_WIDGET ((fns->gdafield_to_widget) (NULL));
		gtk_table_attach_defaults (GTK_TABLE (table), def_val, 1, 2,
					   1, 2);
		gtk_widget_show (def_val);
		gtk_object_set_data (GTK_OBJECT (def_val), "fns", fns);
		gtk_object_set_data (GTK_OBJECT (wid), "val_default",
				     def_val);
	}

	dlginfo->ok_state[3] = activate_ok;
	gnome_dialog_set_sensitive (GNOME_DIALOG (dlginfo->dlg), 0,
				    dlginfo->ok_state[3]);
}

static void
object_add_set_datatypes_list (GtkWidget * wid)
{
	GtkWidget *cdt;
	ObjAddDlgInfo *dlginfo;

	dlginfo = gtk_object_get_data (GTK_OBJECT (wid), "info");
	cdt = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (wid),
					       "val_choice_dt"));
	/* aggregates lists (one per aggregate name) */
	choice_combo_set_content (CHOICE_COMBO (cdt),
				  dlginfo->qe->q->conf->srv->data_types,
				  GTK_STRUCT_OFFSET (SqlDataType, sqlname));
	choice_combo_set_selection_num (CHOICE_COMBO (cdt), 0);
}


static void object_add_close_dlg (GtkWidget * dlg);
static void
object_add_button_dlg_clicked_cb (GnomeDialog * dialog,
				  gint button_number, SqlQueryEdit * qe)
{
	gboolean doremove = TRUE;	/* TRUE => object to be created */
	GtkWidget *wid;
	gchar *namestr, *pnamestr;
	GSList *list;
	ObjAddDlgInfo *dlginfo;

	dlginfo = gtk_object_get_data (GTK_OBJECT (dialog), "info");
	switch (button_number) {
	case 0:		/* Ok button, verifications before adding the object */
		/* checking the object name */
		wid = dlginfo->name;
		namestr = gtk_entry_get_text (GTK_ENTRY (wid));
		list = qe->q->objects;
		while (list && doremove) {
			if (!strcmp
			    (((QueryElement *) (list->data))->name,
			     namestr)) {
				doremove = FALSE;
				gnome_app_error (GNOME_APP (qe->q->conf->app),
						 _
						 ("The object name is already used"));
			}
			list = g_slist_next (list);
		}
		if (doremove && (*namestr == '\0')) {
			gnome_app_error (GNOME_APP (qe->q->conf->app),
					 _("The object name is not valid"));
			doremove = FALSE;
		}
		/* object print name */
		if (doremove) {
			wid = dlginfo->pname;
			pnamestr = gtk_entry_get_text (GTK_ENTRY (wid));
			if (strpbrk (pnamestr, " .'\"\\-(){}[]|@")) {
				gnome_app_error (GNOME_APP (qe->q->conf->app),
						 _
						 ("The object print name is not valid"));
				doremove = FALSE;
			}
		}
		if (doremove) {
			switch (dlginfo->type) {
			case OBJECT_FIELD:	/* Table Field Object */
				{
					SqlMemField *field;

					field = gtk_object_get_data
						(GTK_OBJECT
						 (dlginfo->page_field),
						 "field_selected");
					/* NB: field cannot be null since OK was pressed */
					sql_query_new_elt_field (qe->q,
								 namestr,
								 pnamestr,
								 field);
				}
				break;

			case OBJECT_FUNC:	/* Function Object */
				{
					SqlDataFunction *f;
					GSList *args;
					GtkWidget **args_tab;
					gint i, nbargs;

					wid = GTK_WIDGET (gtk_object_get_data
							  (GTK_OBJECT
							   (dlginfo->
							    page_func),
							   "func_choice_combo"));
					args_tab = (GtkWidget **)
						gtk_object_get_data
						(GTK_OBJECT
						 (dlginfo->page_func),
						 "args_tab");
					f = SQL_DATA_FUNCTION
						(choice_combo_get_selection
						 (CHOICE_COMBO (wid)));
					nbargs = g_slist_length (f->args);
					args = NULL;
					for (i = 0; i < nbargs; i++) {
						args = g_slist_append (args,
								       choice_combo_get_selection
								       (CHOICE_COMBO
									(args_tab
									 [2 *
									  i +
									  1])));
					}
					sql_query_new_elt_function (qe->q,
								    namestr,
								    pnamestr,
								    f, args);
					g_slist_free (args);
				}
				break;

			case OBJECT_AGG:	/* Aggregate Object */
				{
					GtkWidget *combo, *value;
					SqlDataAggregate *a, *agg = NULL;
					SqlDataType *v = NULL;
					QueryElement *elt;

					combo = gtk_object_get_data
						(GTK_OBJECT
						 (dlginfo->page_agg),
						 "agg_choice_agg");
					value = gtk_object_get_data
						(GTK_OBJECT
						 (dlginfo->page_agg),
						 "agg_choice_obj");

					a = SQL_DATA_AGGREGATE
						(choice_combo_get_selection
						 (CHOICE_COMBO (combo)));
					elt = (QueryElement
					       *) (choice_combo_get_selection
						   (CHOICE_COMBO (value)));
					switch (elt->type) {
					case QUERY_ELT_FIELD:
						v = SQL_MEM_FIELD (elt->
								   main)->
							type;
						break;
					case QUERY_ELT_FUNCTION:
						v = SQL_DATA_FUNCTION (elt->
								       main)->
							result_type;
						break;
					case QUERY_ELT_AGGREGATE:	/* cannot occur */
						break;
					case QUERY_ELT_VALUE:
						v = ((QueryElementValue
						      *) (elt->main))->
				      data_type;
						break;
					}

					list = qe->q->conf->srv->
						data_aggregates;
					while (list && !agg) {
						if ((!strcmp
						     (SQL_DATA_AGGREGATE
						      (list->data)->sqlname,
						      a->sqlname))
						    &&
						    ((SQL_DATA_AGGREGATE
						      (list->data)->
						      arg_type == v)
						     ||
						     !SQL_DATA_AGGREGATE
						     (list->data)->arg_type))
							agg = SQL_DATA_AGGREGATE (list->data);
						list = g_slist_next (list);
					}
					if (agg) {
						list = g_slist_append (NULL,
								       elt);
						sql_query_new_elt_aggregate
							(qe->q, namestr,
							 pnamestr, agg, list);
						g_slist_free (list);
					}
				}
				break;

			case OBJECT_VAL:	/* Value Object */
				{
					SqlDataType *dt;
					QueryElement *elt;
					SqlDataDisplayFns *fns;
					gchar *str;
					gboolean mod;

					wid = GTK_WIDGET (gtk_object_get_data
							  (GTK_OBJECT
							   (dlginfo->
							    page_val),
							   "val_choice_dt"));
					dt = SQL_DATA_TYPE
						(choice_combo_get_selection
						 (CHOICE_COMBO (wid)));
					wid = GTK_WIDGET (gtk_object_get_data
							  (GTK_OBJECT
							   (dlginfo->
							    page_val),
							   "val_default"));
					fns = gtk_object_get_data (GTK_OBJECT
								   (wid),
								   "fns");
					if (fns)
						str = (fns->
						       widget_to_sql
						       (DATA_DISPLAY (wid)));
					else	/* it's still the GtkEntry */
						g_error ("fns is null file %s at line %d\n", __FILE__, __LINE__);

					wid = GTK_WIDGET (gtk_object_get_data
							  (GTK_OBJECT
							   (dlginfo->
							    page_val),
							   "val_mod"));
					mod = gtk_toggle_button_get_active
						(GTK_TOGGLE_BUTTON (wid));

					elt = sql_query_new_elt_value (qe->q,
								       namestr,
								       pnamestr,
								       dt,
								       str,
								       mod);
					g_free (str);
				}
				break;
			}
		}
		break;
	case 1:
		/* Forgetting about adding an object */
		break;
	}
	if (doremove)
		object_add_close_dlg (dlginfo->dlg);
}

static void
object_add_close_dlg (GtkWidget * dlg)
{
	gboolean *ok_state;
	GtkWidget *args_tab;
	ObjAddDlgInfo *dlginfo;

	dlginfo = gtk_object_get_data (GTK_OBJECT (dlg), "info");
	gtk_widget_set_sensitive (GTK_WIDGET (dlginfo->qe->add_button), TRUE);

	args_tab =
		gtk_object_get_data (GTK_OBJECT (dlginfo->page_func),
				     "args_tab");
	g_free (args_tab);
	gnome_dialog_close (GNOME_DIALOG (dlg));
	dlginfo->qe->obj_add_dlg = NULL;
	g_free (dlginfo);
}


static void
object_add_data_functions_add_or_del_cb (GtkObject * obj,
					 SqlDataFunction * func,
					 ObjAddDlgInfo * dlginfo)
{
	object_add_set_function_list (dlginfo->page_func);
}



static void
object_add_data_aggregates_add_or_del_cb (GtkObject * obj,
					  SqlDataAggregate * agg,
					  ObjAddDlgInfo * dlginfo)
{
	object_add_set_agg_list (dlginfo->page_agg);
}

static void
object_add_data_datatypes_updated_cb (GtkObject * obj,
				      ObjAddDlgInfo * dlginfo)
{
	object_add_set_datatypes_list (dlginfo->page_val);
}

/* warning: object_edit_dialog_clicked_cb declared as static before because 
 * used when destroying the QueryEdit widget */

static void
object_edit_cb (GtkWidget * widget, SqlQueryEdit * qe)
{
	GtkWidget *dlg, *wid, *table;
	gchar *str;
	QueryElement *elt;
	gboolean exists;
	GSList *list;

	/* Which element */
	elt = (QueryElement *) PACKED_CLIST (qe->clist)->actual_selection;

	/* is there already a dlg opened for that element ? */
	exists = FALSE;
	list = qe->edited_objects;
	while (list && !exists) {
		if (elt ==
		    gtk_object_get_data (GTK_OBJECT (list->data),
					 "QueryElement"))
			exists = TRUE;
		list = g_slist_next (list);
	}

	/* new dialog creation */
	if (elt && !exists) {
		str = g_strdup_printf (_("Edition of object %s for query %s"),
				       elt->name, qe->q->name);
		dlg = gnome_dialog_new (str, GNOME_STOCK_BUTTON_OK,
					GNOME_STOCK_BUTTON_CANCEL, NULL);
		gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, FALSE);
		g_free (str);

		/* widgets at the top */
		table = gtk_table_new (3, 2, FALSE);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
				    table, TRUE, TRUE, 0);
		gtk_widget_show (table);

		wid = gtk_label_new (_("Object type:"));
		gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
		gtk_table_attach_defaults (GTK_TABLE (table), wid,
					   0, 1, 0, 1);
		gtk_widget_show (wid);

		switch (elt->type) {
		case QUERY_ELT_FIELD:
			wid = gtk_label_new (_("Table field"));
			break;
		case QUERY_ELT_AGGREGATE:
			wid = gtk_label_new (_("Aggregate"));
			break;
		case QUERY_ELT_FUNCTION:
			wid = gtk_label_new (_("Function"));
			break;
		case QUERY_ELT_VALUE:
			wid = gtk_label_new (_("Value"));
			break;
		}
		gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
		gtk_table_attach_defaults (GTK_TABLE (table), wid,
					   1, 2, 0, 1);
		gtk_widget_show (wid);

		wid = gtk_label_new (_("Object name:"));
		gtk_table_attach_defaults (GTK_TABLE (table), wid,
					   0, 1, 1, 2);
		gtk_widget_show (wid);

		wid = gtk_label_new (_("Print name:"));
		gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
		gtk_table_attach_defaults (GTK_TABLE (table), wid,
					   0, 1, 2, 3);
		gtk_widget_show (wid);

		wid = gtk_entry_new ();
		gtk_table_attach_defaults (GTK_TABLE (table), wid,
					   1, 2, 1, 2);
		gtk_widget_show (wid);
		gtk_entry_set_text (GTK_ENTRY (wid), elt->name);
		gtk_object_set_data (GTK_OBJECT (dlg), "eltname", wid);

		wid = gtk_entry_new ();
		gtk_table_attach_defaults (GTK_TABLE (table), wid,
					   1, 2, 2, 3);
		gtk_widget_show (wid);
		if (elt->print_name)
			gtk_entry_set_text (GTK_ENTRY (wid), elt->print_name);
		gtk_object_set_data (GTK_OBJECT (dlg), "eltprintname", wid);

		gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
				    GTK_SIGNAL_FUNC
				    (object_edit_dialog_clicked_cb), qe);
		gtk_object_set_data (GTK_OBJECT (dlg), "QueryElement", elt);
		qe->edited_objects = g_slist_append (qe->edited_objects, dlg);
		gtk_widget_show (dlg);
	}
}

static void
object_edit_dialog_clicked_cb (GnomeDialog * dialog,
			       gint button_number, SqlQueryEdit * qe)
{
	QueryElement *elt;
	GtkWidget *name, *pname;
	gchar *namestr, *pnamestr;
	GSList *list;
	gboolean doremove;

	doremove = TRUE;
	elt = gtk_object_get_data (GTK_OBJECT (dialog), "QueryElement");
	switch (button_number) {
	case 0:		/* OK button was pressed */
		name = gtk_object_get_data (GTK_OBJECT (dialog), "eltname");
		pname = gtk_object_get_data (GTK_OBJECT (dialog),
					     "eltprintname");
		namestr = gtk_entry_get_text (GTK_ENTRY (name)), pnamestr =
			gtk_entry_get_text (GTK_ENTRY (pname)),
			/* testing the validity of the object name */
			list = qe->q->objects;
		while (list && doremove) {
			if ((list->data != elt) &&
			    (!strcmp
			     (((QueryElement *) (list->data))->name,
			      namestr))) {
				doremove = FALSE;
				gnome_app_error (GNOME_APP (qe->q->conf->app),
						 _
						 ("The object name is already used"));
			}
			list = g_slist_next (list);
		}
		if (doremove && (*namestr == '\0')) {
			gnome_app_error (GNOME_APP (qe->q->conf->app),
					 _("The object name is not valid"));
			doremove = FALSE;
		}
		/* validity of the object print name */
		if (doremove) {
			if (strpbrk (pnamestr, " .'\"\\-(){}[]|@")) {
				gnome_app_error (GNOME_APP (qe->q->conf->app),
						 _
						 ("The object print name is not valid"));
				doremove = FALSE;
			}
		}

		if (doremove)
			sql_query_modif_elt (qe->q, elt, namestr, pnamestr);
		break;
	case 1:		/* Cancel button was pressed */
		break;
	}

	/* final action */
	if (doremove) {
		qe->edited_objects =
			g_slist_remove (qe->edited_objects, dialog);
		gnome_dialog_close (dialog);
	}
}


static void
obj_del_button (GtkWidget * widget, SqlQueryEdit * qe)
{
	QueryElement *elt;

	elt = (QueryElement *) PACKED_CLIST (qe->clist)->actual_selection;
	if (elt)
		sql_query_del_elt (qe->q, elt);
}




/*
 *
 * Joins page CBs
 *
 */

static void
depend_clist_select_row_cb (GtkWidget * wid,
			    gint row, gint column,
			    GdkEventButton * event, SqlQueryEdit * qe)
{
	QueryLink *ql;
	qe->depend_sel = row;
	ql = gtk_clist_get_row_data (GTK_CLIST (wid), row);
	if (ql) {
		gtk_widget_set_sensitive(qe->join_options[ql->join_type], TRUE);
		gtk_signal_handler_block_by_data(GTK_OBJECT(qe->join_options[ql->join_type]), qe);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(qe->join_options[ql->join_type]), TRUE);
		gtk_signal_handler_unblock_by_data(GTK_OBJECT(qe->join_options[ql->join_type]), qe);
	}
	else
		gtk_widget_set_sensitive(qe->join_options[ql->join_type], FALSE);
}

/* callback function for the join_options buttons */
static void join_options_cb(GtkWidget *bttn, SqlQueryEdit *qe)
{
	QueryLink *ql;
	JoinType jt;
	gchar *jt_str;

	if (qe->depend_sel > -1) { /* there is a join line selected */
		ql = gtk_clist_get_row_data (GTK_CLIST (qe->depend_clist),
					     qe->depend_sel);
		for (jt=0; jt<LAST_JOIN_TYPE; jt++) /* detect what jt is selected */
			if (qe->join_options[jt] == bttn) /* tests selected button */
				break; 
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(qe->join_options[jt]))) {
			ql->join_type = jt;
			
#ifdef debug_signal
			g_print(">> 'CHANGED' from sqlquerywid->join_options_cb\n");
#endif
			gtk_signal_emit_by_name (GTK_OBJECT (qe->q), "changed");
#ifdef debug_signal
			g_print("<< 'CHANGED' from sqlquerywid->join_options_cb\n");
#endif
			jt_str = join_type_get_sql(jt);
			gtk_clist_set_text(GTK_CLIST(qe->depend_clist), qe->depend_sel, 1, jt_str);

			where_manage_depend_ql(qe, ql); /* update the "where" part */
			
			sql_query_edit_changed_cb(GTK_OBJECT(qe->q), qe); /* update the SQL string */   
		}
		/* else the button lost the focus, so ignore */
	}
}

/* TODO FM: callback function to help button in joins page */
static void joins_help(GtkWidget *bttn, SqlQueryEdit *qe) 
{
	g_print ("help button pressed\n");
}


static void
sql_query_edit_auto_links_add_table (SqlQueryEdit * qe, SqlMemTable * t)
{
	GSList *list;
	SqlMemField *f;
	SqlMemTable *tab;

	list = qe->q->conf->db->field_links;
	while (list) {
		f = SQL_MEM_FLINK (list->data)->from;
		tab = sql_db_find_table_from_field (qe->q->conf->db, f);
		if (tab == t)
			sql_query_auto_links_add_link (NULL,
						       SQL_MEM_FLINK (list->
								      data),
						       qe->q);
		else {
			f = SQL_MEM_FLINK (list->data)->to;
			tab = sql_db_find_table_from_field (qe->q->conf->db,
							    f);
			if (tab == t)
				sql_query_auto_links_add_link (NULL,
							       SQL_MEM_FLINK
							       (list->data),
							       qe->q);
		}
		list = g_slist_next (list);
	}
}

static void
sql_query_edit_auto_links_del_table (SqlQueryEdit * qe, SqlMemTable * t)
{
	GSList *list;
	SqlMemField *f;
	SqlMemTable *tab;

	list = qe->q->conf->db->field_links;
	while (list) {
		f = SQL_MEM_FLINK (list->data)->from;
		tab = sql_db_find_table_from_field (qe->q->conf->db, f);
		if (tab == t)
			sql_query_auto_links_del_link (NULL,
						       SQL_MEM_FLINK (list->
								      data),
						       qe->q);
		else {
			f = SQL_MEM_FLINK (list->data)->to;
			tab = sql_db_find_table_from_field (qe->q->conf->db,
							    f);
			if (tab == t)
				sql_query_auto_links_del_link (NULL,
							       SQL_MEM_FLINK
							       (list->data),
							       qe->q);
		}
		list = g_slist_next (list);
	}
}


/* 
 * Adds a relationship to the clist widget in the Join page
 * Creates list item "From Field";"Join Type";"To Field" 
*/
static void
depend_add_to_list (GtkObject * obj, QueryLink * ql, SqlQueryEdit * qe)
{
	/* add an entry to the clist */
	gchar *text[3];
	gchar *whtext[3];
	SqlMemTable *t;
	gboolean ok;
	gint row;

	/* TODO FM: review the logic of this code: why test for t? */
	t = sql_db_find_table_from_field (qe->q->conf->db, ql->link->from);
	if (t) {
		ok = TRUE;
		text[0] = g_strdup_printf("%s.%s", t->name, ql->link->from->name);
	}
	else {
		ok = FALSE;
		text[0] = NULL;
	}

	text[1] = join_type_get_sql(ql->join_type);

	t = sql_db_find_table_from_field (qe->q->conf->db, ql->link->to);
	if (t && ok) {
		text[2] = g_strdup_printf ("%s.%s", t->name, ql->link->to->name);
	}
	else {
		ok = FALSE;
	}

	if (ok) {
		row = gtk_clist_append (GTK_CLIST (qe->depend_clist), text);
		gtk_clist_set_row_data (GTK_CLIST (qe->depend_clist), row, ql);
		g_free (text[0]);
		g_free (text[2]);
	}
	else 
		if (text[0])
			g_free (text[0]);

	/* and in the WHERE part */
	where_manage_depend_ql (qe, ql);

	if (ql->join_type != JOIN_CROSS)
		sql_query_edit_changed_cb (GTK_OBJECT (qe->q), qe);
	qe->q->conf->save_up_to_date = FALSE;
}

static void
depend_remove_from_list (GtkObject * obj, QueryLink * ql, SqlQueryEdit * qe)
{
	gint row;

	row = gtk_clist_find_row_from_data (GTK_CLIST (qe->depend_clist), ql);
	if (row != -1) {
		if (qe->depend_sel == row) {
			qe->depend_sel = -1;
		}
		gtk_clist_remove (GTK_CLIST (qe->depend_clist), row);
	}

	/* and in the WHERE part */
	where_manage_depend_ql (qe, ql);

	/*if (ql->useit)
  	          sql_query_edit_changed_cb (GTK_OBJECT (qe->q), qe);*/
	qe->q->conf->save_up_to_date = FALSE;
}


static void
where_manage_depend_ql (SqlQueryEdit * qe, QueryLink * ql)
{
	GtkCTreeNode *depnode = NULL;
	gchar *text[4];

	depnode =
		gtk_ctree_find_by_row_data (GTK_CTREE
					    (COND_LIST (qe->cdlist)->ctree),
					    NULL, ql);
	/* FIXME: removed by FM
	if (ql->useit && !depnode) {
		text[0] = g_strdup_printf ("%s.%s",
					   sql_db_find_table_from_field (qe->
									 q->
									 conf->
									 db,
									 ql->
									 link->
									 from)->
					   name, ql->link->from->name);
		text[1] = "=";
		text[2] = g_strdup_printf ("%s.%s",
					   sql_db_find_table_from_field (qe->
									 q->
									 conf->
									 db,
									 ql->
									 link->
									 to)->
					   name, ql->link->to->name);
		text[3] = _("automatic dependency");
		depnode =
			cond_list_insert_unselectable_node (COND_LIST
							    (qe->cdlist),
							    (gpointer) ql,
							    NULL, NULL, text,
							    TRUE, TRUE);
		g_free (text[0]);
		g_free (text[2]);
		return;
	}

	if (!ql->useit && depnode)
		cond_list_remove_node_from_data (COND_LIST (qe->cdlist), ql);
	*/
}


/*
 * 
 * Where page CBs
 *
 */
static gchar *render_operand (SqlQuery * q, QueryWhereRightOperandType type,
			      gpointer object);
static gchar *render_logic_op (QueryWhereLogicType op);	/* DO NOT FREE the returned value */
static void
dump_where_mem (SqlQuery * q, GNode * top, gint offset)
{				/* call it with offest=0 */
	GNode *node;
	QueryWhereNode *whnode;

	if (G_NODE_IS_ROOT (top)) {
		g_print ("**********************************************************\n");
		g_print (".\n");
	}
	node = top->children;
	while (node) {
		int i;
		whnode = (QueryWhereNode *) node->data;
		for (i = 1; i <= offset / 4; i++)
			g_print ("|   ");
		if (node->next)
			g_print ("|-- ");
		else
			g_print ("`-- ");

		if (whnode->is_node)
			g_print ("%s ", render_logic_op (whnode->val.logic));
		else
			g_print ("%s %s %s",
				 render_operand (q, QUERY_WHERE_ELT,
						 whnode->val.leaf.l_elt),
				 sql_query_render_operator (whnode->val.leaf.
							    op),
				 render_operand (q, whnode->val.leaf.r_type,
						 whnode->val.leaf.r_object));
		g_print (" (WH=%p GN=%p)\n", whnode, node);
		dump_where_mem (q, node, offset + 4);
		node = node->next;
	}
}

/* here the signal tells that a condition has been added to the SqlQuery, so
   we need to update what we display */
static void
where_cond_created_cb (SqlQuery * q, QueryWhereNode * new_node,
		       SqlQueryEdit * qe)
{
	gchar *text[4];
	GtkCTreeNode *tnde, *tparent = NULL, *tsibling = NULL;
	GNode *gnode;

	g_print ("+ + + + + + + + + + + + + + + + + + + + + + where_cond_created_cb (qe=%p)\n", qe);
	tnde = cond_list_find_node_from_data (COND_LIST (qe->cdlist),
					      new_node);
	if (tnde)
		return;		/* node already in the tree */

	/* here the node is not in the tree yet */
	gnode = sql_query_gnode_find (q, new_node);
	g_print ("the corresponding gnode is GN=%p for WH=%p\n", gnode,
		 new_node);
	if (gnode->parent)
		tparent =
			cond_list_find_node_from_data (COND_LIST (qe->cdlist),
						       gnode->parent->data);

	if (gnode->next)
		tsibling =
			cond_list_find_node_from_data (COND_LIST (qe->cdlist),
						       gnode->next->data);

	gtk_signal_handler_block_by_func (GTK_OBJECT (qe->cdlist),
					  GTK_SIGNAL_FUNC
					  (condlist_node_created_cb), qe);
	if (new_node->is_node) {
		text[0] = render_logic_op (new_node->val.logic);
		text[1] = "";
		text[2] = "";
		text[3] = "";
		cond_list_insert_node (COND_LIST (qe->cdlist), new_node,
				       tparent, tsibling, text, FALSE, TRUE);
	}
	else {
		text[0] =
			render_operand (q, QUERY_WHERE_ELT,
					new_node->val.leaf.l_elt);
		text[1] = sql_query_render_operator (new_node->val.leaf.op);
		text[2] =
			render_operand (q, new_node->val.leaf.r_type,
					new_node->val.leaf.r_object);
		text[3] = "";
		cond_list_insert_node (COND_LIST (qe->cdlist), new_node,
				       tparent, tsibling, text, TRUE, TRUE);
		g_free (text[0]);
		g_free (text[2]);
	}
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (qe->cdlist),
					    GTK_SIGNAL_FUNC
					    (condlist_node_created_cb), qe);
}

static gchar *
render_operand (SqlQuery * q, QueryWhereRightOperandType type,
		gpointer object)
{
	gchar *str = NULL;
	g_return_val_if_fail (q != NULL, g_strdup ("NULL"));

	switch (type) {
	case QUERY_WHERE_ELT:
		{
			QueryElement *elt;

			elt = (QueryElement *) (object);
			switch (elt->type) {
			case QUERY_ELT_FIELD:
				str = g_strdup_printf ("%s.%s",
						       sql_db_find_table_from_field
						       (q->conf->db,
							SQL_MEM_FIELD (elt->
								       main))->
						       name,
						       SQL_MEM_FIELD (elt->
								      main)->
						       name);
				break;
			case QUERY_ELT_FUNCTION:
				{
					GSList *list;
					GString *gstr;
					gboolean first = TRUE;

					list = elt->args;
					gstr = g_string_new ("");
					g_string_sprintf (gstr, "%s (",
							  SQL_DATA_FUNCTION
							  (elt->main)->
							  sqlname);
					while (list) {
						if (first) {
							g_string_sprintfa
								(gstr, "%s",
								 ((QueryElement *) (list->data))->name);
							first = FALSE;
						}
						else
							g_string_sprintfa
								(gstr, ", %s",
								 ((QueryElement *) (list->data))->name);
						list = g_slist_next (list);
					}
					g_string_sprintfa (gstr, ")");
					str = gstr->str;
					g_string_free (gstr, FALSE);
					break;
				}
			case QUERY_ELT_AGGREGATE:
				str = g_strdup_printf ("%s (%s)",
						       SQL_DATA_AGGREGATE
						       (elt->main)->sqlname,
						       ((QueryElement
							 *) (elt->args->
							     data))->name);
				break;
			case QUERY_ELT_VALUE:
				str = g_strdup_printf ("%s",
						       ((QueryElementValue
							 *) (elt->main))->
						       default_val);
				break;
			}
			break;
		}
	case QUERY_WHERE_ASK:
		str = g_strdup (_("Asked at query execution"));
		break;
	}

	if (!str)
		str = g_strdup ("???");
	return str;
}


static gchar *
render_logic_op (QueryWhereLogicType op)
{				/* DO NOT FREE returned value */
	gchar *str;

	switch (op) {
	case QUERY_WHERE_AND:
		str = _("AND");
		break;
	case QUERY_WHERE_OR:
		str = _("OR");
		break;
	case QUERY_WHERE_NOT:
		str = _("NOT");
		break;
	default:
		str = "???";
	}

	return str;
}

static void
where_cond_dropped_cb (SqlQuery * q, QueryWhereNode * node, SqlQueryEdit * qe)
{
	GNode *gnode;
	GtkWidget *dlg;

	g_print ("+ + + + + + + + + + + + + + + + + + + + + + where_cond_dropped_cb\n");

	gnode = sql_query_gnode_find (qe->q, node);

	dlg = g_hash_table_lookup (qe->h_modif_dlg, gnode);
	if (dlg) {
		gtk_object_destroy (GTK_OBJECT (dlg));
		g_hash_table_remove (qe->h_modif_dlg, gnode);
	}

	gtk_signal_handler_block_by_func (GTK_OBJECT (qe->cdlist),
					  GTK_SIGNAL_FUNC
					  (condlist_node_dropped_cb), qe);
	cond_list_remove_node_from_data (COND_LIST (qe->cdlist), node);
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (qe->cdlist),
					    GTK_SIGNAL_FUNC
					    (condlist_node_dropped_cb), qe);
}

static void
where_cond_modified_cb (SqlQuery * q, QueryWhereNode * node,
			SqlQueryEdit * qe)
{
	gchar *text[4];
	GtkCTreeNode *tnde;
	GNode *gnode;

	gnode = sql_query_gnode_find (qe->q, node);

	g_print ("+ + + + + + + + + + + + + + + + + + + + + + where_cond_modified_cb\n");

	tnde = cond_list_find_node_from_data (COND_LIST (qe->cdlist), node);
	if (!tnde)
		return;		/* node not in the tree */

	if (node->is_node) {
		text[0] = render_logic_op (node->val.logic);
		text[1] = "";
		text[2] = "";
		text[3] = "";
		cond_list_update_node (COND_LIST (qe->cdlist), tnde, text);
	}
	else {
		text[0] =
			render_operand (q, QUERY_WHERE_ELT,
					node->val.leaf.l_elt);
		text[1] = sql_query_render_operator (node->val.leaf.op);
		text[2] =
			render_operand (q, node->val.leaf.r_type,
					node->val.leaf.r_object);
		text[3] = "";
		cond_list_update_node (COND_LIST (qe->cdlist), tnde, text);
		g_free (text[0]);
		g_free (text[2]);
	}
}

static void
where_cond_moved_cb (SqlQuery * q, QueryWhereNode * node, SqlQueryEdit * qe)
{
	GtkCTreeNode *tree, *nparent, *nsib;
	GNode *gnode;

	g_print ("+ + + + + + + + + + + + + + + + + + + + + + where_cond_moved_cb\n");
	gnode = sql_query_gnode_find (q, node);
	tree = cond_list_find_node_from_data (COND_LIST (qe->cdlist), node);
	if (gnode->parent != q->where_tree)
		nparent =
			cond_list_find_node_from_data (COND_LIST (qe->cdlist),
						       gnode->parent->data);
	else
		nparent = NULL;

	if (gnode->next)
		nsib = cond_list_find_node_from_data (COND_LIST (qe->cdlist),
						      gnode->next->data);
	else
		nsib = NULL;

	gtk_signal_handler_block_by_func (GTK_OBJECT (qe->cdlist),
					  GTK_SIGNAL_FUNC
					  (condlist_tree_move_cb), qe);
	cond_list_move_node (COND_LIST (qe->cdlist), tree, nparent, nsib);
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (qe->cdlist),
					    GTK_SIGNAL_FUNC
					    (condlist_tree_move_cb), qe);
}

static GtkWidget *cond_dialog_get (SqlQueryEdit * qe);
static void cond_add_dlg_clicked_cb (GnomeDialog * dlg, gint btn,
				     SqlQueryEdit * qe);
static void cond_add_optionmenu1_activate_cb (GtkMenuItem * menu_item,
					      SqlQueryEdit * qe);
static void cond_add_optionmenu2_activate_cb (GtkMenuItem * menu_item,
					      SqlQueryEdit * qe);
static gchar *render_roperand_type (QueryWhereRightOperandType operand);	/* DON'T FREE result */
static void
cond_add_clicked_cb (GtkWidget * btn, SqlQueryEdit * qe)
{
	GtkWidget *dialog;

	if (qe->cond_add_dlg)
		dialog = qe->cond_add_dlg;
	else {
		dialog = cond_dialog_get (qe);
		gtk_signal_connect (GTK_OBJECT (dialog), "clicked",
				    GTK_SIGNAL_FUNC (cond_add_dlg_clicked_cb),
				    qe);
		qe->cond_add_dlg = dialog;
	}
	if (GTK_WIDGET_VISIBLE (dialog))
		gdk_window_raise (dialog->window);
	else
		gtk_widget_show (dialog);
}

/* GnomeDialog signal to be connected by the calling function */
static GtkWidget *
cond_dialog_get (SqlQueryEdit * qe)
{
	GtkWidget *dialog, *vbox, *label, *frame, *hbox, *combo, *button;
	GtkWidget *entry, *optionmenu, *optionmenu_menu, *table,
		*dialog_action_area, *mitem;
	QueryWhereOpType i;
	QueryWhereRightOperandType j;

	dialog = gnome_dialog_new (_("New query condition"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL, NULL);
	vbox = GNOME_DIALOG (dialog)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD / 2.);

	label = gtk_label_new (_("Enter new query condition:"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

	frame = gtk_frame_new (_("Left Operand"));
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD / 2.);
	gtk_widget_show (hbox);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), GNOME_PAD / 2.);

	label = gtk_label_new (_("Operand:"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	combo = choice_combo_new ();
	gtk_object_set_data (GTK_OBJECT (dialog), "lcombo", combo);
	choice_combo_set_content (CHOICE_COMBO (combo), qe->q->objects,
				  GTK_STRUCT_OFFSET (QueryElement, name));
	gtk_widget_show (combo);
	gtk_box_pack_start (GTK_BOX (hbox), combo, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD / 2.);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

	label = gtk_label_new (_("Criteria:"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	optionmenu = gtk_option_menu_new ();
	gtk_object_set_data (GTK_OBJECT (dialog), "criteria", optionmenu);
	gtk_widget_show (optionmenu);
	gtk_box_pack_start (GTK_BOX (hbox), optionmenu, FALSE, FALSE, 0);
	optionmenu_menu = gtk_menu_new ();
	for (i = QUERY_OP_EQUAL; i <= QUERY_OP_IN; i++) {
		mitem = gtk_menu_item_new_with_label
			(sql_query_render_operator (i));
		gtk_signal_connect (GTK_OBJECT (mitem), "activate",
				    GTK_SIGNAL_FUNC
				    (cond_add_optionmenu1_activate_cb), qe);
		gtk_menu_append (GTK_MENU (optionmenu_menu), mitem);
		gtk_widget_show (mitem);
		gtk_object_set_data (GTK_OBJECT (mitem), "op",
				     GINT_TO_POINTER (i));
		gtk_object_set_data (GTK_OBJECT (mitem), "dlg", dialog);
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu),
				  optionmenu_menu);

	frame = gtk_frame_new (_("Right Operand"));
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);

	table = gtk_table_new (2, 2, FALSE);
	gtk_widget_show (table);
	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_container_set_border_width (GTK_CONTAINER (table),
					GNOME_PAD / 2.);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD / 2.);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD / 2.);

	label = gtk_label_new (_("Type:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  (GtkAttachOptions) (0),
			  (GtkAttachOptions) (0), 0, 0);

	optionmenu = gtk_option_menu_new ();
	gtk_object_set_data (GTK_OBJECT (dialog), "rtype", optionmenu);
	gtk_widget_show (optionmenu);
	gtk_table_attach (GTK_TABLE (table), optionmenu, 1, 2, 0, 1,
			  (GtkAttachOptions) (0),
			  (GtkAttachOptions) (0), 0, 0);
	optionmenu_menu = gtk_menu_new ();
	for (j = QUERY_WHERE_ELT; j <= QUERY_WHERE_ASK; j++) {
		mitem = gtk_menu_item_new_with_label (render_roperand_type
						      (j));
		gtk_signal_connect (GTK_OBJECT (mitem), "activate",
				    GTK_SIGNAL_FUNC
				    (cond_add_optionmenu2_activate_cb), qe);
		gtk_menu_append (GTK_MENU (optionmenu_menu), mitem);
		gtk_widget_show (mitem);
		gtk_object_set_data (GTK_OBJECT (mitem), "type",
				     GINT_TO_POINTER (j));
		gtk_object_set_data (GTK_OBJECT (mitem), "dlg", dialog);
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu),
				  optionmenu_menu);

	label = gtk_label_new (_("Operand:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  (GtkAttachOptions) (0),
			  (GtkAttachOptions) (0), 0, 0);

	/* vbox for all the setting widgets depending on the right operand type */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_table_attach (GTK_TABLE (table), vbox, 1, 2, 1, 2,
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_widget_show (vbox);

	/* Element object combo */
	combo = choice_combo_new ();
	choice_combo_set_content (CHOICE_COMBO (combo), qe->q->objects,
				  GTK_STRUCT_OFFSET (QueryElement, name));
	gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);
	gtk_object_set_data (GTK_OBJECT (dialog),
			     render_roperand_type (QUERY_WHERE_ELT), combo);
	gtk_object_set_data (GTK_OBJECT (dialog), "rcombo", combo);
	gtk_widget_show (combo);	/* default option */
	/* Manually entered value */
	label = gtk_label_new (_("To be set at query time"));
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
	gtk_object_set_data (GTK_OBJECT (dialog),
			     render_roperand_type (QUERY_WHERE_ASK), label);

	return dialog;
}

static void
cond_add_dlg_clicked_cb (GnomeDialog * dlg, gint btn, SqlQueryEdit * qe)
{
	switch (btn) {
	case 0:		/* OK */
		{
			QueryElement *lqe, *rqe;
			GtkWidget *omenu, *menu = NULL, *mitem = NULL;
			QueryWhereOpType optype;
			QueryWhereRightOperandType rtype;
			gboolean error = FALSE;

			lqe = choice_combo_get_selection
				(CHOICE_COMBO
				 (gtk_object_get_data
				  (GTK_OBJECT (qe->cond_add_dlg), "lcombo")));
			if (!lqe)
				error = TRUE;
			omenu = gtk_object_get_data (GTK_OBJECT
						     (qe->cond_add_dlg),
						     "criteria");
			if (omenu)
				menu = gtk_option_menu_get_menu
					(GTK_OPTION_MENU (omenu));
			if (menu)
				mitem = gtk_menu_get_active (GTK_MENU (menu));
			if (mitem)
				optype = GPOINTER_TO_INT (gtk_object_get_data
							  (GTK_OBJECT (mitem),
							   "op"));
			else
				error = TRUE;

			menu = mitem = NULL;
			omenu = gtk_object_get_data (GTK_OBJECT
						     (qe->cond_add_dlg),
						     "rtype");
			if (omenu)
				menu = gtk_option_menu_get_menu
					(GTK_OPTION_MENU (omenu));
			if (menu)
				mitem = gtk_menu_get_active (GTK_MENU (menu));
			if (mitem)
				rtype = GPOINTER_TO_INT (gtk_object_get_data
							 (GTK_OBJECT (mitem),
							  "type"));
			else
				error = TRUE;

			g_print ("RTYPE: %d\n", rtype);
			if (!error)
				switch (rtype) {	/* FIXME: put the right value for parent and sibling */
				case QUERY_WHERE_ELT:
					rqe = choice_combo_get_selection
						(CHOICE_COMBO
						 (gtk_object_get_data
						  (GTK_OBJECT
						   (qe->cond_add_dlg),
						   "rcombo")));
					if (rqe)
						sql_query_cond_new_leaf (qe->
									 q,
									 (QueryElement
									  *)
									 lqe,
									 optype,
									 QUERY_WHERE_ELT,
									 rqe,
									 NULL,
									 NULL);
					break;
				case QUERY_WHERE_ASK:
					sql_query_cond_new_leaf (qe->q,
								 (QueryElement
								  *) lqe,
								 optype,
								 QUERY_WHERE_ASK,
								 NULL, NULL,
								 NULL);
					break;
				}

			dump_where_mem (qe->q, qe->q->where_tree, 0);
			break;
		}

	default:
		break;
	}
	gnome_dialog_close (dlg);
	qe->cond_add_dlg = NULL;
}

static void
cond_add_optionmenu1_activate_cb (GtkMenuItem * menu_item, SqlQueryEdit * qe)
{
	GtkWidget *dlg;

	dlg = gtk_object_get_data (GTK_OBJECT (menu_item), "dlg");
	/* Reserved if we want to react to the different styles of criteria, specially for 
	   IN, REGEXP, LIKE */
}

static void
cond_add_optionmenu2_activate_cb (GtkMenuItem * menu_item, SqlQueryEdit * qe)
{
	QueryWhereRightOperandType type, i;
	gchar *str;
	GtkWidget *wid, *dlg;

	dlg = gtk_object_get_data (GTK_OBJECT (menu_item), "dlg");
	type = GPOINTER_TO_INT (gtk_object_get_data
				(GTK_OBJECT (menu_item), "type"));
	for (i = QUERY_WHERE_ELT; i <= QUERY_WHERE_ASK; i++) {
		str = render_roperand_type (i);
		wid = gtk_object_get_data (GTK_OBJECT (dlg), str);
		if (wid) {
			if (i == type)
				gtk_widget_show (wid);
			else
				gtk_widget_hide (wid);
		}
	}
}

static gchar *
render_roperand_type (QueryWhereRightOperandType operand)
{				/* DON'T FREE result */
	gchar *str = NULL;
	switch (operand) {
	case QUERY_WHERE_ELT:
		str = _("Object");
		break;
	case QUERY_WHERE_ASK:
		str = _("Asked at query execution");
		break;
	default:
		str = "???";
		break;
	}

	return str;
}

static GtkWidget *cond_dialog_node_get (SqlQueryEdit * qe);
static void cond_node_modif_dlg_clicked_cb (GnomeDialog * dlg, gint btn,
					    SqlQueryEdit * qe);
static void cond_modif_dlg_clicked_cb (GnomeDialog * dlg, gint btn,
				       SqlQueryEdit * qe);
static void
cond_modif_clicked_cb (GtkWidget * btn, SqlQueryEdit * qe)
{
	/* if this button is clicked, then we only have a single selection */
	GSList *list;
	GNode *gnode;
	QueryWhereNode *whnode;
	GtkWidget *dlg;

	list = COND_LIST (qe->cdlist)->selection;
	if (!list)
		return;
	whnode = (QueryWhereNode *)
		cond_list_find_data_from_node (COND_LIST (qe->cdlist),
					       list->data);
	gnode = sql_query_gnode_find (qe->q, whnode);
	dlg = g_hash_table_lookup (qe->h_modif_dlg, gnode);

	if (dlg)
		gdk_window_raise (dlg->window);
	else {
		if (whnode->is_node) {
			GtkWidget *wid, *combo, *omenu, *menu = NULL, *mitem =
				NULL;
			QueryWhereRightOperandType rtype;
			gchar *str;

			dlg = cond_dialog_node_get (qe);
			g_hash_table_insert (qe->h_modif_dlg, gnode, dlg);
			gtk_object_set_data (GTK_OBJECT (dlg), "gnode",
					     gnode);
			gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
					    GTK_SIGNAL_FUNC
					    (cond_node_modif_dlg_clicked_cb),
					    qe);

			/* setting the right values in the dialog */
			omenu = gtk_object_get_data (GTK_OBJECT (dlg),
						     "logic");
			if (omenu)
				menu = gtk_option_menu_get_menu
					(GTK_OPTION_MENU (omenu));
			gtk_menu_set_active (GTK_MENU (menu),
					     (guint) whnode->val.logic);
			mitem = gtk_menu_get_active (GTK_MENU (menu));
			gtk_menu_shell_activate_item (GTK_MENU_SHELL (menu),
						      mitem, TRUE);

			gtk_widget_show (dlg);
		}
		else {
			GtkWidget *wid, *combo, *omenu, *menu = NULL, *mitem =
				NULL;
			QueryWhereRightOperandType rtype;
			gchar *str;

			dlg = cond_dialog_get (qe);
			g_hash_table_insert (qe->h_modif_dlg, gnode, dlg);
			gtk_object_set_data (GTK_OBJECT (dlg), "gnode",
					     gnode);
			gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
					    GTK_SIGNAL_FUNC
					    (cond_modif_dlg_clicked_cb), qe);

			/* setting the left operand value */
			combo = gtk_object_get_data (GTK_OBJECT (dlg),
						     "lcombo");
			choice_combo_set_selection_ptr (CHOICE_COMBO (combo),
							whnode->val.leaf.
							l_elt);

			/* setting the criteria value */
			omenu = gtk_object_get_data (GTK_OBJECT (dlg),
						     "criteria");
			if (omenu)
				menu = gtk_option_menu_get_menu
					(GTK_OPTION_MENU (omenu));
			gtk_menu_set_active (GTK_MENU (menu),
					     (guint) whnode->val.leaf.op);
			mitem = gtk_menu_get_active (GTK_MENU (menu));
			gtk_menu_shell_activate_item (GTK_MENU_SHELL (menu),
						      mitem, TRUE);

			/* setting the right operand value */
			omenu = gtk_object_get_data (GTK_OBJECT (dlg),
						     "rtype");
			if (omenu)
				menu = gtk_option_menu_get_menu
					(GTK_OPTION_MENU (omenu));
			gtk_menu_set_active (GTK_MENU (menu),
					     (guint) whnode->val.leaf.r_type);
			mitem = gtk_menu_get_active (GTK_MENU (menu));
			gtk_menu_shell_activate_item (GTK_MENU_SHELL (menu),
						      mitem, TRUE);

			for (rtype = QUERY_WHERE_ELT;
			     rtype <= QUERY_WHERE_ASK; rtype++) {
				str = render_roperand_type (rtype);
				wid = gtk_object_get_data (GTK_OBJECT (dlg),
							   str);
				if (wid) {
					if (rtype == whnode->val.leaf.r_type)
						gtk_widget_show (wid);
					else
						gtk_widget_hide (wid);
				}
			}

			/* show the right value if we have a type */
			if (whnode->val.leaf.r_type == QUERY_WHERE_ELT) {
				str = render_roperand_type (QUERY_WHERE_ELT);
				combo = gtk_object_get_data (GTK_OBJECT (dlg),
							     str);
				choice_combo_set_selection_ptr (CHOICE_COMBO
								(combo),
								whnode->val.
								leaf.
								r_object);
			}

			gtk_widget_show (dlg);
		}
	}
}

/* GnomeDialog signal to be connected by the calling function */
static GtkWidget *
cond_dialog_node_get (SqlQueryEdit * qe)
{
	GtkWidget *dialog, *vbox, *label, *frame, *hbox, *combo, *button;
	GtkWidget *entry, *optionmenu, *optionmenu_menu, *table,
		*dialog_action_area, *mitem;
	QueryWhereLogicType i;
	QueryWhereRightOperandType j;

	dialog = gnome_dialog_new (_("Modify node logic"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL, NULL);
	vbox = GNOME_DIALOG (dialog)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD / 2.);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD / 2.);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

	label = gtk_label_new (_("Set the new logic:"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	optionmenu = gtk_option_menu_new ();
	gtk_object_set_data (GTK_OBJECT (dialog), "logic", optionmenu);
	gtk_widget_show (optionmenu);
	gtk_box_pack_start (GTK_BOX (hbox), optionmenu, FALSE, FALSE, 0);
	optionmenu_menu = gtk_menu_new ();
	for (i = QUERY_WHERE_AND; i <= QUERY_WHERE_NOT; i++) {
		mitem = gtk_menu_item_new_with_label (render_logic_op (i));
		gtk_menu_append (GTK_MENU (optionmenu_menu), mitem);
		gtk_widget_show (mitem);
		gtk_object_set_data (GTK_OBJECT (mitem), "op",
				     GINT_TO_POINTER (i));
		gtk_object_set_data (GTK_OBJECT (mitem), "dlg", dialog);
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu),
				  optionmenu_menu);

	return dialog;
}

static void
cond_node_modif_dlg_clicked_cb (GnomeDialog * dlg, gint btn,
				SqlQueryEdit * qe)
{
	GNode *gnode;
	QueryWhereNode *node;

	gnode = gtk_object_get_data (GTK_OBJECT (dlg), "gnode");
	node = (QueryWhereNode *) gnode->data;

	switch (btn) {
	case 0:		/* OK */
		{
			GtkWidget *omenu, *menu = NULL, *mitem = NULL;
			QueryWhereLogicType ltype;
			gboolean error = FALSE;

			omenu = gtk_object_get_data (GTK_OBJECT (dlg),
						     "logic");
			if (omenu)
				menu = gtk_option_menu_get_menu
					(GTK_OPTION_MENU (omenu));
			if (menu)
				mitem = gtk_menu_get_active (GTK_MENU (menu));
			if (mitem)
				ltype = GPOINTER_TO_INT (gtk_object_get_data
							 (GTK_OBJECT (mitem),
							  "op"));
			else
				error = TRUE;

			if (!error)
				sql_query_cond_modif_node (qe->q, node,
							   ltype);

			dump_where_mem (qe->q, qe->q->where_tree, 0);
			break;
		}

	default:
		break;
	}
	gnome_dialog_close (dlg);

	if (gnode)
		g_hash_table_remove (qe->h_modif_dlg, gnode);
}

static void
cond_modif_dlg_clicked_cb (GnomeDialog * dlg, gint btn, SqlQueryEdit * qe)
{
	GNode *gnode;
	QueryWhereNode *node;

	gnode = gtk_object_get_data (GTK_OBJECT (dlg), "gnode");
	node = (QueryWhereNode *) gnode->data;

	switch (btn) {
	case 0:		/* OK */
		{
			QueryElement *lqe, *rqe;
			GtkWidget *omenu, *menu = NULL, *mitem = NULL;
			QueryWhereOpType optype;
			QueryWhereRightOperandType rtype;
			gboolean error = FALSE;

			lqe = choice_combo_get_selection
				(CHOICE_COMBO
				 (gtk_object_get_data
				  (GTK_OBJECT (dlg), "lcombo")));
			if (!lqe)
				error = TRUE;
			omenu = gtk_object_get_data (GTK_OBJECT (dlg),
						     "criteria");
			if (omenu)
				menu = gtk_option_menu_get_menu
					(GTK_OPTION_MENU (omenu));
			if (menu)
				mitem = gtk_menu_get_active (GTK_MENU (menu));
			if (mitem)
				optype = GPOINTER_TO_INT (gtk_object_get_data
							  (GTK_OBJECT (mitem),
							   "op"));
			else
				error = TRUE;

			menu = mitem = NULL;
			omenu = gtk_object_get_data (GTK_OBJECT (dlg),
						     "rtype");
			if (omenu)
				menu = gtk_option_menu_get_menu
					(GTK_OPTION_MENU (omenu));
			if (menu)
				mitem = gtk_menu_get_active (GTK_MENU (menu));
			if (mitem)
				rtype = GPOINTER_TO_INT (gtk_object_get_data
							 (GTK_OBJECT (mitem),
							  "type"));
			else
				error = TRUE;

			if (!error)
				switch (rtype) {
				case QUERY_WHERE_ELT:
					rqe = choice_combo_get_selection
						(CHOICE_COMBO
						 (gtk_object_get_data
						  (GTK_OBJECT (dlg),
						   "rcombo")));
					if (rqe)
						sql_query_cond_modif_leaf
							(qe->q, node,
							 (QueryElement *) lqe,
							 optype,
							 QUERY_WHERE_ELT,
							 rqe);
					break;
				case QUERY_WHERE_ASK:
					sql_query_cond_modif_leaf (qe->q,
								   node,
								   (QueryElement
								    *) lqe,
								   optype,
								   QUERY_WHERE_ASK,
								   NULL);
					break;
				}

			dump_where_mem (qe->q, qe->q->where_tree, 0);
			break;
		}

	default:
		break;
	}
	gnome_dialog_close (dlg);

	if (gnode)
		g_hash_table_remove (qe->h_modif_dlg, gnode);
}


static void
cond_del_clicked_cb (GtkWidget * btn, SqlQueryEdit * qe)
{
	GSList *list;
	GNode *gnode;
	QueryWhereNode *whnode;
	GtkWidget *dlg;		/* associated modif dialog, if existing  */

	while (COND_LIST (qe->cdlist)->selection) {
		list = COND_LIST (qe->cdlist)->selection;
		whnode = (QueryWhereNode *)
			cond_list_find_data_from_node (COND_LIST (qe->cdlist),
						       list->data);
		gnode = sql_query_gnode_find (qe->q, whnode);
		if (gnode) {
			dlg = g_hash_table_lookup (qe->h_modif_dlg, gnode);
			if (dlg) {
				gtk_object_destroy (GTK_OBJECT (dlg));
				g_hash_table_remove (qe->h_modif_dlg, gnode);
			}
		}
		else
			whnode = NULL;

		g_print ("Asked to delete GN=%p WH=%p\n", gnode, whnode);
		if (whnode)
			sql_query_cond_del (qe->q, whnode);
	}
}

/* 
 * initial filling of the WHERE CondList widget: called in a recursive way 
 * tnde is already associated with nde.
 * a call creates all the children of the nde
 * initially called with (qe, NULL, NULL)
 */
static void
initial_where_fill (SqlQueryEdit * qe, GNode * nde, GtkCTreeNode * tnde)
{
	GNode *gnode;
	gchar *text[4];
	GtkCTreeNode *tparent;
	QueryWhereNode *node;

	if (!nde)
		gnode = qe->q->where_tree->children;
	else
		gnode = nde->children;

	while (gnode) {
		node = (QueryWhereNode *) gnode->data;
		if (node->is_node) {
			text[0] = render_logic_op (node->val.logic);
			text[1] = "";
			text[2] = "";
			text[3] = "";
			tparent =
				cond_list_insert_node (COND_LIST (qe->cdlist),
						       node, tnde, NULL, text,
						       FALSE, TRUE);
		}
		else {
			text[0] =
				render_operand (qe->q, QUERY_WHERE_ELT,
						node->val.leaf.l_elt);
			text[1] =
				sql_query_render_operator (node->val.leaf.op);
			text[2] =
				render_operand (qe->q, node->val.leaf.r_type,
						node->val.leaf.r_object);
			text[3] = "";
			tparent =
				cond_list_insert_node (COND_LIST (qe->cdlist),
						       node, tnde, NULL, text,
						       TRUE, TRUE);
			g_free (text[0]);
			g_free (text[2]);
		}
		initial_where_fill (qe, gnode, tparent);
		gnode = gnode->next;
	}
}

/*
 * CBs from the CondList
 * Those signals come from the CondList widget, and in this case the widget
 * is already updated, what needs to be updated is only the SqlQuery object being edited.
 */
static void
condlist_tree_move_cb (CondList * cdlist,
		       GtkCTreeNode * node,
		       GtkCTreeNode * new_parent,
		       GtkCTreeNode * new_sibling, SqlQueryEdit * qe)
{
	GNode *gnode;
	QueryWhereNode *whnode = NULL, *whparent = NULL, *whsibling = NULL;
	/* any node can be moved here */
	g_print ("- - - - - - - - - - - - - - - - - - - - -condlist_tree_move_cb\n");

	whnode = (QueryWhereNode *)
		gtk_ctree_node_get_row_data (GTK_CTREE (cdlist->ctree), node);
	if (whnode)
		gnode = sql_query_gnode_find (qe->q, whnode);
	g_print ("TREE Node Moved... GN=%p ->", gnode);
	whparent =
		(QueryWhereNode *)
		gtk_ctree_node_get_row_data (GTK_CTREE (cdlist->ctree),
					     new_parent);
	if (whparent)
		gnode = sql_query_gnode_find (qe->q, whparent);
	g_print (" GN=%p\n", gnode);

	g_print ("new_sibling = %p\n", new_sibling);
	whsibling =
		(QueryWhereNode *)
		gtk_ctree_node_get_row_data (GTK_CTREE (cdlist->ctree),
					     new_sibling);

	g_print ("Moving WH=%p to parent WH=%p and sibling WH=%p (after blocking signal)\n", whnode, whparent, whsibling);
	gtk_signal_handler_block_by_func (GTK_OBJECT (qe->q),
					  GTK_SIGNAL_FUNC
					  (where_cond_moved_cb), qe);
	sql_query_cond_move (qe->q, whnode, whparent, whsibling);
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (qe->q),
					    GTK_SIGNAL_FUNC
					    (where_cond_moved_cb), qe);

	dump_where_mem (qe->q, qe->q->where_tree, 0);
}

static void
condlist_node_created_cb (CondList * cdlist,
			  GtkCTreeNode * node, SqlQueryEdit * qe)
{
	GtkCTreeNode *nde;
	GtkCTreeRow *row;
	GNode *gnode;
	QueryWhereNode *parent = NULL, *sibling = NULL, *res;
	/* we only wan the AND, OR, NOT nodes (no leaf nodes) */
	g_print ("- - - - - - - - - - - - - - - - - - - - -condlist_node_created_cb\n");

	row = GTK_CTREE_ROW (node);
	if (gtk_ctree_node_get_row_data (GTK_CTREE (cdlist->ctree), node)) {
		g_print ("-> Already there\n");
		return;
	}

	nde = row->parent;
	if (nde)
		parent = (QueryWhereNode *)
			gtk_ctree_node_get_row_data (GTK_CTREE
						     (cdlist->ctree), nde);

	nde = row->sibling;
	if (nde)
		sibling =
			(QueryWhereNode *)
			gtk_ctree_node_get_row_data (GTK_CTREE
						     (cdlist->ctree), nde);

	g_print ("Creating a new AND, OR node in query (after blocking signal)\n");
	gtk_signal_handler_block_by_func (GTK_OBJECT (qe->q),
					  GTK_SIGNAL_FUNC
					  (where_cond_created_cb), qe);
	res = sql_query_cond_new_node (qe->q, QUERY_WHERE_AND, parent,
				       sibling);
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (qe->q),
					    GTK_SIGNAL_FUNC
					    (where_cond_created_cb), qe);
	gnode = sql_query_gnode_find (qe->q, res);

	gtk_ctree_node_set_row_data (GTK_CTREE (cdlist->ctree), node, res);
	g_print ("New AND, OR node: GN=%p WH=%p\n", gnode, res);
	dump_where_mem (qe->q, qe->q->where_tree, 0);
}

static void
condlist_node_dropped_cb (CondList * cdlist,
			  GtkCTreeNode * node, SqlQueryEdit * qe)
{
	/* the nodes created here are AND, OR, NOT nodes ONLY => not leaf nodes */
	GNode *gnode;
	QueryWhereNode *whnode;
	GtkWidget *dlg;

	g_print ("- - - - - - - - - - - - - - - - - - - - -condlist_node_dropped_cb\n");
	whnode = gtk_ctree_node_get_row_data (GTK_CTREE (cdlist->ctree),
					      node);
	gnode = sql_query_gnode_find (qe->q, whnode);

	/* if gnode == NULL, then whnode is not a QueryWhereNode *, but a QueryLink * which comes
	   from the automatic dependencies, so there is nothing to do here! */
	if (!gnode)
		return;

	dlg = g_hash_table_lookup (qe->h_modif_dlg, gnode);
	if (dlg) {
		gtk_object_destroy (GTK_OBJECT (dlg));
		g_hash_table_remove (qe->h_modif_dlg, gnode);
	}

	g_print ("Removing WH=%p, GN=%p (after blocking signal)\n", whnode,
		 gnode);
	gtk_signal_handler_block_by_func (GTK_OBJECT (qe->q),
					  GTK_SIGNAL_FUNC
					  (where_cond_dropped_cb), qe);
	sql_query_cond_del (qe->q, (QueryWhereNode *) gnode->data);
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (qe->q),
					    GTK_SIGNAL_FUNC
					    (where_cond_dropped_cb), qe);
	dump_where_mem (qe->q, qe->q->where_tree, 0);
}

static void
condlist_selection_changed_cb (CondList * cdlist, SqlQueryEdit * qe)
{
	if (!COND_LIST (qe->cdlist)->selection) {
		gtk_widget_set_sensitive (qe->wh_modif_button, FALSE);
		gtk_widget_set_sensitive (qe->wh_del_button, FALSE);
	}
	else {
		gtk_widget_set_sensitive (qe->wh_del_button, TRUE);
		if (g_slist_length (COND_LIST (qe->cdlist)->selection) > 1)
			gtk_widget_set_sensitive (qe->wh_modif_button, FALSE);
		else
			gtk_widget_set_sensitive (qe->wh_modif_button, TRUE);
	}
}

/*
 *
 * Notebook Text only Page CBs
 *
 */

static void
text_only_tbutton_cb (GtkWidget * widget, SqlQueryEdit * qe)
{
	GtkWidget *wid;
	gchar *str;
	gint i;

	if (GTK_TOGGLE_BUTTON (widget)->active) {
		str = gtk_editable_get_chars (GTK_EDITABLE (qe->text), 0, -1);
		sql_query_set_text_sql (qe->q, str);
		g_free (str);
	}
	else
		sql_query_set_text_sql (qe->q, NULL);

	gtk_text_set_editable (GTK_TEXT (qe->text), qe->q->text_only);
	for (i = 0; i < 4; i++) {
		wid = gtk_notebook_get_nth_page (GTK_NOTEBOOK (qe->notebook),
						 i);
		gtk_widget_set_sensitive (wid, !qe->q->text_only);
	}
}

static void
text_only_text_changed_cb (GtkWidget * widget, SqlQueryEdit * qe)
{
	gtk_widget_set_sensitive (qe->text_valid_btn, TRUE);
}

static void
text_only_validate_cb (GtkWidget * widget, SqlQueryEdit * qe)
{
	gchar *str;

	str = gtk_editable_get_chars (GTK_EDITABLE (qe->text), 0, -1);
	sql_query_set_text_sql (qe->q, str);
	g_free (str);
}


/*
 *
 * Functions related to the Dialog in which is packed the SqlQueryEdit obj.
 * 
 */
static void main_dialog_clicked_cb (GnomeDialog * dialog,
				    gint button_number, SqlQueryEdit * qe);

static void
main_dialog_auto_close (GtkObject * obj, GnomeDialog * dlg)
{
	SqlQueryEdit *qe =
		gtk_object_get_data (GTK_OBJECT (dlg), "SqlQueryEdit");
	main_dialog_clicked_cb (dlg, 0, qe);
}

GtkWidget *
sql_query_edit_new_dlg (SqlQuery * q)
{
	GtkWidget *dlg, *wid;
	gchar *str;

	str = g_strdup_printf (_("Edition of query \"%s\""), q->name);
	dlg = gnome_dialog_new (str, GNOME_STOCK_BUTTON_CLOSE, NULL);
	gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, FALSE);
	g_free (str);

	/* widgets at the top */
	wid = sql_query_edit_new (q);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
			    wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
			    GTK_SIGNAL_FUNC (main_dialog_clicked_cb), wid);
	gtk_object_set_data (GTK_OBJECT (dlg), "SqlQueryEdit", wid);
	gtk_signal_connect_while_alive (GTK_OBJECT (q->conf->srv),
					"conn_closed",
					GTK_SIGNAL_FUNC
					(main_dialog_auto_close), dlg,
					GTK_OBJECT (dlg));
	return dlg;
}

static void
main_dialog_clicked_cb (GnomeDialog * dialog,
			gint button_number, SqlQueryEdit * qe)
{
	if (qe->obj_add_dlg)
		object_add_close_dlg (qe->obj_add_dlg);
	gnome_dialog_close (dialog);
}

/* updates the SQL version of the query */
static void
sql_query_edit_changed_cb (GtkObject * obj, SqlQueryEdit * qe)
{
	gchar *str;

	gtk_widget_set_sensitive (qe->text_valid_btn, FALSE);
	if (!qe->q->text_only) {
		/* removing old content */
		gtk_text_freeze (GTK_TEXT (qe->text));
		gtk_text_set_point (GTK_TEXT (qe->text), 0);
		gtk_text_forward_delete (GTK_TEXT (qe->text),
					 gtk_text_get_length (GTK_TEXT
							      (qe->text)));

		/* putting new content */
		str = sql_query_get_select_query (qe->q, NULL);
		gtk_text_insert (GTK_TEXT (qe->text), NULL, NULL, NULL, str,
				 -1);
		g_free (str);
		gtk_text_thaw (GTK_TEXT (qe->text));
	}
}
