/**
 * File:          $RCSfile: png_io.c,v $
 * Module:        PNG format image file I/O functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.28 $
 * Last edited:   $Date: 2005/02/10 20:19:17 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 *
 * Notes:         Works with the libpng & lz libraries
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gandalf/image/io/png_io.h>
#include <gandalf/image/image_bit.h>
#include <gandalf/common/misc_error.h>

/* only compile if you have PNG */
#ifdef HAVE_PNG

#include <png.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageIO
 * \{
 */

/* Prototypes */
static Gan_Bool decode_header_info ( int png_colour_type, 
                                     int png_bit_depth, 
                                     Gan_ImageFormat *format, 
                                     Gan_Type *type );

static Gan_Bool encode_header_info ( Gan_Image *image,
                                     int *png_colour_type, 
                                     int *png_bit_depth,
                                     Gan_Bool *has_alpha );

static Gan_Bool read_header(FILE *infile,
                            png_structp *png_ptr,
                            png_infop *info_ptr,
                            int *pcolour_type,
                            int *pbit_depth,
                            int *pheight,
                            int *pwidth);


/* Function which converts the format specific information into
 * Gandalf format and type information */
static Gan_Bool decode_header_info(int png_colour_type,
                                   int png_bit_depth,
                                   Gan_ImageFormat *format,
                                   Gan_Type *type)
{
   /* Determine format */
   switch(png_colour_type)
   {
      case PNG_COLOR_TYPE_GRAY:       *format = GAN_GREY_LEVEL_IMAGE;
        break;
        
      case PNG_COLOR_TYPE_GRAY_ALPHA: *format = GAN_GREY_LEVEL_ALPHA_IMAGE;
        break;

      case PNG_COLOR_TYPE_PALETTE:    *format = GAN_RGB_COLOUR_IMAGE;
        break;

      case PNG_COLOR_TYPE_RGB:        *format = GAN_RGB_COLOUR_IMAGE;
        break;

      case PNG_COLOR_TYPE_RGB_ALPHA:  *format = GAN_RGB_COLOUR_ALPHA_IMAGE;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "decode_header_info", GAN_ERROR_ILLEGAL_TYPE, "" );
        return GAN_FALSE;
   }

   /* type of pixel values: unsigned char, float etc.
    * and allocate memory*/
   switch(png_bit_depth)
   {
      case 1: *type = GAN_BOOL;
        break;

      case 8: *type = GAN_UINT8;
        break;

      case 16: *type = GAN_UINT16;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "decode_header_info", GAN_ERROR_ILLEGAL_TYPE, "" );
        return GAN_FALSE;
   }

   return GAN_TRUE;
}

static Gan_Bool encode_header_info (
    Gan_Image *image,
    int *png_colour_type, 
    int *png_bit_depth,
    Gan_Bool *has_alpha )
{
   *has_alpha = GAN_FALSE;

   switch(image->format)
   {
      case GAN_GREY_LEVEL_IMAGE:
        *png_colour_type = PNG_COLOR_TYPE_GRAY;
        break;

      case GAN_GREY_LEVEL_ALPHA_IMAGE:
        *png_colour_type = PNG_COLOR_TYPE_GRAY_ALPHA;
        *has_alpha = GAN_TRUE;
        break;
                                        
      case GAN_RGB_COLOUR_IMAGE:
        *png_colour_type = PNG_COLOR_TYPE_RGB;
        break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
        *png_colour_type = PNG_COLOR_TYPE_RGB_ALPHA;
        *has_alpha = GAN_TRUE;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register("encode_header_info", GAN_ERROR_ILLEGAL_TYPE, "" );
        return GAN_FALSE;
   }

   switch(image->type)
   {
      case GAN_BOOL:
        *png_bit_depth = 1;
        break;

      case GAN_UINT8:
        *png_bit_depth = 8;
        break;

      case GAN_UINT16:
        *png_bit_depth = 16;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register("encode_header_info", GAN_ERROR_ILLEGAL_TYPE, "" );
        return GAN_FALSE;
   }

   return GAN_TRUE;
}

/* Function which reads the header of the image file to extract the image 
 * information */
static Gan_Bool read_header(FILE *infile,
                            png_structp *png_ptr,
                            png_infop *info_ptr,
                            int *pcolour_type,
                            int *pbit_depth,
                            int *pheight,
                            int *pwidth)
{
   gan_uint8 sig[8];

   /* Check signature */

   fread(sig, 1, 8, infile);
   if (!png_check_sig(sig, 8))
   {
      gan_err_flush_trace();
      gan_err_register("read_header", GAN_ERROR_CORRUPTED_FILE, "");
      return GAN_FALSE;   
   }

   /* Create read_struct */
   *png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
   if (!*png_ptr)
   {
      gan_err_flush_trace();
      gan_err_register("read_header", GAN_ERROR_MALLOC_FAILED, "");
      return GAN_FALSE;
   }

   /* Create info_struct */
   *info_ptr = png_create_info_struct(*png_ptr);
   if (!(*info_ptr)) 
   {
      gan_err_flush_trace();
      gan_err_register("read_header", GAN_ERROR_MALLOC_FAILED, "");
      return GAN_FALSE;
   }

   /* setjmp() call */
   if (setjmp((*png_ptr)->jmpbuf))
   {
      png_destroy_read_struct(png_ptr, info_ptr, NULL);
      gan_err_flush_trace();
      gan_err_register("read_header", GAN_ERROR_FAILURE, "");
      return GAN_FALSE;        
   }

   /* Initialise IO */
   png_init_io(*png_ptr, infile);
   png_set_sig_bytes(*png_ptr, 8); 

   /* Read image info */
   png_read_info(*png_ptr, *info_ptr); 

   /* Extract necessary image info */
   png_get_IHDR ( *png_ptr, *info_ptr, (png_uint_32 *) pwidth,
                  (png_uint_32 *) pheight, pbit_depth, pcolour_type,
                  NULL, NULL, NULL );
        
   return GAN_TRUE;
}

static void read_cleanup ( Gan_Image *image, 
                           png_structp png_ptr, 
                           png_infop info_ptr )
{
   if (image)
   {
      gan_image_free(image);
      image = NULL;
   }

   if (png_ptr && info_ptr)
   {
      png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
      png_ptr = NULL;
      info_ptr = NULL;
   } 
}

/**
 * \brief Reads an image file in PNG format from a file stream.
 * \param infile The file stream to be read
 * \param image The image structure to read the image data into or NULL
 * \param gam Gamma correction value
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the PNG image from the file stream \a infile into the given \a image.
 * If \a image is \c NULL a new image is dynamically allocated, otherwise the
 * already allocated \a image structure is reused. \a gam identifies the
 * gamma correction which the opened image should have. It should be one if no
 * gamma correction needs to be applied.
 *
 * \sa gan_write_png_image_stream().
 */
Gan_Image *
 gan_read_png_image_stream(FILE *infile, Gan_Image *image, double gam)
{
   double file_gamma;
   Gan_ImageFormat format;
   Gan_Type type;
   int height, width, colour_type, bit_depth;
   png_structp png_ptr = NULL;
   png_infop info_ptr = NULL;

   if ( !read_header ( infile, &png_ptr, &info_ptr, &colour_type, &bit_depth,
                       &height, &width ) )
   {
      read_cleanup(NULL, png_ptr, info_ptr);
      /* ErrorMessage set in read_header */
      gan_err_register("gan_read_png_image", GAN_ERROR_FAILURE, "");
      return NULL;
   }

   /* Establish the Gandalf format and type of image needed */
   if(!decode_header_info(colour_type, bit_depth, &format, &type))
   {
      /* ErrorMessage set in decode_header_info */
      gan_err_register("gan_read_png_image", GAN_ERROR_FAILURE, "");
      return NULL;
   }
   
   /* setjmp() call */
   if (setjmp(png_ptr->jmpbuf))
   {
      png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
      gan_err_flush_trace();
      gan_err_register ( "gan_read_png_image", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   /* Transformations */
   /* Expand palette images to RGB */
   if (colour_type == PNG_COLOR_TYPE_PALETTE)
      png_set_expand(png_ptr);
   
   /* Expand 2,4 and 6 bit grey level to 8 bits */
   if (colour_type == PNG_COLOR_TYPE_GRAY &&
       (1 < bit_depth) && (bit_depth < 8))
      png_set_expand(png_ptr);

   /* Expand transparency bits to full alpha channel;*/
   if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
      png_set_expand(png_ptr);

   /* Get gamma from file. If present use it together with the supplied display gamma */
   if (png_get_gAMA(png_ptr, info_ptr, &file_gamma) && gam != 0.0)
   {
      png_set_gamma(png_ptr, gam, file_gamma);
      
#ifdef GAN_PNG_TEST
      printf("Gamma set\n");
#endif
   }
   
#ifndef WORDS_BIGENDIAN
   /* swap bytes if little endian architecture */
   if(bit_depth == 16)
   {
      png_set_swap(png_ptr);
   }
   else
      if (bit_depth == 1)
      {
         png_set_packswap(png_ptr);
      }
#endif

   /* Update info_ptr data */
   png_read_update_info(png_ptr, info_ptr);

   if(image == NULL)
   {
      /* Allocate memory for the new image */
      image = gan_image_alloc(format, type,
                              (unsigned long) height, (unsigned long) width);
      
      if(image == NULL)
      {
         read_cleanup(image, NULL, NULL);
         gan_err_register ( "gan_read_png_image", GAN_ERROR_FAILURE, "" );
         return NULL;
      }
   }
   else
   {
      /*Use an already allocated image struct*/
      if ( gan_image_set_format_type_dims(image, format, type, height, width)
           == NULL )
      {
         read_cleanup(image, NULL, NULL);
         gan_err_register("gan_read_png_image",GAN_ERROR_FAILURE, "");
         return NULL;
      }
   }

   /* fill binary image with zeros in case PNG and Gandalf differ in the
      word size used to store binary images */
   if ( image->format == GAN_GREY_LEVEL_IMAGE && image->type == GAN_BOOL )
      gan_image_fill_const_b ( image, GAN_FALSE );

   /* Read the image */
   png_read_image(png_ptr, (png_bytepp)image->row_data_ptr);
   png_read_end(png_ptr, NULL);
   
   /* Cleanup */
   read_cleanup(NULL, png_ptr, info_ptr);

   return image;
}

/**
 * \brief Reads an image file in PNG format.
 * \param filename The name of the image file
 * \param image The image structure to read the image data into or NULL
 * \param gam Gamma correction value
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the PNG image stored in the file \a filename into the given \a image.
 * If \a image is passed as \c NULL a new image is dynamically allocated;
 * otherwise the already allocated \a image structure is reused. \a gam
 * identifies the gamma correction which the opened image should have.
 * It should be one if no gamma correction needs to be applied.
 *
 * \sa gan_write_png_image().
 */
Gan_Image *
 gan_read_png_image ( const char *filename, Gan_Image *image, double gam )
{
   FILE *infile;

   /* attempt to open file */
   infile = fopen ( filename, "rb" );
   if ( infile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_png_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return NULL;
   }
 
   image = gan_read_png_image_stream ( infile, image, gam );
   fclose(infile);
   if ( image == NULL )
   {
      gan_err_register ( "gan_read_png_image", GAN_ERROR_FAILURE, filename );
      return NULL;
   }

   return image;
}

static void write_cleanup ( png_structp *p_png_ptr, 
                            png_infop *p_info_ptr )
{
   if(p_png_ptr || p_info_ptr)
      png_destroy_write_struct(p_png_ptr,  p_info_ptr);
}


/**
 * \brief Writes an image file to a stream in PNG format.
 * \param outfile The file stream
 * \param image The image structure to write to the file
 * \param gam Gamma correction value to be applied
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the given \a image into a PNG file using a file stream \a outfile.
 * If \a gam is not one a gamma correction of \a gam is applied.
 */
Gan_Bool
 gan_write_png_image_stream(FILE *outfile, Gan_Image *image, double gam)
{
   png_structp png_ptr;
   png_infop info_ptr;
   int colour_type, bit_depth;
   Gan_Bool has_alpha;

   /* Create the write structure */
   png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

   if (png_ptr == NULL)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_png_image",  GAN_ERROR_MALLOC_FAILED, "" );
      write_cleanup(NULL, NULL);
      return GAN_FALSE;
   }

   /* Allocate/initialize the image information data.*/
   info_ptr = png_create_info_struct(png_ptr);
   if (info_ptr == NULL)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_png_image", GAN_ERROR_MALLOC_FAILED, "" );
      write_cleanup(&png_ptr, NULL);
      return GAN_FALSE;
   }

   /* Set error handling.*/
   if (setjmp(png_ptr->jmpbuf))
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_png_image", GAN_ERROR_MALLOC_FAILED, "" );
      write_cleanup(&png_ptr, &info_ptr);
      return GAN_FALSE;
   }

   /* Initialise the IO streams */
   png_init_io(png_ptr, outfile);

   if(!encode_header_info(image, &colour_type, &bit_depth, &has_alpha))
   {
      /*Error string already set*/
      gan_err_register ( "gan_write_png_image", GAN_ERROR_FAILURE, "" );
      write_cleanup(&png_ptr, &info_ptr);
      return GAN_FALSE;
   }

   /* Set the image information */
   png_set_IHDR ( png_ptr, info_ptr, image->width, image->height, bit_depth,
                  colour_type, PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_BASE,
                  PNG_FILTER_TYPE_BASE );

   /* If the gamma correction applied when reading the image was not 1.0 */
   if ( gam != 0.0 )
      png_set_gAMA(png_ptr, info_ptr, gam);

   /* Write the file header information */
   png_write_info(png_ptr, info_ptr);

   /* pack pixels into bytes if not a boolean image */
   if( 1 < bit_depth && bit_depth < 8)
      png_set_packing(png_ptr);

#ifndef WORDS_BIGENDIAN
   /* swap bytes if little endian architecture */
   if(bit_depth == 16)
   {
      png_set_swap(png_ptr);
   }
   else
      if (bit_depth == 1)
      {
         png_set_packswap(png_ptr);
      }
#endif

   /* Write out the image */
   png_write_image(png_ptr, (png_bytep *) image->row_data_ptr);
   png_write_end(png_ptr, info_ptr);

   /* clean up after the write, and free any memory allocated */
   write_cleanup(&png_ptr, &info_ptr);

   /* that's it */
   return GAN_TRUE;
}

/**
 * \brief Writes an image file in PNG format.
 * \param filename The name of the image file
 * \param image The image structure to write to the file
 * \param gam Gamma correction value to be applied
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the given \a image into PNG file \a filename. If \a gam is not one
 * gamma correction of \a gam is applied.
 *
 * \sa gan_read_png_image().
 */
Gan_Bool
 gan_write_png_image ( const char *filename, Gan_Image *image, double gam )
{
   FILE *outfile;
   Gan_Bool result;

   /* attempt to open file */
   outfile = fopen ( filename, "wb" );
   if ( outfile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_png_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return GAN_FALSE;
   }

   result = gan_write_png_image_stream ( outfile, image, gam );
   fclose(outfile);
   return result;
}

/**
 * \}
 */

/**
 * \}
 */

#endif /* #ifdef HAVE_PNG */
