/**
 * File:          $RCSfile: pgm_io.c,v $
 * Module:        PGM format image file I/O functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.23 $
 * Last edited:   $Date: 2005/02/10 20:19:17 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <gandalf/image/io/pgm_io.h>
#include <gandalf/image/image_gl_uint8.h>
#include <gandalf/image/image_bit.h>
#include <gandalf/common/misc_error.h>
#include <gandalf/common/allocate.h>
#include <gandalf/common/array.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageIO
 * \{
 */

static Gan_Bool read_line ( FILE *infile, char *s )
{
  char *news, *sptr;
  static char line[80]="";

  if ((strlen (line) == 0) || ((sptr = strtok (NULL, " \n\t")) == NULL))
  {
     while ((news = fgets (line, 80, infile)) != NULL)
        if ( (line[0] != '#') && ((sptr = strtok (line, " \n\t")) != NULL) )
           break;

     if (news == NULL)
     {
        gan_err_flush_trace();
        gan_err_register ( "read_line", GAN_ERROR_CORRUPTED_FILE,
                           "reading line");
        return GAN_FALSE;
     }
  }

  strcpy ( s, sptr );
  return GAN_TRUE;
}

/**
 * \brief Reads a grey level image file in PGM format from a stream.
 * \param infile The file stream to be read
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the PGM image from the given file stream \a infile into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated,
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_pgm_image_stream().
 */
Gan_Image *
 gan_read_pgm_image_stream ( FILE *infile, Gan_Image *image )
{
   char s[80]="";
   char *signature = "P5\n";
   int width, height, maxval;
   Gan_Bool result=GAN_TRUE;

   fgets ( s, 3, infile );
   s[2] = '\n';
   if ( strcmp ( s, signature ) != 0 )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_pgm_image", GAN_ERROR_CORRUPTED_FILE,
                         "corrupted PGM file header (no P5)" );
      return NULL;
   }

   result = (Gan_Bool)(result & read_line ( infile, s )); width  = atoi(s);
   result = (Gan_Bool)(result & read_line ( infile, s )); height = atoi(s);
   result = (Gan_Bool)(result & read_line ( infile, s )); maxval = atoi(s);

   /* check whether any of the header lines was corrupted */
   if ( !result )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_pgm_image", GAN_ERROR_CORRUPTED_FILE,
                         "" );
      return NULL;
   }

   /* binary and 8-bit types supported */
   switch ( maxval )
   {
      /* we require characters to be single bytes */
#if (SIZEOF_CHAR != 1)
#error "characters not represented by single bytes"
#endif      
      case 255:
        if ( image == NULL )
           image = gan_image_alloc_gl_ui8 ( height, width );
        else
           image = gan_image_set_gl_ui8 ( image, height, width );

        if ( image == NULL )
        {
           gan_err_register ( "gan_read_pgm_image", GAN_ERROR_FAILURE, "" );
           return NULL;
        }
   
        /* if the image has zero size then we have finished */
        if ( width == 0 || height == 0 ) return image;

        if ( fread ( gan_image_get_pixptr_gl_ui8 ( image, 0, 0 ),
                     sizeof(gan_uint8), height*width, infile )
             != (size_t)(height*width) )
        {
           gan_err_flush_trace();
           gan_err_register ( "gan_read_pgm_image", GAN_ERROR_TRUNCATED_FILE,
                              "" );
           return NULL;
        }

        break;

      case 1:
        /* read binary image */
        if ( image == NULL )
           image = gan_image_alloc_b ( height, width );
        else
           image = gan_image_set_b ( image, height, width );

        if ( image == NULL )
        {
           gan_err_register ( "gan_read_pgm_image", GAN_ERROR_FAILURE, "" );
           return NULL;
        }
   
        /* if the image has zero size then we have finished */
        if ( width == 0 || height == 0 ) return image;
        else
        {
           /* create temporary array to hold bytes */
           gan_uint8 *ui8arr = gan_malloc_array ( gan_uint8, height*width ), *ui8ptr;
           int i, j;

           if ( ui8arr == NULL )
           {
              gan_err_flush_trace();
              gan_err_register ( "gan_read_pgm_image", GAN_ERROR_MALLOC_FAILED,
                                 "" );
              return NULL;
           }

           /* read image data into temporary array */
           if ( fread ( ui8arr, 1, height*width, infile ) !=
                (size_t)(height*width) )
           {
              gan_err_flush_trace();
              gan_err_register ( "gan_read_pgm_image",
                                 GAN_ERROR_TRUNCATED_FILE, "" );
              return NULL;
           }

           /* convert data to binary format */
           gan_image_fill_zero ( image );
           for ( i = 0, ui8ptr = ui8arr; i < height; i++ )
              for ( j = 0; j < width; j++ )
                 if ( *ui8ptr++ )
                    gan_image_set_pix_b ( image, i, j, GAN_TRUE );

           /* free temporary array */
           free(ui8arr);
        }
        
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_read_pgm_image", GAN_ERROR_INCOMPATIBLE, "" );
        return NULL;
   }
   
   /* success */
   return image;
}

/**
 * \brief Reads a grey level image file in PGM format.
 * \param filename The name of the image file
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the PGM image with the in the file \a filename into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated;
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_pgm_image().
 */
Gan_Image *
 gan_read_pgm_image ( const char *filename, Gan_Image *image )
{
   FILE *infile;
   Gan_Image *result;

   /* attempt to open file */
   infile = fopen ( filename, "rb" );
   if ( infile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_pgm_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return NULL;
   }

   result = gan_read_pgm_image_stream ( infile, image );
   fclose(infile);
   return result;
}

/**
 * \brief Writes a grey level image to a file stream in PGM format.
 * \param outfile The file stream
 * \param image The image structure to write to the file
 * \return #GAN_TRUE on successful write operation, or #GAN_FALSE on failure.
 *
 * Writes the \a image into the file stream \a outfile in PGM format.
 *
 * \sa gan_read_pgm_image_stream().
 */
Gan_Bool
 gan_write_pgm_image_stream ( FILE *outfile, Gan_Image *image )
{
   gan_err_test_bool ( image->format == GAN_GREY_LEVEL_IMAGE,
                       "gan_write_pgm_image_stream", GAN_ERROR_INCOMPATIBLE,
                       "" );

   switch ( image->type )
   {
      case GAN_UINT8:
        fprintf ( outfile, "P5\n#Constructed by Gandalf\n%d %d\n255\n",
                  (int)image->width, (int)image->height );

        /* if the image has zero size then we have finished */
        if ( image->width == 0 || image->height == 0 ) return GAN_TRUE;

        /* write image data to stream */
        if ( image->stride == image->width*gan_image_pixel_size(image->format,
                                                                image->type) )
           /* we can write it in one go */
           fwrite ( gan_image_get_pixptr_gl_ui8 ( image, 0, 0 ),
                    sizeof(gan_uint8), image->height*image->width, outfile );
        else
        {
           /* write one line of image data at a time */
           unsigned long r;

           for ( r=0; r < image->height; r++ )
              fwrite ( gan_image_get_pixptr_gl_ui8 ( image, r, 0 ),
                       sizeof(gan_uint8), image->width, outfile );
        }

        break;

      case GAN_BOOL:
      {
         gan_uint8 *ui8arr, *ui8ptr;
         int i, j;

         fprintf ( outfile, "P5\n#Constructed by Gandalf\n%d %d\n1\n",
                   (int)image->width, (int)image->height );

         /* if the image has zero size then we have finished */
         if ( image->width == 0 || image->height == 0 ) return GAN_TRUE;

         /* construct temporary array of bytes */
         ui8arr = gan_malloc_array ( gan_uint8, image->height*image->width );
         if ( ui8arr == NULL )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_write_pgm_image_stream",
                               GAN_ERROR_MALLOC_FAILED, "" );
            return GAN_FALSE;
         }

         /* initialise array of bytes to zero */
         gan_fill_array_ui8 ( ui8arr, image->height*image->width, 1, 0 );

         /* make ucptr point to end of temporary array */
         ui8ptr = ui8arr + image->height*image->width - 1;

         /* fill temporary with ones where a pixel is set */
         for ( i = (int)image->height-1; i >= 0; i-- )
            for ( j = (int)image->width-1; j >= 0; j--, ui8ptr-- )
               if ( gan_image_get_pix_b ( image, i, j ) )
                  *ui8ptr = 1;
                  
         /* write image data to stream */
         fwrite ( ui8arr, 1, image->height*image->width, outfile );

         /* free temporary array */
         free(ui8arr);
      }
      break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_write_pgm_image_stream",
                           GAN_ERROR_INCOMPATIBLE, "" );
        return GAN_FALSE;
   }

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes a grey level image file in PGM format.
 * \param filename The name of the image file
 * \param image The image structure to write to the file
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the \a image into PGM file \a filename.
 *
 * \sa gan_read_pgm_image().
 */
Gan_Bool
 gan_write_pgm_image ( const char *filename, Gan_Image *image )
{
   FILE *outfile;
   Gan_Bool result;

   /* attempt to open file */
   outfile = fopen ( filename, "wb" );
   if ( outfile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_pgm_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return GAN_FALSE;
   }

   result = gan_write_pgm_image_stream ( outfile, image );
   fclose(outfile);
   return result;
}

/**
 * \}
 */

/**
 * \}
 */
