/**
 * File:          $RCSfile: dpx_io.c,v $
 * Module:        DPX format image file I/O functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.5 $
 * Last edited:   $Date: 2005/05/21 22:11:18 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <gandalf/image/io/dpx_io.h>
#include <gandalf/image/image_gl_uint8.h>
#include <gandalf/image/image_rgb_uint8.h>
#include <gandalf/image/image_rgba_uint8.h>
#include <gandalf/image/image_rgb_uint16.h>
#include <gandalf/common/misc_error.h>
#include <gandalf/common/allocate.h>
#include <gandalf/common/compare.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageIO
 * \{
 */

/* file header offsets */
#define OFFSET_MAGIC          0
#define OFFSET_IMAGEOFFSET    4
#define OFFSET_VERSION        8
#define OFFSET_FILESIZE      16
#define OFFSET_DITTOKEY      20
#define OFFSET_GENERICSIZE   24
#define OFFSET_INDUSTRYSIZE  28
#define OFFSET_USERSIZE      32
#define OFFSET_FILENAME      36
#define OFFSET_TIMEDATE     136
#define OFFSET_CREATOR      160
#define OFFSET_PROJECT      260
#define OFFSET_COPYRIGHT    460
#define OFFSET_ENCRYPTKEY   660
#define OFFSET_RESERVED     664

/* image information offsets */
#define OFFSET_ORIENTATION      0
#define OFFSET_ELEMENTNUMBER    2
#define OFFSET_PIXELSPERLINE    4
#define OFFSET_LINESPERIMAGEELE 8
#define OFFSET_ORIENTATION      0
#define OFFSET_ELEMENTNUMBER    2
#define OFFSET_PIXELSPERLINE    4
#define OFFSET_LINESPERIMAGEELE 8
#define OFFSET_DATASIGN0       12
#define OFFSET_DESCRIPTOR0     32
#define OFFSET_TRANSFER0       33
#define OFFSET_COLORIMETRIC0   34
#define OFFSET_BITSIZE0        35
#define OFFSET_PACKING0        36
#define OFFSET_ENCODING0       38
#define OFFSET_DATAOFFSET0     40
#define OFFSET_EOLPADDING0     44
#define OFFSET_EOIMAGEPADDING0 48

#define BIG_BUFFER_SIZE 2048

void
 vReverseEndiannessUI32(gan_uint32* pui32Val)
{
   char *acVal = (char*)pui32Val, cTmp;

   cTmp = acVal[0]; acVal[0] = acVal[3]; acVal[3] = cTmp;
   cTmp = acVal[1]; acVal[1] = acVal[2]; acVal[2] = cTmp;
}

void
 vReverseEndiannessUI16(gan_uint16* pui16Val)
{
   char *acVal = (char*)pui16Val, cTmp;

   cTmp = acVal[0]; acVal[0] = acVal[1]; acVal[1] = cTmp;
}

Gan_Image *
 pgiRead8BitDPXImageData(FILE* pfInFile, Gan_Bool bReversedEndianness, gan_uint16 ui16Packing,
                         gan_uint32 ui32eolPadding, gan_uint32 ui32eoImagePadding,
                         Gan_ImageFormat eFormat, Gan_Type eType,
                         gan_uint32 ui32PixelsPerLine, gan_uint32 ui32LinesPerImageEle,
                         Gan_Image* pgiImage)
{
   Gan_Bool bAllocatedImage=GAN_FALSE;
   unsigned int uiRow;
   int iCol;
   unsigned int uiRowSizeInBytes;
   char *acBuffer, *acAlignedBuffer;

   if(eType != GAN_UINT8)
   {
      gan_err_flush_trace();
      gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "no 8 bit conversions supported" );
      return NULL;
   }

   /* determine row data size in bytes */
   switch(eFormat)
   {
      case GAN_GREY_LEVEL_IMAGE:
        uiRowSizeInBytes = ui32PixelsPerLine;
        break;

      case GAN_RGB_COLOUR_IMAGE:
      case GAN_RGB_COLOUR_ALPHA_IMAGE:
        uiRowSizeInBytes = 4*ui32PixelsPerLine;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "image format not supported" );
        return NULL;
   }        

   /* build a buffer to hold a single line of image data and align it to a four byte boundary*/
   acBuffer = malloc(uiRowSizeInBytes+3);
   if(acBuffer == NULL)
   {
      gan_err_flush_trace();
      gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_MALLOC_FAILED, "" );
      return NULL;
   }

   acAlignedBuffer = (char*)((gan_uint32)acBuffer + 3 - (((gan_uint32)acBuffer + 3) % 4));

   /* allocate the image */
   if ( pgiImage == NULL )
   {
      pgiImage = gan_image_alloc(eFormat, eType, ui32LinesPerImageEle, ui32PixelsPerLine );
      bAllocatedImage = GAN_TRUE;
   }
   else
      pgiImage = gan_image_set_format_type_dims ( pgiImage, eFormat, eType, ui32LinesPerImageEle, ui32PixelsPerLine );

   if ( pgiImage == NULL )
   {
      gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   /* read the image data */
   switch(eFormat)
   {
      case GAN_GREY_LEVEL_IMAGE:
        for(uiRow = 0; uiRow<ui32LinesPerImageEle; uiRow++)
        {
           gan_uint8* pui8Pix;
           gan_uint32* pui32Pix;

           if ( fread ( acAlignedBuffer, 1, uiRowSizeInBytes, pfInFile ) != uiRowSizeInBytes )
           {
              if(bAllocatedImage) gan_image_free(pgiImage);
              gan_err_flush_trace();
              gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_TRUNCATED_FILE, "truncated DPX file" );
              return NULL;
           }

           /* read the last word on the line */
           if((ui32PixelsPerLine % 4) != 0)
           {
              iCol=(int)ui32PixelsPerLine/4;
              pui8Pix = gan_image_get_pixptr_gl_ui8(pgiImage, uiRow, 4*iCol),
              pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol;
              if(bReversedEndianness)
                 vReverseEndiannessUI32(pui32Pix);

              pui8Pix[0]    = (gan_uint8)(((*pui32Pix) & 0xff000000) >> 24);
              if((ui32PixelsPerLine % 4) > 0)
                 pui8Pix[1] = (gan_uint8)(((*pui32Pix) &   0xff0000) >> 16);

              if((ui32PixelsPerLine % 4) > 1)
                 pui8Pix[2] = (gan_uint8)(((*pui32Pix) &     0xff00) >>  8);
           }
           
           for(iCol=(int)ui32PixelsPerLine/4-1, pui8Pix = gan_image_get_pixptr_gl_ui8(pgiImage, uiRow, 4*iCol),
               pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol; iCol >= 0; iCol--, pui8Pix -= 4, pui32Pix--)
           {
              if(bReversedEndianness)
                 vReverseEndiannessUI32(pui32Pix);

              pui8Pix[0] = (gan_uint8)(((*pui32Pix) & 0xff000000) >> 24);
              pui8Pix[1] = (gan_uint8)(((*pui32Pix) &   0xff0000) >> 16);
              pui8Pix[2] = (gan_uint8)(((*pui32Pix) &     0xff00) >>  8);
              pui8Pix[3] = (gan_uint8)(((*pui32Pix) &       0xff));
           }

           // allow for padding at end of line
           if(fseek(pfInFile, ui32eolPadding, SEEK_CUR) != 0)
           {
              if(bAllocatedImage) gan_image_free(pgiImage);
              gan_err_flush_trace();
              gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX data (truncated file?)" );
              return NULL;
           }
        }

        break;

      case GAN_RGB_COLOUR_IMAGE:
        for(uiRow = 0; uiRow<ui32LinesPerImageEle; uiRow++)
        {
           Gan_RGBPixel_ui8* pui8rgbPix;
           gan_uint32* pui32Pix;

           if ( fread ( acAlignedBuffer, 1, uiRowSizeInBytes, pfInFile ) != uiRowSizeInBytes )
           {
              if(bAllocatedImage) gan_image_free(pgiImage);
              gan_err_flush_trace();
              gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_TRUNCATED_FILE, "truncated DPX file" );
              return NULL;
           }

           for(iCol=(int)ui32PixelsPerLine-1, pui8rgbPix = gan_image_get_pixptr_rgb_ui8(pgiImage, uiRow, iCol),
               pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol; iCol >= 0; iCol--, pui8rgbPix--, pui32Pix--)
           {
              if(bReversedEndianness)
                 vReverseEndiannessUI32(pui32Pix);

              pui8rgbPix->R = (gan_uint8)(((*pui32Pix) & 0xff000000) >> 24);
              pui8rgbPix->G = (gan_uint8)(((*pui32Pix) &   0xff0000) >> 16);
              pui8rgbPix->B = (gan_uint8)(((*pui32Pix) &     0xff00) >>  8);
           }

           // allow for padding at end of line
           if(fseek(pfInFile, ui32eolPadding, SEEK_CUR) != 0)
           {
              if(bAllocatedImage) gan_image_free(pgiImage);
              gan_err_flush_trace();
              gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX data (truncated file?)" );
              return NULL;
           }
        }

        break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
        for(uiRow = 0; uiRow<ui32LinesPerImageEle; uiRow++)
        {
           Gan_RGBAPixel_ui8* pui8rgbaPix;
           gan_uint32* pui32Pix;

           if ( fread ( acAlignedBuffer, 1, uiRowSizeInBytes, pfInFile ) != uiRowSizeInBytes )
           {
              if(bAllocatedImage) gan_image_free(pgiImage);
              gan_err_flush_trace();
              gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_TRUNCATED_FILE, "truncated DPX file" );
              return NULL;
           }

           for(iCol=(int)ui32PixelsPerLine-1, pui8rgbaPix = gan_image_get_pixptr_rgba_ui8(pgiImage, uiRow, iCol),
               pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol; iCol >= 0; iCol--, pui8rgbaPix--, pui32Pix--)
           {
              if(bReversedEndianness)
                 vReverseEndiannessUI32(pui32Pix);

              pui8rgbaPix->R = (gan_uint8)(((*pui32Pix) & 0xff000000) >> 24);
              pui8rgbaPix->G = (gan_uint8)(((*pui32Pix) &   0xff0000) >> 16);
              pui8rgbaPix->B = (gan_uint8)(((*pui32Pix) &     0xff00) >>  8);
              pui8rgbaPix->A = (gan_uint8)(((*pui32Pix) &       0xff));
           }

           // allow for padding at end of line
           if(fseek(pfInFile, ui32eolPadding, SEEK_CUR) != 0)
           {
              if(bAllocatedImage) gan_image_free(pgiImage);
              gan_err_flush_trace();
              gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX data (truncated file?)" );
              return NULL;
           }
        }

        break;

      default:
        if(bAllocatedImage) gan_image_free(pgiImage);
        gan_err_flush_trace();
        gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "unsupported format" );
        return NULL;
   }        

   free(acBuffer);

   // allow for padding at end of image
   if(fseek(pfInFile, ui32eoImagePadding, SEEK_CUR) != 0)
   {
      if(bAllocatedImage) gan_image_free(pgiImage);
      gan_err_flush_trace();
      gan_err_register ( "pgiRead8BitDPXImageData", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX data (truncated file?)" );
      return NULL;
   }

   return pgiImage;
}

Gan_Image *
 pgiRead10BitDPXImageData(FILE* pfInFile, Gan_Bool bReversedEndianness, gan_uint16 ui16bPacked,
                          gan_uint32 ui32eolPadding, gan_uint32 ui32eoImagePadding,
                          Gan_ImageFormat eFormat, Gan_Type eType,
                          gan_uint32 ui32PixelsPerLine, gan_uint32 ui32LinesPerImageEle,
                          Gan_Image* pgiImage)
{
   Gan_Bool bAllocatedImage=GAN_FALSE;
   unsigned int uiRow;
   int iCol;
   unsigned int uiRowSizeInBytes;
   char *acBuffer, *acAlignedBuffer;

   if(eType != GAN_UINT16)
   {
      gan_err_flush_trace();
      gan_err_register ( "pgiRead10BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "only 10 --> 16 bit conversion supported" );
      return NULL;
   }

   /* determine row data size in bytes */
   switch(eFormat)
   {
      case GAN_RGB_COLOUR_IMAGE:
        uiRowSizeInBytes = 4*ui32PixelsPerLine;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "pgiRead10BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "image format not supported" );
        return NULL;
   }        

   /* build a buffer to hold a single line of image data and align it to a four byte boundary*/
   acBuffer = malloc(uiRowSizeInBytes+3);
   if(acBuffer == NULL)
   {
      gan_err_flush_trace();
      gan_err_register ( "pgiRead10BitDPXImageData", GAN_ERROR_MALLOC_FAILED, "" );
      return NULL;
   }

   acAlignedBuffer = (char*)((gan_uint32)acBuffer + 3 - (((gan_uint32)acBuffer + 3) % 4));

   /* allocate the image */
   if ( pgiImage == NULL )
   {
      pgiImage = gan_image_alloc(eFormat, eType, ui32LinesPerImageEle, ui32PixelsPerLine );
      bAllocatedImage = GAN_TRUE;
   }
   else
      pgiImage = gan_image_set_format_type_dims ( pgiImage, eFormat, eType, ui32LinesPerImageEle, ui32PixelsPerLine );

   if ( pgiImage == NULL )
   {
      gan_err_register ( "pgiRead10BitDPXImageData", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   /* read the image data */
   switch(eFormat)
   {
      case GAN_RGB_COLOUR_IMAGE:
        for(uiRow = 0; uiRow<ui32LinesPerImageEle; uiRow++)
        {
           Gan_RGBPixel_ui16* pui16rgbPix;
           gan_uint32* pui32Pix;

           if ( fread ( acAlignedBuffer, 1, uiRowSizeInBytes, pfInFile ) != uiRowSizeInBytes )
           {
              if(bAllocatedImage) gan_image_free(pgiImage);
              gan_err_flush_trace();
              gan_err_register ( "pgiRead10BitDPXImageData", GAN_ERROR_TRUNCATED_FILE, "truncated DPX file" );
              return NULL;
           }

           for(iCol=(int)ui32PixelsPerLine-1, pui16rgbPix = gan_image_get_pixptr_rgb_ui16(pgiImage, uiRow, iCol),
               pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol; iCol >= 0; iCol--, pui16rgbPix--, pui32Pix--)
           {
              if(bReversedEndianness)
                 vReverseEndiannessUI32(pui32Pix);

              pui16rgbPix->R = (gan_uint16)(((*pui32Pix) & 0xffc00000) >> 16);
              pui16rgbPix->G = (gan_uint16)(((*pui32Pix) &   0x3ff000) >>  6);
              pui16rgbPix->B = (gan_uint16)(((*pui32Pix) &      0xffc) <<  4);
           }

           // allow for padding at end of line
           if(fseek(pfInFile, ui32eolPadding, SEEK_CUR) != 0)
           {
              if(bAllocatedImage) gan_image_free(pgiImage);
              gan_err_flush_trace();
              gan_err_register ( "pgiRead10BitDPXImageData", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX data (truncated file?)" );
              return NULL;
           }
        }

        break;

      default:
        if(bAllocatedImage) gan_image_free(pgiImage);
        gan_err_flush_trace();
        gan_err_register ( "pgiRead10BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "image format not supported" );
        return NULL;
   }        

   free(acBuffer);

   // allow for padding at end of image
   if(fseek(pfInFile, ui32eoImagePadding, SEEK_CUR) != 0)
   {
      if(bAllocatedImage) gan_image_free(pgiImage);
      gan_err_flush_trace();
      gan_err_register ( "pgiRead10BitDPXImageData", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX data (truncated file?)" );
      return NULL;
   }

   return pgiImage;
}

/**
 * \brief Reads a RGB colour image file in DPX format from a stream.
 * \param infile The file stream to be read
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the DPX image from the given file stream \a infile into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated,
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_dpx_image_stream().
 */
Gan_Image *
 gan_read_dpx_image_stream ( FILE *infile, Gan_Image *image )
{
   char acHeader[BIG_BUFFER_SIZE], *acAlignedHeader;
   Gan_ImageFormat eFormat;
   Gan_Type eType;
   Gan_Bool bReversedEndianness=GAN_FALSE;

   /* DPX file header info */
   gan_uint32 ui32Magic;          /* Magic number */
   gan_uint32 ui32ImageOffset;    /* Offset to start of image data in bytes */

   /* DPX image information variables */
   gan_uint16 ui16Orientation;          /* image orientation */
   gan_uint16 ui16ElementNumber;        /* number of image elements */
   gan_uint32 ui32PixelsPerLine;        /* or x value */
   gan_uint32 ui32LinesPerImageEle;     /* or y value, per element */
   gan_uint8  ui8Descriptor;            /* descriptor for image element */
   gan_uint8  ui8BitSize;               /* bit size for element */
   gan_uint16 ui16Packing;              /* packing for element */
   gan_uint16 ui16Encoding;             /* encoding for element */
   gan_uint32 ui32DataOffset;           /* offset to data of element */
   gan_uint32 ui32eolPadding;           /* end of line padding used in element */
   gan_uint32 ui32eoImagePadding;       /* end of image padding used in element */

   /* align the header array */
   acAlignedHeader = (char*)((gan_uint32)acHeader + 7 - (((gan_uint32)acHeader + 7) % 8));

   /* read the generic file header */
   if(fread(acAlignedHeader, 1, 768, infile) != 768)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX file header (truncated file?)" );
      return NULL;
   }

   /* determine endianness from magic number */
   ui32Magic = *((gan_uint32*)(acAlignedHeader + OFFSET_MAGIC));
   if(ui32Magic == 0x53445058)
      bReversedEndianness = GAN_FALSE;
   else if(ui32Magic == 0x58504453)
      bReversedEndianness = GAN_TRUE;
   else
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX file header (illegal magic number)" );
      return NULL;
   }

   /* get offset of image data */
   ui32ImageOffset = *((gan_uint32*)(acAlignedHeader + OFFSET_IMAGEOFFSET));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32ImageOffset);

   /* read the image information header */
   if(fread(acAlignedHeader, 1, 640, infile) != 640)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX image information header (truncated file?)" );
      return NULL;
   }

   /* decode image information header stuff */
   ui16Orientation = *((gan_uint16*)(acAlignedHeader + OFFSET_ORIENTATION));
   if(bReversedEndianness)
      vReverseEndiannessUI16(&ui16Orientation);

   /* we only support top-to-bottom orientation */
   if(ui16Orientation != 0)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported DPX image orientation" );
      return NULL;
   }

   ui16ElementNumber = *((gan_uint16*)(acAlignedHeader + OFFSET_ELEMENTNUMBER));
   if(bReversedEndianness)
      vReverseEndiannessUI16(&ui16ElementNumber);

   ui32PixelsPerLine = *((gan_uint32*)(acAlignedHeader + OFFSET_PIXELSPERLINE));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32PixelsPerLine);

   ui32LinesPerImageEle = *((gan_uint32*)(acAlignedHeader + OFFSET_LINESPERIMAGEELE));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32LinesPerImageEle);

   /* we only support one DPX image per file */
   if(ui16ElementNumber != 1)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported DPX element number" );
      return NULL;
   }

   /* determine image format */
   ui8Descriptor = *((gan_uint8*)(acAlignedHeader + OFFSET_DESCRIPTOR0));
   switch(ui8Descriptor)
   {
      case 1: /* red */
      case 2: /* green */
      case 3: /* blue */
      case 4: /* alpha */
      case 6: /* luminance */
        eFormat = GAN_GREY_LEVEL_IMAGE;
        break;

      case 50: /* rgb */
        eFormat = GAN_RGB_COLOUR_IMAGE;
        break;

      case 51: /* rgba */
        eFormat = GAN_RGB_COLOUR_ALPHA_IMAGE;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported DPX descriptor" );
        return NULL;
   }

   /* read bits/channel/pixel */
   ui8BitSize = *((gan_uint8*)(acAlignedHeader + OFFSET_BITSIZE0));

   /* Determine packing type */
   ui16Packing = *((gan_uint16*)(acAlignedHeader + OFFSET_PACKING0));
   if(bReversedEndianness)
      vReverseEndiannessUI16(&ui16Packing);

   switch(ui8BitSize)
   {
      case 1:
        eType = GAN_BOOL;
        break;

      case 8:
        eType = GAN_UINT8;
        break;

      case 10:
      case 12:
      case 16:
        eType = GAN_UINT16;
        eType = GAN_UINT16;
        eType = GAN_UINT16;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported DPX bit size" );
        return NULL;
        break;
   }
   
   /* Determine encoding */
   ui16Encoding = *((gan_uint16*)(acAlignedHeader + OFFSET_ENCODING0));
   if(bReversedEndianness)
      vReverseEndiannessUI16(&ui16Encoding);

   /* Determine data offset */
   ui32DataOffset = *((gan_uint32*)(acAlignedHeader + OFFSET_DATAOFFSET0));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32DataOffset);

   /* Determine end-of-line padding */
   ui32eolPadding = *((gan_uint32*)(acAlignedHeader + OFFSET_EOLPADDING0));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32eolPadding);

   /* Determine end-of-image padding */
   ui32eoImagePadding = *((gan_uint32*)(acAlignedHeader + OFFSET_EOIMAGEPADDING0));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32eoImagePadding);

   if(ui32DataOffset != ui32ImageOffset)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "incompatible DPX data offsets" );
      return NULL;
   }

   /* reset file point to start of image data */
   if(fseek(infile, ui32ImageOffset, SEEK_SET) != 0)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_CORRUPTED_FILE, "corrupted DPX image orientation header (truncated file?)" );
      return NULL;
   }

   /* now read the image data */
   switch(ui8BitSize)
   {
      case 10:
        image = pgiRead10BitDPXImageData(infile, bReversedEndianness, ui16Packing, ui32eolPadding, ui32eoImagePadding,
                                         eFormat, eType, ui32PixelsPerLine, ui32LinesPerImageEle, image);
        if(image == NULL)
        {
           gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_FAILURE, "" );
           return NULL;
        }

        break;

      case 8:
        image = pgiRead8BitDPXImageData(infile, bReversedEndianness, ui16Packing, ui32eolPadding, ui32eoImagePadding,
                                        eFormat, eType, ui32PixelsPerLine, ui32LinesPerImageEle, image);
        if(image == NULL)
        {
           gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_FAILURE, "" );
           return NULL;
        }

        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_read_dpx_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported DPX bit depth" );
        return NULL;
   }
        
   /* success */
   return image;
}

/**
 * \brief Reads a RGB colour image file in DPX format.
 * \param filename The name of the image file
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the DPX image with the in the file \a filename into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated;
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_dpx_image().
 */
Gan_Image *
 gan_read_dpx_image ( const char *filename, Gan_Image *image )
{
   FILE *infile;
   Gan_Image *result;

   /* attempt to open file */
   infile = fopen ( filename, "rb" );
   if ( infile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_dpx_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return NULL;
   }

   result = gan_read_dpx_image_stream ( infile, image );
   fclose(infile);
   return result;
}

/**
 * \brief Initialises the write control structure for DPX files
 *
 * This function should be called on the structure to set the parameters to default values.
 * Then set any non-default values yourself before calling gan_write_dpx_image() or
 * gan_write_dpx_image_stream().
 */
void gan_initialise_dpx_write_control_struct(Gan_DPXWriteControlStruct *controlstr)
{
   controlstr->bit_size = 0;
   controlstr->transfer = GAN_DPXTRANSFER_DEFAULT;
   controlstr->colorimetric = GAN_DPXCOLORIMETRIC_DEFAULT;
}

#define IMAGE_DATA_OFFSET 8192

/* file size function assumes unpacked I/O */
static gan_uint32
 ui32DPXFileSize(Gan_ImageFormat eFormat, gan_uint8 ui8BitSize, unsigned int uiHeight, unsigned int uiWidth)
{
   switch(ui8BitSize)
   {
      case 1:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+31)/32)*4);
           default: return UINT_MAX;
        }
        break;
        
      case 8:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+3)/4)*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*uiWidth*4);
           case GAN_RGB_COLOUR_ALPHA_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*uiWidth*4);
           default: return UINT_MAX;
        }
        break;

      case 10:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+2)/3)*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*uiWidth*4);
           default: return UINT_MAX;
        }
        break;
        
      case 12:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+1)/2)*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*(((uiWidth*3)+1)/2)*4);
           default: return UINT_MAX;
        }
        break;

      case 16:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+1)/2)*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*(((uiWidth*3)+1)/2)*4);
           default: return UINT_MAX;
        }
        break;

      case 32:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*uiWidth*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*3*uiWidth*4);
           default: return UINT_MAX;
        }
        break;

      default: return UINT_MAX;
   }

   return UINT_MAX;
}

Gan_Bool
 bWrite8BitDPXImageData(FILE* pfOutFile, Gan_Image* pgiImage, Gan_Bool bReverseBytes)
{
   unsigned int uiRow;
   int iCol;
   unsigned int uiRowSizeInBytes;
   char *acBuffer, *acAlignedBuffer;

   if(pgiImage->type != GAN_UINT8)
   {
      gan_err_flush_trace();
      gan_err_register ( "pgiWrite8BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "no 8 bit conversions supported" );
      return GAN_FALSE;
   }

   /* determine row data size in bytes */
   switch(pgiImage->format)
   {
      case GAN_GREY_LEVEL_IMAGE:
        uiRowSizeInBytes = pgiImage->width;
        break;

      case GAN_RGB_COLOUR_IMAGE:
        uiRowSizeInBytes = 4*pgiImage->width;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "pgiWrite8BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "image format not supported" );
        return GAN_FALSE;
   }        

   /* build a buffer to hold a single line of image data and align it to a four byte boundary*/
   acBuffer = malloc(uiRowSizeInBytes+3);
   if(acBuffer == NULL)
   {
      gan_err_flush_trace();
      gan_err_register ( "pgiWrite8BitDPXImageData", GAN_ERROR_MALLOC_FAILED, "" );
      return GAN_FALSE;
   }

   acAlignedBuffer = (char*)((gan_uint32)acBuffer + 3 - (((gan_uint32)acBuffer + 3) % 4));

   /* write the image data */
   switch(pgiImage->format)
   {
      case GAN_RGB_COLOUR_IMAGE:
        for(uiRow = 0; uiRow<pgiImage->height; uiRow++)
        {
           Gan_RGBPixel_ui8* pui8rgbPix;
           gan_uint32* pui32Pix;

           memset((void*)acAlignedBuffer, 0, uiRowSizeInBytes);
           if(bReverseBytes)
           {
              for(iCol=(int)pgiImage->width-1, pui8rgbPix = gan_image_get_pixptr_rgb_ui8(pgiImage, uiRow, iCol),
                  pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol; iCol >= 0; iCol--, pui8rgbPix--, pui32Pix--)
              {
                 (*pui32Pix) |= ((pui8rgbPix->R >> 6) << 22);
                 (*pui32Pix) |= ((pui8rgbPix->G >> 6) << 12);
                 (*pui32Pix) |= ((pui8rgbPix->B >> 6) <<  2);
                 vReverseEndiannessUI32(pui32Pix);
              }
           }
           else
           {
              for(iCol=(int)pgiImage->width-1, pui8rgbPix = gan_image_get_pixptr_rgb_ui8(pgiImage, uiRow, iCol),
                  pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol; iCol >= 0; iCol--, pui8rgbPix--, pui32Pix--)
              {
                 (*pui32Pix) |= ((pui8rgbPix->R >> 6) << 22);
                 (*pui32Pix) |= ((pui8rgbPix->G >> 6) << 12);
                 (*pui32Pix) |= ((pui8rgbPix->B >> 6) <<  2);
              }
           }
           

           if ( fwrite ( acAlignedBuffer, 1, uiRowSizeInBytes, pfOutFile ) != uiRowSizeInBytes )
           {
              gan_err_flush_trace();
              gan_err_register ( "gan_read_dpx_image", GAN_ERROR_TRUNCATED_FILE, "truncated DPX file" );
              return GAN_FALSE;
           }
        }

        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "pgiWrite8BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "image format not supported" );
        return GAN_FALSE;
   }        

   free(acBuffer);
   return GAN_TRUE;
}

Gan_Bool
 bWrite10BitDPXImageData(FILE* pfOutFile, Gan_Image* pgiImage, Gan_Bool bReverseBytes)
{
   unsigned int uiRow;
   int iCol;
   unsigned int uiRowSizeInBytes;
   char *acBuffer, *acAlignedBuffer;

   if(pgiImage->type != GAN_UINT16)
   {
      gan_err_flush_trace();
      gan_err_register ( "pgiWrite10BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "only 10 --> 16 bit conversion supported" );
      return GAN_FALSE;
   }

   /* determine row data size in bytes */
   switch(pgiImage->format)
   {
      case GAN_RGB_COLOUR_IMAGE:
        uiRowSizeInBytes = 4*pgiImage->width;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "pgiWrite10BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "only RGB format supported" );
        return GAN_FALSE;
   }        

   /* build a buffer to hold a single line of image data and align it to a four byte boundary*/
   acBuffer = malloc(uiRowSizeInBytes+3);
   if(acBuffer == NULL)
   {
      gan_err_flush_trace();
      gan_err_register ( "pgiWrite10BitDPXImageData", GAN_ERROR_MALLOC_FAILED, "" );
      return GAN_FALSE;
   }

   acAlignedBuffer = (char*)((gan_uint32)acBuffer + 3 - (((gan_uint32)acBuffer + 3) % 4));

   /* write the image data */
   switch(pgiImage->format)
   {
      case GAN_RGB_COLOUR_IMAGE:
        for(uiRow = 0; uiRow<pgiImage->height; uiRow++)
        {
           Gan_RGBPixel_ui16* pui16rgbPix;
           gan_uint32* pui32Pix;

           memset((void*)acAlignedBuffer, 0, uiRowSizeInBytes);
           if(bReverseBytes)
           {
              for(iCol=(int)pgiImage->width-1, pui16rgbPix = gan_image_get_pixptr_rgb_ui16(pgiImage, uiRow, iCol),
                  pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol; iCol >= 0; iCol--, pui16rgbPix--, pui32Pix--)
              {
                 (*pui32Pix) |= ((pui16rgbPix->R >> 6) << 22);
                 (*pui32Pix) |= ((pui16rgbPix->G >> 6) << 12);
                 (*pui32Pix) |= ((pui16rgbPix->B >> 6) <<  2);
                 vReverseEndiannessUI32(pui32Pix);
              }
           }
           else
           {
              for(iCol=(int)pgiImage->width-1, pui16rgbPix = gan_image_get_pixptr_rgb_ui16(pgiImage, uiRow, iCol),
                  pui32Pix = ((gan_uint32*)acAlignedBuffer) + iCol; iCol >= 0; iCol--, pui16rgbPix--, pui32Pix--)
              {
                 (*pui32Pix) |= ((pui16rgbPix->R >> 6) << 22);
                 (*pui32Pix) |= ((pui16rgbPix->G >> 6) << 12);
                 (*pui32Pix) |= ((pui16rgbPix->B >> 6) <<  2);
              }
           }

           if ( fwrite ( acAlignedBuffer, 1, uiRowSizeInBytes, pfOutFile ) != uiRowSizeInBytes )
           {
              gan_err_flush_trace();
              gan_err_register ( "gan_read_dpx_image", GAN_ERROR_TRUNCATED_FILE, "truncated DPX file" );
              return GAN_FALSE;
           }
        }

        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "pgiWrite10BitDPXImageData", GAN_ERROR_NOT_IMPLEMENTED, "only RGB format supported" );
        return GAN_FALSE;
   }        

   free(acBuffer);
   return GAN_TRUE;
}

/**
 * \brief Writes a RGB colour image to a file stream in DPX format.
 * \param outfile The file stream
 * \param image The image structure to write to the file
 * \param controlstr Pointer to structure controlling DPX output or \c NULL
 * \return #GAN_TRUE on successful write operation, or #GAN_FALSE on failure.
 *
 * Writes the \a image into the file stream \a outfile in DPX format.
 *
 * \sa gan_read_dpx_image_stream().
 */
Gan_Bool
 gan_write_dpx_image_stream ( FILE *outfile, Gan_Image *image, Gan_DPXWriteControlStruct *controlstr )
{
   char acHeader[BIG_BUFFER_SIZE], *acAlignedHeader;
   gan_uint8 ui8BitSize;
   gan_uint32 ui32eolPadding = 0;

   /* determine bit size to use */
   if(controlstr != NULL && controlstr->bit_size != 0)
      ui8BitSize = (gan_uint8)controlstr->bit_size;
   else
      switch(image->type)
      {
         case GAN_BOOL:    ui8BitSize =  1; break;
         case GAN_UINT8:   ui8BitSize =  8; break;
         case GAN_UINT16:  ui8BitSize = 16; break;
         case GAN_FLOAT32: ui8BitSize = 32; break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_write_dpx_image", GAN_ERROR_NOT_IMPLEMENTED, "unsupported image type" );
           return GAN_FALSE;
           break;
      }

   /* align the header array */
   acAlignedHeader = (char*)((gan_uint32)acHeader + 7 - (((gan_uint32)acHeader + 7) % 8));

   /* build file header */
   memset((void*)acAlignedHeader, 0, 768);
   
   *((gan_uint32*)(acAlignedHeader + OFFSET_MAGIC)) = 0x53445058; /* Magic number */
   *((gan_uint32*)(acAlignedHeader + OFFSET_IMAGEOFFSET)) = IMAGE_DATA_OFFSET;
   strcpy((char *)(acAlignedHeader + OFFSET_VERSION), "V1.0");
   *((gan_uint32*)(acAlignedHeader + OFFSET_FILESIZE)) = ui32DPXFileSize(image->format, ui8BitSize,
                                                                         image->height, image->width);

   /* write file header */
   if(fwrite((const void *)acAlignedHeader, 1, 768, outfile) != 768)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_dpx_image", GAN_ERROR_WRITING_TO_FILE, "");
      return GAN_FALSE;
   }
   
   /* build the image information header */
   memset((void*)acAlignedHeader, 0, 640);

   *((gan_uint16*)(acAlignedHeader + OFFSET_ORIENTATION)) = 0;
   *((gan_uint16*)(acAlignedHeader + OFFSET_ELEMENTNUMBER)) = 1;
   *((gan_uint32*)(acAlignedHeader + OFFSET_PIXELSPERLINE)) = image->width;
   *((gan_uint32*)(acAlignedHeader + OFFSET_LINESPERIMAGEELE)) = image->height;
   *((gan_uint32*)(acAlignedHeader + OFFSET_DATASIGN0)) = 0; /* data sign, 0 = unsigned, the default */

   switch(image->format)
   {
      case GAN_GREY_LEVEL_IMAGE:
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESCRIPTOR0)) = 6; /* luminance; might use 4 for alpha instead */
        break;

      case GAN_RGB_COLOUR_IMAGE:
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESCRIPTOR0)) = 50; /* rgb */
        break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESCRIPTOR0)) = 51; /* rgba */
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_write_dpx_image", GAN_ERROR_NOT_IMPLEMENTED, "unsupported image format" );
        return GAN_FALSE;
   }

   if(controlstr == NULL || controlstr->transfer == GAN_DPXTRANSFER_DEFAULT)
      *((gan_uint8*)(acAlignedHeader + OFFSET_TRANSFER0)) = (gan_uint8)GAN_DPXTRANSFER_LINEAR;
   else
      *((gan_uint8*)(acAlignedHeader + OFFSET_TRANSFER0)) = (gan_uint8)controlstr->transfer;

   if(controlstr == NULL || controlstr->colorimetric == GAN_DPXCOLORIMETRIC_DEFAULT)
      *((gan_uint8*)(acAlignedHeader + OFFSET_COLORIMETRIC0)) = (gan_uint8)GAN_DPXCOLORIMETRIC_PAL;
   else
      *((gan_uint8*)(acAlignedHeader + OFFSET_COLORIMETRIC0)) = (gan_uint8)controlstr->colorimetric;

   *((gan_uint8*)(acAlignedHeader + OFFSET_BITSIZE0)) = ui8BitSize; /* bit size, e.g. 10-bit */
   *((gan_uint16*)(acAlignedHeader + OFFSET_PACKING0)) = 1; /* use unpacked format packing type */
   *((gan_uint16*)(acAlignedHeader + OFFSET_ENCODING0)) = 0; /* no encoding */
   *((gan_uint32*)(acAlignedHeader + OFFSET_DATAOFFSET0)) = IMAGE_DATA_OFFSET;   /* Determine data offset */
   *((gan_uint32*)(acAlignedHeader + OFFSET_EOLPADDING0)) = ui32eolPadding; /* end-of-line padding */
   *((gan_uint32*)(acAlignedHeader + OFFSET_EOIMAGEPADDING0)) = 0; /* no end-of-image padding */

   /* write image information header */
   if(fwrite((const void *)acAlignedHeader, 1, 640, outfile) != 640)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_dpx_image", GAN_ERROR_WRITING_TO_FILE, "");
      return GAN_FALSE;
   }

   /* write zeros for rest of header */
   {
      unsigned int uiNumBytesLeftToWrite = IMAGE_DATA_OFFSET - 768 - 640;

      memset((void*)acHeader, 0, BIG_BUFFER_SIZE);
      while(uiNumBytesLeftToWrite > 0)
      {
         unsigned int uiNumBytes = gan_min2(uiNumBytesLeftToWrite, BIG_BUFFER_SIZE);

         if(fwrite((const void*)acHeader, 1, uiNumBytes, outfile) != uiNumBytes)
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_write_dpx_image", GAN_ERROR_WRITING_TO_FILE, "");
            return GAN_FALSE;
         }

         uiNumBytesLeftToWrite -= uiNumBytes;
      }
   }

   /* now write the image data */
   switch(ui8BitSize)
   {
      case 10:
        if(!bWrite10BitDPXImageData(outfile, image, GAN_FALSE))
        {
           gan_err_register ( "gan_write_dpx_image", GAN_ERROR_FAILURE, "" );
           return GAN_FALSE;
        }

        break;

      case 8:
        if(!bWrite8BitDPXImageData(outfile, image, GAN_FALSE))
        {
           gan_err_register ( "gan_write_dpx_image", GAN_ERROR_FAILURE, "" );
           return GAN_FALSE;
        }

        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_write_dpx_image", GAN_ERROR_NOT_IMPLEMENTED, "unsupported DPX bit depth" );
        return GAN_FALSE;
   }

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes a RGB colour image file in DPX format.
 * \param filename The name of the image file
 * \param image The image structure to write to the file
 * \param controlstr Pointer to structure controlling DPX output or \c NULL
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the \a image into DPX file \a filename.
 *
 * \sa gan_read_dpx_image().
 */
Gan_Bool
 gan_write_dpx_image ( const char *filename, Gan_Image *image, Gan_DPXWriteControlStruct *controlstr )
{
   FILE *outfile;
   Gan_Bool result;

   /* attempt to open file */
   outfile = fopen ( filename, "wb" );
   if ( outfile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_dpx_image", GAN_ERROR_OPENING_FILE, filename );
      return GAN_FALSE;
   }

   result = gan_write_dpx_image_stream ( outfile, image, controlstr );
   fclose(outfile);
   return result;
}

/**
 * \}
 */

/**
 * \}
 */
