/***************************************************************************

  subr_file.c

  The file and input/output subroutines

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/


#include "gb_common.h"
#include "gb_common_buffer.h"
#include "gbx_subr.h"
#include "gb_file.h"
#include "gbx_stream.h"
#include "gbx_archive.h"
#include "gbx_api.h"
#include "gbx_local.h"

#include "gbx_string.h"
#include "gbx_c_file.h"


PRIVATE STREAM *get_stream(VALUE *value, boolean can_default)
{
  STREAM *stream;

  if (TYPE_is_integer(value->type) && can_default)
  {
    switch(value->_integer.value)
    {
      case 0:
        return CSTREAM_stream(&CFILE_in);
      case 1:
        return CSTREAM_stream(&CFILE_out);
      case 2:
        return CSTREAM_stream(&CFILE_err);
      default:
        THROW(E_ILLEGAL);
    }
  }

  if (VALUE_is_null(value))
    THROW(E_NULL);

  /*if (OBJECT_class(value->_object.object) != CLASS_Process)*/
  VALUE_conv(value, (TYPE)CLASS_Stream);

  stream = CSTREAM_stream(value->_object.object);

  if (stream->type == NULL)
    THROW(E_CLOSED);

  return stream;
}


PRIVATE char *get_path(VALUE *param)
{
  char *name;
  long len;

  SUBR_get_string_len(param, &name, &len);

  return STRING_conv_file_name(name, len);
}

PUBLIC void SUBR_open(void)
{
  CFILE *file;
  STREAM stream;
  int mode;

  SUBR_ENTER_PARAM(2);

  SUBR_check_integer(&PARAM[1]);
  mode = PARAM[1]._integer.value;

  STREAM_open(&stream, get_path(PARAM), mode);

  file = CFILE_create(&stream, mode);

  OBJECT_put(RETURN, file);

  SUBR_LEAVE();
}


PUBLIC void SUBR_close(void)
{
  SUBR_ENTER_PARAM(1);

  STREAM_close(get_stream(PARAM, FALSE));

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_flush(void)
{
  SUBR_ENTER_PARAM(1);

  STREAM_flush(get_stream(PARAM, TRUE));

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_print(void)
{
  int i;
  STREAM *stream;
  char *addr;
  long len;

  SUBR_ENTER();

  if (NPARAM < 1)
    THROW(E_NEPARAM);

  stream = get_stream(PARAM, TRUE);

  for (i = 1; i < NPARAM; i++)
  {
    PARAM++;
    VALUE_to_string(PARAM, &addr, &len);
    STREAM_write(stream, addr, len);
    RELEASE_STRING(PARAM);
  }

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_linput(void)
{
  STREAM *stream;
  char *addr;

  SUBR_ENTER_PARAM(1);

  stream = get_stream(PARAM, TRUE);

  STREAM_line_input(stream, &addr);

  RETURN->type = T_STRING;
  RETURN->_string.addr = addr;
  RETURN->_string.start = 0;
  RETURN->_string.len = STRING_length(addr);

  SUBR_LEAVE();
}


PUBLIC void SUBR_input(void)
{
  STREAM *stream;
  char *addr;

  SUBR_ENTER_PARAM(1);

  stream = get_stream(PARAM, TRUE);

  STREAM_input(stream, &addr);

  VALUE_from_string(RETURN, addr, STRING_length(addr));

  if (RETURN->type == T_NULL)
  {
    RETURN->type = T_STRING;
    RETURN->_string.addr = addr;
    RETURN->_string.start = 0;
    RETURN->_string.len = STRING_length(addr);
  }

  SUBR_LEAVE();
}


PUBLIC void SUBR_eof(void)
{
  STREAM *stream;

  SUBR_ENTER_PARAM(1);
  stream = get_stream(PARAM, FALSE);

  RETURN->type = T_BOOLEAN;
  RETURN->_boolean.value = STREAM_eof(stream) ? -1 : 0;

  SUBR_LEAVE();
}


PUBLIC void SUBR_lof(void)
{
  STREAM *stream;

  SUBR_ENTER_PARAM(1);
  stream = get_stream(PARAM, FALSE);

  RETURN->type = T_INTEGER;
  STREAM_lof(stream, &(RETURN->_integer.value));

  SUBR_LEAVE();
}


PUBLIC void SUBR_seek(void)
{
  STREAM *stream;
  long pos;
  long len;
  long whence = SEEK_SET;

  SUBR_ENTER();

  stream = get_stream(PARAM, FALSE);

  VALUE_conv(&PARAM[1], T_INTEGER);
  pos = PARAM[1]._integer.value;

  if (NPARAM == 3)
  {
    VALUE_conv(&PARAM[2], T_INTEGER);
    whence = PARAM[2]._integer.value;
    if (whence != SEEK_SET && whence != SEEK_CUR && whence != SEEK_END)
      THROW(E_ARG);
  }
  else
  {
    if (pos < 0)
    {
      STREAM_lof(stream, &len);
      pos += len;
    }
  }

  RETURN->type = T_INTEGER;
  RETURN->_integer.value = STREAM_tell(stream);

  STREAM_seek(stream, pos, (int)whence);

  SUBR_LEAVE();
}


PUBLIC void SUBR_read(void)
{
  STREAM *stream;
  long len;

  SUBR_ENTER();

  stream = get_stream(PARAM, TRUE);

  if (NPARAM == 3)
  {
    VALUE_conv(&PARAM[2], T_INTEGER);
    len = PARAM[2]._integer.value;
  }
  else
    len = 0;

  STREAM_read_type(stream, PARAM[1].type, RETURN, len);

  SUBR_LEAVE();
}


PUBLIC void SUBR_write(void)
{
  STREAM *stream;
  long len;

  SUBR_ENTER();

  stream = get_stream(PARAM, TRUE);

  if (NPARAM == 3)
  {
    VALUE_conv(&PARAM[2], T_INTEGER);
    len = PARAM[2]._integer.value;
  }
  else
    len = 0;

  STREAM_write_type(stream, PARAM[1].type, &PARAM[1], len);

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_stat(void)
{
  const char *path;
  CFILE *file;
  FILE_STAT info;

  SUBR_ENTER_PARAM(1);

  path = get_path(PARAM);

  FILE_stat(path, &info);
  
  file = CFILE_create(NULL, 0);
  file->info = info;

  OBJECT_put(RETURN, file);

  SUBR_LEAVE();
}


PUBLIC void SUBR_exist(void)
{
  bool exist;
  const char *path;

  SUBR_ENTER_PARAM(1);

  path = get_path(PARAM);

  exist = FILE_exist(path);

  RETURN->type = T_BOOLEAN;
  RETURN->_integer.value = exist ? -1 : 0;

  SUBR_LEAVE();
}


PUBLIC void SUBR_dir()
{
  GB_ARRAY array;
  const char *path;
  char *pattern;
  long len_pattern;
  char *str;

  SUBR_ENTER();

  path = get_path(PARAM);

  if (NPARAM >= 2)
    pattern = SUBR_get_string(&PARAM[1]);
  else
    pattern = NULL;

  FILE_dir_first(path, pattern);

  GB_ArrayNew(&array, T_STRING, 0);

  while (!FILE_dir_next(&pattern, &len_pattern))
  {
    if (!LOCAL_is_UTF8)
    {
      STRING_conv(&str, pattern, len_pattern, LOCAL_encoding, "UTF-8");
      STRING_ref(str);
    }
    else
      STRING_new(&str, pattern, len_pattern);

    *((char **)GB_ArrayAdd(array)) = str;
  }

  RETURN->_object.class = OBJECT_class(array);
  RETURN->_object.object = array;

  SUBR_LEAVE();
}


PUBLIC void SUBR_kill(void)
{
  SUBR_ENTER_PARAM(1);

  FILE_unlink(get_path(PARAM));

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_mkdir(void)
{
  SUBR_ENTER_PARAM(1);

  FILE_mkdir(get_path(PARAM));

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_rmdir(void)
{
  SUBR_ENTER_PARAM(1);

  FILE_rmdir(get_path(PARAM));

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_rename(void)
{
  SUBR_ENTER_PARAM(2);

  FILE_rename(get_path(&PARAM[0]), get_path(&PARAM[1]));

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_temp(void)
{
  char *temp;
  long len;

  temp = FILE_make_temp(&len);
  STRING_new_temp_value(RETURN, temp, len);

  SUBR_leave(0);
}


PUBLIC void SUBR_isdir(void)
{
  bool isdir;
  const char *path;

  SUBR_ENTER_PARAM(1);

  path = get_path(PARAM);

  isdir = FILE_isdir(path);

  RETURN->type = T_BOOLEAN;
  RETURN->_integer.value = isdir ? -1 : 0;

  SUBR_LEAVE();
}


PUBLIC void SUBR_copy(void)
{
  SUBR_ENTER_PARAM(2);

  FILE_copy(get_path(&PARAM[0]), get_path(&PARAM[1]));

  SUBR_LEAVE_VOID();
}


PUBLIC void SUBR_access(void)
{
  int access;

  SUBR_ENTER();

  if (NPARAM == 1)
    access = R_OK;
  else
  {
    VALUE_conv(&PARAM[1], T_INTEGER);
    access = PARAM[1]._integer.value;
  }

  RETURN->type = T_BOOLEAN;
  RETURN->_integer.value = FILE_access(get_path(PARAM), access) ? -1 : 0;

  SUBR_LEAVE();
}



PUBLIC void SUBR_link(void)
{
  SUBR_ENTER_PARAM(2);

  /* Parameters are inverted! */
  FILE_link(get_path(&PARAM[1]), get_path(&PARAM[0]));

  SUBR_LEAVE_VOID();
}


